/*
Copyright (C) 1991 Geoffrey M. Clemm

This file is part of the Odin system.

The Odin system is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public License
as published by the Free Software Foundation (see the file COPYING).

The Odin system is distributed WITHOUT ANY WARRANTY, without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

geoff@boulder.colorado.edu
*/

#include "inc/GMC.h"
#include "inc/FileName.h"
#include "inc/LogLevel_.h"
#include "inc/Status_.h"
#include "inc/Str.h"
#include "inc/Version.h"

boolean		History;
boolean		KeepGoing;
int		ErrLevel;
int		WarnLevel;
tp_LogLevel	LogLevel;
int		HelpLevel;
int		VerifyLevel;


void
Init_Vars(GMC_ARG_VOID)
{
   boolean Abort;
   tp_Str Val;

   Val = GetEnv("ODINKEEPGOING");
   if (Val == NIL) Val = "yes";
   SetVar(&Abort, "keepgoing", Val);
   Val = GetEnv("ODINERRLEVEL");
   if (Val == NIL) Val = "4";
   SetVar(&Abort, "errlevel", Val);
   Val = GetEnv("ODINWARNLEVEL");
   if (Val == NIL) Val = "2";
   SetVar(&Abort, "warnlevel", Val);
   Val = GetEnv("ODINLOGLEVEL");
   if (Val == NIL) Val = "2";
   SetVar(&Abort, "loglevel", Val);
   Val = GetEnv("ODINHELPLEVEL");
   if (Val == NIL) Val = "1";
   SetVar(&Abort, "helplevel", Val);
   Val = GetEnv("ODINVERIFYLEVEL");
   if (Val == NIL) Val = "2";
   SetVar(&Abort, "verifylevel", Val);

   Val = GetEnv("ODINHISTORY");
   if (Val == NIL) Val = "yes";
   SetVar(&Abort, "history", Val);

   Val = GetEnv("ODINBUILDHOSTS");
   if (Val == NIL) Val = "LOCAL:LOCAL";
   SetVar(&Abort, "buildhosts", Val);
   Val = GetEnv("ODINMAXBUILDS");
   if (Val == NIL) Val = "2";
   SetVar(&Abort, "maxbuilds", Val);
   }/*Init_Vars*/


void
Local_LogMessage(
   GMC_ARG(char*, Message)
   )
   GMC_DCL(char*, Message)
{
   Writeln(StdOutFD, Message);
   Flush(StdOutFD);
   }/*Local_LogMessage*/


void
Local_FileErrMessage(
   GMC_ARG(tp_FileName, FileName)
   )
   GMC_DCL(tp_FileName, FileName)
{
   tp_FilDsc FilDsc;
   tps_Str StrBuf;
   tp_Str Str;

   FilDsc = FileName_RFilDsc(FileName, FALSE);
   if (FilDsc == ERROR) {
      Write(StdOutFD, "** Could not read: ");
      Writeln(StdOutFD, FileName);
      return; }/*if*/;
   Unblock_Signals();
   for (Str=Readln(StrBuf, FilDsc); Str!=NIL; Str=Readln(StrBuf, FilDsc)) {
      Local_ErrMessage(Str);
      Local_ErrMessage("\n"); }/*for*/;
   Block_Signals();
   Close(FilDsc);
   }/*Local_FileErrMessage*/


void
ShowVars(GMC_ARG_VOID)
{
   Writeln(StdOutFD, "Dir MaxBuilds BuildHosts Size KeepGoing History");
   Writeln(StdOutFD, "LogLevel ErrLevel WarnLevel HelpLevel VerifyLevel");
   }/*ShowVars*/


void
HelpVar(
   GMC_ARG(tp_Nod, Nod)
   )
   GMC_DCL(tp_Nod, Nod)
{
   tp_Str VarStr;

   VarStr = Sym_Str(Nod_Sym(Nod_Son(1, Nod)));
   /*select*/{
      if (strcasecmp(VarStr, "dir") == 0) {
	 Writeln(StdOutFD, "The current directory.");
      }else if (strcasecmp(VarStr, "buildhosts") == 0) {
	 Writeln(StdOutFD, "Colon separated list of build hosts.");
      }else if (strcasecmp(VarStr, "clients") == 0) {
	 Writeln(StdOutFD,
	  "The current number of clients of the Odin server.");
      }else if (strcasecmp(VarStr, "errlevel") == 0
		|| strcasecmp(VarStr, "warnlevel") == 0) {
	 Writeln(StdOutFD, " 0: No report.");
	 Writeln(StdOutFD, " 1: Status report.");
	 Writeln(StdOutFD, " 2: Incremental and status report.");
	 Writeln(StdOutFD, " 3: Summary and status report.");
	 Writeln(StdOutFD, " 4: Summary, incremental, and status report.");
      }else if (strcasecmp(VarStr, "helplevel") == 0) {
	 Writeln(StdOutFD,
	  "1: Help returns information for common file types.");
	 Writeln(StdOutFD, "2: Help returns information for all file types.");
      }else if (strcasecmp(VarStr, "history") == 0) {
	 Writeln(StdOutFD, "yes: Provide history and command line editing.");
	 Writeln(StdOutFD, " no: Do not provide history or command line editing.");
      }else if (strcasecmp(VarStr, "keepgoing") == 0) {
	 Writeln(StdOutFD, "yes: Build as much as possible.");
	 Writeln(StdOutFD, " no: Stop on first build error.");
      }else if (strcasecmp(VarStr, "loglevel") == 0) {
	 Writeln(StdOutFD, "0: No log information is generated.");
	 Writeln(StdOutFD, "1: Build commands are echoed.");
	 Writeln(StdOutFD, "2: And Odin commands.");
	 Writeln(StdOutFD, "3: And names of objects with errors.");
	 Writeln(StdOutFD,
	  "4: And names of objects generated by tool scripts.");
	 Writeln(StdOutFD,
	  "5: And names of objects generated by internal tools.");
	 Writeln(StdOutFD, "6: And names of objects deleted.");
	 Writeln(StdOutFD, "7: And names of objects touched by broadcast.");
      }else if (strcasecmp(VarStr, "maxbuilds") == 0) {
	 Writeln(StdOutFD, "The maximum number of parallel builds.");
      }else if (strcasecmp(VarStr, "size") == 0) {
	 Writeln(StdOutFD, "The current amount disk space (kilobytes) used by derived objects.");
      }else if (strcasecmp(VarStr, "verifylevel") == 0) {
	 Writeln(StdOutFD, "0: No source file change verification.");
	 Writeln(StdOutFD, "1: Check for changes to source files at the beginning of each session.");
	 Writeln(StdOutFD, "2: Check for changes to source files before each interactive command.");
      }else if (strcasecmp(VarStr, "version") == 0) {
	 Writeln(StdOutFD, "The version of the Odin derived object manager.");
      }else{
	 SystemError("** Unknown variable name, <%s> **.\n", VarStr);
	 };}/*select*/;
   }/*HelpVar*/


void
ShowVar(
   GMC_ARG(tp_Nod, Nod)
   )
   GMC_DCL(tp_Nod, Nod)
{
   tp_Str VarStr;
   int Count, Size;

   VarStr = Sym_Str(Nod_Sym(Nod_Son(1, Nod)));
   /*select*/{
      if (strcasecmp(VarStr, "dir") == 0) {
	 Writeln(StdOutFD, CWDirName);
      }else if (strcasecmp(VarStr, "buildhosts") == 0) {
	 Write_BuildHosts(StdOutFD);
      }else if (strcasecmp(VarStr, "errlevel") == 0) {
	 WriteInt(StdOutFD, ErrLevel);
	 Writeln(StdOutFD, "");
      }else if (strcasecmp(VarStr, "warnlevel") == 0) {
	 WriteInt(StdOutFD, WarnLevel);
	 Writeln(StdOutFD, "");
      }else if (strcasecmp(VarStr, "helplevel") == 0) {
	 WriteInt(StdOutFD, HelpLevel);
	 Writeln(StdOutFD, "");
      }else if (strcasecmp(VarStr, "history") == 0) {
	 Writeln(StdOutFD, (History ? "yes" : "no"));
      }else if (strcasecmp(VarStr, "loglevel") == 0) {
	 WriteInt(StdOutFD, LogLevel);
	 Writeln(StdOutFD, "");
      }else if (strcasecmp(VarStr, "keepgoing") == 0) {
	 Writeln(StdOutFD, (KeepGoing ? "yes" : "no"));
      }else if (strcasecmp(VarStr, "maxbuilds") == 0) {
	 WriteInt(StdOutFD, MaxBuilds);
	 Writeln(StdOutFD, "");
      }else if (strcasecmp(VarStr, "size") == 0) {
	 Get_CurSize(&Size);
	 WriteInt(StdOutFD, Size/1000);
	 Writeln(StdOutFD, "");
      }else if (strcasecmp(VarStr, "clients") == 0) {
	 Get_UseCount(&Count);
	 WriteInt(StdOutFD, Count);
	 Writeln(StdOutFD, "");
      }else if (strcasecmp(VarStr, "verifylevel") == 0) {
	 WriteInt(StdOutFD, VerifyLevel);
	 Writeln(StdOutFD, "");
      }else if (strcasecmp(VarStr, "version") == 0) {
	 Writeln(StdOutFD, ODIN_VERSION_STR);
      }else{
	 SystemError("** Unknown variable name, <%s> **.\n", VarStr);
	 };}/*select*/;
   }/*ShowVar*/


void
SetVar(
   GMC_ARG(boolean*, AbortPtr),
   GMC_ARG(tp_Str, VarStr),
   GMC_ARG(tp_Str, ValStr)
   )
   GMC_DCL(boolean*, AbortPtr)
   GMC_DCL(tp_Str, VarStr)
   GMC_DCL(tp_Str, ValStr)
{
   int Level;
   tps_FileName DirBuf;
   tp_Status Status;
   boolean ExecFlag;

   *AbortPtr = FALSE;
   /*select*/{
      if (strcasecmp(VarStr, "dir") == 0) {
	 Get_OdinFile(DirBuf, &Status, &ExecFlag, ValStr, TRUE);
	 if (Status < STAT_OK) {
	    SystemError("Cannot access directory: <%s>.\n", ValStr);
	    *AbortPtr = TRUE;
	    return; }/*if*/;
	 if (strcmp(DirBuf, "") == 0) {
	    SystemError("No file value associated with: %s\n", ValStr);
	    *AbortPtr = TRUE;
	    return; }/*if*/;
	 ChangeDir(AbortPtr, DirBuf);
	 if (*AbortPtr) {
	    SystemError("Cannot access directory: <%s>.\n", DirBuf);
	    return; }/*if*/;
	 (void)strcpy(CWDirName, DirBuf);
	 Set_CWD(CWDirName);
      }else if (strcasecmp(VarStr, "buildhosts") == 0) {
	 Set_BuildHosts(AbortPtr, ValStr);
	 Extend_Builds(MaxBuilds);
      }else if (strcasecmp(VarStr, "errlevel") == 0) {
	 Level = Str_PosInt(ValStr);
	 if (Level < 0 || Level > 4) {
	    SystemError("** ErrLevel must be between 0 and 4.\n");
	    *AbortPtr = TRUE;
	    return; }/*if*/;
	 ErrLevel = Level;
	 Set_ErrLevel(Level);
      }else if (strcasecmp(VarStr, "warnlevel") == 0) {
	 Level = Str_PosInt(ValStr);
	 if (Level < 0 || Level > 4) {
	    SystemError("** WarnLevel must be between 0 and 4.\n");
	    *AbortPtr = TRUE;
	    return; }/*if*/;
	 WarnLevel = Level;
	 Set_WarnLevel(Level);
      }else if (strcasecmp(VarStr, "helplevel") == 0) {
	 Level = Str_PosInt(ValStr);
	 if (Level < 1 || Level > 2) {
	    SystemError("** HelpLevel must be between 1 and 2.\n");
	    *AbortPtr = TRUE;
	    return; }/*if*/;
	 HelpLevel = Level;
	 Set_HelpLevel(Level);
      }else if (strcasecmp(VarStr, "history") == 0) {
	 ;/*select*/{
	    if (strcasecmp(ValStr, "yes") == 0) {
	       History = TRUE;
	    }else if (strcasecmp(ValStr, "no") == 0) {
	       History = FALSE;
	    }else{
	       SystemError("** History must be \"yes\" or \"no\".\n");
	       *AbortPtr = TRUE;
	       return; };}/*select*/;
      }else if (strcasecmp(VarStr, "keepgoing") == 0) {
	 ;/*select*/{
	    if (strcasecmp(ValStr, "yes") == 0) {
	       KeepGoing = TRUE;
	    }else if (strcasecmp(ValStr, "no") == 0) {
	       KeepGoing = FALSE;
	    }else{
	       SystemError("** KeepGoing must be \"yes\" or \"no\".\n");
	       *AbortPtr = TRUE;
	       return; };}/*select*/;
	 Set_KeepGoing(KeepGoing);
      }else if (strcasecmp(VarStr, "loglevel") == 0) {
	 Level = Str_PosInt(ValStr);
	 if (Level < 0 || Level > LOGLEVEL_MAX) {
	    SystemError("** LogLevel must be between 0 and %d.\n",
			LOGLEVEL_MAX);
	    *AbortPtr = TRUE;
	    return; }/*if*/;
	 LogLevel = Level;
	 Set_LogLevel(Level);
	 Set_HostVar(AbortPtr,
		     "ODINVERBOSE", (Level < LOGLEVEL_Verbose ? "" : "1"));
      }else if (strcasecmp(VarStr, "maxbuilds") == 0) {
	 MaxBuilds = Str_PosInt(ValStr);
	 if (MaxBuilds < 1) {
	    Writeln(StdOutFD, "You won't get much work done that way!");
	    MaxBuilds = 1; }/*if*/;
	 Extend_Builds(MaxBuilds);
	 Set_MaxJobs(MaxBuilds);
      }else if (strcasecmp(VarStr, "size") == 0) {
	 SystemError("** Size is a read-only variable.\n");
	 *AbortPtr = TRUE;
      }else if (strcasecmp(VarStr, "verifylevel") == 0) {
	 Level = Str_PosInt(ValStr);
	 if (Level < 0 || Level > 2) {
	    SystemError("** VerifyLevel must be between 0 and 2.\n");
	    *AbortPtr = TRUE;
	    return; }/*if*/;
	 VerifyLevel = Level;
      }else if (strcasecmp(VarStr, "version") == 0) {
	 SystemError("** Version is a read-only variable.\n");
	 *AbortPtr = TRUE;

      }else if (strcasecmp(VarStr, "clients") == 0) {
	 SystemError("** Clients is a read-only variable.\n");
	 *AbortPtr = TRUE;
      }else if (strcmp(VarStr, "DEBUG") == 0) {
	 Set_Debug(ValStr);

      }else{
	 SystemError("** Unknown variable name, <%s> **.\n", VarStr);
	 *AbortPtr = TRUE; };}/*select*/;
   }/*SetVar*/


void
Set_HostVar(
   GMC_ARG(boolean*, AbortPtr),
   GMC_ARG(tp_Str, VarStr),
   GMC_ARG(tp_Str, ValStr)
   )
   GMC_DCL(boolean*, AbortPtr)
   GMC_DCL(tp_Str, VarStr)
   GMC_DCL(tp_Str, ValStr)
{
   tps_Str StrBuf;
   int status;

   if (IsDef_EnvVar(VarStr)) {
      SystemError("Cannot change value of package variable: %s.\n", VarStr);
      *AbortPtr = TRUE;
      return; }/*if*/;
   (void)sprintf(StrBuf, "%s=%s", VarStr, ValStr);
   status = putenv(Malloc_Str(StrBuf));
   FORBIDDEN(status != 0);
   RBS_VarDef(StrBuf);
   *AbortPtr = FALSE;
   }/*Set_HostVar*/



