/*
 * libmsn
 *
 * Copyright (C) 1999, Shane P. Brady <shane@jxie.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libmsn.h"
#include "parse_utils.h"
#include "msn_commands.h"

/*
 * String representations of commands codes
 */

char CommandString[NUM_OF_COMMANDS][COMMAND_LENGTH] = {
    "ACK", "ADD", "ANS", "BLP", "BYE", "CAL", "CHG", "FLN", "GTC", "INF",
    "ILN", "IRO", "JOI", "LST", "MSG", "NAK", "NLN", "OUT", "REM", "RNG",
    "SYN", "USR", "VER", "XFR"
};

/*
** State strings
*/

char StateStrings[NUM_OF_STATES][4] = {
    "NLN", "FLN", "HDN", "BSY", "IDL", "BRB", "AWY", "PHN", "LUN"
};

/*
** Verbose error handling
*/
char * ErrorMessages[NUM_OF_ERRS];

char FallbackErrorMessage[]="Unknown error";

/*
** Name:    msn_callback_handler
** Purpose: Passes on the call back information to the parsing function
** Input:   data - data passed to handler
**          source - source of event
**          condition - conditions event occurred
** Output:  none
*/

void msn_callback_handler(gpointer data, gint source,
                          GdkInputCondition condition )
{
    ParseForCommand(data);
}

/*
** Main MSN Connection
*/

extern MSN_Conn mainMSNConnection;

/*
** Name:    MSN_RegisterErrorOutput
** Purpose: This function sets the error output function
** Input:   func - function pointer
** Output:  none
*/
 
void MSN_RegisterErrorOutput(ERR_CALLBACK func)
{
    MSN_ErrorOut = func;
}
 
/*
** Name:    VerboseErrorOutput
** Purpose: This function displays a message and interprets the numerical
**          error code that it is passed
** Input:   message - a string showing where it went wrong
**          title - the window title
** Output:  none
*/

void VerboseErrorOutput(char * message, char * title, int errorcode)
{
    char * buf;

    if(errorcode<0 || errorcode>=1000)
    { MSN_ErrorOut(message, title); return; }

    buf = (char *) malloc(
	(strlen(message) + strlen(ErrorMessages[errorcode]) + 8) * sizeof(char)
	);
    strcpy(buf, "MSN: ");
    strcat(buf, message);
    strcat(buf, ": ");
    strcat(buf, ErrorMessages[errorcode]);

    MSN_ErrorOut(buf, title);

    free(buf);
}

/*
** Name:    MSN_RegisterCallback
** Purpose: This function regsiters the call backs for the proper event
** Input:   eventType  - type of server event
**          func       - function pointer
** Output:  none
*/

void MSN_RegisterCallback(int eventType, MSN_CALLBACK func)
{
    msn_event[eventType] = func;
}

/*
** Name:    InitializeMSNConnection
** Purpose: This function initializes a MSN_Conn structure to default values
** Input:   conn - msn connection structure
** Output:  0 on success, -1 on failure
*/

int InitializeMSNConnection(MSN_Conn *conn)
{
    int a;

    for(a=0; a<NUM_OF_ERRS; a++)
    {
      ErrorMessages[a]=FallbackErrorMessage;
    }

    InitialiseErrorMessages();

    conn->serverType = DISPATCH_CONN;
    conn->chatUsers.users = NULL;
    conn->chatUsers.numOfUsers = 0;
    conn->flUsers.users = NULL;
    conn->flUsers.numOfUsers = 0;
    conn->alUsers.users = NULL;
    conn->alUsers.numOfUsers = 0;
    conn->blUsers.users = NULL;
    conn->blUsers.numOfUsers = 0;
    conn->rlUsers.users = NULL;
    conn->rlUsers.numOfUsers = 0;
    conn->cookie[0] = '\0';
    conn->commonName[0] = '\0';
    conn->listenerID = -1;
    conn->unreadMail = 0;
    return 0;
}

/*
** Name:    InitializeErrorMessages
** Purpose: This function initializes the char * array that holds the 
**          verbose error messages.
** Input:   none
** Output:  none
*/

void InitialiseErrorMessages(void)
{
    static char SyntaxError[]="Protocol syntax error";
    static char InvalidParameter[]="Invalid parameter";
    static char InvalidUser[]="Invalid user";
    static char FQDNMissing[]="FQDN missing";
    static char AlreadyLoggedIn[]="Already logged in";
    static char InvalidUsername[]="Invalid username";
    static char InvalidFriendlyName[]="Invalid user-friendly name";
    static char ListFull[]="List full";
    static char AlreadyThere[]="Contact already on this list or in this session";
    static char NotOnList[]="Not on list";
    static char AlreadyInMode[]="Already in this mode";
    static char AlreadyInOppositeList[]="Contact already in the opposite list";
    static char SBFailed[]="Switchboard failed";
    static char NotifyXFRFailed[]="Transfer notify failed";
    static char RequiredFieldsMissing[]="Required fields missing";
    static char NotLoggedIn[]="Not logged in";
    static char InternalServerError[]="Internal server error";
    static char DBServerError[]="Database server error";
    static char FileOperationFailed[]="File operation failed";
    static char MemAllocFailed[]="Memory allocation failed";
    static char ServerBusy[]="Server is too busy";
    static char ServerUnavailable[]="Server is unavailable";
    static char PeerNSDown[]="Peer Notification Server is down";
    static char DBConnectFailed[]="Database connect failed";
    static char ServerGoingDown[]="Server going down for maintainance";
    static char CreateConnection[]="Create connection";
    static char BlockingWrite[]="Blocking write";
    static char SessionOverload[]="Session overload";
    static char UserTooActive[]="User too active - slow down!";
    static char TooManySessions[]="Too many sessions";
    static char NotExpected[]="Command not expected";
    static char BadFriendFile[]="Bad friend file";
    static char AuthenticationFailed[]="Bad username and/or password";
    static char NotAllowedWhenOffline[]="This action is not allowed while offline";
    static char NotAcceptingNewUsers[]="Server is not accepting new users";

    ErrorMessages[200]=SyntaxError;
    ErrorMessages[201]=InvalidParameter;
    ErrorMessages[205]=InvalidUser;
    ErrorMessages[206]=FQDNMissing;
    ErrorMessages[207]=AlreadyLoggedIn;
    ErrorMessages[208]=InvalidUsername;
    ErrorMessages[209]=InvalidFriendlyName;
    ErrorMessages[210]=ListFull;
    ErrorMessages[215]=AlreadyThere;
    ErrorMessages[216]=NotOnList;
    ErrorMessages[218]=AlreadyInMode;
    ErrorMessages[219]=AlreadyInOppositeList;
    ErrorMessages[280]=SBFailed;
    ErrorMessages[281]=NotifyXFRFailed;

    ErrorMessages[300]=RequiredFieldsMissing;
    ErrorMessages[302]=NotLoggedIn;

    ErrorMessages[500]=InternalServerError;
    ErrorMessages[501]=DBServerError;
    ErrorMessages[510]=FileOperationFailed;
    ErrorMessages[520]=MemAllocFailed;

    ErrorMessages[600]=ServerBusy;
    ErrorMessages[601]=ServerUnavailable;
    ErrorMessages[602]=PeerNSDown;
    ErrorMessages[603]=DBConnectFailed;
    ErrorMessages[604]=ServerGoingDown;

    ErrorMessages[707]=CreateConnection;
    ErrorMessages[711]=BlockingWrite;
    ErrorMessages[712]=SessionOverload;
    ErrorMessages[713]=UserTooActive;
    ErrorMessages[714]=TooManySessions;
    ErrorMessages[715]=NotExpected;
    ErrorMessages[717]=BadFriendFile;

    ErrorMessages[911]=AuthenticationFailed;
    ErrorMessages[913]=NotAllowedWhenOffline;
    ErrorMessages[920]=NotAcceptingNewUsers;
}


/*
** Name:    MSN_Login
** Purpose: This function encapsulates the login process to MSN
** Input:   handle   - user's handle
**          passwd   - user's password
**          host     - notification server to use
**          port     - port number of notifcation server
** Output:  0 on success, -1 on failure
*/

int MSN_Login(char *handle, char *passwd, char *host, int port)
{
    int  lID;
    char sp[10];
    int retry = 0;
    int ecode;

    InitializeMSNConnection(&mainMSNConnection);

    strcpy(mainMSNConnection.passwd, passwd);
    strcpy(mainMSNConnection.handle, handle);
    
LOGIN_RETRY:
    retry++;

    if ((ecode=ConnectToServer(&mainMSNConnection, host, port)) != 0) {
        VerboseErrorOutput("Couldn't connect to server", "Server Error", ecode);
        return -1;
    }
    if ((ecode=SetProtocol(&mainMSNConnection, DEFAULT_PROTOCOL)) != 0) {
	if(retry <= 20)
	{
		goto LOGIN_RETRY;
	}
	else
	{
        	VerboseErrorOutput("Couldn't set protocol", "Protocol Error", ecode);
		return -1;
	}
	
    }
    if ((ecode=GetServerPolicyInfo(&mainMSNConnection, sp)) != 0) {
        VerboseErrorOutput("Couldn't retrieve security policy", "Policy Error", ecode);
        return -1;
    }

    if ((ecode=AuthenticateUserMD5(&mainMSNConnection, handle, passwd)) != 0) {
        VerboseErrorOutput("Unable to authenticate user", "Authentication Error", ecode);
        return -1;
    }
   
    Synchronize(&mainMSNConnection);
    ChangeState(&mainMSNConnection, "NLN");

    lID = gdk_input_add(mainMSNConnection.fd, GDK_INPUT_READ,
                        msn_callback_handler, &mainMSNConnection);
    mainMSNConnection.listenerID = lID;
    return 0;
}

/*
** Name:    MSN_SendMessage
** Purpose: This function encapuslates the sending of an instant message
** Input:   handle - user's handle who is receiving the message
**          message - the actual message to send
** Output:  0 on success, -1 on failure
*/

int MSN_SendMessage(char *handle, char *message)
{
    char     *newHandle;
    MSN_Conn *conn;
    int      ecode;

    AddHotmail(handle, &newHandle);
    conn = FindMSNConnectionByHandle(newHandle);
    if (conn == NULL) {
    /* if a connection isn't found try to establish a new connection */
        if ((ecode=RequestSwitchboardSession(&mainMSNConnection, newHandle))!=0) {
            VerboseErrorOutput("Switchboard error", "Could not initiate a switchboard session with this user", ecode);
            free(newHandle);
            return -1;
        }
        conn = FindMSNConnectionByHandle(newHandle);
        if (conn == NULL) {
            free(newHandle);
            return -1;
        }
    } 
    SendMessage(conn, message);
    free(newHandle);
    return 0;

}

/*
** Name:    MSN_AddContact
** Purpose: This function sends a add to forward contact list message to the
**          server
** Input:   handle  - handle to add to the list
** Output:  0 on success, -1 on failure
*/

int MSN_AddContact(char *handle)
{
    if (handle == NULL)
        return -1;

    AddContact(&mainMSNConnection, handle);
    return 0;    
}

/*
** Name:    MSN_RemoveContact
** Purpose: This function sends a remove to forward contact list message to
**          the server
** Input:   handle - handle to remove from the list
** Output:  0 on success, -1 on failure
*/

int MSN_RemoveContact(char *handle)
{
    if (handle == NULL)
        return -1;

    RemoveContact(&mainMSNConnection, handle);
    return 0;
}

/*
** Name:    MSN_AuthorizeContact
** Purpose: This function sends an authorize message to the server
** Input:   conn   - MSN connection structure
**          handle - handle to authorize
** Output:  0 on success, -1 on failure
*/

int MSN_AuthorizeContact(MSN_Conn *conn, char *handle)
{
    return AuthorizeContact(conn, handle);
}

/*
** Name:    MSN_EndChat
** Purpose: This function sends an OUT mesage to the server to end a 
**          chat with a user
** Input:   handle - handle to end chat with 
** Output:  0 on success, -1 on failure
*/

int MSN_EndChat(char *handle)
{
    MSN_Conn *conn;

    conn = FindMSNConnectionByHandle(handle);
    if (conn == NULL)
        return -1;
    return SendSBYE(conn);
}

/*
** Name:    MSN_Logout
** Purpose: This function properly logs out of the MSN service
** Input:   none
** Output:  0 on success, -1 on failure
*/

int MSN_Logout(void)
{
    return SendBYE(&mainMSNConnection);
}

/*
** Name:    MSN_ChangeState
** Purpose: This function changes the current state of the user
** Input:   state - integer representation of the state
** Output:  0 on success, -1 on failure
*/

int MSN_ChangeState(int state)
{
    if (state > (NUM_OF_STATES-1))
        return -1;

    return ChangeState(&mainMSNConnection, StateStrings[state]);
}

