/*
    ext2_buffer.c -- ext2 buffer cache
    Copyright (C) 1998, 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_buffer_c[] = "$Id: ext2_buffer.c,v 1.12 2000/10/16 17:10:35 adilger Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ext2.h"

/* pseudo-header */

static __inline__ int ext2_block_hash(blk_t block)
{
	unsigned long x;

	x = block ^ (block >> 8) ^ (block >> 16) ^ (block >> 24);
	return x & ((1 << ext2_hash_bits) - 1);
}

static struct ext2_buffer_head *ext2_bh_alloc   (struct ext2_buffer_cache *, blk_t);
static void			ext2_bh_dealloc (struct ext2_buffer_head *);
static struct ext2_buffer_head *ext2_bh_find    (struct ext2_buffer_cache *, blk_t);
static void			ext2_bh_do_read (struct ext2_buffer_head *);
static void			ext2_bh_do_write(struct ext2_buffer_head *);
static void			ext2_bh_hash    (struct ext2_buffer_head *);
static void			ext2_bh_unhash  (struct ext2_buffer_head *);



static int try_to_flush(struct ext2_buffer_cache *bc)
{
	struct ext2_buffer_head *bh;
	int i;

	for (i = 0, bh = bc->heads; i < bc->size; i++, bh++) {
		if (bh->alloc && !bh->usecount && !bh->dirty) {
			ext2_bh_dealloc(bh);
			return 1;
		}
	}

	for (i = 0, bh = bc->heads; i < bc->size; i++, bh++) {
		if (bh->alloc && !bh->usecount && bh->dirty) {
			ext2_bh_do_write(bh);
			ext2_bh_dealloc(bh);

			return 1;
		}
	}

	fprintf(stderr, "ext2resize: couldn't flush!\n");
	return 0;
}





static struct ext2_buffer_head *ext2_bh_alloc(struct ext2_buffer_cache *bc,
					      blk_t block)
{
	struct ext2_buffer_head *bh;
	int i;

tryagain:
	for (i = 0, bh = bc->heads; i < bc->size; i++, bh++) {
		if (!bh->alloc)
			break;
	}

	if (i == bc->size) {
		if (!try_to_flush(bc))
			return NULL;
		goto tryagain;
	}

	bh->next = NULL;
	bh->prev = NULL;
	bh->block = block;
	bh->usecount = 0;
	bh->dirty = 0;
	bh->alloc = 1;
	bc->numalloc++;

	ext2_bh_hash(bh);

	return bh;
}

static void ext2_bh_dealloc(struct ext2_buffer_head *bh)
{
	if (bh->dirty)
		fprintf(stderr, "deallocing a dirty buffer! %i\n", bh->block);

	ext2_bh_unhash(bh);
	bh->alloc = 0;
	bh->bc->numalloc--;
}

static struct ext2_buffer_head *ext2_bh_find(struct ext2_buffer_cache *bc, blk_t block)
{
	struct ext2_buffer_head *a;
	struct ext2_buffer_head *b;
	int			 hash;

	hash = ext2_block_hash(block);
	a = bc->hash[hash];

	if (a != NULL) {
		b = a;
		do {
			if (a->block == block)
				return a;

			a = a->next;
		} while (a != b);
	}

	return NULL;
}

static void ext2_bh_do_read(struct ext2_buffer_head *bh)
{
	ext2_read_blocks(bh->bc->fs, bh->data, bh->block, 1);
}

static void ext2_bh_do_write(struct ext2_buffer_head *bh)
{
	ext2_write_blocks(bh->bc->fs, bh->data, bh->block, 1);
	bh->dirty = 0;
}

static void ext2_bh_hash(struct ext2_buffer_head *bh)
{
	int hash;

	hash = ext2_block_hash(bh->block);
	if (bh->bc->hash[hash] != NULL) {
		bh->next = bh->bc->hash[hash];
		bh->prev = bh->next->prev;
		bh->next->prev = bh;
		bh->prev->next = bh;
		return;
	}

	bh->bc->hash[hash] = bh;
	bh->next = bh->prev = bh;
}

static void ext2_bh_unhash(struct ext2_buffer_head *bh)
{
	int hash;

	hash = ext2_block_hash(bh->block);

	bh->prev->next = bh->next;
	bh->next->prev = bh->prev;

	if (bh->bc->hash[hash] == bh) {
		if (bh->next != bh)
			bh->bc->hash[hash] = bh->next;
		else
			bh->bc->hash[hash] = NULL;
	}

	bh->next = NULL;
	bh->prev = NULL;
}







static int breadimmhits = 0;
static int breadindhits = 0;
static int breadmisses = 0;

void ext2_bcache_deinit(struct ext2_fs *fs)
{
	ext2_bcache_sync(fs);
	free(fs->bc->buffermem);
	free(fs->bc->hash);
	free(fs->bc->heads);
	free(fs->bc);

	if (fs->flags & FL_VERBOSE)
		printf("cache direct hits: %i, indirect hits: %i, misses: %i\n",
		       breadimmhits, breadindhits, breadmisses);
}

void ext2_bcache_dump(struct ext2_fs *fs)
{
	int i;

	if (fs->flags & FL_DEBUG)
		printf(__FUNCTION__ "\n");

	for (i = 0; i < (1 << ext2_hash_bits); i++)
		if (fs->bc->hash[i] != NULL) {
			struct ext2_buffer_head *a;
			struct ext2_buffer_head *b;

			printf("%i: ", i);

			a = b = fs->bc->hash[i];
			do {
				printf("%i ", a->block);
				a = a->next;
			} while (a != b);

			printf("\n");
		}
}

void ext2_bcache_flush(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;

	if ((bh = ext2_bh_find(fs->bc, block)) == NULL)
		return;

	if (fs->flags & FL_DEBUG && bh->usecount)
		fprintf(stderr, "flushing buffer that's in use! [%i,%i]\n",
			bh->block, bh->usecount);

	if (bh->dirty)
		ext2_bh_do_write(bh);

	ext2_bh_dealloc(bh);
}

void ext2_bcache_flush_range(struct ext2_fs *fs, blk_t block, blk_t num)
{
	blk_t end = block + num;

	for (; block < end; block++)
		ext2_bcache_flush(fs, block);
}

int ext2_bcache_init(struct ext2_fs *fs)
{
	struct ext2_buffer_cache *bc;
	int i;
	int size;

	if (fs->flags & FL_DEBUG)
		printf(__FUNCTION__ "\n");

	size = ext2_buffer_cache_pool_size >> (fs->logsize - 10);

	bc = (struct ext2_buffer_cache *)
		malloc(sizeof(struct ext2_buffer_cache));
	if (bc == NULL) {
		fprintf(stderr, "%s: error allocating buffer cache\n",fs->prog);
		goto error;
	}

	bc->heads = (struct ext2_buffer_head *)
		malloc(size * sizeof(struct ext2_buffer_head));
	if (bc->heads == NULL) {
		fprintf(stderr, "%s: error allocating buffer heads\n",fs->prog);
		goto error_bc;
	}

	bc->hash = (struct ext2_buffer_head **)
		malloc(sizeof(struct ext2_buffer_head *) << ext2_hash_bits);
	if (bc->hash == NULL) {
		fprintf(stderr, "%s: error allocating buffer hash\n", fs->prog);
		goto error_heads;
	}

	bc->buffermem = (unsigned char *)
		malloc(ext2_buffer_cache_pool_size << 10);
	if (bc->buffermem == NULL) {
		fprintf(stderr, "%s: error allocating cache pool\n", fs->prog);
		goto error_hash;
	}

	bc->cache = &bc->heads[0];
	bc->fs = fs;
	bc->size = size;
	bc->numalloc = 0;

	for (i = 0; i < size; i++) {
		bc->heads[i].data = bc->buffermem + (i << fs->logsize);
		bc->heads[i].bc = bc;
		bc->heads[i].alloc = 0;
	}

	for (i = 0; i < (1 << ext2_hash_bits); i++)
		bc->hash[i] = NULL;

	fs->bc = bc;

	return 1;

error_hash:
	free(bc->hash);
error_heads:
	free(bc->heads);
error_bc:
	free(bc);
error:
	return 0;
}

void ext2_bcache_sync(struct ext2_fs *fs)
{
	int i;
	struct ext2_buffer_head *bh;

	for (i = 0, bh = fs->bc->heads; i < fs->bc->size; i++, bh++) {
		if (bh->dirty) {
			if (fs->flags & FL_VERBOSE)
				printf("   ...flushing buffer %d/block %d\r",
				       i, bh->block);
			ext2_bh_do_write(bh);
		}
	}
	if (fs->flags & FL_VERBOSE)
		printf("\n");
}

struct ext2_buffer_head *ext2_bcreate(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;

	if ((bh = ext2_bh_find(fs->bc, block)) != NULL) {
		bh->usecount++;
	} else {
		bh = ext2_bh_alloc(fs->bc, block);
		bh->usecount = 1;
	}

	memset(bh->data, 0, fs->blocksize);
	bh->dirty = 1;

	return bh;
}

struct ext2_buffer_head *ext2_bread(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;

	if ((bh = fs->bc->cache)->block == block) {
		breadimmhits++;
		bh->usecount++;
		return bh;
	}

	if ((bh = ext2_bh_find(fs->bc, block)) != NULL) {
		fs->bc->cache = bh;
		breadindhits++;
		bh->usecount++;
		return bh;
	}

	breadmisses++;

	bh = ext2_bh_alloc(fs->bc, block);
	if (!bh)
		return NULL;
	ext2_bh_do_read(bh);
	fs->bc->cache = bh;
	bh->usecount = 1;

	return bh;
}

void ext2_brelse(struct ext2_buffer_head *bh, int forget)
{
	if (bh->usecount-- == 1 && forget) {
		if (bh->dirty)
			ext2_bh_do_write(bh);

		ext2_bh_dealloc(bh);
	}
}
