// dynbuff.cpp
/*
 * (C) 1998-2000 Murat Deligonul
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 * ---
 * dynbuff class: basically an array of char on steroids
 * ---
 * FIXME/TODO:              
 *        Should we use unsigned char here instead?
 * ---
 * 04-03-99: write() may not write everything we wanted
 * 03-21-99: last-char null thingy really fixed
 * 01-10-99: return failed read if eof in add()
 * 01-9-99:  due to c++ or egcs sillyness, static constants don't work
 *           well in switch statements. Basically they need to be defined
 *           in that particular file. Its stupid to define them in conn.cpp,
 *           and now I need them in dcc.cpp. So out they go, and in come some
 *           enums which will hopefully work ;P  
 * 10-7-98:  switched to four-space tabbing :)
 * 7-19-98:  made it so that one extra byte is allocated for contents
 *            and that it is set to 0. Several functions treated
 *            contents as a string which was unsafe as there was no
 *            gurantee that it would end w/ null character. This was 
 *            probably a better way to do this than have the functions
 *            make a copy of contents and make it a real string.
 */


    
#include "autoconf.h"

#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#ifdef HAVE_SYS_FILIO_H
#   include <sys/filio.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
#   include <sys/ioctl.h>
#endif
#include <string.h>
#include "dynbuff.h"


/*  
 *  Check if sizes are valid. allocate something of 
 *  minsize size.
 *  maxsize can be 0 for unlimited length.
 *  If maxsize is smaller than min, we assume unlimited length as well
 */ 
dynbuff::dynbuff(u_long minsize, u_long maxsize) 
{
    if (maxsize < minsize)
        maxsize = 0;
    if (minsize == 0)
        minsize = 1024;
    min = minsize;
    max = (maxsize == 0) ? 0xFFFFFFF : maxsize;    // ok so not really unlimited
    /* add a 0 to the end */
    contents = new char[minsize + 1];
    alloced = minsize;
    contents[alloced] = 0;
    bytes_in = 0;
}

dynbuff::~dynbuff()
{
    delete[] contents;
}


long dynbuff::add(const char *data, u_long len)
{
    if (len == 0)
        len = strlen(data);
    if (len == 0)
        return 0;
    return append(data, len);
}

/*
 * Note: for nonblocking sockets, if we get EAGAIN,
 * we return 0 (nothing to add)
 */
long dynbuff::add(int fd, u_long len)
{
    if (len == 0)
    {
        ioctl(fd, FIONREAD, &len);
        if (len == 0)
            return 0;
    }

    char *tmp = new char[len];
    if (!tmp)
        return DYNBUFF_ERR_MEM;
    switch ((signed long) (len = read(fd, tmp, len)))
    {
    case -1:
        delete[] tmp;
        return (errno == EAGAIN) ? 0 : DYNBUFF_ERR_READ;
    case 0:
        delete[] tmp;
        return DYNBUFF_ERR_READ;
    }

    fd = append(tmp, len);
    delete[] tmp;
    return long(fd);
}


/*
 *  Right now these remove 'buffsize' amount from the start of the 
 *  data. You can't specify where to start. I might code it if i
 *  find a need for it.
 *  
 *  Return value for peek: Amount of stuff copied
 *  Return value for get : New size of internal buffer
 */
long dynbuff::peek(char *buffer, u_long buffsize)
{
    /* check if requested amount is reasonable */
    u_long bytes2copy = (buffsize > bytes_in) ? 
                        bytes_in : buffsize;
    memcpy(buffer, contents, bytes2copy);
    return bytes2copy;
}

long dynbuff::get(char *buffer, u_long buffsize)
{
    peek(buffer, buffsize);
    return remove(0, buffsize);
}

/* 
 * Returns # found.
 * FIXME: not sure if it should be went < bytes_in ..
 */
u_long dynbuff::have_char(char x) const
{
    int num = 0; 
    unsigned went;
    char *f;

    for (f = contents, went = 0; went < bytes_in; went++, f++)
        if (*f == x) num++;
    return num;
}


/* 
 *  Obtains a line from this buffer.
 *  Arguments:
 *     line        - 0 based index
 *     buffer      - where to store line
 *     maxsize     - how much to store at most
 *     leave       - leave the line in the internal buffer or remove it? 
 *                    (optional, defaults to 0)
 *   must_have_crlf - line must end w/ cr, lf, or both.
 *   
 *   Entire line will be removed if leave is 0, even if the length of the 
 *      whole line is less than maxsize.
 *
 *  Return values: 0+   how long the string is in buffer. Can be 0, 
 *                       if line was empty, as the ending \n or \r is not copied.
 *                -1:   line was not found
 *
 * FIXME: we check for 'out of bounds': this is wrong. What is going on?
 *        if we can peek out of bounds it's a segfault waiting to happen.
 *        properly it's all good, but the check there is simply one 2 years old,
 *        from the days of classic ezbounce where murat forgot to terminate
 *        his buffer with a null char and tried to run string functions on it.
 */ 
long dynbuff::get_line(unsigned line, char *buffer, u_long maxsize, bool leave, 
                       bool need_crlf)
{
    const char *p, *cur = contents;
    unsigned curline = 0;
    
    /* Keep finding \n's until we find the line we need */
    while ((p = strchr(cur, '\n')) && (curline != line) 
           && ((u_long) (p - contents) < bytes_in)) {
        cur = p + 1;                       
        curline++;
    }

    if (curline == line && bytes_in)
    {
        const char *next = strchr(cur, '\n');
        if (!next && need_crlf)
            return -1;           /* incomplete line, ignore it */
        if (!next || 
            ((u_long) (next - contents) > bytes_in))    /*  any more \n's ?  */
            next = contents + bytes_in;                 /* or is it out of bounds? */
        else
            next++;                                     /* include this \n so it will be removed */
        u_long len = next - cur;
        u_long real_len = len;

        if (len >= maxsize)
            len = maxsize - 1;
        memcpy(buffer, cur, len);
        buffer[len] = 0;
        /* get rid of any lf or cr at the end */
        if ((buffer[len - 1] == '\r') || (buffer[len - 1] == '\n')) 
            buffer[--len] = 0;
        if (len && buffer[len - 1] == '\r') 
            buffer[--len] = 0;

        /* remove this line if requested */
        if (!leave)
            remove(u_long(cur - contents), real_len);
        return len;
    }

    return -1;
}

/* 
 *  adds data to buffer. resizes if needed
 *  if data is big enough and buffer can be enlarged, it is done.
 *  
 *  FIXME: bettererror checking 
 *  FIXME: seems to truncate?? 
 *  return value is new size of buffer. 0 on error
 * 
 */
long dynbuff::append(const char *data, u_long len)
{
    if (bytes_in == max)
        return DYNBUFF_ERR_FULL;
    if (len == 0)
        return 0;

    u_long newsize = (len + bytes_in) > max ? max : len + bytes_in;
    u_long bytes2copy = newsize - bytes_in;

    if (newsize > alloced)
    {
        /* 
         * The new size will be too big for this buffer.
         *  we need to allocate bigger one.
         */
        char *newbuff;
        newbuff = new char[newsize + 1];
        memcpy(newbuff, contents, bytes_in);
        memcpy(newbuff + bytes_in, data, bytes2copy);

        /* clean up */
        delete[] contents;
        contents = newbuff;
        contents[newsize] = 0;  /* add a null char */
        alloced = newsize;
        bytes_in += bytes2copy;
    }
    else if (newsize <= alloced)
    {
        memcpy(contents + bytes_in, data, bytes2copy);
        bytes_in += bytes2copy;
    }
    contents[bytes_in] = 0;
    return bytes_in;
}

/* 
 * Attempt to dump all contents to a file descriptor.
 * Return new size of the buffer
 */
long dynbuff::flush(int fd, unsigned int bytes)
{
    if (!bytes || bytes > bytes_in)
        bytes = bytes_in;

    int wr = write(fd, contents, bytes);
    if (wr < 0)
        return DYNBUFF_ERR_WRITE;
    /* We must NOT assume all of the contents were written 
     * (especially for sockets)!!!! */
    return remove(0, wr);
}   

bool dynbuff::flush(void)
{
    bytes_in = 0;
    return 1;
}

bool dynbuff::reset(void)
{
    return flush();
}

/* 
 *  Starts at 'start' and removes 'howmuch' stuff 
 *  FIXME: what if its > max ?? 
 *  
 *  Returns: new size of buffer
 *  0-based index
 */
long dynbuff::remove(u_long start, u_long howmuch)
{
    u_long end = start + howmuch;
    u_long part2 = (end > bytes_in) ? end : bytes_in - end; 

    /* start at 'end'. move it to 'start' */
    memmove(contents + start, contents + end, part2);
    /* re-adjust */
    bytes_in = start + part2;
    contents[alloced] = 0;
    contents[bytes_in] = 0;
    return bytes_in;
}

/* 
 *  Finds first 'x' in the internal buffer.
 *  Returns its position relative to the begining of the buffer.
 *  -1 is nothing is found.
 */
long dynbuff::get_pos(char x) const
{
    char *where = strchr(contents, x);
    if (!where || unsigned(where - contents) > bytes_in)
        return -1;
    return (long) (where - contents + 1);
}

/*
 *  Optimize memory usage, i.e:
 *  if allocated amount is > minimum and bytes_in is also 
 *  < allocated amount, then resize it.
 *   Make size no smaller than min.
 */
u_long dynbuff::optimize(void)
{
    if (alloced > min && bytes_in < alloced)
    {
        char *_new = new char[min + 1];
        if (!_new)
            return 0;
        memcpy(_new, contents, bytes_in);
        delete[] contents;
        alloced = min;
        contents = _new;
        contents[alloced] = 0;
        contents[bytes_in] = 0;
        return alloced;
    }
    return 0;
}

