/*
 * logfile.cpp
 *
 * (c) 1999, 2000 Murat Deligonul
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */
 
        
/*
 * TODO:
 * - Size variables updated incorrectly if not logging seperately (still true?)
 */
 
                
/*
 * some extended info is encryped into the log file name; the format is:
 * pid username userid detachedpasswordinsomeweirdencrpytion private channel [nothing].log
 *
 * We use spaces for seperators. The main reason for this is to keep me sane.
 * It is hard enough manipulating C style strings, it is a total nightmare when
 * the seperator character is not unique. I.e., using, say, '-' as the seperator
 * sounds cool but really it is a legal character to use in Nicknames and passwords,
 * and will therefore be a mess to parse. 
 *
 * We use spaces. Murat is happy.
 */
#include "autoconf.h"

                   
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <limits.h>
#include <stdlib.h>
#include <ctype.h>
#include <dirent.h>

#include "linkedlist.h"                     
#include "ezbounce.h"                            
#include "general.h"
#include "logfile.h"
#include "objset.h"
#include "server.h"

static char * cheesy_encrypt(char *);
static char * cheesy_decrypt(char *);

linkedlist<char> logfile::active_logs;

/*
 * open()'s a new logfile, ensuring that is unique and not locked by anyone
 * else
 */
/* static */ int logfile::mkname(const char * basedir, const char * owner, int id, 
                                  const char * pw, const char * type, char * buff)
{
    static const char *extra[] = { "", "1.", "2.", "3.", "4.", "5.", "6." };
    int c = 0, fd = -1;
    while (1)
    {
        sprintf(buff, "%s/%d %s %d %s %s.%slog", basedir, getpid(), owner, id, pw, type, extra[c]); 
        if (is_locked(buff)
           || (fd = open(buff, O_CREAT | O_APPEND | O_WRONLY, 0600)) < 0)
        {
            if (fd > -1)
            {
                close(fd);
                fd = -1;
            }
            if (++c > 6)
                return -1;
            continue;
        } else if (fd > -1)
            break;
    }
    return fd;
}
/*
 * Updates result on leave. Can be:
 *  1 = Ok
 *  0 = Errors, but can go on
 * -1 = Fatal
 */
logfile::logfile(const char * basedir, const char * owner, int id, int options, int maxsize, const char * password, int * result)
{
    char filebuff[PATH_MAX + 1];
    char * pw = my_strdup(password);
    
    pw = cheesy_encrypt(pw);
    
    log1 = log2 = NULL;
    fd1 = fd2 = -1;
    this->options = options;
    this->maxsize = maxsize;
    
    /* Create the log files now */
    if (options & LOG_SEPERATE)
    {
        if (options & LOG_CHANNEL)
        {
            /* Assemble the file name */
            fd1 = mkname(basedir, owner, id, pw, "(channel)", filebuff);
            if (fd1 < 0)
                *result = 0;
            else
                log1 = my_strdup(filebuff);
        }
        if (options & LOG_PRIVATE)
        {
            /* Assemble the file name */
            fd2 = mkname(basedir, owner, id, pw, "(private)", filebuff);
            if (fd2 < 0)
                *result = 0;
            else
                log2 = my_strdup(filebuff);
        }
        
        if (fd1 < 0 && fd2 >= 0)
            fd1 = fd2;
        else if (fd2 < 0 && fd1 >= 0)
            fd2 = fd1;
    } else {
        if ((options & LOG_CHANNEL) || (options & LOG_PRIVATE))
            fd1 = mkname(basedir, owner, id, pw, "(all)", filebuff);
        if (fd1 < 0)
        {
            *result = -1;
            delete[] pw;
            return;
        } 
        log1 = my_strdup(filebuff);
    }

    delete[] pw;            

    if (fd2 < 0 && fd1 < 0)
    {
        *result = -1;
        return;
    }
    
    struct stat st;

    fstat(fd1, &st);
    size1 = st.st_size;
    fstat(fd2, &st);
    size2 = st.st_size;

    if ((maxsize) && (size1 > (unsigned) maxsize || size2 > (unsigned) maxsize))
    {
        *result = -1;
        DEBUG("Log file already too big; aborting.\n");
        close(fd1);
        if (fd2 >= 0 && fd2 != fd1) 
            close(fd2); 
        fd1 = fd2 = -1;
        return;
    }

    /* And I think we're ready after this */
    if (log1)
        lock(log1);
    if (fd1 != fd2 && log2) 
        lock(log2);
    
    fdprintf(fd1, "******************************************************\n"
                  "ezbounce log file started at %s\n"
                  "******************************************************\n", 
                   timestamp());  
    if (fd2 != fd1)
        fdprintf(fd2, "******************************************************\n"
                      "ezbounce log file started at %s\n"
                      "******************************************************\n", 
                   timestamp());  
    *result = 1;
}                            


void logfile::stop(void)
{
    if (fd1 > -1)
    {
        fdprintf(fd1, "******************************************************\n"
                      "ezbounce log file stopped at %s\n"
                      "******************************************************\n", 
                     timestamp());
        close(fd1);
    }
    if (fd2 != fd1)
    {
        fdprintf(fd2, "******************************************************\n"
                      "ezbounce log file stopped at %s\n"
                      "******************************************************\n", 
                     timestamp());
        close(fd2);
    }
    fd1 = fd2 = -1;
}

logfile::~logfile(void)
{
    stop();
    if (log1)
        release(log1);
    if (log2)
        release(log2);
    delete[] log1;
    delete[] log2;
}


/* Written number of bytes written
 * 0 if nothing could be */
int logfile::write(const char * raw)
{
    char source[ADDRESS_LENGTH + 1];
    char target[CHANNEL_LENGTH + 1];
    char target2[NICKNAME_LENGTH + 1] = "";
    char text[TEXT_LENGTH + 1] = "";
    char extra[CHANNEL_LENGTH + 5] = ":";
    int target_fd = fd1;
    unsigned long * size = &size1;
    int written = 0;

    int flags = parse(raw, source, target, target2, text);
    if (flags & OTHER)
        return 0;

    if (flags & PRIVATE)
    {
        if (!(options & LOG_PRIVATE))
            return 0;
        extra[0] = 0;
        target_fd = fd2;
        size = &size2;
    }

    /* Check file size now */
    if ((maxsize) && (*size > (unsigned) maxsize))
    {
        fdprintf(target_fd, "*** [NOTE] Logfile size limit of %d bytes exceeded; stopping log ***\n", maxsize);
        fdprintf(target_fd, "******************************************************\n"
                        "ezbounce log file stopped at %s\n"
                        "******************************************************\n", 
                 timestamp());
        close(target_fd);
        if (target_fd == fd1)
            fd1 = -1;
        else if (target_fd == fd2)
            fd2 = -1;
        return 0;
    }
        

    else if (flags & PUBLIC)
    {
        /* Check if we really need to log (though this is redundant, checks
         * happen elsewhere */
        if (!(options & LOG_CHANNEL))
            return 0;
        /* Include full address in public channel messages ? */
        if (!(options & LOG_FULL_ADDRS) && (flags & MESSAGE))
        {
            /* If not, truncate after the nickname */
            char * pos = strchr(source, '!');
            if (pos)
                *pos = 0;
        }
        strcat(extra, target);
    }
                             
    strip_burc_codes(text);

    if (options & LOG_TIMESTAMP)
        written += fdprintf(target_fd, "%s ", timestamp(1));
    if (flags & NOTICE)
        written += fdprintf(target_fd, "-%s%s- %s\n", no_leading(source), extra, no_leading(text));
    else if (flags & JOIN)
        written += fdprintf(target_fd, "*** %s has JOINed %s\n", no_leading(source), no_leading(target));
    else if (flags & PART)
        written += fdprintf(target_fd, "*** %s has PARTed %s (%s)\n", no_leading(source), no_leading(target), 
                           no_leading(text));
    else if (flags & TOPIC)
        written += fdprintf(target_fd, "*** %s changes %s TOPIC to: %s\n", no_leading(source), no_leading(target), 
                           no_leading(text));
    else if (flags & KICK)
        written += fdprintf(target_fd, "*** %s has KICKed %s from %s (%s)\n", no_leading(source), no_leading(target2), 
                           target, no_leading(text));
    else if (flags & MODE)
        written += fdprintf(target_fd, "*** %s changes %s MODE to: %s\n", no_leading(source), target, no_leading(text));
    else if (flags & CTCP)                            
    {
        /* Remove ending 001 char */
        char * dummy = &text[strlen(text)] - 1;
        if (*dummy == '\001')
            *dummy = 0;

        /* Check for ACTION ctcps and do nicer formatting */
        char tmp[10];
        gettok(no_leading(text) + 1, tmp, sizeof(tmp), ' ', 1);
        if (strcmp("ACTION", tmp) == 0)
            written += fdprintf(target_fd, "* %s%s %s\n", no_leading(source), extra, no_leading(text) + 1 + 7);
        else
            written += fdprintf(target_fd, "*** CTCP %s from %s to %s\n", no_leading(text) + 1, no_leading(source),
                                        target);
    } else
        written += fdprintf(target_fd, "<%s%s> %s\n", no_leading(source), extra, no_leading(text));
    
    if (written >= 0)
        *size += (unsigned long) written;
    return written;
}


/*
 * Split the message up into little chunks
 * Return WTF it is!
 */
int logfile::parse(const char * raw, char * source, char * target, char * target2, char * text)
{
    /* Valid stuff handled is in the form:
     * :[source] [PRIVMSG/NOTICE/JOIN/PART/TOPIC/MODE] [target] :text 
     * :[source] KICK [target] [target2] :reason
     */

    int flags = 0;
    char type[10];
    
    if (!gettok(raw, type, sizeof(type),' ', 2))
        return OTHER;

    if (strcmp(type, "PRIVMSG") == 0)
        flags |= MESSAGE; 
    else if (strcmp(type, "NOTICE") == 0)
        flags |= NOTICE;
    else if (options & LOG_CHANNEL)
    {
        if (strcmp(type, "JOIN") == 0)
            flags |= JOIN;
        else if (strcmp(type, "PART") == 0)
            flags |= PART;
        else if (strcmp(type, "TOPIC") == 0)
            flags |= TOPIC;
        else if (strcmp(type, "MODE") == 0)
            flags |= MODE;
        else if (strcmp(type, "KICK") == 0)
        {
            flags |= KICK;
            /* Get target 2 -- that is, the person being kicked, as well 
            as the reason */
            if (!gettok(raw, target2, NICKNAME_LENGTH, ' ', 4) || 
                !gettok(raw, text, TEXT_LENGTH, ' ', 5, 1))
                return OTHER;
            gettok(raw, target, CHANNEL_LENGTH , ' ', 3);
            gettok(raw, source, ADDRESS_LENGTH, ' ', 1);
            return flags;
        }
        else
            return OTHER;
    }
    else
        return OTHER;

    /* Get the target */
    if (!gettok(raw, target, CHANNEL_LENGTH , ' ', 3))
        return OTHER;

    if ((flags & NOTICE) || (flags & MESSAGE))
    {
        switch (target[0])
        {
        case '#':
        case '+':       
        case '&':
        case '!':               /* new IRCNET stuff? */
            if (!(options & LOG_CHANNEL))
                return OTHER;
            flags |= PUBLIC;
            break;
        default:
            if (!(options & LOG_PRIVATE))
                return OTHER;
            flags |= PRIVATE;
            flags &= ~PUBLIC;
        }
    }

    gettok(raw, text, TEXT_LENGTH, ' ', 4,1);
    gettok(raw, source, ADDRESS_LENGTH, ' ', 1);
    
    if ((flags & MESSAGE) && (text[0] == '\001' || text[1] == '\001'))
        flags |= CTCP;
    
    return flags;
}

    
/*
 * Convert character log control codes to logfile class
 * flags. Valid codes:
 * f -  full user address in public channel messages
 * p -  private
 * c -  channel
 * a -  both channel and private
 * n -  none (causes immediate return 
 * s -  log seperately
 * t -  timestamp all events
 */
/* static */ int logfile::charflags_to_int(const char * stuff)
{
    int flags = 0;
    for (unsigned x = 0; x < strlen(stuff); x++)
    {
        switch (tolower(stuff[x]))
        {
        case 'c':
            flags |= LOG_CHANNEL;
            break;
        case 'p':
            flags |= LOG_PRIVATE;
            break;
        case 'a':
            flags |= LOG_ALL;
            break;
        case 'n':            
            return LOG_NONE;
        case 's':
            flags |= LOG_SEPERATE;
            break;
        case 't':
            flags |= LOG_TIMESTAMP;
            break;
        case 'f':
            flags |= LOG_FULL_ADDRS;
            break;
        }
    }
    return flags;
}

/* 
 * Not going to bother with checks.
 * Buffer had better be able to hold 15 chars.  
 * Return size, not including the null char */
/* static */ int logfile::intflags_to_char(int flags, char * buff)
{
    int p = 0;
    if (flags & LOG_SEPERATE)
        buff[p++] = 's';
    if ((flags & LOG_ALL) == LOG_ALL)
        buff[p++] = 'a';
    if (flags & LOG_PRIVATE)
        buff[p++] = 'p';
    if (flags & LOG_CHANNEL)
        buff[p++] = 'c';
    if (flags & LOG_FULL_ADDRS)
        buff[p++] = 'f';
    if (flags & LOG_TIMESTAMP)
        buff[p++] = 't';
    if (flags == LOG_NONE)
    {
        p = 0;
        buff[p++] = 'n';
    }
    buff[p] = 0;
    
    return p;
}

/* Not going to bother checking for exceeding size limit. */ 
int logfile::dump(const char * raw)
{
    int ret = -1;
    if (fd1 > -1)
    {
        ret = fdprintf(fd1, "%s", raw);
        size1 += ret;
    }
        
    if (fd2 != fd1)
        size2 += fdprintf(fd2, "%s", raw);

    return ret;
}


/* 
 * Fill an array with our file names.
 * We will always fill both elements.
 * f[0] will be all OR channel logs
 * f[1] will be NULL OR private logs
 */
int logfile::get_filenames(const char *f[2])
{
    f[0] = my_strdup(log1);
    f[1] = my_strdup(log2);
    return 1;
}

/*
 * Remove Bold, Underline, Reverse, and Color codes from
 * the text. 
 */
int logfile::strip_burc_codes(char * text)
{
    static const char BOLD  = 2,
                      COLOR = 3,
                      ULINE = 21,
                      REVERSE = 18;
    
    /* Speed hack.. */

    if (!strchr(text,BOLD) && !strchr(text, COLOR) 
        && !strchr(text,ULINE) && !strchr(text, REVERSE))
        return 1;

    char * n = new char[strlen(text) + 1];    
    char * orig = text, *origN = n;

    while (*text)
    {
        if (*text == BOLD || *text == COLOR 
            || *text == ULINE || *text == REVERSE)
        {
            text++;
            continue;
        }
        if (*text == COLOR)
        {
            /* From what i can tell.. there seems to be no clear
             * standard as to how many digits the number after the
             * ^C will be, and how many digits the number after the
             * comma will be.. and I think the valid range of any
             * color code number is 0-15.
             */ 
            text++;

            if (isdigit(*text))
                text++;
            if (*text <= '5' && *text >= '0' && *(text - 1) == '1')
                text++;

            if (*text == ',')
            {
                text++;
                if (isdigit(*text))
                    text++;
                if (*text <= '5' && *text >= '0' && *(text - 1) == '1')
                    text++;            
            }

            continue;
        }

        *n++ = *text++;
    }
    *n = 0;
    strcpy(orig, origN); 

    /* there's gotta be a better way.. */
    return 1;
}
    


static char * cheesy_encrypt(char * text)
{
    unsigned int i;
    char c;
    for (i = 0; i < strlen(text); i++)
	{
        c = text[i];
        if (!(i % 2))
            c = (9 ^ c);
        else
            c = c ^ (i % c);    
        text[i] = c;
	}
	return text;
}

static char * cheesy_decrypt(char * text)
{
	unsigned int i;
    char c;
	for (i = 0; i < strlen(text); i++)
	{
            c = text[i];
            if (!(i % 2))
                c = (9 ^ c);
            else
                c = c ^ (i % c);
            text[i] = c;            
	}
	return text;         
}


/*
 * Finds log files matching requested properties in basedir.
 * First checks for ones under this pid then everything else is fair game.
 * Makes sure to not match log files that are currently being written
 * to. 
 *
 * Store up to two results in logfiles. Return number of results found.
 */

struct logfile_ent {
    int uid;
    pid_t pid;
    char * filename;
    char file_nick[NICKNAME_LENGTH];
    char file_pw[PASSWORD_LENGTH];
};


/*
 * Arguments: 
 *      basedir - where to find logs
 *      nickname, password, id, self explanatatory
 *      logfiles[2] - holds names of users active log files AND will be used
 *      to store the new ones
 */
/* static */ int logfile::find_log_files(const char * basedir, const char * /* nickname */, 
                                         const char * password, int id, obj_set<char> * plist, int max)
{
    DIR *dir = opendir(basedir);
    struct dirent *d = 0;
    int num_found = 0;
    static pid_t mypid = getpid();

    if (!dir)
        return -1;
    
    obj_set<struct logfile_ent> list;
    struct logfile_ent * lt = 0;

    errno = 0;

    while ((d = readdir(dir)))
    {
        int args;

        if (is_locked(d->d_name))
        {
            DEBUG("Log file %s found to be locked, skipping.\n", d->d_name);
            continue;
        }
        
        lt = (lt ? lt : new struct logfile_ent);
        
        /* First of all, check that the name matches the known pattern we use */
        args = sscanf(d->d_name, "%d %" STR_NICKNAME_LENGTH "s %d %" STR_PASSWORD_LENGTH "s", &lt->pid, lt->file_nick, &lt->uid, lt->file_pw);

        if (args < 4) 
            continue;

        /* Compare the id and password -- 
         * NOTE: right now we don't check for nickname */
        if (lt->uid == id && !strcasecmp(password, cheesy_decrypt(lt->file_pw)))
        {
            num_found++;
            lt->filename = new char[strlen(basedir) + strlen(d->d_name) + 5];
            sprintf(lt->filename, "%s/%s", basedir, my_strdup(d->d_name));
            list.add(lt);
            lt = 0; 
        }
    }
    
    closedir(dir);

    if (lt)
    {
        delete lt;
        lt = 0;
    }
    

    if (num_found)
    {
        int cur = 0;
        /* Go through the whole list, favor ones with current PID */
        for (int i = 0; i < num_found && cur < max; i++)
        {
            lt = list.objs[i];
            if (lt->pid == mypid)
            {
                cur++;
                plist->add(my_strdup(lt->filename));
            }
        }

        /* Fill in other matches.    */
        for (int i = 0; i < num_found && cur < max; i++)
        {
            lt = list.objs[i];
            if (lt->pid != mypid)
            {
                cur++;
                plist->add(my_strdup(lt->filename));
            }
        }

        lt = 0;
        
        /* We got the filenames now, so clean up the mess */
        for (int i = 0; i < num_found; i++)
        {
            delete[] list.objs[i]->filename;
            delete list.objs[i];
        }

        return num_found;
    }    
    return 0;
}


/* static */ int logfile::lock(const char * file)
{
    DEBUG("logfile::lock() called on %s\n", file);
    file = nopath(file);
    if (!file || is_locked(file))
    {
        DEBUG("logfile::lock() on alreday locked file %s !!\n", file);
        return 0;
    }
    active_logs.add(my_strdup(file));
    return 1;
}

/* static */ int logfile::release(const char * file)
{
    list_iterator<char> i (&active_logs);
    char * f;
    file = nopath(file);
    DEBUG("logfile::release() called on %s\n", file);
    for (; (f = i.get()); ++i)
    {
        if (strcmp(file, f) == 0)
        {
            active_logs.remove(f);
            delete[] f;
            return 1;
        }
    }
    return 0;
}


/* static */ int logfile::is_locked(const char * file)
{
    list_iterator<char> i (&active_logs);
    char * f;
    file = nopath(file);
    DEBUG("Lock Check on file %s\n", file);
    for (; (f = i.get()); ++i)
        if (strcmp(file, f) == 0)
            return 1;

    return 0;
}


const char * logfile::fixup_logname(const char * filename)
{
    static char * fixup_buff;
    if (fixup_buff)
    {
        delete[] fixup_buff;
        fixup_buff = 0;
    }
    
    filename = nopath(filename);
    if (!fixup_buff)
        fixup_buff = new char[strlen(filename) + 20];
    
    char nick[NICKNAME_LENGTH], id[10], suffix[20];
    time_t blah = ircproxy_time();
    struct tm * t = localtime(&blah);

    gettok(filename, nick, sizeof(nick), ' ', 2);
    gettok(filename, id, sizeof(id), ' ', 3);
    gettok(filename, suffix, sizeof(suffix), ' ', 5);

    sprintf(fixup_buff, "%d%02d%02d-%s-[%s]-%s", 1900 + t->tm_year, t->tm_mon + 1, t->tm_mday, nick, id, suffix);
    return fixup_buff;
}
