/*
 * logfile.h
 *
 * (c) 1999,2000 Murat Deligonul
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

 
#ifndef __LOGFILE_H
#define __LOGFILE_H

class logfile;

#include "objset.h"
#include "linkedlist.h"

           
class logfile
{
public:
    logfile(const char * basedir, const char * owner, int id, int options, int maxsize, const char *, int * result);

    ~logfile();

    int  write(const char *);   /* Parse raw IRC text and write to a log file */
    int  dump(const char *);    /* Write raw text to log */
    void stop(void);            /* Stop logging */
    int  get_filenames(const char *f[2]);
    
    enum {                      /* Log file options */
        LOG_PRIVATE  = 2,       /* Log private messages */
        LOG_CHANNEL  = 4,       /* Log messages to channel */
        LOG_PUBLIC   = 4,
        LOG_ALL      = 6,       /* Log all of them */
        LOG_SEPERATE = 8,       /* Log to seperate files */
        LOG_NONE     = 16,      /* Do not log */
        LOG_FULL_ADDRS = 32,    /* Full-user address in public messages */
        LOG_TIMESTAMP = 64      /* Log all events with a timestamp */
    };
        
    static int charflags_to_int(const char *);
    static int intflags_to_char(int, char *);
    static int find_log_files(const char * basedir, const char * nickname, 
                                         const char * password, int id, obj_set<char> * plist, int max);
private:
    /* fd2 is always private message log file fd.
     * fd1 is always chan message log file fd.
     * They are both always valid file descriptors
     * (so the may equal each other)
     */
    int options, fd1, fd2, maxsize;
    unsigned long size1, size2; /* How big are the files */
    char * log1;                /* Both only needed when sending */
    char * log2;                /* log files back to user */
    
public:
    const char * get_log1(void) { return log1; }
    const char * get_log2(void) { return log2; }


    static const char * fixup_logname(const char * filename);

    /* need to keep track of which log files are active, that is:
     *   -- currently being written to
     *   -- waiting to be sent
     *   -- being sent
     */
    static int lock(const char * file);
    static int release(const char * file);
    static int is_locked(const char * file);

private:
    static linkedlist<char> active_logs;


protected:
    enum {
        OTHER   =  1,
        PRIVATE =  2,
        PUBLIC  =  4,
        CTCP    =  8,
        NOTICE  = 16,
        KICK    = 32,  
        JOIN    = 64,
        PART    = 128,
        MODE    = 256,
        TOPIC   = 512,
        MESSAGE = 1024
    };

    /* Parse and split */
    int parse(const char *, char * source, char * target, char * target2, char * text);    
    static int strip_burc_codes(char * text);
    static int mkname(const char * basedir, const char * owner, int id, 
                                 const char * pw, const char * type, char * buff);
    

};
  

#endif /* __LOGFILE_H */
