/*
 * $Id: system.c,v 1.443 2009-10-16 08:47:55 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "system.h"
#include "glue-main.h"
#include "glue-gui.h"
#include "glue-log.h"

#include "sig_boolean.h"
#include "sig_cardbus.h"
#include "sig_dio24.h"
#include "sig_dio48.h"
#include "sig_eth.h"
#include "sig_opt_rgb.h"
#include "sig_host_bus.h"
#include "sig_ide_bus.h"
#include "sig_integer.h"
#include "sig_isa_bus.h"
#include "sig_magneto_optical.h"
#include "sig_manage.h"
#include "sig_match.h"
#include "sig_mem_bus.h"
#include "sig_parallel.h"
#include "sig_pci_bus.h"
#include "sig_power.h"
#include "sig_ps2.h"
#include "sig_scsi_bus.h"
#include "sig_serial.h"
#include "sig_shugart_bus.h"
#include "sig_sound.h"
#include "sig_std_logic.h"
#include "sig_telephone.h"
#include "sig_usb_bus.h"
#include "sig_vga.h"
#include "sig_video.h"

#if 0
#include "cim_audio.h"
#include "cim_boolean.h"
#include "cim_cardbus.h"
#include "cim_dio24.h"
#include "cim_dio48.h"
#include "cim_eth.h"
#include "cim_opt_rgb.h"
#include "cim_integer.h"
#include "cim_parallel.h"
#include "cim_serial.h"
#include "cim_string.h"
#include "cim_telephone.h"
#include "cim_usb.h"
#include "cim_video.h"
#endif

#include "simulator/system.inc1"

static void *cpssp = &cpssp;

#if 0
/* CIM Ports */
static struct cim_audio port_audio[100];
static struct cim_boolean port_boolean[1000];
static struct cim_cardbus port_cardbus[100];
static struct cim_dio24 port_dio24[100];
static struct cim_dio48 port_dio48[100];
static struct cim_eth port_eth[100];
static struct cim_integer port_integer[100];
static struct cim_parallel port_parallel[100];
static struct cim_serial port_serial[100];
static struct cim_string port_string[100];
static struct cim_telephone port_telephone[100];
static struct cim_usb port_usb_bus[100];
static struct cim_video port_video[100];
#endif

static const struct {
	const char *type;
	void *(*create)(char *);
	void (*destroy)(void *);
	void *(*merge)(void *, void *);
	void (*split)(void *);
} sig_type[] = {
	{ "agp_bus",
		(void *(*)(char *)) sig_agp_bus_create,
		(void (*)(void *)) sig_agp_bus_destroy
	}, { "agp_bus_main",
		(void *(*)(char *)) sig_agp_bus_main_create,
		(void (*)(void *)) sig_agp_bus_main_destroy
	}, { "boolean",
		(void *(*)(char *)) sig_boolean_create,
		(void (*)(void *)) sig_boolean_destroy
	}, { "boolean_or",
		(void *(*)(char *)) sig_boolean_or_create,
		(void (*)(void *)) sig_boolean_or_destroy
	}, { "cardbus",
		(void *(*)(char *)) sig_cardbus_create,
		(void (*)(void *)) sig_cardbus_destroy
	}, { "cs",
		(void *(*)(char *)) sig_cs_create,
		(void (*)(void *)) sig_cs_destroy
	}, { "dio24",
		(void *(*)(char *)) sig_dio24_create,
		(void (*)(void *)) sig_dio24_destroy
	}, { "dio48",
		(void *(*)(char *)) sig_dio48_create,
		(void (*)(void *)) sig_dio48_destroy
	}, { "eth",
		(void *(*)(char *)) sig_eth_create,
		(void (*)(void *)) sig_eth_destroy
	}, { "floppy",
		(void *(*)(char *)) sig_floppy_create,
		(void (*)(void *)) sig_floppy_destroy,
		(void *(*)(void *, void *)) sig_floppy_merge,
		(void (*)(void *)) sig_floppy_split
	}, { "host_bus",
		(void *(*)(char *)) sig_host_bus_create,
		(void (*)(void *)) sig_host_bus_destroy
	}, { "host_bus_main",
		(void *(*)(char *)) sig_host_bus_main_create,
		(void (*)(void *)) sig_host_bus_main_destroy
	}, { "i2c_bus",
		(void *(*)(char *)) sig_i2c_bus_create,
		(void (*)(void *)) sig_i2c_bus_destroy
	}, { "icc_bus",
		(void *(*)(char *)) sig_icc_bus_create,
		(void (*)(void *)) sig_icc_bus_destroy
	}, { "ide_bus",
		(void *(*)(char *)) sig_ide_bus_create,
		(void (*)(void *)) sig_ide_bus_destroy
	}, { "integer",
		(void *(*)(char *)) sig_integer_create,
		(void (*)(void *)) sig_integer_destroy
	}, { "isa_bus",
		(void *(*)(char *)) sig_isa_bus_create,
		(void (*)(void *)) sig_isa_bus_destroy
	}, { "isa_bus_dma",
		(void *(*)(char *)) sig_isa_bus_dma_create,
		(void (*)(void *)) sig_isa_bus_dma_destroy
	}, { "isa_bus_main",
		(void *(*)(char *)) sig_isa_bus_main_create,
		(void (*)(void *)) sig_isa_bus_main_destroy
	}, { "magneto_optical",
		(void *(*)(char *)) sig_magneto_optical_create,
		(void (*)(void *)) sig_magneto_optical_destroy,
		(void *(*)(void *, void *)) sig_magneto_optical_merge,
		(void (*)(void *)) sig_magneto_optical_split
	}, { "manage",
		(void *(*)(char *)) sig_manage_create,
		(void (*)(void *)) sig_manage_destroy
	}, { "match",
		(void *(*)(char *)) sig_match_create,
		(void (*)(void *)) sig_match_destroy
	}, { "mem_bus",
		(void *(*)(char *)) sig_mem_bus_create,
		(void (*)(void *)) sig_mem_bus_destroy
	}, { "mem_bus_main",
		(void *(*)(char *)) sig_mem_bus_main_create,
		(void (*)(void *)) sig_mem_bus_main_destroy
	}, { "opt_rgb",
		(void *(*)(char *)) sig_opt_rgb_create,
		(void (*)(void *)) sig_opt_rgb_destroy
	}, { "parallel",
		(void *(*)(char *)) sig_parallel_create,
		(void (*)(void *)) sig_parallel_destroy
	}, { "pci_bus",
		(void *(*)(char *)) sig_pci_bus_create,
		(void (*)(void *)) sig_pci_bus_destroy
	}, { "pci_bus_idsel",
		(void *(*)(char *)) sig_pci_bus_idsel_create,
		(void (*)(void *)) sig_pci_bus_idsel_destroy
	}, { "pci_bus_main",
		(void *(*)(char *)) sig_pci_bus_main_create,
		(void (*)(void *)) sig_pci_bus_main_destroy
	}, { "power_board",
		(void *(*)(char *)) sig_power_board_create,
		(void (*)(void *)) sig_power_board_destroy
	}, { "power_board_at",
		(void *(*)(char *)) sig_power_board_at_create,
		(void (*)(void *)) sig_power_board_at_destroy
	}, { "power_device",
		(void *(*)(char *)) sig_power_device_create,
		(void (*)(void *)) sig_power_device_destroy
	}, { "ps2",
		(void *(*)(char *)) sig_ps2_create,
		(void (*)(void *)) sig_ps2_destroy
	}, { "ps2_main",
		(void *(*)(char *)) sig_ps2_main_create,
		(void (*)(void *)) sig_ps2_main_destroy
	}, { "scsi_bus",
		(void *(*)(char *)) sig_scsi_bus_create,
		(void (*)(void *)) sig_scsi_bus_destroy
	}, { "serial",
		(void *(*)(char *)) sig_serial_create,
		(void (*)(void *)) sig_serial_destroy
	}, { "shugart_bus",
		(void *(*)(char *)) sig_shugart_bus_create,
		(void (*)(void *)) sig_shugart_bus_destroy
	}, { "sound",
		(void *(*)(char *)) sig_sound_create,
		(void (*)(void *)) sig_sound_destroy
	}, { "std_logic",
		(void *(*)(char *)) sig_std_logic_create,
		(void (*)(void *)) sig_std_logic_destroy
	}, { "string",
		(void *(*)(char *)) sig_string_create,
		(void (*)(void *)) sig_string_destroy
	}, { "telephone",
		(void *(*)(char *)) sig_telephone_create,
		(void (*)(void *)) sig_telephone_destroy
	}, { "usb_bus",
		(void *(*)(char *)) sig_usb_bus_create,
		(void (*)(void *)) sig_usb_bus_destroy
	}, { "usb_bus_main",
		(void *(*)(char *)) sig_usb_bus_main_create,
		(void (*)(void *)) sig_usb_bus_main_destroy
	}, { "vga",
		(void *(*)(char *)) sig_vga_create,
		(void (*)(void *)) sig_vga_destroy
	}, { "video",
		(void *(*)(char *)) sig_video_create,
		(void (*)(void *)) sig_video_destroy
	}
};

static const struct {
	const char *type;
	const char *generic[50];
	const char *port[150];
	enum system_gender gender[150];
	void *(*create)(const char *name, ...);
	void (*destroy)(void *);
	void *(*gui_create)(unsigned int page, const char *name, ...);
	void (*gui_destroy)(void *);
	void *(*aui_create)(const char *name, ...);
	void (*aui_destroy)(void *);
} comp_type[] = {
#include "simulator/system.inc2"
};

/* Generic GUI pages. */
static struct {
	unsigned int id;
	char *name;
} page[64];
static unsigned int page_count = 0;

/* Generic signals. */
static struct {
	unsigned int id;
	char *type;
	char *name;

	void *ptr;
} sig[10000];

/* Generic components. */
static struct {
	char *type;
	char *name;
	unsigned int node;
	unsigned int page;
	unsigned int id;

	void *ptr;
	void *aui_ptr;
	void *gui_ptr;
} comp[200];

/* Generic generics. */
static struct {
	unsigned int comp_id;
	char *type;
	char *name;
	char *value;
} generic[2000];
static unsigned int generic_count = 0;

/* Generic connections. */
static struct {
	unsigned int comp_id;
	char *port;
	enum system_gender gender;
	unsigned int sig_id;
	void *sig_ptr;

	void *merge;
	int peer;
} conn[2000];

static const char *system_name[64];
static unsigned int system_level;

static unsigned int
system_id(void)
{
	static unsigned int id = 0;

	return id++;
}

void
system_name_push(const char *name)
{
	assert(name);

	assert(system_level < sizeof(system_name) / sizeof(system_name[0]));
	system_name[system_level++] = name;
}

void
system_name_pop(void)
{
	assert(1 <= system_level);
	system_level--;
}

const char *
system_path(void)
{
	static char path[1024];
	unsigned int level;

	assert(1 <= system_level);

	assert(strlen(system_name[0]) + 1 <= sizeof(path));
	strcpy(path, system_name[0]);
	for (level = 1; level < system_level; level++) {
		assert(strlen(path) + 1 + strlen(system_name[level]) + 1
			<= sizeof(path));
		strcat(path, ":");
		strcat(path, system_name[level]);
	}

	return path;
}

static void
local_page_create(unsigned int page_id, const char *name)
{
	unsigned int pi;

	for (pi = 0; pi < page_count; pi++) {
		if (strcasecmp(page[pi].name, name) == 0) {
			/* Found */
			assert(0); /* FIXME */
			return;
		}
	}

	page[page_count].id = page_id;
	page[page_count].name = strdup(name);
	assert(page[page_count].name);

	gui_page_add(page_id, page[page_count].name);
}

int
system_page_create(const char *name)
{
	unsigned int page_id;

	page_id = system_id();

	local_page_create(page_id, name);

	return page_id;
}

static void
local_page_destroy(unsigned int page_id)
{
	/* FIXME */
}

int
system_page_destroy(unsigned int page_id)
{
	local_page_destroy(page_id);

	return 0;
}

int
system_page_lookup(const char *name)
{
	unsigned int pi;

	for (pi = 0; ; pi++) {
		if (pi == sizeof(page) / sizeof(page[0])) {
			/* Page not found. */
			return -1;
		}

		if (strcasecmp(page[pi].name, name) == 0) {
			break;
		}
	}

	return page[pi].id;
}

static void
local_sig_create(unsigned int sig_id, const char *type, const char *name)
{
	unsigned int si;

	for (si = 0; si < sizeof(sig) / sizeof(sig[0]); si++) {
		if (! sig[si].type
		 || ! sig[si].name) {
			continue;
		}
		if (strcasecmp(sig[si].name, name) == 0) {
			/* Found */
			assert(0);
			return;
		}
	}
	for (si = 0; ; si++) {
		if (si == sizeof(sig) / sizeof(sig[0])) {
			/* Table Full */
			assert(0);
			return;
		}
		if (! sig[si].name) {
			break;
		}
	}

	sig[si].id = sig_id;

	sig[si].type = strdup(type);
	assert(sig[si].type);

	sig[si].name = strdup(name);
	assert(sig[si].name);

	sig[si].ptr = NULL;
}

int
system_sig_create(const char *type, const char *name)
{
	unsigned int sig_id;

	sig_id = system_id();

	local_sig_create(sig_id, type, name);

	return sig_id;
}

static void *
local_sig_get(unsigned int sig_id)
{
	unsigned int si;

	/* Lookup signal. */
	for (si = 0; ; si++) {
		if (si == sizeof(sig) / sizeof(sig[0])) {
			/* Not found. */
			/* FIXME */
			assert(0);
		}
		if (! sig[si].type
		 || ! sig[si].name) {
			continue;
		}
		if (sig[si].id == sig_id) {
			/* Found */
			break;
		}
	}

	assert(sig[si].id == sig_id);
	assert(sig[si].type);
	assert(sig[si].name);

	if (! sig[si].ptr) {
		unsigned int j;

		for (j = 0; ; j++) {
			if (j == sizeof(sig_type) / sizeof(sig_type[0])) {
				/* FIXME */
				fprintf(stderr, "Type: %s.\n", sig[si].type);
				assert(0);
			}
			if (strcasecmp(sig[si].type, sig_type[j].type) == 0) {
				sig[si].ptr = sig_type[j].create(sig[si].name);
				break;
			}
		}
	}

	return sig[si].ptr;
}

void *
system_sig_get(unsigned int sig_id)
{
	return local_sig_get(sig_id);
}

static int
local_sig_unget(void *ptr)
{
	unsigned int si;
	unsigned int j;

	for (si = 0; ; si++) {
		if (si == sizeof(sig) / sizeof(sig[0])) {
			/* Signal not found. */
			assert(0); /* FIXME */
		}
		if (! sig[si].type
		 || ! sig[si].name) {
			continue;
		}
		if (sig[si].ptr == ptr) {
			break;
		}
	}

	for (j = 0; ; j++) {
		if (j == sizeof(sig_type) / sizeof(sig_type[0])) {
			/* FIXME */
			assert(0);
		}
		if (strcasecmp(sig[si].type, sig_type[j].type) == 0) {
			sig_type[j].destroy(sig[si].ptr);
			break;
		}
	}

	sig[si].ptr = NULL;

	return 0;
}

int
system_sig_unget(void *ptr)
{
	return local_sig_unget(ptr);
}

static void
local_sig_destroy(unsigned int sig_id)
{
	unsigned int si;

	/* Lookup signal. */
	for (si = 0; ; si++) {
		if (si == sizeof(sig) / sizeof(sig[0])) {
			/* Not found. */
			return; /* FIXME */
		}
		if (! sig[si].type
		 || ! sig[si].name) {
			continue;
		}
		if (sig[si].id == sig_id) {
			/* Found. */
			break;
		}
	}

	if (sig[si].ptr) {
		/* Hack - FIXME */
		system_sig_unget(sig[si].ptr);
	}

	free(sig[si].type);
	sig[si].type = NULL;

	free(sig[si].name);
	sig[si].name = NULL;
}

void
system_sig_destroy(unsigned int sig_id)
{
	local_sig_destroy(sig_id);
}

int
system_sig_info(unsigned int sig_id, char *type, char *name)
{
	unsigned int si;

	/* Lookup signal. */
	for (si = 0; ; si++) {
		if (si == sizeof(sig) / sizeof(sig[0])) {
			/* Not found. */
			return -1;
		}
		if (! sig[si].type
		 || ! sig[si].name) {
			continue;
		}
		if (sig[si].id == sig_id) {
			/* Found. */
			break;
		}
	}

	strcpy(type, sig[si].type);
	strcpy(name, sig[si].name);

	return 0;
}

static void
local_comp_create(
	unsigned int comp_id,
	const char *type,
	const char *name,
	unsigned int node,
	unsigned int page
)
{
	unsigned int ci;

	for (ci = 0; ci < sizeof(comp) / sizeof(comp[0]); ci++) {
		if (! comp[ci].name) {
			continue;
		}
		if (strcasecmp(comp[ci].name, name) == 0) {
			/* Found */
			assert(0); /* FIXME */
			return;
		}
	}
	for (ci = 0; ; ci++) {
		if (ci == sizeof(comp) / sizeof(comp[0])) {
			/* Table Full */
			assert(0); /* FIXME */
			return;
		}
		if (! comp[ci].name) {
			break;
		}
	}

	comp[ci].id = comp_id;

	comp[ci].type = strdup(type);
	assert(comp[ci].type);

	comp[ci].name = strdup(name);
	assert(comp[ci].name);

	comp[ci].node = node;
	comp[ci].page = page;
}

int
system_comp_create(
	const char *type,
	const char *name,
	unsigned int node,
	unsigned int page
)
{
	int comp_id;

	if (strchr(name, '/')) {
		/* Name mustn't contain '/'. */
		fprintf(stderr, "%s: WARNING: %s: %s.\n", progname,
				name, "Bad name");
		return -1;
	}

	comp_id = system_id();

	local_comp_create(comp_id, type, name, node, page);

	return comp_id;
}

static void
local_comp_generic_set(
	unsigned int comp_id,
	const char *t,
	const char *n,
	const char *v
)
{
	char *type;
	char *name;
	char *value;
	unsigned int i;

	/* check if generic with that name is already present */
	for (i = 0; i < generic_count; i++) {
		if (generic[i].comp_id != comp_id) {
			continue;
		}

		if (strcasecmp(generic[i].name, n) == 0) {
			faum_log(FAUM_LOG_WARNING, "system", __func__,
				"Generic %s of component %d already set!\n",
				n, comp_id);
		}
	}

	generic[generic_count].comp_id = comp_id;

	assert(t);
	type = strdup(t);
	assert(type);
	assert(n);
	name = strdup(n);
	assert(name);
	generic[generic_count].type = type;
	generic[generic_count].name = name;

	if (v) {
		value = strdup(v);
		assert(value);
	} else {
		value = NULL;
	}
	generic[generic_count].value = value;

	generic_count++;
}

int
system_comp_generic_set(
	unsigned int comp_id,
	const char *t,
	const char *n,
	const char *v
)
{
	local_comp_generic_set(comp_id, t, n, v);

	return 0;
}

static void
local_comp_port_connect(
	unsigned int comp_id,
	const char *port,
	unsigned int sig_id
)
{
	unsigned int conn_id;
	unsigned int ci;
	unsigned int si;

	for (ci = 0; ; ci ++) {
		if (ci == sizeof(comp) / sizeof(comp[0])) {
			/* Component not found. */
			assert(0); /* FIXME */
			return;
		}
		if (comp[ci].id == comp_id) {
			/* Component found. */
			break;
		}
	}
	for (si = 0; ; si++) {
		if (si == sizeof(sig) / sizeof(sig[0])) {
			/* Signal not found. */
			assert(0); /* FIXME */
			return;
		}
		if (sig[si].id == sig_id) {
			/* Signal found. */
			break;
		}
	}

	for (conn_id = 0; conn_id < sizeof(conn) / sizeof(conn[0]); conn_id++) {
		if (! conn[conn_id].port) {
			continue;
		}

		if (conn[conn_id].comp_id == comp_id
		 && strcasecmp(conn[conn_id].port, port) == 0) {
			/* Port already defined. */
			assert(0); /* FIXME */
			return;
		}
	}

	/* Lookup free connection entry. */
	for (conn_id = 0; ; conn_id++) {
		if (conn_id == sizeof(conn) / sizeof(conn[0])) {
			/* Table full. */
			assert(0); /* FIXME */
			return;
		}
		if (! conn[conn_id].port) {
			break;
		}
	}

	conn[conn_id].comp_id = comp_id;
	conn[conn_id].port = strdup(port);
	assert(conn[conn_id].port);
	conn[conn_id].sig_id = sig_id;
	conn[conn_id].sig_ptr = system_sig_get(sig_id);

	conn[conn_id].merge = NULL;
	conn[conn_id].peer = -1;
}

int
system_comp_port_connect(
	unsigned int comp_id,
	const char *port,
	unsigned int sig_id
)
{
	local_comp_port_connect(comp_id, port, sig_id);

	return 0;
}

static const char *
generic_get(unsigned int comp_id, const char *name)
{
	unsigned int i;

	for (i = 0; ; i++) {
		if (i == generic_count) {
			fprintf(stderr, "WARNING: Generic %s not set for component!\n",
					name);
			return NULL;
		}
		if (generic[i].comp_id == comp_id
		 && strcasecmp(generic[i].name, name) == 0) {
			return generic[i].value;
		}
	}
}

static void *
sig_lookup(unsigned int comp_id, const char *port, enum system_gender gender)
{
	unsigned int conn_id;

	for (conn_id = 0; ; conn_id++) {
		if (conn_id == sizeof(conn) / sizeof(conn[0])) {
			assert(0); /* FIXME */
		}
		assert(conn_id < sizeof(conn) / sizeof(conn[0]));

		if (! conn[conn_id].port) {
			continue;
		}

		if (conn[conn_id].comp_id == comp_id
		 && strcasecmp(conn[conn_id].port, port) == 0) {
			if (gender != SYSTEM_GENDER_NEUTRAL) {
				assert(conn[conn_id].gender == 0
				    || conn[conn_id].gender == gender);
				conn[conn_id].gender = gender;
			}
			return conn[conn_id].sig_ptr;
		}
	}
}

static void
local_comp_init(unsigned int comp_id)
{
	unsigned int ci;
	unsigned int i;

	/* Lookup component. */
	for (ci = 0; ; ci++) {
		if (ci == sizeof(comp) / sizeof(comp[0])) {
			/* Component not found. */
			assert(0); /* FIXME */
			return;
		}
		if (comp[ci].id == comp_id
		 && comp[ci].type
		 && comp[ci].name) {
			/* Component found. */
			break;
		}
	}

	for (i = 0; ; i++) {
		if (i == sizeof(comp_type) / sizeof(comp_type[0])) {
			fprintf(stderr, "%s\n", comp[ci].type);
			assert(0); /* FIXME */
		}
		if (strcasecmp(comp[ci].type, comp_type[i].type) == 0) {
			const void *ptr[200];
			unsigned int nptrs;
			unsigned int j;

			nptrs = 0;
			for (j = 0; comp_type[i].generic[j]; j++) {
				ptr[nptrs++] = generic_get(comp_id, comp_type[i].generic[j]);
			}
			for (j = 0; comp_type[i].port[j]; j++) {
				ptr[nptrs++] = sig_lookup(comp_id, comp_type[i].port[j], comp_type[i].gender[j]);
			}
			while (nptrs < 200) {
				ptr[nptrs++] = NULL;
			}

			if (comp_type[i].create) {
				comp[ci].ptr = comp_type[i].create(comp[ci].name,
					ptr[0], ptr[1], ptr[2], ptr[3], ptr[4],
					ptr[5], ptr[6], ptr[7], ptr[8], ptr[9],
					ptr[10], ptr[11], ptr[12], ptr[13], ptr[14],
					ptr[15], ptr[16], ptr[17], ptr[18], ptr[19],
					ptr[20], ptr[21], ptr[22], ptr[23], ptr[24],
					ptr[25], ptr[26], ptr[27], ptr[28], ptr[29],
					ptr[30], ptr[31], ptr[32], ptr[33], ptr[34],
					ptr[35], ptr[36], ptr[37], ptr[38], ptr[39],
					ptr[40], ptr[41], ptr[42], ptr[43], ptr[44],
					ptr[45], ptr[46], ptr[47], ptr[48], ptr[49],
					ptr[50], ptr[51], ptr[52], ptr[53], ptr[54],
					ptr[55], ptr[56], ptr[57], ptr[58], ptr[59],
					ptr[60], ptr[61], ptr[62], ptr[63], ptr[64],
					ptr[65], ptr[66], ptr[67], ptr[68], ptr[69],
					ptr[70], ptr[71], ptr[72], ptr[73], ptr[74],
					ptr[75], ptr[76], ptr[77], ptr[78], ptr[79],
					ptr[80], ptr[81], ptr[82], ptr[83], ptr[84],
					ptr[85], ptr[86], ptr[87], ptr[88], ptr[89],
					ptr[90], ptr[91], ptr[92], ptr[93], ptr[94],
					ptr[95], ptr[96], ptr[97], ptr[98], ptr[99],
					ptr[100], ptr[101], ptr[102], ptr[103], ptr[104],
					ptr[105], ptr[106], ptr[107], ptr[108], ptr[109],
					ptr[110], ptr[111], ptr[112], ptr[113], ptr[114],
					ptr[115], ptr[116], ptr[117], ptr[118], ptr[119],
					ptr[120], ptr[121], ptr[122], ptr[123], ptr[124],
					ptr[125], ptr[126], ptr[127], ptr[128], ptr[129],
					ptr[130], ptr[131], ptr[132], ptr[133], ptr[134],
					ptr[135], ptr[136], ptr[137], ptr[138], ptr[139],
					ptr[140], ptr[141], ptr[142], ptr[143], ptr[144],
					ptr[145], ptr[146], ptr[147], ptr[148], ptr[149],
					ptr[150], ptr[151], ptr[152], ptr[153], ptr[154],
					ptr[155], ptr[156], ptr[157], ptr[158], ptr[159],
					ptr[160], ptr[161], ptr[162], ptr[163], ptr[164],
					ptr[165], ptr[166], ptr[167], ptr[168], ptr[169],
					ptr[170], ptr[171], ptr[172], ptr[173], ptr[174],
					ptr[175], ptr[176], ptr[177], ptr[178], ptr[179],
					ptr[180], ptr[181], ptr[182], ptr[183], ptr[184],
					ptr[185], ptr[186], ptr[187], ptr[188], ptr[189],
					ptr[190], ptr[191], ptr[192], ptr[193], ptr[194],
					ptr[195], ptr[196], ptr[197], ptr[198], ptr[199]);
			} else {
				comp[ci].ptr = NULL;
			}
			if (comp_type[i].gui_create) {
				comp[ci].gui_ptr = comp_type[i].gui_create(comp[ci].page, comp[ci].name,
					ptr[0], ptr[1], ptr[2], ptr[3], ptr[4],
					ptr[5], ptr[6], ptr[7], ptr[8], ptr[9],
					ptr[10], ptr[11], ptr[12], ptr[13], ptr[14],
					ptr[15], ptr[16], ptr[17], ptr[18], ptr[19],
					ptr[20], ptr[21], ptr[22], ptr[23], ptr[24],
					ptr[25], ptr[26], ptr[27], ptr[28], ptr[29],
					ptr[30], ptr[31], ptr[32], ptr[33], ptr[34],
					ptr[35], ptr[36], ptr[37], ptr[38], ptr[39],
					ptr[40], ptr[41], ptr[42], ptr[43], ptr[44],
					ptr[45], ptr[46], ptr[47], ptr[48], ptr[49],
					ptr[50], ptr[51], ptr[52], ptr[53], ptr[54],
					ptr[55], ptr[56], ptr[57], ptr[58], ptr[59],
					ptr[60], ptr[61], ptr[62], ptr[63], ptr[64],
					ptr[65], ptr[66], ptr[67], ptr[68], ptr[69],
					ptr[70], ptr[71], ptr[72], ptr[73], ptr[74],
					ptr[75], ptr[76], ptr[77], ptr[78], ptr[79],
					ptr[80], ptr[81], ptr[82], ptr[83], ptr[84],
					ptr[85], ptr[86], ptr[87], ptr[88], ptr[89],
					ptr[90], ptr[91], ptr[92], ptr[93], ptr[94],
					ptr[95], ptr[96], ptr[97], ptr[98], ptr[99],
					ptr[100], ptr[101], ptr[102], ptr[103], ptr[104],
					ptr[105], ptr[106], ptr[107], ptr[108], ptr[109],
					ptr[110], ptr[111], ptr[112], ptr[113], ptr[114],
					ptr[115], ptr[116], ptr[117], ptr[118], ptr[119],
					ptr[120], ptr[121], ptr[122], ptr[123], ptr[124],
					ptr[125], ptr[126], ptr[127], ptr[128], ptr[129],
					ptr[130], ptr[131], ptr[132], ptr[133], ptr[134],
					ptr[135], ptr[136], ptr[137], ptr[138], ptr[139],
					ptr[140], ptr[141], ptr[142], ptr[143], ptr[144],
					ptr[145], ptr[146], ptr[147], ptr[148], ptr[149],
					ptr[150], ptr[151], ptr[152], ptr[153], ptr[154],
					ptr[155], ptr[156], ptr[157], ptr[158], ptr[159],
					ptr[160], ptr[161], ptr[162], ptr[163], ptr[164],
					ptr[165], ptr[166], ptr[167], ptr[168], ptr[169],
					ptr[170], ptr[171], ptr[172], ptr[173], ptr[174],
					ptr[175], ptr[176], ptr[177], ptr[178], ptr[179],
					ptr[180], ptr[181], ptr[182], ptr[183], ptr[184],
					ptr[185], ptr[186], ptr[187], ptr[188], ptr[189],
					ptr[190], ptr[191], ptr[192], ptr[193], ptr[194],
					ptr[195], ptr[196], ptr[197], ptr[198], ptr[199]);
			} else {
				comp[ci].gui_ptr = NULL;
			}
			if (comp_type[i].aui_create) {
				comp[ci].aui_ptr = comp_type[i].aui_create(comp[ci].name,
					ptr[0], ptr[1], ptr[2], ptr[3], ptr[4],
					ptr[5], ptr[6], ptr[7], ptr[8], ptr[9],
					ptr[10], ptr[11], ptr[12], ptr[13], ptr[14],
					ptr[15], ptr[16], ptr[17], ptr[18], ptr[19],
					ptr[20], ptr[21], ptr[22], ptr[23], ptr[24],
					ptr[25], ptr[26], ptr[27], ptr[28], ptr[29],
					ptr[30], ptr[31], ptr[32], ptr[33], ptr[34],
					ptr[35], ptr[36], ptr[37], ptr[38], ptr[39],
					ptr[40], ptr[41], ptr[42], ptr[43], ptr[44],
					ptr[45], ptr[46], ptr[47], ptr[48], ptr[49],
					ptr[50], ptr[51], ptr[52], ptr[53], ptr[54],
					ptr[55], ptr[56], ptr[57], ptr[58], ptr[59],
					ptr[60], ptr[61], ptr[62], ptr[63], ptr[64],
					ptr[65], ptr[66], ptr[67], ptr[68], ptr[69],
					ptr[70], ptr[71], ptr[72], ptr[73], ptr[74],
					ptr[75], ptr[76], ptr[77], ptr[78], ptr[79],
					ptr[80], ptr[81], ptr[82], ptr[83], ptr[84],
					ptr[85], ptr[86], ptr[87], ptr[88], ptr[89],
					ptr[90], ptr[91], ptr[92], ptr[93], ptr[94],
					ptr[95], ptr[96], ptr[97], ptr[98], ptr[99],
					ptr[100], ptr[101], ptr[102], ptr[103], ptr[104],
					ptr[105], ptr[106], ptr[107], ptr[108], ptr[109],
					ptr[110], ptr[111], ptr[112], ptr[113], ptr[114],
					ptr[115], ptr[116], ptr[117], ptr[118], ptr[119],
					ptr[120], ptr[121], ptr[122], ptr[123], ptr[124],
					ptr[125], ptr[126], ptr[127], ptr[128], ptr[129],
					ptr[130], ptr[131], ptr[132], ptr[133], ptr[134],
					ptr[135], ptr[136], ptr[137], ptr[138], ptr[139],
					ptr[140], ptr[141], ptr[142], ptr[143], ptr[144],
					ptr[145], ptr[146], ptr[147], ptr[148], ptr[149],
					ptr[150], ptr[151], ptr[152], ptr[153], ptr[154],
					ptr[155], ptr[156], ptr[157], ptr[158], ptr[159],
					ptr[160], ptr[161], ptr[162], ptr[163], ptr[164],
					ptr[165], ptr[166], ptr[167], ptr[168], ptr[169],
					ptr[170], ptr[171], ptr[172], ptr[173], ptr[174],
					ptr[175], ptr[176], ptr[177], ptr[178], ptr[179],
					ptr[180], ptr[181], ptr[182], ptr[183], ptr[184],
					ptr[185], ptr[186], ptr[187], ptr[188], ptr[189],
					ptr[190], ptr[191], ptr[192], ptr[193], ptr[194],
					ptr[195], ptr[196], ptr[197], ptr[198], ptr[199]);
			} else {
				comp[ci].aui_ptr = NULL;
			}
			break;
		}
	}

	if (strcasecmp(comp[ci].type, "power_supply") == 0) {
		sig_boolean_connect_out(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 0);
		sig_boolean_connect_out(sig_lookup(comp_id, "mech_power_switch", 0),
				cpssp, 0);
		sig_boolean_set(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 1);
		sig_boolean_set(sig_lookup(comp_id, "mech_power_switch", 0),
				cpssp, 1);

	} else if (strcasecmp(comp[ci].type, "power_supply_at") == 0) {
		sig_boolean_connect_out(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 0);
		sig_boolean_connect_out(sig_lookup(comp_id, "mech_power_switch", 0),
				cpssp, 0);
		sig_boolean_set(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 1);

	} else if (strcasecmp(comp[ci].type, "monitor") == 0) {
		sig_boolean_connect_out(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 1);
		sig_boolean_set(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 1);

	} else if (strcasecmp(comp[ci].type, "telephone_isp") == 0) {
		sig_boolean_connect_out(sig_lookup(comp_id, "switch", 0), cpssp, 0);
		sig_boolean_set(sig_lookup(comp_id, "switch", 0), cpssp, 1);

	} else if (strcasecmp(comp[ci].type, "telephone_switch") == 0) {
		sig_boolean_connect_out(sig_lookup(comp_id, "switch", 0), cpssp, 0);
		sig_boolean_set(sig_lookup(comp_id, "switch", 0), cpssp, 1);
	}
}

int
system_comp_init(unsigned int comp_id)
{
	local_comp_init(comp_id);

	return 0;
}

static void
local_comp_exit(unsigned int comp_id)
{
	unsigned int ci;

	for (ci = 0; ; ci++) {
		if (ci == sizeof(comp) / sizeof(comp[0])) {
			/* Component not found. */
			assert(0); /* FIXME */
			return;
		}
		if (! comp[ci].type
		 || ! comp[ci].name) {
			continue;
		}
		if (comp[ci].id == comp_id) {
			/* Component found. */
			break;
		}
	}

	/* Nothing to do, jet. */
}

int
system_comp_exit(unsigned int comp_id)
{
	local_comp_exit(comp_id);

	return 0;
}

static void
local_comp_destroy(unsigned int comp_id)
{
	unsigned int ci;
	unsigned int i;

	for (ci = 0; ; ci++) {
		if (ci == sizeof(comp) / sizeof(comp[0])) {
			/* Component not found. */
			assert(0); /* FIXME */
			return;
		}
		if (comp[ci].id == comp_id
		 && comp[ci].type
		 && comp[ci].name) {
			/* Component found. */
			break;
		}
	}

	for (i = 0; ; i++) {
		if (i == sizeof(comp_type) / sizeof(comp_type[0])) {
			fprintf(stderr, "%s\n", comp[ci].type);
			assert(0); /* FIXME */
		}
		if (strcasecmp(comp[ci].type, comp_type[i].type) == 0) {
			if (comp_type[i].destroy) {
				comp_type[i].destroy(comp[ci].ptr);
			}
			if (comp_type[i].gui_destroy) {
				comp_type[i].gui_destroy(comp[ci].gui_ptr);
			}
			if (comp_type[i].aui_destroy) {
				comp_type[i].aui_destroy(comp[ci].aui_ptr);
			}
			break;
		}
	}

	free(comp[ci].type);
	comp[ci].type = NULL;

	free(comp[ci].name);
	comp[ci].name = NULL;
}

int
system_comp_destroy(unsigned int comp_id)
{
	local_comp_destroy(comp_id);

	return 0;
}

int
system_comp_info(
	unsigned int comp_id,
	char *type,
	char *name,
	unsigned int *nodep,
	unsigned int *pagep
)
{
	unsigned int ci;

	for (ci = 0; ; ci++) {
		if (ci == sizeof(comp) / sizeof(comp[0])) {
			/* Component not found. */
			assert(0); /* FIXME */
			return -1;
		}
		if (! comp[ci].type
		 || ! comp[ci].name) {
			continue;
		}
		if (comp[ci].id == comp_id) {
			/* Component found. */
			break;
		}
	}

	strcpy(type, comp[ci].type);
	strcpy(name, comp[ci].name);
	*nodep = comp[ci].node;
	*pagep = comp[ci].page;

	return 0;
}

int
system_comp_generic_info(
	unsigned int comp_id,
	unsigned int generic_id,
	char *type,
	char *name,
	char *value
)
{
	unsigned int id;

	for (id = 0; ; id++) {
		if (id == sizeof(generic) / sizeof(generic[0])) {
			/* Not found. */
			return -1;
		}
		if (! generic[id].name
		 || ! generic[id].value) {
			continue;
		}
		if (generic[id].comp_id == comp_id) {
			if (generic_id == 0) {
				strcpy(type, generic[id].type);
				strcpy(name, generic[id].name);
				strcpy(value, generic[id].value);
				return 0;
			} else {
				generic_id--;
			}
		}
	}
}

int
system_comp_port_info(
	unsigned int comp_id,
	unsigned int port_id,
	char *port,
	unsigned int *sigidp
)
{
	unsigned int id;

	for (id = 0; ; id++) {
		if (id == sizeof(conn) / sizeof(conn[0])) {
			/* Not found. */
			return -1;
		}
		if (! conn[id].port) {
			continue;
		}
		if (conn[id].comp_id == comp_id) {
			if (port_id == 0) {
				strcpy(port, conn[id].port);
				*sigidp = conn[id].sig_id;
				return 0;
			} else {
				port_id--;
			}
		}
	}
}

int
system_arch_create(
	const char *type,
	const char *name,
	unsigned int node,
	unsigned int page
)
{
	return system_comp_create(type, name, node, page);
}

int
system_arch_generic_set(
	unsigned int id,
	const char *t,
	const char *n,
	const char *v
)
{
	return system_comp_generic_set(id, t, n, v);
}

int
system_arch_port_connect(
	unsigned int arch_id,
	const char *port,
	unsigned int sig_id
)
{
	return system_comp_port_connect(arch_id, port, sig_id);
}

int
system_arch_init(unsigned int id)
{
	return system_comp_init(id);
}

int
system_arch_exit(unsigned int id)
{
	return system_comp_exit(id);
}

int
system_arch_destroy(unsigned int id)
{
	return system_comp_destroy(id);
}

void
system_comp_connect(unsigned int comp_id, const char *port, unsigned int sig_id)
{
	struct sig_manage *sig_manage;
	void *sig_ptr;

	sig_manage = sig_lookup(comp_id, "manage", 0);
	assert(sig_manage);

	sig_ptr = system_sig_get(sig_id);
	assert(sig_ptr);

	sig_manage_connect_port(sig_manage, port, sig_ptr);

	/* FIXME */
	// ret = system_sig_unget(sig_ptr);
	// assert(0 <= ret);
}

void
system_comp_disconnect(unsigned int comp_id, const char *port)
{
	struct sig_manage *sig_manage;

	sig_manage = sig_lookup(comp_id, "manage", 0);
	assert(sig_manage);

	sig_manage_disconnect_port(sig_manage, port);
}

int
system_comp_lookup(const char *name)
{
	unsigned int id;

	for (id = 0; id < sizeof(comp) / sizeof(comp[0]); id++) {
		if (comp[id].name == NULL) {
			/* past last entry */
			return -1;
		}

		if (strcasecmp(comp[id].name, name) == 0) {
			return comp[id].id;
		}
	}

	return -1;
}

int
system_sig_lookup(const char *name)
{
	unsigned int si;

	for (si = 0; ; si++) {
		if (si == sizeof(sig) / sizeof(sig[0])) {
			/* Signal not found. */
			return -1;
		}

		if (sig[si].name
		 && strcasecmp(sig[si].name, name) == 0) {
			break;
		}
	}

	return sig[si].id;
}

int
system_port_lookup(
	const char *type,
	enum system_gender gender,
	unsigned int n,
	const char **comp_name,
	const char **port_name
)
{
	unsigned int conn_id;
	unsigned int comp_id;
	unsigned int sig_id;

	for (conn_id = 0; ; conn_id++) {
		if (conn_id == sizeof(conn) / sizeof(conn[0])) {
			return -1;
		}
		if (! conn[conn_id].port) {
			/* Empty slot. */
			continue;
		}
		if (conn[conn_id].gender != gender) {
			/* Wrong gender. */
			continue;
		}
		for (sig_id = 0; ; sig_id++) {
			assert(sig_id < sizeof(sig) / sizeof(sig[0]));
			if (! sig[sig_id].type
			 || ! sig[sig_id].name) {
				continue;
			}
			if (sig[sig_id].id == conn[conn_id].sig_id) {
				break;
			}
		}
		if (strcasecmp(sig[sig_id].type, type) != 0) {
			/* Wrong type. */
			continue;
		}
		for (comp_id = 0; ; comp_id++) {
			assert(comp_id < sizeof(comp) / sizeof(comp[0]));
			if (! comp[comp_id].type
			 || ! comp[comp_id].name) {
				continue;
			}
			if (comp[comp_id].id == conn[conn_id].comp_id) {
				break;
			}
		}

		if (n == 0) {
			*comp_name = comp[comp_id].name;
			*port_name = conn[conn_id].port;
			return 1;
		} else {
			n--;
		}
	}
}

int
system_port_connect(
	const char *comp_name0,
	const char *port_name0,
	const char *comp_name1,
	const char *port_name1
)
{
	unsigned int comp_id0;
	unsigned int comp_id1;
	unsigned int conn_id0;
	unsigned int conn_id1;
	unsigned int sig_id0;
	unsigned int sig_id1;
	unsigned int type_id;

	/*
	 * Lookup component 0.
	 */
	for (comp_id0 = 0; ; comp_id0++) {
		if (comp_id0 == sizeof(comp) / sizeof(comp[0])) {
			/* Component not found. */
			return -1;
		}
		if (! comp[comp_id0].type
		 || ! comp[comp_id0].name) {
			/* Empty slot. */
			continue;
		}
		if (strcasecmp(comp[comp_id0].name, comp_name0) == 0) {
			/* Component found. */
			break;
		}
	}

	/*
	 * Lookup component 1.
	 */
	for (comp_id1 = 0; ; comp_id1++) {
		if (comp_id1 == sizeof(comp) / sizeof(comp[0])) {
			/* Component not found. */
			return -1;
		}
		if (! comp[comp_id1].type
		 || ! comp[comp_id1].name) {
			/* Empty slot. */
			continue;
		}
		if (strcasecmp(comp[comp_id1].name, comp_name1) == 0) {
			/* Component found. */
			break;
		}
	}

	/*
	 * Lookup port 0.
	 */
	for (conn_id0 = 0; ; conn_id0++) {
		if (conn_id0 == sizeof(conn) / sizeof(conn[0])) {
			/* Connection not found. */
			return -1;
		}
		if (! conn[conn_id0].port) {
			/* Empty slot. */
			continue;
		}
		if (conn[conn_id0].comp_id == comp[comp_id0].id
		 && strcasecmp(conn[conn_id0].port, port_name0) == 0) {
			/* Connection found. */
			break;
		}
	}

	/*
	 * Lookup port 1.
	 */
	for (conn_id1 = 0; ; conn_id1++) {
		if (conn_id1 == sizeof(conn) / sizeof(conn[0])) {
			/* Connection not found. */
			return -1;
		}
		if (! conn[conn_id1].port) {
			/* Empty slot. */
			continue;
		}
		if (conn[conn_id1].comp_id == comp[comp_id1].id
		 && strcasecmp(conn[conn_id1].port, port_name1) == 0) {
			/* Connection found. */
			break;
		}
	}

	/*
	 * Lookup signal 0.
	 */
	for (sig_id0 = 0; ; sig_id0++) {
		if (sig_id0 == sizeof(sig) / sizeof(sig[0])) {
			/* Signal not found. */
			assert(0); /* Mustn't happen. */
		}
		if (! sig[sig_id0].type
		 || ! sig[sig_id0].name) {
			/* Empty slot. */
			continue;
		}
		if (sig[sig_id0].id == conn[conn_id0].sig_id) {
			/* Signal found. */
			break;
		}
	}

	/*
	 * Lookup signal 1.
	 */
	for (sig_id1 = 0; ; sig_id1++) {
		if (sig_id1 == sizeof(sig) / sizeof(sig[0])) {
			/* Signal not found. */
			assert(0); /* Mustn't happen. */
		}
		if (! sig[sig_id1].type
		 || ! sig[sig_id1].name) {
			/* Empty slot. */
			continue;
		}
		if (sig[sig_id1].id == conn[conn_id1].sig_id) {
			/* Signal found. */
			break;
		}
	}

	/*
	 * Lookup signal type.
	 */
	for (type_id = 0; ; type_id++) {
		if (type_id == sizeof(sig_type) / sizeof(sig_type[0])) {
			/* Signal type not found. */
			assert(0); /* Mustn't happen. */
		}
		if (strcasecmp(sig_type[type_id].type, sig[sig_id0].type) == 0) {
			/* Signal type found. */
			break;
		}
	}

	/*
	 * Check if signals connectable.
	 */
	if (conn[conn_id0].peer != -1
	 || conn[conn_id1].peer != -1) {
		/* Port already connected. */
		return -1;
	}
	if (strcasecmp(sig[sig_id0].type, sig[sig_id1].type) != 0) {
		/* Signals of different type. */
		return -1;
	}
	if ((conn[conn_id0].gender | conn[conn_id1].gender) != 0
	 && (conn[conn_id0].gender | conn[conn_id1].gender) != 3) {
		/* Wrong genders. */
		return -1;
	}

	/*
	 * Connect signals.
	 */
	assert(conn[conn_id0].sig_ptr); /* FIXME */
	assert(conn[conn_id1].sig_ptr); /* FIXME */
	assert(! conn[conn_id0].merge);
	assert(! conn[conn_id1].merge);
	assert(conn[conn_id0].peer == -1);
	assert(conn[conn_id1].peer == -1);

	conn[conn_id0].peer = conn_id1;
	conn[conn_id1].peer = conn_id0;

	conn[conn_id0].merge =
	conn[conn_id1].merge = (*sig_type[type_id].merge)(
			conn[conn_id0].sig_ptr, conn[conn_id1].sig_ptr);

	return 0;
}

int
system_port_disconnect(
	const char *comp_name0,
	const char *port_name0
)
{
	unsigned int comp_id0;
	unsigned int conn_id0;
	unsigned int conn_id1;
	unsigned int sig_id0;
	unsigned int type_id;

	/*
	 * Lookup component 0.
	 */
	for (comp_id0 = 0; ; comp_id0++) {
		if (comp_id0 == sizeof(comp) / sizeof(comp[0])) {
			/* Component not found. */
			return -1;
		}
		if (! comp[comp_id0].type
		 || ! comp[comp_id0].name) {
			/* Empty slot. */
			continue;
		}
		if (strcasecmp(comp[comp_id0].name, comp_name0) == 0) {
			/* Component found. */
			break;
		}
	}

	/*
	 * Lookup port 0.
	 */
	for (conn_id0 = 0; ; conn_id0++) {
		if (conn_id0 == sizeof(conn) / sizeof(conn[0])) {
			/* Connection not found. */
			return -1;
		}
		if (! conn[conn_id0].port) {
			/* Empty slot. */
			continue;
		}
		if (conn[conn_id0].comp_id == comp[comp_id0].id
		 && strcasecmp(conn[conn_id0].port, port_name0) == 0) {
			/* Connection found. */
			break;
		}
	}

	/*
	 * Lookup signal 0.
	 */
	for (sig_id0 = 0; ; sig_id0++) {
		if (sig_id0 == sizeof(sig) / sizeof(sig[0])) {
			/* Signal not found. */
			assert(0); /* Mustn't happen. */
		}
		if (! sig[sig_id0].type
		 || ! sig[sig_id0].name) {
			/* Empty slot. */
			continue;
		}
		if (sig[sig_id0].id == conn[conn_id0].sig_id) {
			/* Signal found. */
			break;
		}
	}

	/*
	 * Lookup signal type.
	 */
	for (type_id = 0; ; type_id++) {
		if (type_id == sizeof(sig_type) / sizeof(sig_type[0])) {
			/* Signal type not found. */
			assert(0); /* Mustn't happen. */
		}
		if (strcasecmp(sig_type[type_id].type, sig[sig_id0].type) == 0) {
			/* Signal type found. */
			break;
		}
	}

	/*
	 * Checks if signal disconnectable.
	 */
	if (conn[conn_id0].peer == -1) {
		/* Not connected. */
		return -1;
	}

	/*
	 * Disconnect.
	 */
	conn_id1 = conn[conn_id0].peer;
	assert(conn[conn_id1].peer == conn_id0);

	assert(conn_id0 < sizeof(conn) / sizeof(conn[0]));
	assert(conn_id1 < sizeof(conn) / sizeof(conn[0]));
	assert(conn[conn_id0].merge);
	assert(conn[conn_id1].merge);

	(*sig_type[type_id].split)(conn[conn_id0].merge);

	conn[conn_id0].merge = NULL;
	conn[conn_id0].peer = -1;
	conn[conn_id1].merge = NULL;
	conn[conn_id1].peer = -1;

	return 0;
}
