/*
 * Copyright (C) 2003-2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include "glue.h"

#include "misc_analog_rcsound.h"

#define COMP_(x) misc_analog_rcsound_ ## x

struct cpssp {
	struct sig_sound *port_snd_out_left;
	struct sig_sound *port_snd_out_right;
	unsigned int state_ton_in;
	unsigned int state_rc_in_0;
	unsigned int state_rc_in_1;
	unsigned int state_rc_in_2;
	unsigned int state_rc_in_3;
	unsigned char state_vol_in;

	unsigned long long state_time;
	unsigned int state_nsamples;
	int16_t state_sample[
		SIG_SOUND_RATE / SIG_SOUND_PROCESS_FREQ * SIG_SOUND_CHANNELS];
	int16_t state_last_sample;
};

static unsigned int
COMP_(std_to_01)(unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_0:
	case SIG_STD_LOGIC_L:
		return 0;

	case SIG_STD_LOGIC_Z:
	case SIG_STD_LOGIC_H:
	case SIG_STD_LOGIC_1:
		return 1;

	default:
		break;
	}

	return 0;
}

static void
COMP_(flush_audio)(struct cpssp *cpssp)
{
	sig_sound_samples_set(cpssp->port_snd_out_left, cpssp,
			cpssp->state_sample);
	sig_sound_samples_set(cpssp->port_snd_out_right, cpssp,
			cpssp->state_sample);
	cpssp->state_nsamples = 0;
}

static void
COMP_(update_snd_buf)(struct cpssp *cpssp, int16_t sample)
{
	const unsigned long long delta = TIME_HZ / SIG_SOUND_RATE;
	const unsigned long long now = time_virt();
	const int16_t last_val = cpssp->state_last_sample;

	/* TODO smooth wave via RC-members */

	while (cpssp->state_time + delta < now) {
		cpssp->state_sample[cpssp->state_nsamples] = last_val;
		cpssp->state_nsamples++;

		if (SIG_SOUND_RATE / SIG_SOUND_PROCESS_FREQ
		    <= cpssp->state_nsamples) {

			COMP_(flush_audio)(cpssp);
		}

		cpssp->state_time += delta;
	}

	if (cpssp->state_time < now) {
		cpssp->state_sample[cpssp->state_nsamples] = sample;
		cpssp->state_nsamples++;
		cpssp->state_time += delta;

		if (SIG_SOUND_RATE / SIG_SOUND_PROCESS_FREQ
		    <= cpssp->state_nsamples) {

			COMP_(flush_audio)(cpssp);
		}
	}

	cpssp->state_last_sample = sample;
}

static void
COMP_(ton_in_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp*)_cpssp;
	unsigned int v;
	int16_t sample = INT16_MIN;

	v = COMP_(std_to_01)(val);
	/* force sample to become positive if v is 1 */
	sample -= v;

	/* adjust volume */
	assert(cpssp->state_vol_in < 4);
	sample /= 4;
	sample *= cpssp->state_vol_in;

	COMP_(update_snd_buf)(cpssp, sample);
}

static void
COMP_(vol_in_change)(struct cpssp *cpssp, unsigned int val, unsigned int bit)
{
	unsigned int v = COMP_(std_to_01)(val);

	/* circuit: ___                      ___
	 * ton_in  -___---.----.----.----.---___-----.---- SPKR
	 *                |    |    |    |           |
	 *               | |  | |  | |  | |   (configurable capacity)
	 *               | |  | |  | |  | |
	 *                |    |    |    |
	 * volume_0  -----/    |    |    |
	 * volume_1  ----------/    |    |
	 * volume_2  ---------------/    |
	 * volume_3  --------------------/
	 *
	 *
	 * only one of volume_0 - volume_3 should be set.
	 */

	if (! v) {
		cpssp->state_vol_in = bit;
	}
}

static void
COMP_(rc_in_0_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	cpssp->state_rc_in_0 = val;
}

static void
COMP_(rc_in_1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	cpssp->state_rc_in_1 = val;
}

static void
COMP_(rc_in_2_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	cpssp->state_rc_in_2 = val;
}

static void
COMP_(rc_in_3_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	cpssp->state_rc_in_3 = val;
}

static void
COMP_(vol_in_0_set)(void *_cpssp, unsigned int val)
{
	COMP_(vol_in_change)((struct cpssp *)_cpssp, val, 0);
}

static void
COMP_(vol_in_1_set)(void *_cpssp, unsigned int val)
{
	COMP_(vol_in_change)((struct cpssp *)_cpssp, val, 1);
}

static void
COMP_(vol_in_2_set)(void *_cpssp, unsigned int val)
{
	COMP_(vol_in_change)((struct cpssp *)_cpssp, val, 2);
}

static void
COMP_(vol_in_3_set)(void *_cpssp, unsigned int val)
{
	COMP_(vol_in_change)((struct cpssp *)_cpssp, val, 3);
}


void *
COMP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_std_logic *port_ton_in,
	struct sig_std_logic *port_rc_in_0,
	struct sig_std_logic *port_rc_in_1,
	struct sig_std_logic *port_rc_in_2,
	struct sig_std_logic *port_rc_in_3,
	struct sig_std_logic *port_vol_in_0,
	struct sig_std_logic *port_vol_in_1,
	struct sig_std_logic *port_vol_in_2,
	struct sig_std_logic *port_vol_in_3,
	struct sig_sound *port_snd_out_left,
	struct sig_sound *port_snd_out_right
)
{
	static const struct sig_std_logic_funcs ton_in_funcs = {
		.std_logic_set = COMP_(ton_in_set)
	};
	static const struct sig_std_logic_funcs rc_in_0_funcs = {
		.std_logic_set = COMP_(rc_in_0_set)
	};
	static const struct sig_std_logic_funcs rc_in_1_funcs = {
		.std_logic_set = COMP_(rc_in_1_set)
	};
	static const struct sig_std_logic_funcs rc_in_2_funcs = {
		.std_logic_set = COMP_(rc_in_2_set)
	};
	static const struct sig_std_logic_funcs rc_in_3_funcs = {
		.std_logic_set = COMP_(rc_in_3_set)
	};
	static const struct sig_std_logic_funcs vol_in_0_funcs = {
		.std_logic_set = COMP_(vol_in_0_set)
	};
	static const struct sig_std_logic_funcs vol_in_1_funcs = {
		.std_logic_set = COMP_(vol_in_1_set)
	};
	static const struct sig_std_logic_funcs vol_in_2_funcs = {
		.std_logic_set = COMP_(vol_in_2_set)
	};
	static const struct sig_std_logic_funcs vol_in_3_funcs = {
		.std_logic_set = COMP_(vol_in_3_set)
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->state_time = time_virt();
	cpssp->state_nsamples = 0;
	memset(cpssp->state_sample, 0, sizeof(cpssp->state_sample));
	cpssp->state_last_sample = 0;

	/* Call */
	/* Out */
	cpssp->port_snd_out_left = port_snd_out_left;
	cpssp->port_snd_out_right = port_snd_out_right;

	/* In */
	cpssp->state_ton_in = 0;
	sig_std_logic_connect_in(port_ton_in, cpssp, &ton_in_funcs);

	cpssp->state_rc_in_0 = 0;
	sig_std_logic_connect_in(port_rc_in_0, cpssp, &rc_in_0_funcs);

	cpssp->state_rc_in_1 = 0;
	sig_std_logic_connect_in(port_rc_in_1, cpssp, &rc_in_1_funcs);

	cpssp->state_rc_in_2 = 0;
	sig_std_logic_connect_in(port_rc_in_2, cpssp, &rc_in_2_funcs);

	cpssp->state_rc_in_3 = 0;
	sig_std_logic_connect_in(port_rc_in_3, cpssp, &rc_in_3_funcs);

	cpssp->state_vol_in = 0;
	sig_std_logic_connect_in(port_vol_in_0, cpssp, &vol_in_0_funcs);
	sig_std_logic_connect_in(port_vol_in_1, cpssp, &vol_in_1_funcs);
	sig_std_logic_connect_in(port_vol_in_2, cpssp, &vol_in_2_funcs);
	sig_std_logic_connect_in(port_vol_in_3, cpssp, &vol_in_3_funcs);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
COMP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
