/* polyline.c  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* contains functions assoc. with line drawing, motion, and handling */ 
/*
 * $Log: polyline.c,v $
 * Revision 1.2  2000/12/06 20:56:03  moz
 * GPL stuff.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:31  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:30  moz
 * CVS Import
 *
 * Revision 1.15  2000/03/15 00:15:47  moz
 * Deal with disable_motion.
 *
 * Revision 1.14  2000/03/08 22:26:20  moz
 * Compile fixes.
 *
 * Revision 1.13  2000/02/18 21:18:47  moz
 * polyline.pic usage changed.
 *
 * Revision 1.12  2000/01/31 01:06:12  moz
 * Tidied code.
 *
 * Revision 1.11  2000/01/29 21:01:10  moz
 * Use get_nearest_point correctly.
 *
 * Revision 1.10  2000/01/28 16:44:11  moz
 * Use edit_view to stop droppings on other views.
 *
 * Revision 1.9  1999/11/15 02:05:45  moz
 * Use rounded trig.
 *
 * Revision 1.8  1999/05/23 20:28:34  moz
 * Remove line dropping.
 *
 * Revision 1.7  1999/05/22 02:52:34  moz
 * Don't jump to a point when rotating a regular polygon.
 *
 * Revision 1.6  1999/05/19 17:08:29  moz
 * 1.0 Checkin.
 *
 * Revision 1.5  1999/04/27 07:40:37  moz
 * Remove extraneous register_undo
 *
 * Revision 1.4  1999/04/27 04:11:49  moz
 * -Wall appeasement.
 *
 * Revision 1.3  1999/04/23 00:40:02  moz
 * redraw_view_window change.
 *
 * Revision 1.2  1999/04/08 02:06:57  moz
 * Added setting fillcolour for completeness.
 *
 * Revision 1.1  1999/03/30 00:05:48  moz
 * Initial revision
 *
 */    

/* see mouse_button.c for general behaviour */  
 
#include <math.h> 
#include "include/figurine.h"  
#include "include/extern.h"
 
static Boolean held = FALSE;
static ListEl *edited_point=NULL; 
static Object *cpoly=NULL; 
static View *edit_view=NULL; 
static int pcount=0; 
static long orig_x=0;
static long orig_y=0;

extern XPoint xp[];
extern XPoint hp[];

extern int disable_motion;

/* construct an arrow  */  
Arrow *
make_arrow(Arrow arrow)
{
	Arrow *a;

	a = (Arrow *)malloc(sizeof(Arrow));

	a->type = arrow.type;
	a->filled = arrow.filled;
	a->w = arrow.w;
	a->h = arrow.h;
	a->lw = arrow.lw;
	return a;
}

void 
polyline_button(BEvent *bev, View *view)
{
	Object *ob; 
	VPoint *p1,*p2; 
	long x,y; 
	List l; 
	long ax,ay; 

	if (bev->button==Button2) 
		return; 

	if (bev->button==Button3 && state.busy_drawing)
		{

		/* cancel the current line */ 
		if (!held) 
			XUngrabPointer(display,CurrentTime);
		
		toggle_polyline(view, bev->x, bev->y);
		
		if (cpoly->ob.polyline.points->next==NULL)
			{
			view->doc->o = trash_object(view->doc->o, &view->doc->lo,  cpoly);
			kill_object(cpoly);
			cpoly = NULL;
			};
			
		state.busy_drawing = FALSE;
		held = FALSE;
		if (cpoly!=NULL)
			send_redraw_object(view,cpoly);
		cpoly = NULL;
		edit_view=NULL; 
		pcount = 0;
		 
		}
	else if (bev->button==Button1)
		{
		if (!P_IN_DOC(bev->x, bev->y, view) && state.busy_drawing)
			{
			/* ok, cancel the particular line we're drawing  */
			state.busy_drawing = FALSE;
			pcount = 0;
			 
			if (cpoly->ob.polyline.points->next==NULL)
				{
				view->doc->o = trash_object(view->doc->o, &view->doc->lo,  cpoly);
				kill_object(cpoly);
				cpoly = NULL;
				
				};
			if (cpoly!=NULL)
				send_redraw_object(view,cpoly);
			cpoly = NULL;
			 
			if (!held)
				XUngrabPointer(display,CurrentTime);
			else 
				held = FALSE;
			
			SET_CLIP_WINDOW(blackxorgc,view->draw_window); 
 			toggle_polyline(view, bev->x, bev->y); 
			edit_view=NULL; 
			}
		else
			{
			Object *rob=NULL; 
			 
			switch (bev->type)
				{
				case BUTTON_HELD:
					if (state.busy_drawing)
						break; 
						 
					state.busy_drawing = TRUE;
					edit_view=view;
					held = TRUE;

					if (state.shift_down && !(state.current_icon==POLYGON && view->regular)
					    && get_nearest_point(view,view->doc->o, XP2D(bev->x,view),YP2D(bev->y,view),&x,&y,&rob))
						{
						if (view->guide_lines && view->guide_lines_displayed)
							{
							toggle_guidelines(view);
							view->guide_lines_displayed = FALSE;
							};
						orig_x = x;
						orig_y = y;
						bev->x = XD2P(x,view); 
						bev->y = YD2P(y,view); 
						mouse_x = bev->x;
						mouse_y = bev->y;
						XWarpPointer(display, None, view->draw_window->win, 0, 0, 1, 1, 
							 bev->x,bev->y);
						}
					else
						{	
						if (view->gridon)
							{
							orig_x = GXP2D(bev->x,view);
							orig_y = GYP2D(bev->y,view);
							}
						else
							{
							orig_x = XP2D(bev->x,view);
							orig_y = YP2D(bev->y,view);
							};	
						};

					toggle_polyline(view, bev->x, bev->y);

					/* add start of polyline !  */ 
					ob = (Object *)malloc(sizeof(Object));
					p1 = (VPoint *)malloc(sizeof(VPoint));
					p1->x = 0;
					p1->y = 0;
					p1->derried = rob!=NULL;
					ob->bbox.x1 = orig_x;
					ob->bbox.y1 = orig_y;
					ob->bbox.x2 = orig_x;
					ob->bbox.y2 = orig_y;
					ob->ls = view->linestyle; 
					ob->lw = view->linewidth; 
					ob->js = view->joinstyle;
					ob->es = view->endstyle;
					if (view->barrow_on)
						ob->barrow = make_arrow(view->barrow);
					else
						ob->barrow = NULL;
					if (view->farrow_on)
						ob->farrow = make_arrow(view->farrow);
					else
						ob->farrow = NULL;
					ob->colour = view->colour; 
					ob->fillcolour = view->fillcolour; 
					if (state.current_icon==POLYGONICON)
						ob->type = POLYGON;
					else
						ob->type = POLYLINE;
					ob->fs = view->fillstyle; 
					ob->ticket = ob_ticket++; 
					ob->derries = NULL; 
					if (rob!=NULL)
						add_derry(rob,ob,p1);
					ob->depth = view->doc->ob_depth--; 
					 
					ob->ob.polyline.points = create_list(0,0,(void *)p1);
					ob->ob.polyline.pic = NULL;
					cpoly = ob; 
					pcount = 1; 

					view->doc->o = add_object(view->doc->o, &view->doc->lo,  ob);
					register_undo(UNDO_PASTE,ob,view->doc); 
					
					/* start of drawing  */  

					break; /* BUTTON_HELD  */ 

				case BUTTON_CLICKED:
					if (held)
						{
						XGrabPointer(display, view->draw_window->win, False, 
							ButtonPressMask | ButtonReleaseMask | 
							PointerMotionMask | ButtonMotionMask | 
							Button1MotionMask, GrabModeAsync, 
							GrabModeAsync, None, None, CurrentTime); 
						}
					else
						{
						VRegion bb; 

						toggle_polyline(view, bev->x, bev->y);
						 
						/* add line to existing object */ 
						bb.x1 = orig_x;
						bb.y1 = orig_y;
						if (state.shift_down && !(state.current_icon==POLYGON && view->regular)
						    && get_nearest_point(view,view->doc->o, XP2D(bev->x,view),YP2D(bev->y,view),&x,&y,&rob))
							{
							if (view->guide_lines && view->guide_lines_displayed)
								{
								toggle_guidelines(view);
								view->guide_lines_displayed = FALSE;
								};
							bb.x2 = x;
							bb.y2 = y;
							bev->x = XD2P(x,view); 
							bev->y = YD2P(y,view); 
							mouse_x = bev->x;
							mouse_y = bev->y;
							XWarpPointer(display, None, view->draw_window->win, 0, 0, 1, 1, 
								 	 bev->x,bev->y);
							}
						else 
							{ 
							if (view->gridon) 
								{ 
								bb.x2 = GXP2D(bev->x,view);
								bb.y2 = GYP2D(bev->y,view);
								} 
							else
								{
								bb.x2 = XP2D(bev->x,view);
								bb.y2 = YP2D(bev->y,view);
								}; 
							};
						p1 = (VPoint *)malloc(sizeof(VPoint));
						/* constrain to 90 degrees  */
						if (state.control_down)
							constrain_line(orig_x, orig_y, &bb.x2, &bb.y2);

						p1->x = bb.x2;
						p1->y = bb.y2;
						p1->derried = rob!=NULL;
						orig_x = p1->x;
						orig_y = p1->y;
						normalise_rectangle(&bb.x1,&bb.y1,&bb.x2, &bb.y2);
						l = cpoly->ob.polyline.points;

						/* unrelavitise coordinates before bb merge  */  
						while (l != NULL)
							{
							((VPoint *)l->data)->x = ((VPoint *)l->data)->x+cpoly->bbox.x1;
							((VPoint *)l->data)->y = ((VPoint *)l->data)->y+cpoly->bbox.y1;
							l = l->next; 
							}; 

						cpoly->bbox = merge_boxes(cpoly->bbox,bb); 
						view->doc->o = change_bbox(view->doc->o, cpoly); 
						cpoly->ob.polyline.points = add_to_list(cpoly->ob.polyline.points, (ulong)pcount, 0, (void *)p1); 
						pcount++; 
					
						l = cpoly->ob.polyline.points;
						while (l != NULL)
							{
							((VPoint *)l->data)->x = ((VPoint *)l->data)->x-cpoly->bbox.x1;
							((VPoint *)l->data)->y = ((VPoint *)l->data)->y-cpoly->bbox.y1;
							l = l->next; 
							}; 

						bb.x2++; bb.y2++; 
						 
						if (rob!=NULL)
							add_derry(rob,cpoly,p1);
 						send_redraw_object(view,cpoly); 
						/* carry on drawing  */  
						toggle_polyline(view,bev->x, bev->y);
						  
					
						}; 
					 
					held = FALSE; 
					break;
					 
					  
				case BUTTON_RELEASED:
					toggle_polyline(view, bev->x, bev->y);
					 
					if (held)
						{
						VRegion bb; 
						 
						/* end drawing  */ 
						state.busy_drawing = FALSE;
						held = FALSE;
						 
						if (state.shift_down && !(state.current_icon==POLYGON && view->regular)
						    && get_nearest_point(view,view->doc->o, XP2D(bev->x,view),YP2D(bev->y,view),&ax,&ay,&rob))
							{
							if (view->guide_lines && view->guide_lines_displayed)
								{
								toggle_guidelines(view);
								view->guide_lines_displayed = FALSE;
								};
							x = ax;
							y = ay;
							bev->x = XD2P(ax,view); 
							bev->y = YD2P(ay,view); 
							mouse_x = bev->x;
							mouse_y = bev->y;
							XWarpPointer(display, None, view->draw_window->win, 0, 0, 1, 1, 
							 	 bev->x,bev->y);
							}
						else
							{
							if (view->gridon) 
								{ 
								x = GXP2D(bev->x,view);
								y = GYP2D(bev->y,view);
								}
							else
								{
								x = XP2D(bev->x,view);
								y = YP2D(bev->y,view);
								};
							}; 
							 
						if (state.control_down)
							constrain_line(orig_x, orig_y, &x, &y);

						p2 = (VPoint *)malloc(sizeof(VPoint));
						p2->x = x;
						p2->y = y;
						p2->derried = rob!=NULL;
						 
						normalise_rectangle(&orig_x,&orig_y,&x, &y);
						bb.x1 = orig_x;
						bb.y1 = orig_y;
						bb.x2 = x;
						bb.y2 = y;
						
						l = cpoly->ob.polyline.points;
						 
						while (l != NULL)
							{
							((VPoint *)l->data)->x = ((VPoint *)l->data)->x+cpoly->bbox.x1;
							((VPoint *)l->data)->y = ((VPoint *)l->data)->y+cpoly->bbox.y1;
							l = l->next; 
							};

						if (state.current_icon==POLYGONICON)
							cpoly->type = POLYGON;
						else
							cpoly->type = POLYLINE;
							 
						cpoly->fs = view->fillstyle; 
						cpoly->ticket = ob_ticket++;
						cpoly->derries = NULL; 
						cpoly->depth = view->doc->ob_depth--; 
						cpoly->ls = view->linestyle; 
						cpoly->lw = view->linewidth; 
						cpoly->js = view->joinstyle;
						cpoly->es = view->endstyle;
						if (view->barrow_on)
							cpoly->barrow = make_arrow(view->barrow);
						else
							cpoly->barrow = NULL;
						if (view->farrow_on)
							cpoly->farrow = make_arrow(view->farrow);
						else
							cpoly->farrow = NULL;
						cpoly->colour = view->colour; 
						cpoly->fillcolour = view->fillcolour; 
						cpoly->bbox = merge_boxes(cpoly->bbox,bb); 
						view->doc->o = change_bbox(view->doc->o, cpoly); 
						 
						cpoly->ob.polyline.points = add_to_list(cpoly->ob.polyline.points, (ulong)pcount, 0, (void *)p2); 
						cpoly->ob.polyline.pic = NULL;

						l = cpoly->ob.polyline.points;
						 
						while (l != NULL)
							{
							((VPoint *)l->data)->x = ((VPoint *)l->data)->x-cpoly->bbox.x1;
							((VPoint *)l->data)->y = ((VPoint *)l->data)->y-cpoly->bbox.y1;
							l = l->next; 
							};  
						if (rob!=NULL)
							add_derry(rob,cpoly,p2);
 						send_redraw_object(view,cpoly);
						/* no more drawing  */ 
						edit_view=NULL; 
						cpoly = NULL; pcount = 0; 
						}
					else
						{
						VRegion bb; 

						toggle_polyline(view, bev->x, bev->y);
						 
						/* add line to existing object */ 
						bb.x1 = orig_x;
						bb.y1 = orig_y;
						if (state.shift_down && !(state.current_icon==POLYGON && view->regular)
						    && get_nearest_point(view,view->doc->o, XP2D(bev->x,view),YP2D(bev->y,view),&ax,&ay,&rob))
							{
							if (view->guide_lines && view->guide_lines_displayed)
								{
								toggle_guidelines(view);
								view->guide_lines_displayed = FALSE;
								};
							bb.x2 = ax;
							bb.y2 = ay;
							bev->x = XD2P(ax,view); 
							bev->y = YD2P(ay,view); 
							mouse_x = bev->x;
							mouse_y = bev->y;
							XWarpPointer(display, None, view->draw_window->win, 0, 0, 1, 1, 
							 	 bev->x,bev->y);
							} 
						else
							{
							if (view->gridon) 
								{ 
								bb.x2 = GXP2D(bev->x,view);
								bb.y2 = GYP2D(bev->y,view);
								}
							else
								{
								bb.x2 = XP2D(bev->x,view);
								bb.y2 = YP2D(bev->y,view);
								};
							}; 
						ax = bb.x2;
						ay = bb.y2;
						p1 = (VPoint *)malloc(sizeof(VPoint));
						p1->x = bb.x2; 
						p1->y = bb.y2;
						p1->derried = rob!=NULL;
						
						l = cpoly->ob.polyline.points;
						 
						while (l != NULL)
							{
							((VPoint *)l->data)->x = ((VPoint *)l->data)->x+cpoly->bbox.x1;
							((VPoint *)l->data)->y = ((VPoint *)l->data)->y+cpoly->bbox.y1;
							l = l->next; 
							};   
							 
						normalise_rectangle(&bb.x1,&bb.y1,&bb.x2, &bb.y2);
						cpoly->bbox = merge_boxes(cpoly->bbox,bb); 
						view->doc->o = change_bbox(view->doc->o, cpoly); 
						cpoly->ob.polyline.points = add_to_list(cpoly->ob.polyline.points, (ulong)pcount, 0, (void *)p1); 
						pcount++; 
					
						l = cpoly->ob.polyline.points;
						 
						while (l != NULL)
							{
							((VPoint *)l->data)->x = ((VPoint *)l->data)->x-cpoly->bbox.x1;
							((VPoint *)l->data)->y = ((VPoint *)l->data)->y-cpoly->bbox.y1;
							l = l->next; 
							};   
							 
						if (rob!=NULL)
							add_derry(rob,cpoly,p1);
 						send_redraw_object(view,cpoly); 
						/* carry on drawing  */  
						orig_x = ax;
						orig_y = ay;
						toggle_polyline(view,bev->x, bev->y);
						};

					break;
				};
			}; 
		}; 
}

void 
toggle_polyline(View *view, int x, int y)
{
	VLine *pl;
	VLine line;
	Boolean clipped; 

	if (view!=edit_view)
		return;

 	line.x1 = XD2P(orig_x,view);
	line.y1 = YD2P(orig_y,view);
	 
	if (view->gridon)
		{
		line.x2 = XD2P(GXP2D(x,view),view);
		line.y2 = YD2P(GYP2D(y,view),view);
		}
	else
		{
		line.x2 = x;
		line.y2 = y;
		};

	/* constrain to 90 degrees  */
	if (state.control_down)
		constrain_line(line.x1,line.y1,&line.x2,&line.y2); 

	pl = clip_line(0,0,(int)view->draw_window->w,(int)view->draw_window->h, &line, &clipped, 1); 
		 
  	if (pl!=NULL)    
		XDrawLine(display,view->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
}

/* expects doc coord, x, y is a point RELATIVE to object origin */  
ListEl *
polyline_nearest_point(List points, long x, long y, double *nearest)
{
	ListEl *n;
	double measure; 

	measure = sqrt((double)(abs((POINT(points)->x-x)*(POINT(points)->x-x))+
				abs((POINT(points)->y-y)*(POINT(points)->y-y))));
	n = points;
	
	while (points->next!=NULL)
		{
		points = points->next;

		if (sqrt((double)(abs((POINT(points)->x-x)*(POINT(points)->x-x))+abs((POINT(points)->y-y)*(POINT(points)->y-y)))) < measure)
			{ 
			measure = sqrt((double)(abs((POINT(points)->x-x)*(POINT(points)->x-x))+
						abs((POINT(points)->y-y)*(POINT(points)->y-y))));
			n = points;
			};
		};

	if (nearest!=NULL)
		*nearest = measure;

	return n;
}

void 
polyline_point_button(BEvent *bev, View *v)
{
	long x,y; 
	 
	if (bev->type==BUTTON_HELD)
		{
		if (!state.editing_point && !is_in_bbox(XP2D(bev->x,v),YP2D(bev->y,v),
			 v->edited_object->bbox.x1-P2D(HANDLE_PIXEL_SIZE,v),
			 v->edited_object->bbox.y1-P2D(HANDLE_PIXEL_SIZE,v), 
			 v->edited_object->bbox.x2+P2D(HANDLE_PIXEL_SIZE,v),
			 v->edited_object->bbox.y2+P2D(HANDLE_PIXEL_SIZE,v)))
			{
			if (state.editing_point) 
				{ 
				toggle_point_polyline(v,bev->x,bev->y); 
				state.editing_point = FALSE; 
				};
			send_redraw_object(v,v->edited_object->ob);
			v->edited_object = NULL;
			} 
		else if (!state.editing_point)
			{
			double near; 
			ListEl *n;

			n = polyline_nearest_point(v->edited_object->ob->ob.polyline.points, 
						XP2D(bev->x,v)-v->edited_object->ob->bbox.x1, 
						YP2D(bev->y,v)-v->edited_object->ob->bbox.y1, &near);

			if (D2P(near,v)<POINTER_JUMP_SIZE)
				{ 
				XWarpPointer(display, None, v->draw_window->win, 0, 0, 1, 1, 
								 XD2P(POINT(n)->x+v->edited_object->ob->bbox.x1,v), 
								 YD2P(POINT(n)->y+v->edited_object->ob->bbox.y1,v));
				state.editing_point = TRUE;
				edited_point = n; 
				toggle_point_polyline(v,bev->x, bev->y);
				orig_x = bev->x;
				orig_y = bev->y;
				};
			}
		else if (bev->button == Button3)
			{
			toggle_point_polyline(v,bev->x, bev->y);
			state.editing_point = FALSE;
			}; 
		}
	else
		{ /* CLICKED or RELEASED  */ 
		/* if released, place it and redraw object  */  
		if (bev->type == BUTTON_RELEASED && state.editing_point && P_IN_DOC(bev->x,bev->y,v))
			{
			Object *rob=NULL;
			Object *s=NULL; 

			toggle_point_polyline(v,bev->x, bev->y);
  			register_undo(UNDO_OB_PROP,v->edited_object->ob,v->doc);  

			if (state.shift_down && !(state.current_icon==POLYGON && v->regular)
			    && get_nearest_point(v,v->doc->o, XP2D(bev->x,v),YP2D(bev->y,v),&x,&y,&rob))
				{
				if (v->guide_lines && v->guide_lines_displayed)
					{
					toggle_guidelines(v);
					v->guide_lines_displayed = FALSE;
					};
				POINT(edited_point)->x = x-v->edited_object->ob->bbox.x1; 
				POINT(edited_point)->y = y-v->edited_object->ob->bbox.y1; 
				bev->x = XD2P(x,v); 
				bev->y = YD2P(y,v); 
				mouse_x = bev->x;
				mouse_y = bev->y;
				XWarpPointer(display, None, v->draw_window->win, 0, 0, 1, 1, 
				 	 bev->x,bev->y);
				} 
			else
				{
				if (v->gridon)
					{
					POINT(edited_point)->x = GXP2D(bev->x, v)-v->edited_object->ob->bbox.x1;
					POINT(edited_point)->y = GYP2D(bev->y, v)-v->edited_object->ob->bbox.y1;
					}
				else
					{
					POINT(edited_point)->x = XP2D(bev->x, v)-v->edited_object->ob->bbox.x1;
					POINT(edited_point)->y = YP2D(bev->y, v)-v->edited_object->ob->bbox.y1;
					}; 
				};
			if (POINT(edited_point)->derried)
				remove_derry(v,v->edited_object->ob,POINT(edited_point));
			POINT(edited_point)->derried = rob!=NULL;
			if (rob!=NULL)
				add_derry(rob,v->edited_object->ob,POINT(edited_point));
			store_redraw_object(v->edited_object->ob); 
			recalc_polyline_bbox(v->edited_object->ob, FALSE);	 
			s = v->edited_object->ob; 
			v->doc->o = change_bbox(v->doc->o, v->edited_object->ob);	
			v->edited_object = get_object_node(v->doc->o, s);
			send_stored_redraw_object(v,v->edited_object->ob); 
			state.editing_point = FALSE; 
			}
		else if (state.editing_point)
			{
			toggle_point_polyline(v,bev->x,bev->y);
			state.editing_point = FALSE;
			};
		}; 
}

void 
polyline_point_motion(View *v, long x, long y)
{
	long tx, ty;
	 
	/* if drawing, resize as appropriate */  
	if (state.editing_point)
		{
		Object *rob=NULL; 
		int test=-1,test2=-1; /* func returns -1 if no change */
		 
		/* complex logic for after a zoom */ 
 		if (!disable_motion)
			toggle_point_polyline(v,orig_x,orig_y);
		else if (disable_motion==1)
			disable_motion--;
		else
			{
			disable_motion--;
			return;
			}; 
				  
		if (state.shift_down && !(state.current_icon==POLYGON && v->regular)
		    && get_nearest_point(v,v->doc->o, XP2D(x,v),YP2D(y,v),&tx,&ty,&rob))
			{
			if (v->guide_lines && v->guide_lines_displayed)
				{
				toggle_guidelines(v);
				v->guide_lines_displayed = FALSE;
				};
			 
			mouse_x = XD2P(tx,v);
			mouse_y = YD2P(ty,v);
			x = mouse_x;
			y = mouse_y;
			XWarpPointer(display, None, v->draw_window->win, 0, 0, 1, 1, 
			 	 mouse_x,mouse_y);
			}; 
							  
		orig_x = x;
		orig_y = y;
		 
		/* if nudged borders of window then scroll view window  */  
		if ( x < MOVE_BOUNDARY) /* move left */  
			test = nudge_ruler_x(v->ruler_x_window, v, LEFT);
		else if ( x >= ((int)v->draw_window->w)-MOVE_BOUNDARY) /* move right */ 
			test = nudge_ruler_x(v->ruler_x_window, v, RIGHT);
				
		if ( y < MOVE_BOUNDARY) /* move up */  
			test2 = nudge_ruler_y(v->ruler_y_window, v, UP); 
		else if ( y >= ((int)v->draw_window->h)-MOVE_BOUNDARY) /* move down */ 
			test2 = nudge_ruler_y(v->ruler_y_window, v, DOWN); 
				
		if (test==-1 && test2==-1) /* i.e. not scrolled  */ 
			{
			if (!disable_motion)
				toggle_point_polyline(v, x, y);
			else
				disable_motion--;
			}
		else
			{ 
			state.rubberon = FALSE; 
			redraw_view_window(v); 
			toggle_point_polyline(v,mouse_x,mouse_y); 
			};
		}; 
}

/* draw rubber line for point editing  */  
void 
toggle_point_polyline(View *v, long x, long y)
{
	List l;
	VLine line; VLine *pl; Boolean clipped; 

	if (edited_point->prev!=NULL)
		{
		if (v->gridon)
			{
			line.x1 = XD2P(POINT(edited_point->prev)->x+v->edited_object->ob->bbox.x1,v);
			line.y1 = YD2P(POINT(edited_point->prev)->y+v->edited_object->ob->bbox.y1,v);
			line.x2 = XD2P(GXP2D(x,v),v);
			line.y2 = YD2P(GYP2D(y,v),v);
			pl = clip_line(0,0,(int)v->draw_window->w,(int)v->draw_window->h,&line,&clipped, 2);
			if (pl!=NULL)
				XDrawLine(display,v->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
			}
		else
			{ 
			line.x1 = XD2P(POINT(edited_point->prev)->x+v->edited_object->ob->bbox.x1,v);
			line.y1 = YD2P(POINT(edited_point->prev)->y+v->edited_object->ob->bbox.y1,v);
			line.x2 = x;
			line.y2 = y;
			pl = clip_line(0,0,(int)v->draw_window->w,(int)v->draw_window->h,&line,&clipped, 2);
			if (pl!=NULL)
				XDrawLine(display,v->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
			};
		}
	else if (v->edited_object->ob->type == POLYGON)
		{
		l = v->edited_object->ob->ob.polyline.points;
		while (l->next!=NULL)
			l = l->next;
			 
		if (v->gridon)
			{
			line.x1 = XD2P(POINT(l)->x+v->edited_object->ob->bbox.x1,v);
			line.y1 = YD2P(POINT(l)->y+v->edited_object->ob->bbox.y1,v);
			line.x2 = XD2P(GXP2D(x,v),v);
			line.y2 = YD2P(GYP2D(y,v),v);
			pl = clip_line(0,0,(int)v->draw_window->w,(int)v->draw_window->h,&line,&clipped, 2);
			if (pl!=NULL)
				XDrawLine(display,v->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
			}
		else
			{ 
			line.x1 = XD2P(POINT(l)->x+v->edited_object->ob->bbox.x1,v);
			line.y1 = YD2P(POINT(l)->y+v->edited_object->ob->bbox.y1,v);
			line.x2 = x;
			line.y2 = y;
			pl = clip_line(0,0,(int)v->draw_window->w,(int)v->draw_window->h,&line,&clipped, 2);
			if (pl!=NULL)
				XDrawLine(display,v->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
			};
		};
	
	if (edited_point->next!=NULL)
		{
		if (v->gridon)
			{
			line.x1 = XD2P(POINT(edited_point->next)->x+v->edited_object->ob->bbox.x1,v);
			line.y1 = YD2P(POINT(edited_point->next)->y+v->edited_object->ob->bbox.y1,v);
			line.x2 = XD2P(GXP2D(x,v),v);
			line.y2 = YD2P(GYP2D(y,v),v);
			pl = clip_line(0,0,(int)v->draw_window->w,(int)v->draw_window->h,&line,&clipped, 2);
			if (pl!=NULL)
				XDrawLine(display,v->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
			}
		else
			{ 
			line.x1 = XD2P(POINT(edited_point->next)->x+v->edited_object->ob->bbox.x1,v);
			line.y1 = YD2P(POINT(edited_point->next)->y+v->edited_object->ob->bbox.y1,v);
			line.x2 = x;
			line.y2 = y;
			pl = clip_line(0,0,(int)v->draw_window->w,(int)v->draw_window->h,&line,&clipped, 2);
			if (pl!=NULL)
				XDrawLine(display,v->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
			};
		}
	else if (v->edited_object->ob->type==POLYGON) /* next is null, wrap round to start  */ 
		{
		if (v->gridon)
			{
			line.x1 = XD2P(POINT(v->edited_object->ob->ob.polyline.points)->x+v->edited_object->ob->bbox.x1,v);
			line.y1 = YD2P(POINT(v->edited_object->ob->ob.polyline.points)->y+v->edited_object->ob->bbox.y1,v);
			line.x2 = XD2P(GXP2D(x,v),v);
			line.y2 = YD2P(GYP2D(y,v),v);
			pl = clip_line(0,0,(int)v->draw_window->w,(int)v->draw_window->h,&line,&clipped, 2);
			if (pl!=NULL)
				XDrawLine(display,v->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
			}
		else
			{ 
			line.x1 = XD2P(POINT(v->edited_object->ob->ob.polyline.points)->x+v->edited_object->ob->bbox.x1,v);
			line.y1 = YD2P(POINT(v->edited_object->ob->ob.polyline.points)->y+v->edited_object->ob->bbox.y1,v);
			line.x2 = x;
			line.y2 = y;
			pl = clip_line(0,0,(int)v->draw_window->w,(int)v->draw_window->h,&line,&clipped, 2);
			if (pl!=NULL)
				XDrawLine(display,v->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
			};
		};
}

/* recalculates the bounding box of an object.
	If a spline, the bbox must also include the final
	curve, contained in ob->ob.spline.cache */  
void 
recalc_polyline_bbox(Object *ob, Boolean is_spline)
{
	List l;
	long lx;
	long ly;
	long hx;
	long hy;

	if (!is_spline)
		{
		l = ob->ob.polyline.points;

		POINT(l)->x = POINT(l)->x + ob->bbox.x1; 
		POINT(l)->y = POINT(l)->y + ob->bbox.y1; 
		lx = POINT(l)->x; 
		ly = POINT(l)->y; 
		hx = POINT(l)->x; 
		hy = POINT(l)->y; 
		 
		/* determine lowest and highest  */  
		/* we unrelativise now, then relativise again next  */ 
		l = l->next;  
		while (l!=NULL)
			{
			POINT(l)->x = POINT(l)->x + ob->bbox.x1; 
			POINT(l)->y = POINT(l)->y + ob->bbox.y1; 
			lx = min(lx, POINT(l)->x);
			ly = min(ly, POINT(l)->y);
			hx = max(hx, POINT(l)->x);
			hy = max(hy, POINT(l)->y);
			l = l->next;
			};
		}
	else
		{
		l = ob->ob.spline.points;

		SPOINT(l)->x = SPOINT(l)->x + ob->bbox.x1; 
		SPOINT(l)->y = SPOINT(l)->y + ob->bbox.y1; 
		lx = SPOINT(l)->x; 
		ly = SPOINT(l)->y; 
		hx = SPOINT(l)->x; 
		hy = SPOINT(l)->y; 
		 
		/* determine lowest and highest  */  
		/* we unrelativise now, then relativise again next  */ 
		l = l->next;  
		while (l!=NULL)
			{
			SPOINT(l)->x = SPOINT(l)->x + ob->bbox.x1; 
			SPOINT(l)->y = SPOINT(l)->y + ob->bbox.y1; 
			lx = min(lx, SPOINT(l)->x);
			ly = min(ly, SPOINT(l)->y);
			hx = max(hx, SPOINT(l)->x);
			hy = max(hy, SPOINT(l)->y);
			l = l->next;
			};
		};

	if (is_spline)
		{ 
		l = ob->ob.spline.cache;

		POINT(l)->x = POINT(l)->x + ob->bbox.x1; 
		POINT(l)->y = POINT(l)->y + ob->bbox.y1; 
		lx = min(lx,POINT(l)->x); 
		ly = min(ly,POINT(l)->y); 
		hx = max(hx,POINT(l)->x); 
		hy = max(hy,POINT(l)->y); 
		 
			/* determine lowest and highest of control points */  
			/* we unrelativise now, then relativise again next  */ 
		l = l->next;  
		while (l!=NULL)
			{
			POINT(l)->x = POINT(l)->x + ob->bbox.x1; 
			POINT(l)->y = POINT(l)->y + ob->bbox.y1; 
			lx = min(lx, POINT(l)->x);
			ly = min(ly, POINT(l)->y);
			hx = max(hx, POINT(l)->x);
			hy = max(hy, POINT(l)->y);
			l = l->next;
			};
		};

	ob->bbox.x1 = lx;
	ob->bbox.y1 = ly;
	ob->bbox.x2 = hx;
	ob->bbox.y2 = hy;
	
	/* now must recalculate relative coordinates */ 

	if (is_spline)
		{ 
		l = ob->ob.spline.points;

		while (l!=NULL)
			{
			SPOINT(l)->x = SPOINT(l)->x - ob->bbox.x1;
			SPOINT(l)->y = SPOINT(l)->y - ob->bbox.y1;
			l = l->next;
			};
		}
	else
		{
		l = ob->ob.polyline.points;

		while (l!=NULL)
			{
			POINT(l)->x = POINT(l)->x - ob->bbox.x1;
			POINT(l)->y = POINT(l)->y - ob->bbox.y1;
			l = l->next;
			};
		};
		 
	if (is_spline)
		{ 
		l = ob->ob.spline.cache;

		while (l!=NULL)
			{
			POINT(l)->x = POINT(l)->x - ob->bbox.x1;
			POINT(l)->y = POINT(l)->y - ob->bbox.y1;
			l = l->next;
			};
		}; 
}	

/* this attempts to draw as many points in one XDrawPoints as possible, for showing join style  */  
void 
draw_polyline(Object *ob, View *view, GC gc, long x, long y, double rx, double ry, double angle)
{
	int xpp=0,hpp;
	List l;
	VLine line; 
	VLine *pl; 
	Boolean clipped=FALSE; 
	
	if (state.tied_corner!=NOTSCALING)
		corner_magic(ob, &x, &y, rx, ry);

	rx = abs(rx);
	ry = abs(ry);
	
	if (angle!=0.0) /* ************* Rotating  */ 
		{
		double sina,cosa; 
		long cx, cy;
		sina = sinround(angle);
		cosa = cosround(angle);
		cx = (ob->bbox.x2-ob->bbox.x1)/2;
		cy = (ob->bbox.y2-ob->bbox.y1)/2;
		 
		l = ob->ob.polyline.points; 

		xpp=0; 
		l = ob->ob.polyline.points; 
		line.x1 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(l)->x-cx)),view) - D2P((sina*(POINT(l)->y-cy)),view));
		line.y1 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l)->x-cx)),view) + D2P((cosa*(POINT(l)->y-cy)),view));
		xp[xpp].x = line.x1;
		xp[xpp].y = line.y1;
		xpp++;

		while (l->next!=NULL)
			{
			line.x1 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(l)->x-cx)),view) - D2P((sina*(POINT(l)->y-cy)),view));
			line.y1 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l)->x-cx)),view) + D2P((cosa*(POINT(l)->y-cy)),view));
			line.x2 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(l->next)->x-cx)),view) - D2P((sina*(POINT(l->next)->y-cy)),view));
			line.y2 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l->next)->x-cx)),view) + D2P((cosa*(POINT(l->next)->y-cy)),view));
			pl = clip_line(0,0,(int)view->draw_window->w,(int)view->draw_window->h, &line, &clipped, 2*ob->lw); 
			if (pl!=NULL)   
				{
				xp[xpp].x = pl->x2;
				xp[xpp].y = pl->y2;
				xpp++;
				};
			
			if (clipped || xpp==MAXPOINTS)
				{
				XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
				/* continue from last line  */  
				xp[0].x = XD2P(x,view) + D2P(cx,view) +  (D2P((cosa*(POINT(l->next)->x-cx)),view) - D2P((sina*(POINT(l->next)->y-cy)),view));
				xp[0].y = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l->next)->x-cx)),view) + D2P((cosa*(POINT(l->next)->y-cy)),view));
				xpp=1;
				};
			l = l->next;
			};
			 
		if (xpp!=0)
			XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
			
		/* close a polygon  */
		if (ob->type==POLYGON)
			{
			l = ob->ob.polyline.points;
			while (l->next!=NULL)
				l = l->next;

			line.x1 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(l)->x-cx)),view) - D2P((sina*(POINT(l)->y-cy)),view));
			line.y1 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l)->x-cx)),view) + D2P((cosa*(POINT(l)->y-cy)),view));
			line.x2 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(ob->ob.polyline.points)->x-cx)),view) - D2P((sina*(POINT(ob->ob.polyline.points)->y-cy)),view));
			line.y2 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(ob->ob.polyline.points)->x-cx)),view) + D2P((cosa*(POINT(ob->ob.polyline.points)->y-cy)),view));
			pl = clip_line(0,0,(int)view->draw_window->w,(int)view->draw_window->h, &line, &clipped, 2*ob->lw); 
			 
			if (pl!=NULL)   
				XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
			}; 
		}
	else  /* scaling or normal drawing */ 
		{
		l = ob->ob.polyline.points; 
		hpp=0;
		while (l!=NULL && hpp<MAXPOINTS)
			{
			hp[hpp].x = XD2P(POINT(l)->x+x,view);
			hp[hpp].y = YD2P(POINT(l)->y+y,view);
			hpp++;
			l = l->next;
			};

		if (ob->type==POLYGON && ob->fs!=NONE && gc!=blackxorgc)  
			XFillPolygon(display,view->draw_window->win,fillgc, hp, hpp, Complex, CoordModeOrigin);    

		if (gc==blackxorgc || ob->lw!=0)
			{
			xpp=0; 
			l = ob->ob.polyline.points; 
			line.x1 = XD2P(R(POINT(l)->x,rx)+x,view);
			line.y1 = YD2P(R(POINT(l)->y,ry)+y,view);
			xp[xpp].x = line.x1;
			xp[xpp].y = line.y1;
			xpp++;

			while (l->next!=NULL)
				{
				line.x1 = XD2P(R(POINT(l)->x,rx)+x,view);
				line.y1 = YD2P(R(POINT(l)->y,ry)+y,view);
				line.x2 = XD2P(R(POINT(l->next)->x,rx)+x,view);
				line.y2 = YD2P(R(POINT(l->next)->y,ry)+y,view);
				pl = clip_line(0,0,(int)view->draw_window->w,(int)view->draw_window->h, &line, &clipped, 2*ob->lw); 
				if (pl!=NULL)   
					{
					xp[xpp].x = pl->x2;
					xp[xpp].y = pl->y2;
					xpp++;
					};
				
				if (clipped || xpp==MAXPOINTS)
					{
					XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
					/* continue from last line  */  
					xp[0].x = XD2P(R(POINT(l->next)->x,rx)+x,view);
					xp[0].y = YD2P(R(POINT(l->next)->y,ry)+y,view);
					xpp=1;
					};
				l = l->next;
				};
				 
			if (xpp!=0)
				{
				if (ob->type==POLYGON)
					{
					/* close polygon */  
					l = ob->ob.polyline.points;
					while (l->next!=NULL)
						l = l->next;

					if (xpp==MAXPOINTS)
						{
						XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
						/* continue from last line  */  
						xpp=0;
						};
						 
					line.x1 = XD2P(R(POINT(l)->x,rx)+x,view);
					line.y1 = YD2P(R(POINT(l)->y,ry)+y,view);
					line.x2 = XD2P(R(POINT(ob->ob.polyline.points)->x,rx)+x,view);
					line.y2 = YD2P(R(POINT(ob->ob.polyline.points)->y,ry)+y,view);
					pl = clip_line(0,0,(int)view->draw_window->w,(int)view->draw_window->h, &line, &clipped, 2*ob->lw); 
					if (pl!=NULL)
						{
						xp[xpp].x = pl->x1;
						xp[xpp].y = pl->y1;
						xpp++;
						xp[xpp].x = pl->x2;
						xp[xpp].y = pl->y2;
						xpp++; 
						}; 
					}; 
				XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
				}; 
				
			if (ob->barrow!=NULL && rx==1.0 && ry==1.0 && cpoly!=ob)
				{
				long x1,y1,x2,y2;

				x2 = XD2P(POINT(ob->ob.polyline.points)->x+x,view); 
				y2 = YD2P(POINT(ob->ob.polyline.points)->y+y,view); 
				if (ob->ob.polyline.points->next!=NULL)
					{
					x1 = XD2P(POINT(ob->ob.polyline.points->next)->x+x,view); 
					y1 = YD2P(POINT(ob->ob.polyline.points->next)->y+y,view); 
					}
				else
					{
					x1 = XD2P(POINT(ob->ob.polyline.points)->x+x,view); 
					y1 = YD2P(POINT(ob->ob.polyline.points)->y+y,view); 
					};
				draw_arrow(view,gc,ob->barrow,x1,y1,x2,y2);
				};

			if (ob->farrow!=NULL && rx==1.0 && ry==1.0 && cpoly!=ob)
				{
				/* init to zero to appease -Wall  */  
				long x1=0,y1=0,x2,y2;

				l = ob->ob.polyline.points;

				while (l->next!=NULL)
					{
					if (l->next->next==NULL)
						{
						x1 = XD2P(POINT(l)->x+x,view);
						y1 = YD2P(POINT(l)->y+y,view); 
						}; 
					l = l->next;
					};

				x2 = XD2P(POINT(l)->x+x,view);
				y2 = YD2P(POINT(l)->y+y,view); 
	
				draw_arrow(view,gc,ob->farrow,x1,y1,x2,y2);
				};
				
		/* handles after everything else  */  
		if (view->edited_object!=NULL && view->edited_object->ob == ob)
			{
			int i;
			for (i=0; i<hpp; i++)
				{
				if (P_IN_DOC(hp[i].x,hp[i].y,view))
					draw_handle(view, hp[i].x-3, hp[i].y-3);
				};
			};
		}; 
	};
}
