/*---[ ipchains-script.c ]-----------------------------------------------
 * Copyright (C) 2000 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Functions to write the ipchains shell scripts
 *-------------------------------------------------------------------*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>


#include "druid.h"
#include "ipchains-script.h"

/* [ write_ipchains_script ]
 * Creates the ipchains shell script
 */
void
write_ipchains_script (Druid *data)
{
	gboolean masq = GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->masq));
	gboolean tos = GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->tos));

	gchar *scriptpath = FIRESTARTER_RULES_DIR "/firestarter/firewall.sh";

	FILE *script = fopen (scriptpath, "w");
        if (script == NULL) {
                /* Use perror to get sane error messages */
                perror(scriptpath);
                g_printerr("Script not written!");

		return;
	}

	chmod (scriptpath, 00700);

/* Init variables and external files */
	fprintf (script, "#!/bin/sh\n");
	fprintf (script, "#Generated by Firestarter " VERSION ", IPCHAINS in use\n\n");
	
/* in an effort to clean up the code - variables get moved to the top - at least that way, if a user
  needs to alter a variable themselves (eg. if a cable modem misbehaves) - it's easy to find */
  

    fprintf (script, "# --------( Initial Setup - Variables (required) )--------\n\n");

	fprintf (script, "# Type of Service (TOS) parameters\n");
	fprintf (script, "# 0x08: Maximum Throughput - Minimum Delay\n");
	fprintf (script, "# 0x04: Minimize Delay - Maximize Reliability\n");
	fprintf (script, "# 0x10: No Delay - Moderate Throughput - High Reliability\n\n");
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_option_throughput)))
			fprintf (script, "TOSOPT=\"0x01 0x08\"\n\n");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_option_reliability)))
			fprintf (script, "TOSOPT=\"0x01 0x04\"\n\n");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_option_delay)))
			fprintf (script, "TOSOPT=\"0x01 0x10\"\n\n");

	fprintf (script, "# Default Packet Rejection Type\n");
	fprintf (script, "# ( do NOT change this here - set it in the GUI instead )\n\n");
	
	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE"))
		fprintf (script, "STOP=DENY\n\n\n");
	else
		fprintf (script, "STOP=REJECT\n\n\n");

    fprintf (script, "# --------( Initial Setup - Network Information (required) )--------\n\n");
    
	/* The external network interface */
	fprintf (script, "IF=%s\n", gtk_entry_get_text (GTK_ENTRY (data->extdevice)));
	/* The internal network interface */
	if (masq)
		fprintf (script, "INIF=%s\n", gtk_entry_get_text (GTK_ENTRY (data->intdevice)));

	/* IP address of external interface */
	fprintf (script, "IP=`/sbin/ifconfig $IF | grep inet | cut -d : -f 2 | cut -d \\  -f 1`\n");
	/* Netmask/net of external interface */
	fprintf (script, "MASK=`/sbin/ifconfig $IF | grep Mas | cut -d : -f 4`\n");
	fprintf (script, "NET=$IP/$MASK\n\n");

	if (masq) {
		if (GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->auto_int_ip)))
	/* Netmask/net of internal interface (autodetected) */
			fprintf (script, "INNET=%s\n\n", gtk_entry_get_text (GTK_ENTRY (data->intrange)));
		else {
	/* IP address of internal interface */
			fprintf (script, "INIP=`/sbin/ifconfig $INIF | grep inet | cut -d : -f 2 | cut -d \\  -f 1`\n");
	/* Netmask/net of internal interface */
			fprintf (script, "INMASK=`/sbin/ifconfig $INIF | grep Mas | cut -d : -f 4`\n");
			fprintf (script, "INNET=$INIP/$INMASK\n");
		}
	}

    fprintf (script, "# --------( Initial Setup - Firewall Location Check )--------\n\n");
    			
	if (gnome_config_get_bool ("/firestarter/Druid/locatesbins=FALSE")) {
		fprintf (script, "IPC=`which ipchains`\nMPB=`which modprobe`\nLSM=`which lsmod`\n");
		fprintf (script, "MSQADM=`which ipmasqadm`\n\n");
	 } else {
		fprintf (script, "IPC=/sbin/ipchains\nMPB=/sbin/modprobe\nLSM=/sbin/lsmod\n");
		fprintf (script, "MSQADM=/usr/sbin/ipmasqadm\n\n");
	}

    fprintf (script, "# --------( Initial Setup - Firewall Modules Autoloader )--------\n\n");
    


	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pppcheck))) {
    fprintf (script, "# --------( Initial Setup - Additional Firewall Modules (PPP) )--------\n\n");
    
    fprintf (script, "if ! ( $LSM | /bin/grep bsd_comp > /dev/null ); then\n");
    fprintf (script, "$MPB bsd_comp\nfi\n\n");
	}
	    
	/* Pipe in the modrules files */

   fprintf (script, "# --------( Initial Setup - External Lists )--------\n\n");
   
	fprintf (script, "# Deny all traffic on these ports, without logging\n");
	fprintf (script, "if [ -e "FIRESTARTER_RULES_DIR "/firestarter/block-ports ]\n then\n"
		"source " FIRESTARTER_RULES_DIR "/firestarter/block-ports\nfi\n\n");

	fprintf (script, "# Deny all traffic from these machines\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/deny-all\n\n");

	fprintf (script, "# Allow all traffic from these machines\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-all\n\n");
	fprintf (script, "# Allow a specific service to a specific machine\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-service-machine\n\n");
	fprintf (script, "# Allow a specific service to everyone\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-service-all\n\n");
	
	fprintf (script, "# Portforwarding rules\n");
	fprintf (script, "if [ -e "FIRESTARTER_RULES_DIR "/firestarter/portfw ]\n then\n"
		"source " FIRESTARTER_RULES_DIR "/firestarter/portfw\nfi\n\n");    


   fprintf (script, "# --------( Chain Configuration - Flush Existing Chains )--------\n\n");
   
	fprintf (script, "#Delete user made chains. Flush and zero any existing chains.\n");
	fprintf (script, "$IPC -X\n$IPC -F\n$IPC -Z\n\n");
	
   fprintf (script, "# --------( Chain Configuration - Set Default Policy )--------\n\n");
   
	fprintf (script, "$IPC -P input $STOP\n$IPC -P output $STOP\n$IPC -P forward $STOP\n\n\n");
	
   fprintf (script, "# --------( Sysctl Tuning - Recommended Parameters )--------\n\n");
   
	fprintf (script, "# Turn off IP forwarding by default\n");
	fprintf (script, "# (this will be enabled if you require masquerading)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_forward ]\n then\n"
	"  echo 0 > /proc/sys/net/ipv4/ip_forward\nfi\n\n");
	
	fprintf (script, "# Log 'odd' IP addresses (excludes 0.0.0.0 & 255.255.255.255)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/log_martians ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/conf/all/log_martians\nfi\n\n");	

   fprintf (script, "# --------( Sysctl Tuning - TCP Parameters )--------\n\n");
   
    fprintf (script, "# Turn off TCP Timestamping in kernel\n");
    fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_timestamps ]; then\n"
    "  echo 0 > /proc/sys/net/ipv4/tcp_timestamps\nfi\n\n");    
      
    fprintf (script, "# Turn off TCP ACK in kernel\n");
    fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_sack ]; then\n"
    "  echo 0 > /proc/sys/net/ipv4/tcp_sack\nfi\n\n");
    
    fprintf (script, "# Turn off TCP Window Scaling in kernel\n");
    fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_window_scaling ]; then\n"
    "  echo 0 > /proc/sys/net/ipv4/tcp_window_scaling\nfi\n\n");

    fprintf (script, "# Set Keepalive timeout to 30 seconds\n");
    fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_keepalive_time ]; then\n"
    "  echo 1800 > /proc/sys/net/ipv4/tcp_keepalive_time\nfi\n\n");
    
    fprintf (script, "# Set FIN timeout to 30 seconds\n");
    fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_fin_timeout ]; then\n"
    "  echo 1800 > /proc/sys/net/ipv4/tcp_fin_timeout\nfi\n\n");

    fprintf (script, "# Set TCP retry count to 3\n");
    fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_retries1 ]; then\n"
    "  echo 3 > /proc/sys/net/ipv4/tcp_retries1\nfi\n\n");
            
   fprintf (script, "# --------( Sysctl Tuning - SYN Parameters )--------\n\n");
   
	fprintf (script, "# Turn on SYN cookies protection in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_syncookies ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/tcp_syncookies\nfi\n\n");
	
	fprintf (script, "# Set SYN backlog buffer to '64'\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_max_syn_backlog ]; then\n"
	"  echo 64 > /proc/sys/net/ipv4/tcp_max_syn_backlog\nfi\n\n");
	
	fprintf (script, "# Set SYN retry attempts to '6'\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_syn_retries ]; then\n"
	"  echo 6 > /proc/sys/net/ipv4/tcp_syn_retries\nfi\n\n");
	
   fprintf (script, "# --------( Sysctl Tuning - Routing / Redirection Parameters )--------\n\n");

/* under 2.2 - source route verification has 0 (off), 1 (weak detection) & 2 (RFC compliant) */

	fprintf (script, "# Turn on source address verification in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/rp_filter ]; then\n"
	"  for f in /proc/sys/net/ipv4/conf/*/rp_filter\n  do\n   echo 2 > $f\n  done\nfi\n\n");
	
	fprintf (script, "# Turn off source routes in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/accept_source_route ]; then\n"
	"  for f in /proc/sys/net/ipv4/conf/*/accept_source_route\n  do\n   echo 0 > $f\n  done\nfi\n\n");

	fprintf (script, "# Do not respond to 'redirected' packets\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/secure_redirects ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/secure_redirects\nfi\n\n");
	
	fprintf (script, "# Do not reply to 'redirected' packets if requested\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/send_redirects ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/send_redirects\nfi\n\n");
    
	fprintf (script, "# Do not reply to 'proxyarp' packets\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/proxy_arp ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/proxy_arp\nfi\n\n");
	
    fprintf (script, "# Set FIB model to be RFC1812 Compliant\n");
    fprintf (script, "# (certain policy based routers may break with this - if you find\n");
    fprintf (script, "#  that you can't access certain hosts on your network - please set\n");
    fprintf (script, "#  this option to '0' - which is the default)\n\n");
    fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_fib_model ]; then\n"
    "  echo 2 > /proc/sys/net/ipv4/ip_fib_model\nfi\n\n");

   fprintf (script, "# --------( Sysctl Tuning - ICMP/IGMP Parameters )--------\n\n");
   
	fprintf (script, "# ICMP Dead Error Messages protection\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses\nfi\n\n");

	fprintf (script, "# ICMP Broadcasting protection\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts\nfi\n\n");
	
	fprintf (script, "# IGMP Membership 'overflow' protection\n");
	fprintf (script, "# (if you are planning on running your box as a router - you should either\n");
	fprintf (script, "#  set this option to a number greater than 5, or disable this protection\n");
	fprintf (script, "#  altogether by commenting out this option)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/igmp_max_memberships ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/igmp_max_memberships\nfi\n\n");

   fprintf (script, "# --------( Sysctl Tuning - Miscellanous Parameters )--------\n\n");
   
   	fprintf (script, "# Set TTL to '64' hops\n");
   	fprintf (script, "# (If you are running a masqueraded network, or use policy-based\n");
   	fprintf (script, "#  routing - you may want to increase this value depending on the load\n");
   	fprintf (script, "#  on your link.)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/ip_default_ttl ]; then\n"
	"  for f in /proc/sys/net/ipv4/conf/*/ip_default_ttl\n  do\n   echo 64 > $f\n  done\nfi\n\n");

  	fprintf (script, "# Always defragment incoming packets\n");
   	fprintf (script, "# (Some cable modems [ Optus @home ] will suffer intermittent connection\n");
   	fprintf (script, "#  droputs with this setting. If you experience problems, set this to '0')\n\n");	
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_always_defrag ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/ip_always_defrag\nfi\n\n");
	
  	fprintf (script, "# Keep packet fragments in memory for 8 seconds\n");
   	fprintf (script, "# (Note - this option has no affect if you turn packet defragmentation\n");
   	fprintf (script, "#  (above) off!)\n\n");	
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ipfrag_time ]; then\n"
	"  echo 8 > /proc/sys/net/ipv4/ipfrag_time\nfi\n\n");

  	fprintf (script, "# Do not reply to Address Mask Notification Warnings\n");
   	fprintf (script, "# (If you are using your machine as a DMZ router or a PPP dialin server\n");
   	fprintf (script, "#  that relies on proxy_arp requests to provide addresses to it's clients\n");
   	fprintf (script, "#  you may wish to disable this option by setting the value to '1'\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_addrmask_agent ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/ip_addrmask_agent\nfi\n\n");
		
		
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pppcheck))) {
		fprintf (script, "# Turn on dynamic TCP/IP address hacking\n");
   	    fprintf (script, "# (Some broken PPPoE clients require this option to be enabled)\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]; then\n"
		"  echo 1 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n\n");
	} else {
		fprintf (script, "# Turn off dynamic TCP/IP address hacking\n");
   	    fprintf (script, "# (Some broken PPPoE clients have issues when this is disabled\n");
   	    fprintf (script, "#  If you experience problems with DSL or Cable providers, set this to '1')\n\n");		
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]; then\n"
		"  echo 0 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n\n");
	}

   fprintf (script, "# --------( Rules Configuration - Specific Rule - Loopback Interfaces )--------\n\n");
	fprintf (script, "# Allow all traffic on the loopback interface\n");
	fprintf (script, "$IPC -A input -i lo -s 0/0 -d 0/0 -j ACCEPT\n");
	fprintf (script, "$IPC -A output -i lo -s 0/0 -d 0/0 -j ACCEPT\n\n\n");


/* TOS RULES */
if (tos) {

   fprintf (script, "# --------( Rules Configuration - Type of Service (ToS) - Ruleset Filtered by GUI )--------\n\n");
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_client))) {
	fprintf (script, "# ToS: Client Applications\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 20:21 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 22 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 68 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 80 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 443 -t $TOSOPT\n\n");
}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_server))) {
	fprintf (script, "# ToS: Server Applications\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 20:21 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 22 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 25 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 53 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 67 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 80 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 110 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 143 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 443 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 1812 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 1813 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 2401 -t $TOSOPT\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 8080 -t $TOSOPT\n");
}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_X))) {
	fprintf (script, "# ToS: The X Window System\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 22 -t 0x01 0x10\n");
	fprintf (script, "$IPC -A output -p tcp -d 0/0 6000:6015 -t 0x01 0x08\n\n");
 }
}

/* ICMP RULES */

if GTK_WIDGET_IS_SENSITIVE (data->icmp)
{
       fprintf (script, "# --------( Rules Configuration - ICMP - Ruleset Filtered by GUI )--------\n\n");
       
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_echo))) {
		fprintf (script, "# ICMP: Ping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-request -j REJECT\n");
	}
		else {
		fprintf (script, "# ICMP: Ping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-request -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-reply -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_traceroute))) {
		fprintf (script, "# ICMP: Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 33434 -j REJECT -l\n");
	}
		else {
		fprintf (script, "# ICMP: Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 33434 -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_tracert))) {
		fprintf (script, "# ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j REJECT -l\n");
	}
		else {
		fprintf (script, "# ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_unreach))) {
		fprintf (script, "# ICMP: Unreachable Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j REJECT -l\n");
	}
		else {
		fprintf (script, "# ICMP: Unreachable Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_timestamp))) {
		fprintf (script, "# ICMP: Timestamping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j REJECT -l\n");
	}
		else {
		fprintf (script, "# ICMP: Timestamping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j ACCEPT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_masking))) {
		fprintf (script, "# ICMP: Address Masking\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j REJECT -l\n");
	}
		else {
		fprintf (script, "# ICMP: Address Masking\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j ACCEPT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_redir))) {
		fprintf (script, "# ICMP: Redirection Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type redirect -j REJECT -l\n");
	}
		else {
		fprintf (script, "# ICMP: Redirection Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type redirect -j ACCEPT -l\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_quench))) {
		fprintf (script, "# ICMP: Source Quench Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type source-quench -j REJECT -l\n");
	}
		else {
		fprintf (script, "# ICMP: Source Quench Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type source-quench -j ACCEPT -l\n");
	}
}
else {
       fprintf (script, "# --------( Rules Configuration - ICMP - Default Ruleset )--------\n\n");
       
		fprintf (script, "# ICMP: Ping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-request -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type echo-reply -j ACCEPT\n");
		fprintf (script, "# ICMP: Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 33434 -j REJECT -l\n");
		fprintf (script, "# ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j ACCEPT -l\n");
		fprintf (script, "# ICMP: Unreachable Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j ACCEPT -l\n");
		fprintf (script, "# ICMP: Timestamping Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j ACCEPT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j ACCEPT -l\n");
		fprintf (script, "# ICMP: Address Masking\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j ACCEPT -l\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j ACCEPT -l\n");
		fprintf (script, "# ICMP: Redirection Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type redirect -j ACCEPT -l\n");
		fprintf (script, "# ICMP: Source Quench Requests\n");
		fprintf (script, "$IPC -A input -p icmp -s 0/0 -d $NET --icmp-type source-quench -j ACCEPT -l\n\n");
}

/* MASQ RULES */
	if (masq) {

       fprintf (script, "\n# --------( Rules Configuration - Masquerading )--------\n\n");
       
       fprintf (script, "# --------( Rules Configuration - Masquerading - Firewall Modules Autoloader )--------\n\n");
		fprintf (script, "if ! ( $LSM | /bin/grep masq_ftp > /dev/null ); then\n");
		fprintf (script, "$MPB ip_masq_ftp\nfi\n");
		fprintf (script, "if ! ( $LSM | /bin/grep masq_irc > /dev/null ); then\n");
		fprintf (script, "$MPB ip_masq_irc\nfi\n");
		fprintf (script, "if ! ( $LSM | /bin/grep masq_quake > /dev/null ); then\n");
		fprintf (script, "$MPB ip_masq_quake\nfi\n");
		fprintf (script, "if ! ( $LSM | /bin/grep masq_raudio > /dev/null ); then\n");
		fprintf (script, "$MPB ip_masq_raudio\nfi\n");

       fprintf (script, "# --------( Rules Configuration - Masquerading - Timeouts / Defaults )--------\n\n");
		fprintf (script, "$IPC -M -S 14400 60 600\n\n");

       fprintf (script, "# --------( Rules Configuration - Masquerading - Sysctl Modifications )--------\n\n");
       
		fprintf (script, "# Turn on IP forwarding\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_forward ]\n then\n"
		"  echo 1 > /proc/sys/net/ipv4/ip_forward\nfi\n\n");

       fprintf (script, "# --------( Rules Configuration - Masquerading - Default Policy )--------\n\n");
		fprintf (script, "# Set forward (masquerading) policy to DENY\n");
		fprintf (script, "$IPC -P forward DENY\n\n");
		
      fprintf (script, "# --------( Rules Configuration - Masquerading - Default Ruleset )--------\n\n");		

		fprintf (script, "# Don't masquerade internal-internal traffic\n");
		fprintf (script, "$IPC -A forward -s $INNET -d $INNET -j ACCEPT\n\n");

		fprintf (script, "# Don't masquerade external interface traffic\n");
		fprintf (script, "$IPC -A forward -s $NET -d 0/0 -j ACCEPT\n\n");

		fprintf (script, "# Masquerade outbound local net traffic\n");
		fprintf (script, "$IPC -A forward -s $INNET -d 0/0 -j MASQ\n\n");

		fprintf (script, "# Allow traffic from internal network going anywhere\n");
		fprintf (script, "$IPC -A input -s $INNET -d 0/0 -j ACCEPT\n");
		fprintf (script, "$IPC -A output -s $INNET -d 0/0 -j ACCEPT\n");
		fprintf (script, "$IPC -A output -p icmp -s $INNET -d 0/0 -j ACCEPT\n\n");
	}

	
   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block nonroutable IP Addresses )--------\n\n");

	fprintf (script, "#Block nonroutable IPs\n");
	fprintf (script, "$IPC -A input -s 1.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 2.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 5.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 7.0.0.0/8 -d $NET -i $IF -j $STOP\n");
/*	fprintf (script, "$IPC -A input -s 10.0.0.0/8 -d $NET -i $IF -j $STOP\n"); */
	fprintf (script, "$IPC -A input -s 23.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 27.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 31.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 36.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 37.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 39.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 41.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 42.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 58.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 59.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 60.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 69.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 70.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 71.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 72.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 73.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 74.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 75.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 76.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 77.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 78.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 79.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 82.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 83.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 84.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 85.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 86.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 87.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 88.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 89.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 90.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 91.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 92.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 93.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 94.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 95.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 96.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 97.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 98.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 99.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 100.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 101.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 102.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 103.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 104.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 105.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 106.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 107.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 108.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 109.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 110.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 111.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 112.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 113.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 114.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 115.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 116.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 117.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 118.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 119.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 120.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 121.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 122.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 123.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 124.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 125.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 126.0.0.0/8 -d $NET -i $IF -j $STOP\n");
/*	fprintf (script, "$IPC -A input -s 127.0.0.0/8 -d $NET -i $IF -j $STOP\n"); */
	fprintf (script, "$IPC -A input -s 128.66.0.0/16 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 172.16.0.0/12 -d $NET -i $IF -j $STOP\n");
/*	fprintf (script, "$IPC -A input -s 192.168.0.0/16 -d $NET -i $IF -j $STOP\n"); */
	fprintf (script, "$IPC -A input -s 197.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 221.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 222.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 223.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPC -A input -s 240.0.0.0/4 -d $NET -i $IF -j $STOP\n");
	
   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block known Trojan Ports )--------\n\n");
	
    fprintf (script, "# Block Back Orifice\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 31337 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 31337 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -s $NET -d 0/0 31337 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p udp -s $NET -d 0/0 31337 -j $STOP -l\n\n");

	fprintf (script, "# Block Trinity v3\n");	
	fprintf (script, "$IPC -A input -p tcp -y -s 0/0 -d $NET 33270 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 33270 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -y -s $NET -d 0/0 33270 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p udp -s $NET -d 0/0 33270 -j $STOP -l\n\n");

	fprintf (script, "# Block Subseven (1.7/1.9)\n");	
	fprintf (script, "$IPC -A input -p tcp -y -s 0/0 -d $NET 1234 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p tcp -y -s 0/0 -d $NET 6711 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -y -s $NET -d 0/0 1234 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -y -s $NET -d 0/0 6711 -j $STOP -l\n\n");

	fprintf (script, "# Block Stacheldraht\n");	
	fprintf (script, "$IPC -A input -p tcp -y -s 0/0 -d $NET 16660 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p tcp -y -s 0/0 -d $NET 60001 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -y -s $NET -d 0/0 16660 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -y -s $NET -d 0/0 60001 -j $STOP -l\n\n");

	fprintf (script, "# Block MS-RPC (dce)\n");	
	fprintf (script, "$IPC -A input -p tcp -y -s 0/0 -d $NET 135 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 135 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -y -s $NET -d 0/0 135 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p udp -s $NET -d 0/0 135 -j $STOP -l\n\n");

	fprintf (script, "# Block NetBus\n");	
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 12345:12346 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 12345:12346 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -s $NET -d 0/0 12345:12346 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p udp -s $NET -d 0/0 12345:12346 -j $STOP -l\n\n");

	fprintf (script, "# Block Trin00\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 1524 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 27665 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 27444 -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 31335 -j $STOP -l\n\n");
	fprintf (script, "$IPC -A output -p tcp -s $NET -d 0/0 1524 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p tcp -s $NET -d 0/0 27665 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p udp -s $NET -d 0/0 27444 -j $STOP -l\n");
	fprintf (script, "$IPC -A output -p udp -s $NET -d 0/0 31335 -j $STOP -l\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block Multicast Traffic )--------\n\n");
   fprintf (script, "# (some cable/DSL providers require their clients to accept multicast transmissions\n");
   fprintf (script, "#  you should remove the following four rules if you are affected by multicasting\n");
   
	fprintf (script, "$IPC -A input -s 224.0.0.0/8 -d 0/0 -j $STOP\n");
	fprintf (script, "$IPC -A input -s 0/0 -d 224.0.0.0/8 -j $STOP\n");
	fprintf (script, "$IPC -A output -s 224.0.0.0/8 -d 0/0 -j $STOP\n");
	fprintf (script, "$IPC -A output -s 0/0 -d 224.0.0.0/8 -j $STOP\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block Traffic w/ Stuffed Routing )--------\n\n");
   fprintf (script, "# (early versions of PUMP - (the DHCP client application included in RH / Mandrake) require\n");
   fprintf (script, "#  inbound packets to be accepted from a source address of 255.255.255.255.  If you have issues\n");
   fprintf (script, "#  with DHCP clients on your local LAN - either update PUMP, or remove the first rule below)\n");
	fprintf (script, "$IPC -A input -s 255.255.255.255 -j $STOP\n");
	fprintf (script, "$IPC -A input -d 0.0.0.0 -j $STOP\n");
	fprintf (script, "$IPC -A output -s 255.255.255.255 -j $STOP\n");
	fprintf (script, "$IPC -A output -d 0.0.0.0 -j $STOP\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block Traffic w/ Excessive Fragmented Packets )--------\n\n");
	fprintf (script, "$IPC -A input -f -j $STOP -l\n\n");

/* RULES BASED ON WIZARD SELECTIONS */

    fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Ruleset Filtered by GUI )--------\n\n");

	/* Allow DHCP */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dhcpcheck))) {
		fprintf (script, "#DHCP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 67:68 -j ACCEPT\n\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 67:68 -j ACCEPT\n\n");
	}

if GTK_WIDGET_IS_SENSITIVE (data->services)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ftp))) {
		fprintf (script, "#FTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 20 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 21 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ssh))) {
		fprintf (script, "#SSH\n");		
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 22 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->telnet))) {
		fprintf (script, "#Telnet\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 23 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->smtp))) {
		fprintf (script, "#SMTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 25 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dns))) {
		fprintf (script, "#DNS\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 53 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 53 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->finger))) {
		fprintf (script, "#Finger\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 79 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->www))) {
		fprintf (script, "#HTTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 80 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->sslweb))) {
		fprintf (script, "#SSL HTTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 443 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pop))) {
		fprintf (script, "#POP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 110 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->imap))) {
		fprintf (script, "#IMAP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 143 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ident))) {
		fprintf (script, "#IDENT\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 113 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 113 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->nntp))) {
		fprintf (script, "#NNTP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 119 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ntp))) {
		fprintf (script, "#NTP\n");		
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 123 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 123 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->samba))) {
		fprintf (script, "#SAMBA\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 137:139 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 137:139 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 445 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 445 -i $IF -j ACCEPT\n\n");
 	}
	else {
		fprintf (script, "#Block SAMBA (Silently Rejected)\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A output -p tcp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A output -p udp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A output -p tcp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n");
		fprintf (script, "$IPC -A output -p udp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ipsec))) {
		fprintf (script, "#IPSec / KLIPS\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 500 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p 51 -s 0/0 -d 0/0 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->routed))) {
		fprintf (script, "#Routed\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 520 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->nfs))) {
		fprintf (script, "#NFS\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 2049 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 2049 -i $IF -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "#Block NFS\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 2049 -i $IF -j $STOP -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 2049 -i $IF -j $STOP -l\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->x))) {
		fprintf (script, "#Xwindows\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 6000:6015 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 6000:6015 -i $IF -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "#Block Xwindows\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 6000:6015 -i $IF -j $STOP -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 6000:6015 -i $IF -j $STOP -l\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dhcp))) {
		fprintf (script, "#DHCP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 67:68 -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 67:68 -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "#Block DHCP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 67:68 -j $STOP\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 67:68 -j $STOP\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->upnp))) {
		fprintf (script, "#uPNP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 5000 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 5000 -i $IF -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "#Block uPNP\n");
		fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 5000 -i $IF -j $STOP -l\n");
		fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 5000 -i $IF -j $STOP -l\n\n");
	}

}
else {
    
   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Default Ruleset )--------\n\n");
	fprintf (script, "#Block NFS\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 2049 -i $IF -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 2049 -i $IF -j $STOP -l\n\n");
	
	fprintf (script, "#Block SAMBA (Silently Rejected)\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n\n");
	fprintf (script, "$IPC -A output -p tcp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A output -p udp -s 0/0 -d 0/0 137:139 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A output -p tcp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n");
	fprintf (script, "$IPC -A output -p udp -s 0/0 -d 0/0 445 -i $IF -j REJECT -l\n\n");

	fprintf (script, "#Block uPNP\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 5000 -i $IF -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 5000 -i $IF -j $STOP -l\n\n");

	fprintf (script, "#Block Xwindows\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d 0/0 6000:6015 -i $IF -j $STOP -l\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d 0/0 6000:6015 -i $IF -j $STOP -l\n\n");
}

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Highport Connections )--------\n\n");
	fprintf (script, "$IPC -A input -p tcp -s 0/0 -d $NET 20 -j ACCEPT\n");
	fprintf (script, "$IPC -A input ! -y -p tcp -s 0/0 -d $NET 1023:65535 -j ACCEPT\n");
	fprintf (script, "$IPC -A input -p udp -s 0/0 -d $NET 1023:65535 -j ACCEPT\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Outbound Traffic - Default Ruleset )--------\n\n");
	fprintf (script, "$IPC -A output -p icmp -s $NET -d 0/0 -j ACCEPT\n\n");
	fprintf (script, "$IPC -A output -j ACCEPT\n\n");

/* REQUIRED RULES */

   fprintf (script, "\n# --------( Catch all Rules (required) )--------\n\n");
	fprintf (script, "# Deny everything not let through earlier\n");
	fprintf (script, "$IPC -A input -j $STOP -l\n\n");

	fclose (script);

	g_print (_("Firewall script saved as %s\n"), scriptpath);
}
