/********************************************************************************
*                                                                               *
*                          T I F F   I n p u t / O u t p u t                    *
*                                                                               *
*********************************************************************************
* Copyright (C) 2001 Eric Gillet <egillet@ibelgique.com>. All Rights Reserved.  *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id: fxtifio.cpp,v 1.7 2001/07/24 21:49:59 jeroen Exp $                       *
********************************************************************************/
#include "fx.h"


/*
  Notes:
  - Made it thread-safe.
  - Made error and warning handlers call FOX's warning handler.
  - References:
    http://www.libtiff.org/
    ftp://ftp.onshore.com/pub/libtiff/TIFF6.ps.Z
    ftp://ftp.sgi.com/graphics/tiff/TTN2.draft.txt
    http://partners.adobe.com/asn/developer/technotes.html
*/


/*******************************************************************************/

/// Load a tiff from a stream
extern FXAPI FXbool fxloadTIF(FXStream& store,FXuchar*& data,FXColor& transp,FXint& width,FXint& height,FXushort& codec);


/// Save a tiff to a stream
extern FXAPI FXbool fxsaveTIF(FXStream& store,const FXuchar* data,FXColor transp,FXint width,FXint height,FXushort codec);


/*******************************************************************************/

#ifdef HAVE_TIFF_H
#include <tiffio.h>


#define TIFF_SWAP(p) (((p)&0xff)<<24 | ((p)&0xff00)<<8 | ((p)&0xff0000)>>8 | ((p)&0xff000000)>>24)


// Stuff being passed around
struct tiff_store_handle {
  FXStream     *store;
  unsigned long begin;
  unsigned long end;
  FXbool        error;
  };


// Report an error
static void fxerrorhandler(const char* module,const char* format,va_list args){
  FXchar message[1024];
  vsprintf(message,format,args);
  fxwarning("error: in %s: %s\n",module,message);
  }


// Report a warning
static void fxwarninghandler(const char* module,const char* format,va_list args){
  FXchar message[1024];
  vsprintf(message,format,args);
  fxwarning("warning: in %s: %s\n",module,message);
  }


// Read bytes from stream
static tsize_t tif_read_store(thandle_t handle,tdata_t data,tsize_t size){
  tiff_store_handle *h=(tiff_store_handle*)handle;
  h->store->load((FXuchar*)data,size);
  if(h->store->status()!=FXStreamOK) return 0;
  if(h->store->position() > h->end) h->end=h->store->position();
  return size;
  }


// Dummy read bytes
static tsize_t tif_dummy_read_store(thandle_t,tdata_t,tsize_t){
  return 0;
  }


// Write bytes to stream
static tsize_t tif_write_store(thandle_t handle,tdata_t data,tsize_t size){
  tiff_store_handle *h=(tiff_store_handle*)handle;
  h->store->save((FXuchar*)data,size);
  if(h->store->status()!=FXStreamOK) return 0;
  if(h->store->position()>h->end) h->end=h->store->position();
  return size;
  }


// Seek to a position in the stream
static toff_t tif_seek_store(thandle_t handle,toff_t offset,int whence){
  tiff_store_handle *h=(tiff_store_handle*)handle;
  unsigned long off;
  if(whence==SEEK_SET){
    off=h->begin+offset;
    }
  else if(whence==SEEK_CUR){
    off=h->store->position()+offset;
    }
  else{ // SEEK_END
    off=h->end+offset;
    }
  h->store->position(off);
  return off;
  }


// Dummy close store
static int tif_close_store(thandle_t){
  return 0;
  }


// Compute size of what's been written
static toff_t tif_size_store(thandle_t handle){
  tiff_store_handle *h=(tiff_store_handle*)handle;
  return (h->end-h->begin);
  }


// Load a TIFF image
FXbool fxloadTIF(FXStream& store,FXuchar*& data,FXColor&,FXint& width,FXint& height,FXushort& codec){
  TIFF *image;
  tiff_store_handle s_handle;
  FXbool ret = FALSE;

  // Set error/warning handlers
  TIFFSetErrorHandler(fxerrorhandler);
  TIFFSetWarningHandler(fxwarninghandler);

  // Initialize
  s_handle.store=&store;
  s_handle.begin=store.position();
  s_handle.end=store.position();
  s_handle.error=FALSE;

  // Open image
  image=TIFFClientOpen("tiff","rm",(thandle_t)&s_handle,tif_read_store,tif_write_store,tif_seek_store,tif_close_store,tif_size_store,NULL,NULL);
  if(image){
    TIFFRGBAImage img;
    char emsg[1024];

    // We try to remember the codec for later when we save the image back out...
    TIFFGetField(image,TIFFTAG_COMPRESSION,&codec);
    FXTRACE((100,"fxloadTIF: codec=%d\n",codec));

    ret = (TIFFRGBAImageBegin(&img, image, 0, emsg) != 0);    // FIXME TIFFRGBAImage{Begin,Get,End} is too broken!
    if(ret){
      FXTRACE((100,"FXTIF: width=%ld height=%ld\n", img.width , img.height));
      width = (FXint)img.width;
      height = (FXint)img.height;
      // Make room for data
      unsigned long size=4*img.width*(img.height+((img.orientation==ORIENTATION_TOPLEFT)?1:0));
      FXMALLOC(&data,FXuchar,size);
      if(!data){
        ret=FALSE;
        }
      if(ret && !TIFFRGBAImageGet(&img,(unsigned long *)data,img.width,img.height)){
        ret=FALSE;
        }
      ret = !s_handle.error;
      if(ret && (img.orientation == ORIENTATION_TOPLEFT)){ // vertical mirroring
        FXuchar *up, *down, *temp;
        size = 4 * img.width;
        up = data;
        down = data + size * (img.height -1);
        temp = data + size * img.height;
        while(up < down){
          memcpy(temp,up,size);
          memcpy(up,down,size);
          memcpy(down,temp,size);
          up+=size;
          down-=size;
          }
        }
#if FOX_BIGENDIAN == 1
      if(ret){
	FXPixel *pix_data = (FXPixel*)data;
	size = img.width * img.height;
	while(size--){
	  *pix_data = TIFF_SWAP(*pix_data);
	  pix_data++;
	  }
	}
#endif
      TIFFRGBAImageEnd(&img);
      }
    TIFFClose(image);
    if(!ret && data) FXFREE(&data);
    }
  return ret;
  }


/*******************************************************************************/

// Save a TIFF image
FXbool fxsaveTIF(FXStream& store,const FXuchar* data,FXColor,FXint width,FXint height,FXushort codec){
  TIFF *image;
  tiff_store_handle s_handle;
  long rows_per_strip,line,h,size;
  const TIFFCodec* coder;
  FXbool ok=FALSE;

  // Correct for unsupported codecs
  coder=TIFFFindCODEC(codec);
  if(coder==NULL) codec=COMPRESSION_PACKBITS;

  // Due to the infamous UNISYS patent, we can read LZW TIFF's but not
  // write them back as that would require the LZW compression algorithm!
  if(codec==COMPRESSION_LZW) codec=COMPRESSION_PACKBITS;

  FXTRACE((100,"fxsaveTIF: codec=%d\n",codec));

  // Set error/warning handlers
  TIFFSetErrorHandler(fxerrorhandler);
  TIFFSetWarningHandler(fxwarninghandler);

  // Initialize
  s_handle.store=&store;
  s_handle.begin=store.position();
  s_handle.end=store.position();
  s_handle.error=FALSE;

  // Open image
  image=TIFFClientOpen("tiff","w",(thandle_t)&s_handle,tif_dummy_read_store,tif_write_store,tif_seek_store,tif_close_store,tif_size_store,NULL,NULL);
  if(image){

    // Size of a strip is 16kb
    rows_per_strip=16*1024/width;
    if(rows_per_strip<1) rows_per_strip=1;

    // Set fields
    TIFFSetField(image,TIFFTAG_IMAGEWIDTH,width);
    TIFFSetField(image,TIFFTAG_IMAGELENGTH,height);
    TIFFSetField(image,TIFFTAG_COMPRESSION,codec);
    TIFFSetField(image,TIFFTAG_ORIENTATION,ORIENTATION_TOPLEFT);
    TIFFSetField(image,TIFFTAG_ROWSPERSTRIP,rows_per_strip);
    TIFFSetField(image,TIFFTAG_BITSPERSAMPLE,8);
    TIFFSetField(image,TIFFTAG_SAMPLESPERPIXEL,4);
    TIFFSetField(image,TIFFTAG_PLANARCONFIG,1);
    TIFFSetField(image,TIFFTAG_PHOTOMETRIC,PHOTOMETRIC_RGB);

    // Dump each line
    h=height;
    size=4*width;
    for(line=0; line<h; line++){
      if(TIFFWriteScanline(image,(void*)data,line,1)!=1 || s_handle.error) goto x;
      data+=size;
      }
    ok=TRUE;
x:  TIFFClose(image);
    }
  return ok;
  }


/*******************************************************************************/


#else


// Stub routine
FXbool fxloadTIF(FXStream&,FXuchar*& data,FXColor& transp,FXint& width,FXint& height,FXushort& codec){
  data=NULL;
  transp=0;
  width=0;
  height=0;
  codec=0;
  return FALSE;
  }


// Stub routine
FXbool fxsaveTIF(FXStream&,const FXuchar*,FXColor transp,FXint,FXint,FXushort){
  return FALSE;
  }

// Stub routine
FXuint fxcodecTIF(FXuint codec){
  return 1;
  }

#endif
