//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Scene.hpp,v 1.2 2003/08/31 15:13:08 delpinux Exp $



#ifndef _SCENE_HPP_
#define _SCENE_HPP_

#ifdef __GNUG__
#pragma interface
#endif // __GNUG__

#include <Object.hpp>
#include <Index.hpp>

#include <map>
#include <StreamCenter.hpp>

/**
 * @file   Scene.hpp
 * @class  Scene
 * @author Stephane Del Pino
 * @date   Sun Jun  9 19:22:48 2002
 * 
 * @brief  Set of ojects
 *
 * The Scene class is the "main" class of the computation.  It
 * contains all objects of the scene, the mesh, boundary conditions,
 * variables, ...
 * 
 */

class Scene
{
private:
  //! The vector containing objects.
  std::vector<Object> __objects;

  struct VectorOrder
  {
    bool operator()(const TinyVector<3>& a,
		    const TinyVector<3>& b) const
    {
      bool less;
      if (a[0] < b[0]) {
	less = true;
      } else if (a[0] == b[0]) {
	if (a[1] < b[1]) {
	  less = true;
	} else if (a[1] == b[1]) {
	  if (a[2] < b[2]) {
	    less = true;
	  } else {
	    less = false;
	  }
	} else {
	  less = false;
	}
      } else {
	less = false;
      }
      return less;
    }
  };

  //! references list
  typedef std::map<TinyVector<3>, int, Scene::VectorOrder> ReferenceMap;
  ReferenceMap __references;

public:
  //! Returns the number of Object.
  inline const size_t nbObjects() const
  {
    return __objects.size();
  }

  /** 
   * Stores the object o and checks its reference
   * 
   * @param o the object to add
   */
  inline void add(Object& o)
  {
    if (o.hasReference()) {
      ReferenceMap::iterator i = __references.find(o.reference());
      if (i == __references.end()) {
	int size = __references.size();
	__references[o.reference()] = size;
      }
    }

    // finaly store the object
    __objects.push_back(o);
  }

  //! Access to the Object \a i.
  inline Object& object(const size_t i)
  {
    assert (i<__objects.size());
    return __objects[i];
  }

  //! Read-only access to the Object \a i.
  inline const Object& object(const size_t i) const
  {
    assert (i<__objects.size());
    return __objects[i];
  }

  const int getReferenceNumber(const TinyVector<3>& t) const
  {
    ReferenceMap::const_iterator i = __references.find(t);
    if (i == __references.end()) {
      fferr(2) << '\n' << t << " reference has not been found\n";
      std::exit(1);
    }
    return (*i).second;
  }

  /** 
   * Default Constructor
   * 
   */
  Scene();

  /** 
   * Copy constructor
   * 
   * @param S a Scene
   */
  Scene(const Scene& S);

  /** 
   * Destructor
   * 
   */
  ~Scene()
  {
    ;
  }
};

#endif // _SCENE_HPP_

