//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MeshReader.hpp,v 1.9 2004/04/17 15:33:49 delpinux Exp $

#ifndef MESHREADER_HPP
#define MESHREADER_HPP

#include <MeshGenerator.hpp>
#include <istream>
#include <string>
#include <sstream>

#include <ReferenceCounting.hpp>

#include <VerticesSet.hpp>

#include <Vector.hpp>
#include <Vertex.hpp>

#include <Hexahedron.hpp>
#include <Tetrahedron.hpp>
#include <Triangle.hpp>

#include <set>

/**
 * @file   MeshReader.hpp
 * @author Stphane Del Pino
 * @date   Sun Dec 15 17:45:02 2002
 * 
 * @brief  Reads mesh from file in the 'mesh' format
 * 
 */

class MeshReader
  : public MeshGenerator
{
private:
  struct VertexError {};

  std::stringstream __is;	/**< The input stream (without comments) */
  std::string __fileName;	/**< The file name if appropriate */

  /**
   * List of vertices.
   * 
   */
  ReferenceCounting<VerticesSet> __vertices;

  /**
   * List of hexahedra.
   * 
   */
  ReferenceCounting<Vector<Hexahedron> > __hexahedra;

  /**
   * List of quadrilaterals
   * 
   */
  ReferenceCounting<Vector<Quadrangle> > __quadrilaterals;

  /**
   * List of tetrahedra.
   * 
   */
  ReferenceCounting<Vector<Tetrahedron> > __tetrahedra;

  /**
   * List of triangles.
   * 
   */
  ReferenceCounting<Vector<Triangle> > __triangles;

  /**
   * List of allowed keyword in mesh file
   * 
   */
  enum KeywordType {
    EndOfFile,
    Unknown,
    MeshVersionFormatted,
    MeshDimension,
    Vertices,
    Triangles,
    Quadrilaterals,
    Tetrahedra,
    Hexahedra,
    Normals,
    Tangents,
    Corners,
    Edges,
    Ridges
  };

  /**
   * List of known keywords
   * 
   */
  typedef std::map<std::string, int> KeywordList;

  KeywordList __keywordList;	/**< The keyword list */

  /**
   * Type for keyword
   * 
   */
  typedef std::pair<std::string, int> Keyword;

  /** 
   * Reads the next keyword and returns its token
   * 
   * @return KeywordToken
   */
  MeshReader::Keyword __nextKeyword();

  /** 
   * Removes all comments from input.
   * 
   * @param is 
   */
  inline void __removeComments(std::istream& is);

  /** 
   * Checks if a number is a valid vertex number
   * 
   * @param i a vertex number
   */
  inline void __checkVertex(const size_t& i);

  /** 
   * Gets a vertex number
   * 
   * @return next integer
   */
  inline size_t __getVertexNumber();

  /** 
   * Tries to read next integer (unsigned)
   * 
   * @return next integer
   */
  inline size_t __getInteger();

  /** 
   * Tries to read next real
   * 
   * @return next real
   */
  inline real_t __getReal();

  /** 
   * get list of vertices
   * 
   */
  void __getVertices();

  /**
   * Read list of vertices
   * 
   */
  void __readVertices();

  /**
   * Read list of hexahedra
   * 
   */
  void __readHexahedra();

  /**
   * Read list of quadrilaterals
   * 
   */
  void __readQuadrilaterals();

  /**
   * Read list of tetrahedra
   * 
   */
  void __readTetrahedra();

  /**
   * Read list of triangles
   * 
   */
  void __readTriangles();

  /**
   * Read list of vertices
   * 
   */
  void __readElements();

  /** 
   * Copy constructor is forbidden
   * 
   * @param M 
   */
  MeshReader(const MeshReader& M)
  {
    ;
  }

  /** 
   * Common interface for writing references
   * 
   * @param references the set of computed references
   * @param ojectName the type of refernces
   */
  void __writeReferences(const std::set<size_t>& references,
			 std::string objectName);
public:

  /** 
   * Access to the read mesh
   * 
   * 
   * @return __mesh
   */
  ReferenceCounting<Mesh> mesh()
  {
    return __mesh;
  }

  /** 
   * Constructor
   * 
   * @param is the input stream
   * @param s  the string containing the filename
   */
  MeshReader(std::istream& is,
	     std::string s = "");

  /** 
   * Destructor
   * 
   */
  ~MeshReader()
  {
    ;
  }
};

#endif // MESHREADER_HPP
