package freenet.client;

import freenet.support.Bucket;
import freenet.support.BucketSink;
import freenet.support.BucketFactory;
import freenet.client.metadata.SplitFile;
import java.io.IOException;

/*
  This code is distributed under the GNU Public Licence (GPL)
  version 2.  See http://www.gnu.org/ for further details of the GPL.
*/

/**
 * Interface for plugging Forward Error Correction (FEC) encoder
 * implementaions into fproxy. 
 * <p>
 * Requirements:<br>
 * <ul>
 * <li> Implementations are not guaranteed to be 
 *      thread safe. The client code is responsible
 *      for serializing access.
 * <li> Implementations must have a no args constructor in order
 *      to be loadable by FECFactory.  All initialization should
 *      be done in init().
 * <li> All check blocks and data blocks are 
 *      the same size. Smaller trailing data blocks must
 *      be zero padded.
 * <li> The encoder may ask for extra trailing data blocks.
 *      i.e. getN() * getBlockSize() * getSegments()  can
 *      be greater than the blockSize padded length.
 * <li> Implementations should avoid segmenting if possible. <p>
 *      You get better robustness if all check blocks contain 
 *      information about all data blocks.
 *      However, some FEC codecs just can't
 *      handle really huge files at reasonable
 *      block sizes.  In this case, segementing
 *      is the only way out.
 *   
 * </ul>
 **/
public interface FECEncoder {
    /**
     * Initializes the encoder to encode a file of
     * length len.
     * @return true on success, false otherwise.
     **/
    boolean init(int len, BucketFactory factory);

    /**
     * Releases all resources (memory, file handles, temp files, etc.)
     * used by the encoder instance.
     * <p>
     * Implementations must support calling this function 
     * irregardless of the state of the encoder instance. e.g.
     * it isn't an error to call release on an encoder that
     * has already been released or has never been initialized.
     **/
    void release();

    /**
     * Polite implementations should use block sizes 
     * between 256K and 3Mb. 
     * @return The block size for data and check blocks.
     **/
    int getBlockSize();
    
    /** 
     * @return the total number of data and check blocks per segment.
     */
    int getN(int segment);

    /**
     * @return the number of data blocks per segment.
     **/
    int getK(int segment);

    /**
     * @return the number of segments.
     **/
    int getSegmentCount();

    /**
     * Every segment except for the last must be this long.
     * <p>
     * @return the segment size.
     **/
    int getSegmentSize();

    /**
     * Make check blocks for a single segment. This may take
     * a very long time.  Polite implementations should 
     * abort as quickly as possible and throw an 
     * InterruptedIOException if the thread is interrupted.
     * <p>
     * REQUIRES: blocks.length == getK(segmentNumber)
     * <p>
     * @return the check blocks.
     **/
    Bucket[] encode(int segmentNumber, Bucket[] blocks) throws IOException;

    /**
     * Writes the information necessary to retrieve
     * and decode the segmented file into a
     * SplitFile metadata object.
     * <p>
     * Implementations should write parameters
     * necessary for decoding in the SplitFile's
     * decoderParams FieldSet.
     * <p>
     * @return the SplitFile.
     **/
    SplitFile makeMetadata(String[] dataURIs, String[] checkURIs);
}





