/*
  This code is part of fproxy, an HTTP proxy server for Freenet.
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

package freenet.client.http;

import freenet.*;
import freenet.client.*;
import freenet.client.metadata.*;
import freenet.support.*;

import java.io.*;
import java.net.*;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;

/**
 * Stateful "sub-servlet" to provide user feedback for
 * SplitFile requests.
 **/
class SplitFileRequestContext implements Reapable {

    private int requestState = INIT;
    private SplitFileDownloader.Status finalDownloadStatus = 
        new SplitFileDownloader.Status();

    final static int INIT = 1;
    final static int STARTING_REQUEST = 2;
    final static int WORKING = 3;
    final static int SUCCEEDED = 4;
    final static int FAILED = 5;
    final static int ABORTED = 6;
    
    final static String PREFIX = "/__INTERNAL__01/";

    // If we don't hear from the downloader for longer
    // than this amount of time we assume that something
    // went badly wrong allow the context to be reaped.
    //
    // N.B.: This is so long because FEC decoding might
    //       take a long time on slow machines.
    private final static long DOWNLOADTIMEOUT_MS =  60 * 60000;

    // The amount of time to keep the context around after 
    // a request finishes.
    private final static long LINGERTIMEOUT_MS = 15 * 60000;

    // Must set before creating instances
    static ContextManager contextManager = null;
    static Reaper reaper = null;
    static ClientFactory factory = null;
    static BucketFactory bucketFactory = null;
    static FECFactory fecFactory = null;

    private String contextID = null;

    private int htl;
    private int retryHtlIncrement;
    private int retries;
    private int threads;
    private int refreshInterval;
    private boolean useUI;
    private boolean forceSave;
    private boolean pollForDroppedConnection = true;

    private SplitFile splitFile = null;
    private String mimeType = null;

    private String uri = null;
    private String contentDesc = null;

    private SplitFileDownloader downloader = null;

    private Logger logger = null;

    ////////////////////////////////////////////////////////////

    // REQUIRES: urlValue.startsWith(PREFIX)
    final static void handle(SplitFileRequestContext context, String urlValue,
                             HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {

        //  if (!urlValue.startsWith(PREFIX)) {
        //     throw new RuntimeException("Assertion Failure: Bad Request");
        //  }

        if (context != null) {
            // Let it render the page.
            context.renderPage(req, resp);
            return;
        }
        
        // We can get here if the Reaper released the context.
        sendError(resp, "<h1>Request for an unknown, finished or expired context.</h1>");
    }

    ////////////////////////////////////////////////////////////

    // This bad boy sure is ugly...
    SplitFileRequestContext(SplitFile sf,
                            String uri,
                            String mimeType,
                            int htl,
                            int retryHtlIncrement,
                            int retries,
                            int threads,
                            boolean useUI,
                            boolean forceSave,
                            boolean pollForDroppedConnection, 
                            int refreshInterval,
                            Logger logger) {

        this.splitFile = sf;
        this.uri = uri;
        this.mimeType = mimeType;
        this.htl = htl;
        this.retryHtlIncrement = retryHtlIncrement; 
        this.retries = retries;
        this.threads = threads;
        this.useUI = useUI;
        this.forceSave = forceSave;
        this.pollForDroppedConnection = pollForDroppedConnection;
        this.logger = logger;
        if (logger == null) {
            logger = Core.logger;
        }
        this.refreshInterval = refreshInterval;

        String fecInfo = "none";
        if ((sf.getDecoderName() != null) && (sf.getCheckBlockCount() > 0)) {
            int redundancy = (100 * sf.getCheckBlockCount()) / sf.getBlockCount();
            fecInfo = sf.getDecoderName() + " (" + 
                Integer.toString(redundancy) + "% redundancy)";
        }

        // e.g. "120MB video/avi, FEC:OnionDecoder_0 (50% redundancy)";
        contentDesc = ", &nbsp &nbsp " + formatByteCount(sf.getSize()) + " " +
            mimeType + ", FEC decoder: " + fecInfo;

        // So that we can find this instance for future
        // GET's
        contextID = contextManager.add(this);

        // Reaper will release our resources if we time out.
        reaper.add(this);
    }

    final String getRedirectURL() {
        return PREFIX + contextID + "/";
    }

    ////////////////////////////////////////////////////////////
    // Reapable implementation
    public boolean reap() {
        try {
            cancel();
        }
        catch (InterruptedException ie) {
            // NOP
        }
        contextManager.remove(contextID);

        return true;
    }

    private final static long intervalMs(long startMs, long endMs) {
        if ((startMs == 0) || (endMs == 0)) {
            return 0;
        }

        if (startMs > endMs) {
            return 0;
        }

        return endMs - startMs;
    }

    public boolean isExpired() {
        SplitFileDownloader.Status downloadStatus = null;
        int contextStatus = -1;
        long nowMs = System.currentTimeMillis();
        synchronized (this) {
            if (downloader != null) {
                // nested locks
                downloadStatus = downloader.getStatus();
            }
            else {
                downloadStatus = finalDownloadStatus;
            }

            contextStatus = requestState;
        }

        if (contextStatus == WORKING) {
            return intervalMs(downloadStatus.lastActiveMs, nowMs) > DOWNLOADTIMEOUT_MS;
        }
        else {
            return intervalMs(downloadStatus.lastActiveMs, nowMs) > LINGERTIMEOUT_MS;
        }
    }
    ////////////////////////////////////////////////////////////

    final void cancel() throws InterruptedException {
        SplitFileDownloader needCancel = null;
        synchronized (this) {
            // Wait until downloader is in a cancelable 
            // state.
            while (requestState == STARTING_REQUEST) {
                this.wait();
            }
            if (requestState == INIT) {
                // Keep another thread from calling 
                // sendData().
                setRequestState(ABORTED);
            }
            else if (requestState == WORKING) {
                needCancel = downloader;
            }
        }

        // Avoid nested locks. Canceling when we are not
        // working is a legal nop.
        if (needCancel != null) {
            needCancel.cancel();
        }

        // The downloader might not be stopped, but at least
        // it is on the way to stopping.
    }

    // Renders top level 2 frame download / status page.
    void renderPage(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {
        
        SplitFileDownloader.Status downloadStatus = null;
        int contextStatus = -1;
        synchronized (this) {
            if (downloader != null) {
                // nested locks
                downloadStatus = downloader.getStatus();
            }
            else {
                downloadStatus = finalDownloadStatus;
            }

            // The SplitFileDownloader knows its exit status
            // before sendData() does because it doesn't have
            // to wait for the data to stream out of the
            // BucketSequence. This hack keeps the display in
            // sync.
            contextStatus = requestState;
            if (contextStatus == WORKING) {
                switch(downloadStatus.state) {
                case SplitFileDownloader.FINISHED:
                    contextStatus = SUCCEEDED;
                    break;
                case SplitFileDownloader.ABORTED:
                    contextStatus = ABORTED;
                    break;
                case SplitFileDownloader.FAILED:
                    contextStatus = FAILED;
                }
            }

        }

        String urlValue = null; 
        try {
            urlValue = freenet.support.URLDecoder.decode(req.getRequestURI());
        }
        catch (URLEncodedFormatException  fe) {
            throw new IOException(fe.toString()); // hmmm...
        }

        if (urlValue.startsWith(PREFIX + contextID + "/download")) {
            switch(contextStatus) {
            case INIT: 
                // SplitFile data
                updateParameters(req);
                sendData(req, resp);
                break;
            case WORKING:
            case STARTING_REQUEST:
                sendError(resp, "<h1>Download already in progress.</h1>");
                break;
            default:
                sendError(resp, "<h1>Download already finished.</h1>");
            }
        }
        else if (urlValue.equals(PREFIX + contextID + "/")) {
            // Main UI Page
            if (contextStatus == INIT) {
                updateParameters(req);
            }
            renderFrameset(resp);
        }
        else if (urlValue.equals(PREFIX + contextID + "/link_frame.html")) {
            renderLinkFrame(resp, contextStatus);
        }
        else if (urlValue.equals(PREFIX + contextID + "/status_frame.html")) {
            // Status frame
            renderDownloadStatusFrame(resp, downloadStatus, contextStatus);
        }
        else {
            // WE SHOULD NEVER GET HERE.
            sendError(resp, "<h1>Request for an unknown, finished or expired SplitFile process.</h1>");
            return;
        }
        resp.flushBuffer(); 
    }
    
    // Call updateParameters before using this.
    final boolean useUI() {
        return useUI;
    }

    // Sets parameters from a query.
    // Package scope on purpose.
    final synchronized void updateParameters(HttpServletRequest req) {
        // LATER: hard coded limit constants ?
        htl = FproxyServlet.readInt(req, logger, "htl", htl, 0, 100);
        retryHtlIncrement = FproxyServlet.readInt(req, logger,
                                                  "retryHtlIncrement", retryHtlIncrement, 0, 100);
        retries = FproxyServlet.readInt(req, logger, "retries", retries, 0, 10);
        threads = FproxyServlet.readInt(req, logger, "threads", threads, 0, 100);
        useUI = FproxyServlet.readBoolean(req, logger, "useUI", useUI);
        // NOTE: checkbox's return nothing at all in the unchecked
        //       state.
        if (FproxyServlet.readBoolean(req, logger, "usedForm", false)) {
            forceSave = FproxyServlet.readBoolean(req, logger, "forceSaveCB", false);
        }
        else {
            forceSave = FproxyServlet.readBoolean(req, logger, "forceSave", forceSave);
        }
    }
        
    private void renderLinkFrame(HttpServletResponse resp, int contextStatus) 
        throws IOException {
        String linkURI = null;

        // LATER: clean up so browsers get better filename?
        linkURI = java.net.URLEncoder.encode(PREFIX + contextID + "/download/" + uri);

        String resetURI = PREFIX + contextID + "/reset";

        String htlAsString = Integer.toString(htl);
        String retryHtlIncrementAsString = Integer.toString(retryHtlIncrement);
        String retriesAsString = Integer.toString(retries);
        String threadsAsString = Integer.toString(threads);
        String forceSaveAsString = "";
        if (forceSave) {
            forceSaveAsString = " checked ";
        }
        
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");
        PrintWriter pw = resp.getWriter();

        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println(uri);
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        pw.println("<h1>SplitFile download</h1>");
        pw.println("<u>" + uri + "</u> ");
        pw.println(" <b><code>" + contentDesc + "</code></b>");
        pw.println("");
        pw.println("<p>");
        
        if (contextStatus == INIT) {
            pw.println("");
            pw.println("Downloading large SplitFile's");
            pw.println("can be a resource intensive operation.  Hit the Back button on you browser");
            pw.println("if you want to abort.  You can also abort the download once it starts by hitting");
            pw.println("the Back button on your browser or canceling the file save dialog if you're saving");
            pw.println("to a file.");
            pw.println("<form method=\"GET\" action=\"" + PREFIX + contextID + "/download/" +  uri + "\">");
            // Hack to make checkbox work correctly.
            pw.println("<input type=\"hidden\" name=\"usedForm\" value=\"true\" >");
            pw.println("<table border>");
            pw.println("<tr>");
            pw.println("    <td> <b> htl </b> </td>");
            pw.println("    <td> <input type=\"text\" name=\"htl\" value=\"" +
                       htlAsString  + "\"> </td>");
            pw.println("    <td> <b> htlRetryIncrement </b> </td>");
            pw.println("    <td> <input type=\"text\" name=\"retryHtlIncrement\" value=\"" +
                   retryHtlIncrementAsString + "\"></td>");

            pw.println("    <td> <b> forceSave </b> </td>");
            pw.println("    <td> <input type=\"checkBox\" name=\"forceSaveCB\" value=\"true\"" +
                       forceSaveAsString + " ></td>");
            pw.println("</tr>");
            pw.println("<tr>");
            pw.println("    <td> <b> retries </b> </td>");
            pw.println("    <td> <input type=\"text\" name=\"retries\" value=\"" + 
                       retriesAsString + "\"></td>");
            pw.println("    <td> <b> threads </b> </td>");
            pw.println("    <td> <input type=\"text\" name=\"threads\" value=\"" +
                       threadsAsString + "\"> </td>");
            pw.println("</tr>");
            pw.println("</table>");
            pw.println("<p>");
            pw.println("<input type=\"submit\" value=\"Start Download\">");
            pw.println("</form>");
            pw.println("");
        }
        else {
            switch(contextStatus) {
            case WORKING: 
                pw.println("Download in progress.");
                break;
            case SUCCEEDED: 
                pw.println("Download finished successfully.");
                break;
            case FAILED: 
                pw.println("Download failed.");
                break;
            case ABORTED: 
                pw.println("Download aborted by client or timed out.");
                break;
            }
        }

        pw.println("</body>");
        pw.println("</html>");
    }

    private final void renderRunningStatus(PrintWriter pw, SplitFileDownloader.Status status)
        throws  IOException {

        String segmentInfo = Integer.toString(status.segment + 1) + "/" +
            Integer.toString(status.segments);
        String blocksRequired = Integer.toString(status.blocksRequired) +
            " (" + formatByteCount(status.blocksRequired * status.blockSize) + ")";
        String blocksDownloaded = Integer.toString(status.blocksDownloaded);
        if (status.blocksDownloaded > 0) {
            blocksDownloaded += " (" + formatByteCount(status.blocksDownloaded * status.blockSize) + ")";
        }

        String threads = Integer.toString(status.runningThreads);
        String blocksQueued = Integer.toString(status.blocksQueued);
        String retries = Integer.toString(status.retries);

        String blocksFailed = Integer.toString(status.blocksFailed);
        String dnfCount = Integer.toString(status.dnfCount);
        String rnfCount = Integer.toString(status.rnfCount);
        if (status.localRNFCount > 0) {
            rnfCount +=  " <font color=\"red\"> (" + Integer.toString(status.localRNFCount) + 
                ") </font> ";
        }
        long deltatSecs = (System.currentTimeMillis() - status.lastActiveMs) / 1000;

        pw.println("<table border>");
        pw.println("<tr>");
        pw.println("    <td> <b> Segment </b> </td>");
        pw.println("    <td>" + segmentInfo + " </td>");
        pw.println("    <td> <b> Blocks failed </b> </td>");
        pw.println("    <td> " + blocksFailed + "</td>");
        pw.println("</tr>");
        pw.println("<tr>");
        pw.println("    <td> <b> Blocks required </b> </td>");
        pw.println("    <td> " + blocksRequired + " </td>");
        pw.println("    <td> <b> DNF count </b> </td>");
        pw.println("    <td> " + dnfCount + "</td>");
        pw.println("</tr>");
        pw.println("<tr>");
        pw.println("    <td> <b> Blocks downloaded </b> </td>");
        pw.println("    <td> " + blocksDownloaded + " </td>");
        pw.println("    <td> <b> RNF count </b> </td>");
        pw.println("    <td> " + rnfCount + "</td>");
        pw.println("</tr>");
        pw.println("<tr>");
        pw.println("    <td> <b> Threads running</b> </td>");
        pw.println("    <td> " + threads + " </td>");
        pw.println("    <td> <b> Retries </b> </td>");
        pw.println("    <td> " + retries + "</td>");
        pw.println("</tr>");
        pw.println("<tr>");
        pw.println("    <td> <b> Blocks queued </b> </td>");
        pw.println("    <td> " + blocksQueued + "</td>");
        pw.println("    <td> <b> Inactive </b> </td>");
        pw.println("    <td> " + deltatSecs + " secs. </td>");
        pw.println("</tr>");
        pw.println("");
        pw.println("</table>");
        pw.println("");
    }

    private void renderDownloadStatusFrame(HttpServletResponse resp, 
                                          SplitFileDownloader.Status status,
                                          int contextStatus) 
        throws IOException {
        PrintWriter pw = resp.getWriter();
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");
        final boolean updating = (contextStatus == INIT) || (contextStatus == WORKING); 
        if ((refreshInterval > 0) && updating) {
            // Optional client pull updating.
            resp.setHeader("Refresh", Integer.toString(refreshInterval));
        }
        // hmmm... don't set content length...

        
        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Download status: " + uri);
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        pw.println("");
        pw.println("<h1>Download status: " + uri + "</h1>");
        pw.println("");
        
        switch (contextStatus) {
        case INIT:
        case STARTING_REQUEST:
            pw.println("Waiting for user to start by clicking on the download link...");
            break;
        case WORKING:
            if (status.state == SplitFileDownloader.FEC_DECODING) {
                pw.println("FEC decoding missing data blocks...");
                pw.println("<p>");
                pw.println("This can take a long time. Be patient.");
                pw.println("");
            }
            else {
                renderRunningStatus(pw, status);
            }
            break;
        case SUCCEEDED:
            pw.println("Download finished successfully.");
            break;
        case FAILED:
            pw.println("Download failed.  Couldn't download enough blocks to reconstruct segment: " +
                       status.segment + ".");
            pw.println("<p>");
            pw.println("required: " + status.blocksRequired + " <br>");
            pw.println("downloaded: " + status.blocksDownloaded + " <br>");
            pw.println("<p>");
            // REDFLAG: retry link, how do I dismiss the frames?
            break;

        case ABORTED:
            pw.println("Download aborted.  The client dropped the connection or the " +
                       "Freenet request timed out.");
            break;
        default:
            pw.println("Assertion Failure: Unexpected contextStatus: " + contextStatus);
        }

        if (updating) {
            pw.println("<p>");
            pw.println("<a href=\"" + PREFIX + contextID + "/status_frame.html" +
                       "\"> [Update Status Info] </a> ");
        }

        pw.println("</body>");
        pw.println("</html>");
    }

    private void renderFrameset(HttpServletResponse resp) throws IOException {
        String linkURL = PREFIX + contextID + "/link_frame.html";
        String statusURL = PREFIX + contextID + "/status_frame.html";

        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");
        PrintWriter pw = resp.getWriter();

        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>SplitFile Download: " + uri + "</title>");
        pw.println("</head>");
        pw.println("<frameset rows =\"50%,50%\">");
        pw.println("  <frame src=\"" + linkURL + "\">");
        pw.println("  <frame src=\"" + statusURL + "\">");
        pw.println("</frameset>");
        pw.println("</html>");
    }

    // REDFLAG: C&P from manifest tools
    private final static String formatByteCount(long nBytes) {
	String unit = "";
	double scaled = 0.0;
	if (nBytes >= (1024*1024) ) {
	    scaled = ((double)nBytes)/(1024*1024);
	    unit = "M";
	}
	else if (nBytes >= 1024) {
	    scaled = ((double)nBytes)/1024;
	    unit = "K";
	}
	else {
	    scaled = (double)nBytes;
	    unit = "bytes";
	}
	
	// Truncate to one digit after the decimal point.
	String value = Double.toString(scaled);

	int p = value.indexOf(".");
	if ((p != -1) && (p < value.length() - 2)) {
	    value = value.substring(0, p + 2);
	}
	
	if (value.endsWith(".0")) {
	    value = value.substring(0, value.length() - 2);
	}

	return value + unit;
    }

   static void sendError(HttpServletResponse resp, String htmlFormattedMsg)
        throws IOException {

        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");
        PrintWriter pw = resp.getWriter();

        pw.println("<html>");
        pw.println("<head><title> Error </title></head>");
        pw.println("<body>");
        pw.println(htmlFormattedMsg);
        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer(); 
    }

    private final boolean enforceSingleSend(HttpServletResponse resp)
        throws ServletException, IOException {
        boolean alreadySending = false;
        synchronized (this) {
            alreadySending = (requestState != INIT);
            if (!alreadySending) {
                requestState = STARTING_REQUEST;
            }
        }
        
        if (alreadySending) {
            String msg = 
                "<h1>Stream already started</h1> \n" +
                "You can only start downloading the SplitFile stream once. <p> \n" +
                "Go back to the original link if you want to start downloading again.\n";
            sendError(resp, msg);
            return false;
        }

        return true;
    }

    ////////////////////////////////////////////////////////////
    void sendData(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {
        
        // You can only download the data stream once.
        if (!enforceSingleSend(resp)) {
            return;
        }
        
        // INIT --> STARTING_REQUEST  
        String mimeTypeUsed = mimeType;
        if (forceSave) {
            // Keep the browser from trying to display
            // the data.
            mimeTypeUsed = "application/octet-stream";
        }

        boolean flushed = false;
        InputStream in = null;
        OutputStream out = null;
        try {
            // Special case SplitFiles to support streaming.
            boolean sendingPartial = false;
            RangeParams range = 
                parseRangeHeader(req.getHeader("range"), splitFile.getSize());
            if ((range != null) && range.handled && splitFile.getBlockSize() != -1) {
                // Handle partial requests.
                // Set response headers, 206
                resp.setStatus(HttpServletResponse.SC_PARTIAL_CONTENT);
                resp.setContentType(mimeTypeUsed);
                resp.setHeader("Content-Range", range.contentRangeHeader);
                // IMPORTANT: length of data sent.
                resp.setContentLength((int)range.nBytes);
                sendingPartial = true;
            } 
            else {
                // No range specified, or we couldn't parse the range
                // header so we send the entire split file.
                
                // Set response headers, 200
                resp.setStatus(HttpServletResponse.SC_OK);
                resp.setContentType(mimeTypeUsed);
                resp.setContentLength((int)splitFile.getSize());
            }
        
            // HACK: If the connection is available the SplitFileDownloader
            //       polls it to make sure the client hasn't gone
            //       away.
            Connection conn = null;
            if (pollForDroppedConnection) {
                conn = (Connection)req.getAttribute("Freenet.Connection");
            }
        
            synchronized (this) {
                downloader = new SplitFileDownloader(threads);
            }

            downloader.setHtl(htl);
            downloader.setMaxRetries(retries);
            downloader.setRetryHtlIncrement(retryHtlIncrement);
        
            BucketSequence bucketSequence = null;
            if (sendingPartial) {
                // IMPORTANT: BlockSize wasn't spec'd in the SplitFile
                //            metadata spec until 20011002.  SplitFile
                //            trys to guess the BlockSize when it isn't
                //            present in order to allow range support.
                //            If it guesses wrong, bad things will happen...
                bucketSequence = downloader.start(splitFile, factory,
                                                           bucketFactory,
                                                           range.offset, range.nBytes,
                                                           conn);

                setRequestState(WORKING);
                // STARTING_REQUEST --> WORKING  
            }
            else {
                // Don't depend on fixed block size if we don't need to.
                bucketSequence = downloader.start(splitFile, factory,
                                                           bucketFactory, fecFactory,
                                                           conn);
                setRequestState(WORKING);
                // STARTING_REQUEST --> WORKING  
            }
        
            in = bucketSequence.getInputStream();
            // set up for binary output
            // can't do this earlier in case we need resp.getWriter()
            out = resp.getOutputStream();
                
            // It might be a long time before 
            // the SplitFileDownloader actually sends
            // the first block. Send headers immediately so the
            // browser can throw up a status dialog if it is
            // saving to file.
            out.flush();
            // This following line is a workaround for an IE bug
            // if (mimeType.equals("application/octet-stream")) {
            //  out.write("Content-Disposition: attachment; ".getBytes());
            //  out.write(("filename=" + key + "\015\012\015\012").getBytes());
            // }
            FproxyServlet.copy(in, out);
            in.close();
            in = null;
        }
        catch (Exception e) {
            //e.printStackTrace();
            logger.log(this, "Error sending data to browser: " + e,
                       Logger.ERROR);
            sendError(resp, "<h1>Error sending data to browser: " + e + " </h1>");
            flushed = true;
        }
        finally {
            if (in != null) {
                // Don't leak file handles.
                // Also, releases BucketSequence resources if
                // in came from BucketSequence.getInputStream().
                try {in.close();} catch (IOException e) {}
            }
            
            if ((downloader != null) && 
                downloader.isRunning()) {
                downloader.cancel();
                synchronized (downloader) {
                    try {
                        while (downloader.isRunning()) {
                            downloader.wait();
                        }
                    }
                    catch (InterruptedException ie) {
                        throw new InterruptedIOException();
                    }
                }
            }

            // Unlocked scope on purpose.
            finalDownloadStatus = downloader.getStatus();
            synchronized (this) {
                // Save status info.
                switch (finalDownloadStatus.state) {
                case SplitFileDownloader.FINISHED:
                    setRequestState(SUCCEEDED);
                    break;
                case SplitFileDownloader.ABORTED:
                    setRequestState(ABORTED);
                    break;
                default:
                    setRequestState(FAILED);
                }
                // Release ref
                downloader = null;
            }

            // commit response
            if (!flushed) {
                resp.flushBuffer();
            }
        }
    }

    ////////////////////////////////////////////////////////////
    // Helper functions to do range parsing.
    static class RangeParams {
        String rangeAsString;
        String contentRangeHeader;
        int offset;
        int nBytes;
        // false for multiple ranges
        boolean handled;
    }

    // This is more of a grepper than a parser...
    final RangeParams parseRangeHeader(String text, int length) {
        if (text == null) {
            return null;
        }
        text = text.trim();
        if (text.equals("")) {
            return null;
        }

        RangeParams ret = new RangeParams();
        ret.rangeAsString = text;

        if (text.indexOf(",") != -1) {
            // We don't handle multi-segement requests.
            ret.handled = false;
            logger.log(this, "Multi-segment Range: headers not supported, " + text, Logger.DEBUGGING);
            return ret;
        }

        logger.log(this, "Parsing Range: header, " + text, Logger.DEBUGGING);

        boolean gotDash = false;
        boolean gotStart = false;
        int startPos = 0;
        int endPos = length -1;
        StringTokenizer strTok = new StringTokenizer(text, "-=", true);
        try {
            while (strTok.hasMoreTokens()) {
                String token = strTok.nextToken().toLowerCase();
                if (token.equals("-")) {
                    gotDash = true;
                    continue;
                }
                else if (token.equals("bytes") || token.equals("=")) {
                    continue;
                }
                else {
                    // Try to parse a number
                    int value = Integer.parseInt(token);
                    // Cases
                    // 0) start-end
                    // 1) start-
                    // 2) -count, i.e. the last count bytes
                    if (!gotDash) {
                        startPos = value;
                        gotStart = true;
                    }
                    else {
                        if (gotStart) {
                            endPos = value;
                        }
                        else {
                            endPos = length - value;
                        }
                    }
                }
            }
        }
        catch (NumberFormatException nfe) {
            logger.log(this, "Couldn't parse Range: header, " + text, Logger.DEBUGGING);
            ret.handled = false;
            return ret;
        }

        if (startPos > endPos) {
            logger.log(this, "Couldn't parse Range: header, " + text, Logger.DEBUGGING);
            ret.handled = false;
            return ret;
        }

        ret.contentRangeHeader = Integer.toString(startPos) + "-" + Integer.toString(endPos) +
            "/" + Integer.toString(length);
        ret.offset = startPos;
        ret.nBytes = endPos - startPos + 1;
        ret.handled = true;
        logger.log(this, "Parsed Range: header, " + text + " " + ret.contentRangeHeader,
                   Logger.DEBUGGING);
        return ret;
    }

    private final synchronized void setRequestState(int value) {
        requestState = value;
        notifyAll();
    }
}

    






    






