package freenet.client.metadata;
import freenet.FieldSet;
import freenet.client.RequestProcess;
import freenet.client.GetRequestProcess;
import freenet.client.PutRequestProcess;
import freenet.client.FreenetURI;
import freenet.support.Bucket;
import freenet.support.BucketFactory;
import freenet.support.Fields;
//import Freenet.client.Metadata;

public class DateRedirect extends Redirect {
    
    public static final String name = "DateRedirect";

    public static final long DEFAULT_OFFSET = 0;
    public static final int  DEFAULT_INCREMENT = 0x15180;

    private long currentTime; // time use as current in seconds...

    private int increment;
    private long offset;
    //    private FreenetURI target;

    public DateRedirect(int increment, long offset, FreenetURI target) {
        this(increment, offset, System.currentTimeMillis() / 1000, 
             target);
    }

    public DateRedirect(int increment, long offset, long currentTime, 
                        FreenetURI target) {
        super(target);
        this.increment = increment;
        this.offset = offset;
        this.target = target;
        this.currentTime = currentTime;
    }
        
    public DateRedirect(FieldSet rawFields) 
        throws InvalidPartException {

        super(rawFields);
        currentTime = System.currentTimeMillis() / 1000;
        String offsets = rawFields.get("Offset");
        String increments = rawFields.get("Increment");

        try {
            if (offsets == null)
                offset = DEFAULT_OFFSET;
            else
                offset = Fields.stringToLong(offsets);
        
            if (increments == null)
                increment = DEFAULT_INCREMENT;
            else
                increment = (int) Fields.stringToLong(increments);
        } catch (NumberFormatException e) {
           throw new InvalidPartException("DateRedirect: Malformed numeric.");
        }
    }

    public void addTo(FieldSet fs) {
        super.addTo(fs);
        FieldSet me = fs.getSet(name());

        if (offset != DEFAULT_OFFSET)
            me.put("Offset", Fields.longToHex(offset));
        if (increment != DEFAULT_INCREMENT)
            me.put("Increment", Fields.longToHex(increment));
    }

    public String name() {
        return name;
    }

    protected FreenetURI getCurrentTarget(FreenetURI furi) {
        long reqTime = 
            offset + increment * ( ( currentTime - offset ) / increment);
        FreenetURI ruri = getRequestTarget(furi);
        return ruri.setDocName(Fields.longToHex(reqTime) + "-" 
                               + target.getDocName());
    }

    public FreenetURI getTargetForTime(FreenetURI furi, long timeMs) {
        long reqTime = 
            offset + increment * ( ( timeMs - offset ) / increment);
        FreenetURI ruri = getRequestTarget(furi);
        return ruri.setDocName(Fields.longToHex(reqTime) + "-" 
                               + target.getDocName());
    }


    public RequestProcess getGetProcess(FreenetURI furi, int htl, 
                                        Bucket data,
                                        BucketFactory ptBuckets, 
                                        int recursionLevel) {
        
        return new GetRequestProcess(getCurrentTarget(furi), htl, data, 
                                     ptBuckets,
                                     ++recursionLevel);
    }

    /*
     * We can't update the target here, but then Data redirects can only
     * work on URIs that are independant of the document, so...
     */
    public RequestProcess getPutProcess(FreenetURI furi, int htl, 
                                        String cipher, Metadata next, 
                                        Bucket data, BucketFactory ptBuckets,
                                        int recursionLevel, boolean descend) {



        return new PutRequestProcess(getCurrentTarget(furi), htl, cipher, 
                                     next, 
                                     data, ptBuckets, ++recursionLevel, 
                                     descend);
    }

    public String toString() {
        return "DateRedirect -> " + target + " (" + offset + " + n*" +
            increment + ')';
    }

}
