package freenet.diagnostics;
import freenet.FieldSet;
import freenet.support.Fields;
import java.util.Enumeration;
import java.io.DataOutputStream;
import java.io.DataInputStream;
import java.io.IOException;
/**
 * Implementation of a the counting process.
 *
 * @author oskar
 */


class CountingProcess extends RandomVar {

    static {
        try {
            RandomVar.registerType("CountingProcess", RandomVar.class);
        } catch (NoSuchMethodException e) {
        }
    }

    public static int VERSION = 2;
    
    private long totalEvents;
    private long lastEventTime;

    public CountingProcess(StandardDiagnostics sd, String name, int period,
                           String comment) {
        super(sd, name, period, comment);
        lastEventTime = System.currentTimeMillis(); // FIXME
        totalEvents = 0; // reset when the var is created.
    }

    public String getType() {
        return "CountingProcess";
    }
    
    public void add(long time, long occurrences, long lasttime) {

        totalEvents += occurrences;
        super.add(new CountingEvent(time, 1, occurrences, (double) totalEvents,
                                    time - lastEventTime), lasttime);
        lastEventTime = time;
    }

    public VarEvent aggregate(EventList ll, long time) {
        long nsum = 0, eventsum = 0, totalTime = 0;
        double totalEventsAverage = 0;
        for (Enumeration e = ll.elements() ; 
            e.hasMoreElements();) {

            CountingEvent ce = (CountingEvent) e.nextElement();

            if (ce.time() > time)
                break;
            nsum += ce.n;
            eventsum += ce.events;
            totalTime += ce.timeForEvents;
            if (!Double.isNaN(ce.totalEvents)) {
                totalEventsAverage += ce.timeForEvents * ce.totalEvents;
            }
        }
        totalEventsAverage = totalEventsAverage / totalTime; // note, may be NaN

        return new CountingEvent(time, eventsum, nsum, 
                                 totalEventsAverage, 
                                 totalTime);
    }

    public VarEvent readEvent(DataInputStream in) throws IOException {
        long version = in.readLong();
        long time = version;
        if (version > StandardDiagnostics.y2k)
            version = 1;
        if (version < 2) {
            long n = in.readLong();
            return new CountingEvent(time, n, n, (double) 0, 0);
        } else {
            time = in.readLong(); 
            long events = in.readLong();
            long n = in.readLong();
            double totalEvents = in.readDouble();
            long timeForEvents = in.readLong();
            return new CountingEvent(time, events, n, totalEvents, 
                                     timeForEvents);
        }
    }

    public String[] headers() {
        return new String[] { 
            "Events", "Value Change", "Mean Total Value",
            "Mean Time Between Events"
        };
    }
}

class CountingEvent extends VarEvent {
    
    public long n;
    public long events;
    public double totalEvents;
    // Time Between Events
    public long timeForEvents;
    
    public CountingEvent(long time, long events, long n, double totalEvents,
                         long timeForEvents) {
        super(time);
        this.events = events;
        this.n = n;
        this.totalEvents = totalEvents;
        this.timeForEvents = timeForEvents;
    }

    public String toString() {
        return "Counted " + n + " events.";
    }

    public double timeBetweenEvents() {
        return (double) timeForEvents / events;
    }

    public String[] fields() {
        return new String[] { 
            Long.toString(events), Long.toString(n),
            Double.toString(totalEvents), Double.toString(timeBetweenEvents())
        };
    }

    public double getValue(int type) {
        switch (type) {
        case Diagnostics.NUMBER_OF_EVENTS:
            return events;
        case Diagnostics.MEAN_TIME_BETWEEN_EVENTS:
            return timeBetweenEvents();
        case Diagnostics.MEAN_RUNTIME_COUNT:
            return totalEvents;
        case Diagnostics.COUNT_CHANGE:
            return n;
        default:
            throw new IllegalArgumentException("Unsupported value type.");
        }
    }

    public void write(DataOutputStream out) throws IOException {
        super.write(CountingProcess.VERSION, out);
        out.writeLong(events);
        out.writeLong(n);
        out.writeDouble(totalEvents);
        out.writeLong(timeForEvents);
    }

}
