package freenet.interfaces;

import freenet.*;
import freenet.thread.ThreadFactory;
import freenet.support.Logger;

/**
 * A PublicInterface is connected to by other nodes and can limit the
 * number of simultaneous connections, and prune idle connections.
 */
public class PublicInterface extends Interface {

    protected ThreadFactory tf;
    protected OpenConnectionManager ocm; 
    protected ConnectionRunner runner;
    protected ContactCounter inboundContacts;
    
    protected int maxConnections;
    protected volatile int activeConnections = 0;
    
    /**
     * @param maxConnections  if > 0, the maximum number of simultaneous
     *                        connections to allow
     * @param tm              the ThreadManager for connection threads
     * @param ocm             the OCM for pruning
     * @param runner          handles the connection thread
     * @param inboundContacts counter for incoming contacts. Can be null.
     */
    public PublicInterface(ListeningAddress listenAddr, int maxConnections,
                           ThreadFactory tf, OpenConnectionManager ocm,
                           ConnectionRunner runner, ContactCounter inboundContacts) 
        throws ListenException {
        super(listenAddr);
        this.maxConnections = (maxConnections <= 0 ? Integer.MAX_VALUE
                                                   : maxConnections);
        this.tf = tf;
        this.ocm = ocm;
        this.runner = runner;
        this.inboundContacts = inboundContacts;
    }
    
    protected void dispatch(Connection conn) throws RejectedConnectionException {

        Core.diagnostics.occurrenceCounting("inboundConnectionsDispatched", 1);

        if (inboundContacts != null) {
            inboundContacts.incTotal(conn.getPeerAddress().toString());
        }

        ConnectionShell shell = new ConnectionShell(conn);

        Thread thread = tf.getThread(shell, false);

        if (thread == null) {
            Core.diagnostics.occurrenceCounting("inboundConnectionsThreadLimitRej", 1);
            throw new RejectedConnectionException("thread limit reached");
        }
        Core.diagnostics.occurrenceCounting("inboundConnectionsAccepted", 1);

        if (inboundContacts != null) {
            inboundContacts.incSuccesses(conn.getPeerAddress().toString());
        }

        thread.start();
    }
    
    protected class ConnectionShell implements Runnable {
        protected final Connection conn;
        protected ConnectionShell(Connection conn) {
            this.conn = conn;
        }
        public void run() {
            try {
                // ++ and -- are *NOT* atomic operations.
                synchronized (PublicInterface.this) {
                    ++activeConnections;
                }

                if (inboundContacts != null) {
                    inboundContacts.incActive(conn.getPeerAddress().toString());
                }

                Core.diagnostics.occurrenceCounting("incomingConnections", 1);
                runner.handle(conn);
            }
            catch (Throwable t) {
                Core.logger.log(PublicInterface.this,
                    "Unhandled throwable while handling connection",
                    t, Logger.NORMAL);
                conn.close();
            }
            finally {
                synchronized (PublicInterface.this) {
                    --activeConnections;
                }

                if (inboundContacts != null) {
                    inboundContacts.decActive(conn.getPeerAddress().toString());
                }

                // FIXME.  this is too soon.
                // but hey, before it wasn't being called at all :)
                Core.diagnostics.occurrenceCounting("incomingConnections", -1);
            }
        }
    }
}


