package freenet.interfaces.servlet;

import freenet.*;
import freenet.node.Node;
import freenet.support.Logger;
import freenet.client.ClientFactory;
import freenet.support.servlet.http.*;
import javax.servlet.*;
import javax.servlet.http.*;
import java.io.IOException;
import java.util.Hashtable;

/**
 * General functionality for dealing with HTTP servlets.
 * The contexts and servlet pools are left in the domain
 * of the implementation.
 * @author tavin
 */
public abstract class HttpServletContainer extends ServletContainer {

    public static final String SERVER_INFO =
        Version.nodeName + " " + Version.nodeVersion
        + " (build " + Version.buildNumber + ")"
        + " HTTP Servlets";

    public static final int INPUT_BUFFER  = 0x10000,
                            OUTPUT_BUFFER = 0x10000;

    
    protected final SessionHolder sessionHolder = new SessionHolderImpl();
                                                  // FIXME - this thing
                                                  // will never ever forget
                                                  // a session..

    
    /** live ServletRequests and their associated ServletPool */
    protected final Hashtable liveTable = new Hashtable();
    

    /**
     * In-process container.
     */
    public HttpServletContainer(Node node) {
        super(node);
    }
    
    /**
     * Out-of-process container.
     */
    public HttpServletContainer(Logger logger, ClientFactory factory) {
        super(logger, factory);
    }

    
    /**
     * @return  the name of this Service (node-plugin)
     */
    public String name() {
        return SERVER_INFO;
    }

    /**
     * @return  some descriptive nonsense about the container
     */
    public String getServerInfo() {
        return SERVER_INFO;
    }

    
    /**
     * @return  the part of the URI path specifying the context
     */
    public abstract String getContextPath(String uripath);
    
    /**
     * @return  the part of the URI path specifying the servlet
     */
    public abstract String getServletPath(String uripath);

    /**
     * @return  the part of the URI path passed to the servlet
     *          as "path info" (the end)
     */
    public abstract String getPathInfo(String uripath);

    
    /**
     * @return  the ServletPool registered for the path
     *          (associated with a specific Servlet class)
     *          or null if not found
     */
    protected abstract ServletPool getServletPool(String uripath);

        
    protected ServletRequest getNextRequest(ServletRequest lastRequest,
                                            ServletResponse lastResponse,
                                            Connection conn) throws IOException {
        
        if (lastResponse == null
            || ((HttpServletResponseImpl) lastResponse).keepAlive()) {
            
            return new HttpServletRequestImpl(this, conn, INPUT_BUFFER,
                                              sessionHolder, "BASIC");
        }
        else return null;
    }

    protected ServletResponse getResponseFor(ServletRequest req)
                                            throws IOException {
        
        return new HttpServletResponseImpl((HttpServletRequestImpl) req,
                                           OUTPUT_BUFFER);
    }
    
    protected Servlet getServletFor(ServletRequest req_) throws ServletException,
                                                                UnavailableException {

        HttpServletRequestImpl req = (HttpServletRequestImpl) req_;

        if (!req.isValid())
            return new DumbServlet(HttpServletResponse.SC_BAD_REQUEST,
                                   req.getBadRequestMessage());
        
        String uri = req.getRequestPath();
        ServletPool pool = getServletPool(uri);
        
        if (pool == null) {
            // directory servlet check for URIs missing the terminal slash
            if (!uri.endsWith("/") && getServletPool(uri + "/") != null) {
                return new MovedServlet(uri + "/");
            }
            return new DumbServlet(HttpServletResponse.SC_NOT_FOUND);
        }        

        Servlet servlet = pool.getServlet();
        liveTable.put(req, pool);
        return servlet;
    }

    protected void returnServlet(ServletRequest req, Servlet servlet) {
        ServletPool pool = (ServletPool) liveTable.remove(req);
        if (pool != null)
            pool.returnServlet(servlet);
    }


    /**
     * Used to return canned responses.
     */
    protected final class DumbServlet extends HttpServlet {
        
        private final int sc;
        private final String msg;
        
        DumbServlet(int sc) {
            this(sc, null);
        }
        
        DumbServlet(int sc, String msg) {
            this.sc  = sc;
            this.msg = msg;
        }

        public void service(HttpServletRequest request,
                            HttpServletResponse response) throws IOException {
            if (msg == null)
                response.sendError(sc);
            else
                response.sendError(sc, msg);
        }
    }

    /**
     * Used for moved resources.
     */
    protected final class MovedServlet extends HttpServlet {

        private String location;

        public MovedServlet(String location) {
            this.location = location;
        }

        public void service(HttpServletRequest request,
                            HttpServletResponse response) throws IOException {

            response.setStatus(response.SC_MOVED_PERMANENTLY);
            response.addHeader("location", location);
        }

    }
}



