/*
 *$Id: FNPRawMessage.java,v 1.2 2002/03/10 22:50:29 hobbex Exp $

  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU General Public Licence (GPL) 
  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */

/**
 * This class represents a raw message in the Freenet Protocol
 * expected by Freenet.  It can be created either from
 * an InputStream or manually created from scratch.  It can
 * then by piped out to an OutputStream.  Methods are provided
 * for manipulation of normal fields, however the type of the
 * message and the trailing field should be set by direct
 * manipulation of fields.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <A HREF="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</A>
 **/
package freenet.presentation;
import freenet.*;
import freenet.support.*;
import freenet.support.io.*;
import java.io.*;
import java.io.EOFException;
import java.util.*;

public class FNPRawMessage extends RawMessage {
    
    // Constructors

    // FIXME -- what is the TransportHandler for??  -t.c.
    // god knows - oskar (consider it zapped)

    /**
     * Constructs a new RawMessage off an FNP Stream
     * @param i An InputStream of decrypted FNP data
     * @param th  The nodes TransportHandler 
     **/
    public FNPRawMessage(InputStream i) 
        throws InvalidMessageException, EOFException {

        //Core.logger.log(this,"Reading message",Logger.DEBUGGING);
        ReadInputStream ris = new ReadInputStream(i);
        //        PushbackInputStream in = new PushbackInputStream(i);
        fs = new FieldSet();

        try {
            // Read message type
            messageType = ris.readToEOF('\n','\r');

            //            System.out.println(messageType);
            trailingFieldName = fs.parseFields(ris,
                                               '\n', // ends a field
                                               '\r', // ignore before end
                                               '=', // delimits field
                                               '.'); // delimits subset
            setFields(ris);
        } catch (EOFException e) {
            if (messageType != null) {
                Core.logger.log(this, "Stream died while reading message of type: " + messageType, Logger.ERROR); 
            } else {
                // stream closed without getting a new message
                Core.logger.log(this, "Stream closed", Logger.DEBUGGING);
            }
            throw e;
        } catch (IOException e) {
            throw new EOFException("Could not parse message from stream : " + e);
        } catch(Exception e) {
            Core.logger.log(this, "Exception in RawMessage()", Logger.ERROR);
            e.printStackTrace();
        }
    }

    private void setFields(ReadInputStream in) {
        // Read and set the presentation related fields

        // setting KeepAlive
        String cvalue = fs.get("Connection");
        close   = cvalue != null && cvalue.equals("close");
        sustain = cvalue != null && cvalue.equals("sustain");

        // setting DataLength and trailing
        String dlvalue = fs.get("DataLength");
        if (dlvalue == null) {
            trailingFieldLength = 0;
        }
        else {
            trailingFieldLength = Fields.stringToLong(dlvalue);
            fs.remove("DataLength");
        }
        /*
        trailingFieldLength = dlvalue == null ? 0 : Fields.stringToLong(dlvalue);
        if (dlvalue != null) 
            fs.remove("DataLength");
        if (trailingFieldLength != 0) {// we have a trailing
            trailingFieldStream = in;
        } else {
            trailingFieldName = null; // no trailing
        }
        */
    }

    protected FNPRawMessage(String messageType, boolean close, 
                            boolean sustain, FieldSet fs,
                            long trailingLength, String trailingName, 
                            DiscontinueInputStream trailing) {
        super(messageType, close, sustain, fs == null ? new FieldSet() : fs, 
              trailingLength, trailingName, trailing);
    }
    
    // Public Methods

    public void writeMessage(OutputStream out) throws IOException {

        String key, value;
        
        WriteOutputStream writer=new WriteOutputStream(out);
        
        // Output message type
        writer.writeUTF(messageType, '\n');

        // Output tansport options

        if (close)
            fs.add("Connection","close");
        else if (sustain)
            fs.add("Connection","sustain");

        if (trailingFieldLength != 0)
            fs.add("DataLength",Fields.longToString(trailingFieldLength));

        // Output message fields

        fs.writeFields(writer,
                       (trailingFieldName == null ? 
                        "EndMessage" : 
                        trailingFieldName),
                       '\n', // ends a pair
                       '=',  // delimits a pair
                       '.'); // delimits a subset
        // empty writer
        writer.flush();
    }

    public String toString() {
        StringBuffer sb = new StringBuffer(400);
        sb.append(messageType).append("\n{Close=").append(close);
        sb.append(",Sustain=").append(sustain);
        sb.append(",DataLength=").append(trailingFieldLength);
        sb.append(",").append(fs.toString()).append("}");
        return sb.toString();
    }
}

/*
 *$Log: FNPRawMessage.java,v $
 *Revision 1.2  2002/03/10 22:50:29  hobbex
 *Added first concept implementation of a DistributionServlet, and did some small changes to help memory deallocation
 *
 *Revision 1.1.1.1  2002/01/13 05:24:53  sanity
 *Imported source
 *
 *Revision 1.10  2001/08/23 01:22:11  oierw
 *merge 0.4 to HEAD (hopefully)
 *
 *Revision 1.8.6.10  2001/08/22 05:10:28  scipient
 *removed a couple probably unneeded log messages
 *
 *Revision 1.8.6.9  2001/08/11 20:29:38  hobbex
 *ServletService plus fixes for sustained connections
 *
 *Revision 1.8.6.8  2001/08/08 02:26:51  scipient
 *now doing discontinue() on ReceiveInputStream after CB_RESTARTED/ABORTED
 *otherwise, now closing the conn. if read < toRead  (sorry)
 *
 *Revision 1.8.6.7  2001/07/20 19:52:51  hobbex
 *Implemented closing of connections the way I think I wanted it - ie it sends a Void with Connection=close, and the reading side always does the transport closing
 *
 *Revision 1.8.6.6  2001/07/14 19:56:23  hobbex
 *FNPClient builds now
 *
 *Revision 1.8.6.5  2001/04/22 18:30:44  scipient
 *ended up needing to make separate presentation classes for FCP after all
 *
 *Revision 1.8.6.4  2001/04/19 04:45:56  scipient
 *it lives.. Fred says NodeHello
 *
 *Revision 1.8.6.3  2001/04/12 22:20:49  hobbex
 *Touched fucking everything
 *
 *Revision 1.8.6.2  2001/01/24 22:22:14  hobbex
 *build fixes
 *
 *Revision 1.8.6.1  2001/01/18 02:36:24  hobbex
 *New Transport handeling structure for 0.4
 *
 *Revision 1.8  2000/09/16 18:34:42  hobbex
 *Further bugfixes
 *
 *Revision 1.7  2000/09/13 22:18:14  hobbex
 *Added Size bound byte to keys
 *
 *Revision 1.6  2000/09/08 00:28:53  bemann
 *This should now work with Kaffe.
 *
 *Revision 1.5  2000/08/25 15:46:08  hobbex
 *Bugfixes and some class moving (io support classes are now in Freenet.support.io)
 *
 *Revision 1.4  2000/08/21 18:32:42  hobbex
 *Added the ability for clients to encrypt data written out locally, and the Persist option to the protocol
 *
 *Revision 1.3  2000/08/13 10:29:22  hobbex
 *more stuff
 *
 *Revision 1.2  2000/08/12 16:15:57  hobbex
 *bugfix
 *
 *Revision 1.1  2000/08/12 06:40:35  hobbex
 *Renamed incorrectly named 'Transport' layer to Presentation
 *
 *Revision 1.5  2000/08/08 06:46:18  hobbex
 *Changed logger and fixed up ClientCore, ClientMessageHandler. The syntax to log a comment is now Freenet.Core.logger.log(javaa.lang.Object,java.lang.String,int) , see support.Logger and support.StandardLogger for more.
 *
 *Revision 1.4  2000/08/02 16:25:40  hobbex
 *CHK, KSK, and SVK support. Stream verifying, stream forking, and more.
 *
 *Revision 1.3  2000/06/13 18:09:55  hobbex
 *Reverted a bunch of my changes from a month ago because I wasn't allow to go on an do it my way (blah). Made FieldSet generic not presentation specific, and made it get used by DataProperties as well. Rewrote the message parsing code mostly, UTF8 encoding and decoding should work now, and Internet EOLs (\r\n) are better supported.
 *
 *Revision 1.2  2000/05/17 08:10:15  blanu
 *Moved things around to make code more reusable: broke up methods into smaller methods, made methods which didn't change state static. This is in preparation to make clients read FNP-form information out of the trailing fields of messages which contain metadata.
 *
 *Revision 1.1  2000/05/09 00:11:06  hobbex
 *The new Presentation layer for the standard Free Network Protocol (FNP - anyone have a better name?)
 *
 *Revision 1.21  2000/05/03 02:40:06  scgmille
 *
 *Crypto so far... WARNING: Breaks server
 *
 *Revision 1.20  2000/04/16 23:26:54  thong
 *Changed assorted Longs to longs
 *
 *Revision 1.19  2000/04/07 20:34:09  hobbex
 *Fixed building and running with Kaffe (scripts/kbuild.sh). Added canceling of timer callback and sending of requestfailed on loosing MessageMemory (not too well tested). And fixed a bug in the datastore + some exception catching in requestfailed.
 *
 *Revision 1.18  2000/04/05 22:14:08  hobbex
 *Unknown message fields are passed, and source is now optional for clients
 *
 *Revision 1.17  2000/03/31 14:36:13  hobbex
 *Abstracted returning a message, and made replies set KeepAlive=false if the original message did
 *
 *Revision 1.16  2000/03/31 12:51:38  thong
 *Make "stream died" error more intelligible
 *
 *Revision 1.15  2000/03/30 11:32:27  hobbex
 *Close connections after sending KeepAlive=false message
 *
 *Revision 1.14  2000/03/29 00:30:45  hal
 *Check for CR at end of EndMessage line.
 *
 *Revision 1.13  2000/03/27 21:02:32  hobbex
 *Sustained streams
 *
 *Revision 1.12  2000/02/29 06:46:56  blanu
 *Stray debugging println removed.
 *
 *Revision 1.11  2000/02/29 06:35:36  blanu
 *Handshaking is now in a message-like format. It actually uses RawMessage to construct the handshake.
 *RawMessage now supports the EndMessage field to end a message instead of EOF.
 *This is useful for handshaking and also for future Keepalive connections.
 *
 *Revision 1.10  2000/02/17 14:12:54  hobbex
 *catch unknown message types
 *
 *Revision 1.9  2000/02/17 12:52:23  hobbex
 *removed runtime exception on bad connection (annoying)
 *
 *Revision 1.8  2000/02/12 13:59:57  hobbex
 *fixed tunneling on inserts for large files. Britney never sounded so good!
 *
 *Revision 1.7  2000/02/11 20:12:01  hobbex
 *Fixed tunneling to use only one thread. Changed Rawmessage to write to output streams rather than return inputstreams. Fixed releasing of DataStream when a message is dropped.
 *
 *Revision 1.6  2000/02/09 23:56:39  hobbex
 *Preliminary tunneling of DataReplies and DataInserts (for now)
 *
 *Revision 1.5  2000/01/27 15:45:39  sanity
 *Some bug fixes, added localAddress field to Node.java which is used by
 *DataRequest and DataReply messages to set dataSource field.  Also removed
 *untidy debugging stuff from RawMessage.
 *
 *Revision 1.4  2000/01/11 06:32:05  blanu
 *fixed problem with hex uniqueid and name=value field format
 *
 *Revision 1.3  2000/01/11 02:02:51  blanu
 *changed from -: to = field format
 *
 *Revision 1.2  2000/01/03 16:32:07  hobbex
 *Changed various println to Logger - think I got them all
 *
 *Revision 1.1.1.1  1999/12/31 11:53:02  sanity
 *Initial import from OpenProjects
 *
 *Revision 1.4  1999/12/29 07:14:06  michael
 *Additional LF after CR of trailing field - bug fixed
 *
 *
 */
