/* 

                          Firewall Builder

                 Copyright (C) 2000,2001 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: policy.c,v 1.39 2001/12/14 22:22:01 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Puplic License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "iptables.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
 
static GSList     *ready_subrules;
static GSList     *new_chains=NULL;
static int         srcdstrule_num=0;

/*
 *  Rule options:
 */
char  log_level[256];
char  log_prefix[256];
char  log_tcp_seq[256];
char  log_tcp_opt[256];
char  log_ip_opt[256];
int   use_numeric_log_levels;
char  limit[256];
char  action_on_reject[256];
int   no_optimisation;
int   no_state;
int   manage_virtual_addr;    /* this does not apply as it may be only global
			option, but we have it here to satisfy list of
			arguments for parseOptions function */

void rulePrologue(xmlNodePtr firewall,
		  xmlNodePtr iface,
		  xmlNodePtr rule,
		  int subrules_n)
{
    xmlNodePtr src,dst,srv;
    xmlNodePtr  options;

    int           num=getInt(rule,"position");
    const char* comm =getStr(rule,"comment");
    

    strcpy(log_level,"");
    strcpy(log_prefix,"");
    strcpy(log_tcp_seq,"");
    strcpy(log_tcp_opt,"");
    strcpy(log_ip_opt,"");
    strcpy(limit,"");
    strcpy(action_on_reject,"");

    no_optimisation=0;
    no_state=0;

    options=getFirstChildXmlNode(rule,"PolicyRuleOptions");
    
    if ( options!=NULL )
	parseOptions( options,
		      log_level,
		      log_prefix,
		      log_tcp_seq,
		      log_tcp_opt,
		      log_ip_opt,
		      &use_numeric_log_levels,
		      limit,
		      action_on_reject,
		      &no_optimisation,
		      &no_state,
		      &manage_virtual_addr);


    
    if (iface)  sprintf(rule_chain,"IRULE_%d_%s"  ,num ,getName(iface));
    else        sprintf(rule_chain,"RULE_%d"      ,num);


    fprintf(ofile,"#\n");
    fprintf(ofile,"#   %s Rule #%d\n", (iface!=NULL)?"Interface":"" , num );
    fprintf(ofile,"#\n");
    if (comm) {
	fprintf(ofile,"#    %s\n",comm);
	fprintf(ofile,"#\n");
    }

    src=getFirstChildXmlNode(rule,"Src");
    dst=getFirstChildXmlNode(rule,"Dst");
    srv=getFirstChildXmlNode(rule,"Srv");

    if (getBool(src,"neg") || getBool(dst,"neg") || getBool(srv,"neg") ) {

	if (iface)  {
	    sprintf(temp_i_chain,"I_IRULE_%d_%s",num ,getName(iface));
	    sprintf(temp_o_chain,"O_IRULE_%d_%s",num ,getName(iface));
	    sprintf(temp_f_chain,"F_IRULE_%d_%s",num ,getName(iface));
	}
	else {
	    sprintf(temp_i_chain,"I_RULE_%d"    ,num);
	    sprintf(temp_o_chain,"O_RULE_%d"    ,num);
	    sprintf(temp_f_chain,"F_RULE_%d"    ,num);
	}
	fprintf(ofile,"\n");
	
    }
    else {
	strcpy(temp_i_chain,"");
	strcpy(temp_o_chain,"");
	strcpy(temp_f_chain,"");
    }
}

void ruleEpilogue(xmlNodePtr firewall,
		  xmlNodePtr iface,
		  xmlNodePtr rule,
		  GSList     *subrules)
{
    GSList   *list_ptr1;
    elementaryRule *erule, *trule, *t2rule;

    xmlNodePtr src,dst,srv;
    char  tmpstr[1024], tmp2str[1024];
    char *cptr;
    char  action_chain[64];
    char  log_options[4096];

    const char  *action;
    int          num, log;


    src=getFirstChildXmlNode(rule,"Src");
    dst=getFirstChildXmlNode(rule,"Dst");
    srv=getFirstChildXmlNode(rule,"Srv");

    if (getBool(src,"neg") || getBool(dst,"neg") || getBool(srv,"neg")) {

	for (list_ptr1=subrules;
	     list_ptr1!=NULL;
	     list_ptr1=g_slist_next(list_ptr1)) {

	    erule=(elementaryRule*)(list_ptr1->data);

	    if (erule->num==-1)  continue;
	    if (erule->final==1) continue;
	    
	    trule=createElementaryPolicyRule();
	    copyElementaryPolicyRule(trule,erule);

	    if (trule->src_neg)  {
		trule->src=NULL;
		trule->t->p_src="0/0";
	    }
	    if (trule->dst_neg)  {
		trule->dst=NULL;
		trule->t->p_dst="0/0";
	    }
	    if (trule->srv_neg)  {
		trule->srv=NULL;
		trule->t->p_sprt=NULL;
		trule->t->p_dprt=NULL;
		trule->t->p_proto=NULL;
		trule->t->p_icmp_type=NULL;
		trule->t->p_icmp_code=NULL;
	    }

	    
	    trule->p_action  = copyStr(rule_chain);

	    if ( strcasecmp(erule->master_group,"INPUT")==SAME)
		trule->p_group   = copyStr(temp_i_chain);
	    if ( strcasecmp(erule->master_group,"OUTPUT")==SAME)
		trule->p_group   = copyStr(temp_o_chain);
	    if ( strcasecmp(erule->master_group,"FORWARD")==SAME)
		trule->p_group   = copyStr(temp_f_chain);

	    trule->t->state_new  = 0;
	    
	    if ( iface && (erule->p_direction==NULL || 
		 strcasecmp(erule->p_direction,"both")==SAME ) ) {

		t2rule=createElementaryPolicyRule();
		copyElementaryPolicyRule(t2rule,trule);
		
		trule->p_direction="inbound";
		trule->final     = 1;
		processElementaryPolicyRule(firewall,iface,rule,trule);

		t2rule->p_direction="outbound";
		t2rule->final     = 1;
		processElementaryPolicyRule(firewall,iface,rule,t2rule);

	    } else {
		trule->final     = 1;
		processElementaryPolicyRule(firewall,iface,rule,trule);
	    }
	}
    }

    action=getStr(rule,"action");
    log=getBool(rule,"log");
    num=getInt(rule,"position");

    strcpy(action_chain,"");
    strcpy(log_options,"");
    
    if (strcasecmp(action,"accept")==0)  strcat(action_chain,"ACCEPT");
    if (strcasecmp(action,"reject")==0) {
	strcat(action_chain,"REJECT");
	if (strlen(action_on_reject)) {
	    strcat(action_chain," ");
	    strcat(action_chain,action_on_reject);
	} else
	    if (strlen(global_action_on_reject)) {
		strcat(action_chain," ");
		strcat(action_chain,global_action_on_reject);
	    }
    }
    if (strcasecmp(action,"deny")==0)  strcat(action_chain,"DROP");
	
    if (log) {

	strcpy(tmpstr,(strlen(log_level)!=0)?log_level:global_log_level);
	strcat(tmpstr,(strlen(log_prefix)!=0)?log_prefix:global_log_prefix);
	strcat(tmpstr,(strlen(log_tcp_seq)!=0)?log_tcp_seq:global_log_tcp_seq);
	strcat(tmpstr,(strlen(log_tcp_opt)!=0)?log_tcp_opt:global_log_tcp_opt);
	strcat(tmpstr,(strlen(log_ip_opt)!=0)?log_ip_opt:global_log_ip_opt);
	strcat(tmpstr,(strlen(limit)!=0)?limit:global_log_limit);

	if ( (cptr=strstr(tmpstr,"%I"))!=NULL) {
	    strncpy(tmp2str, tmpstr, cptr-tmpstr);
	    tmp2str[cptr-tmpstr]='\0';
	    strcat(tmp2str , getName(iface) );
	    strcat(tmp2str, cptr+2 );
	    strcpy(tmpstr,tmp2str);
	}
	
	if ( (cptr=strstr(tmpstr,"%N"))!=NULL) {
	    strncpy(tmp2str, tmpstr, cptr-tmpstr);
	    tmp2str[cptr-tmpstr]='\0';
	    sprintf(tmp2str+strlen(tmp2str),"%d",num);
	    strcat(tmp2str, cptr+2 );
	    strcpy(tmpstr,tmp2str);
	}
	
	if ( (cptr=strstr(tmpstr,"%A"))!=NULL) {
	    strncpy(tmp2str, tmpstr, cptr-tmpstr);
	    tmp2str[cptr-tmpstr]='\0';
	    strcat(tmp2str , action );
	    strcat(tmp2str, cptr+2 );
	    strcpy(tmpstr,tmp2str);
	}
	
	strcpy(log_options, global_log_parameters);
	strcat(log_options, " ");
	strcat(log_options, tmpstr);
	
	trule=createElementaryPolicyRule();
	trule->p_group      = copyStr(rule_chain);
	trule->p_action       = "LOG";
	trule->t->p_options = copyStr(log_options);
	trule->t->state_new  = 0;
	if (no_state) trule->t->state_new= 0;
	storePolicyRule(trule);

    }
    
    trule=createElementaryPolicyRule();
    trule->p_group      = copyStr(rule_chain);
    trule->p_action       = copyStr(action_chain);
    trule->t->p_options = copyStr(limit);
    trule->t->state_new  = 0;
    if (no_state) trule->t->state_new= 0;
    storePolicyRule(trule);
}



void processElementaryPolicyRule(xmlNodePtr firewall,
				 xmlNodePtr iface,
				 xmlNodePtr rule,
				 elementaryRule *erule)
{
    xmlNodePtr      c1;
    elementaryRule *trule;
    const char     *ifaddr;
    int             fs,fd;
    int             on_loopback;
    
    if (iface!=NULL && erule->p_iface==NULL) erule->p_iface=getName(iface);

    
    if (erule->final) {
	parsePolicyRule( rule,erule);
	return;
    }


    if ( iface && ( erule->p_direction==NULL ||
	 strcasecmp(erule->p_direction,"both")==SAME ) ) {

	trule=createElementaryPolicyRule();
	copyElementaryPolicyRule( trule , erule );
	trule->p_direction="inbound";
	if (no_state) trule->t->state_new= 0;
	processElementaryPolicyRule(firewall,iface,rule,trule);
    
	trule=createElementaryPolicyRule();
	copyElementaryPolicyRule( trule , erule );
	trule->p_direction="outbound";
	if (no_state) trule->t->state_new= 0;
	processElementaryPolicyRule(firewall,iface,rule,trule);

	return;
    }

    
    if ( (erule->p_direction==NULL ||
	  strcasecmp(erule->p_direction,"outbound")==0) &&
	 (generate_input_output_groups_for_any &&
	  erule->src && isAny(erule->src) ) ) {
	
	trule=createElementaryPolicyRule();
	copyElementaryPolicyRule( trule , erule );
	trule->src=firewall;
	if (no_state) trule->t->state_new= 0;
	
	processElementaryPolicyRule(firewall,iface,rule,trule);
    }

    if ( (erule->p_direction==NULL ||
	  strcasecmp(erule->p_direction,"inbound")==0) &&
	 (generate_input_output_groups_for_any &&
	  erule->dst && isAny(erule->dst) ) ) {
	
	trule=createElementaryPolicyRule();
	copyElementaryPolicyRule( trule , erule );
	trule->dst=firewall;
	if (no_state) trule->t->state_new= 0;

	processElementaryPolicyRule(firewall,iface,rule,trule);
    }

    if ( ( erule->p_direction==NULL ||
	 strcasecmp(erule->p_direction,"both")==SAME ) &&
	 cmpObjects(erule->src , firewall)==SAME &&  
	 cmpObjects(erule->dst , firewall)==SAME ) {

	trule=createElementaryPolicyRule();
	copyElementaryPolicyRule( trule , erule );
	trule->p_direction="inbound";
	if (no_state) trule->t->state_new= 0;
	processElementaryPolicyRule(firewall,iface,rule,trule);
    
	trule=createElementaryPolicyRule();
	copyElementaryPolicyRule( trule , erule );
	trule->p_direction="outbound";
	if (no_state) trule->t->state_new= 0;
	processElementaryPolicyRule(firewall,iface,rule,trule);

	return;

    }

/* decide on the chain this rule should go to and on the action chain  */

    erule->p_group="FORWARD";
    
    if ( (erule->p_direction==NULL ||
	  strcasecmp(erule->p_direction,"inbound")==0 ) &&
	 cmpObjects(erule->dst , firewall)==SAME )  erule->p_group="INPUT";

    if ( (erule->p_direction==NULL ||
	  strcasecmp(erule->p_direction,"outbound")==0 ) &&
	 cmpObjects(erule->src , firewall)==SAME )  erule->p_group="OUTPUT";

/* check if destination address is broadcast */
    for(c1=firewall->xmlChildrenNode; c1; c1=c1->next) {
	if ( xmlIsBlankNode(c1) ) continue;
	if (isElement(c1,"Interface")) {
	    if ( isDynamicInterface(c1) ) continue;

	    if ( getBroadcastAddress(c1) == convertAddress(erule->dst) ) 
		erule->p_group="INPUT";

	}
    }
    
    
    erule->master_group = copyStr( erule->p_group );
    
    if (erule->src_neg || erule->dst_neg || erule->srv_neg) {

	if ( strcasecmp(erule->p_group,"INPUT")==SAME) {
	    trule=createElementaryPolicyRule();
	    copyElementaryPolicyRule( trule , erule );

	    trule->src=NULL;
	    trule->dst=NULL;
	    trule->srv=NULL;
	    trule->p_action  = copyStr(temp_i_chain);
	    trule->t->state_new  = 0;
	    trule->final     = 1;
	    processElementaryPolicyRule(firewall,iface,rule,trule);
	}

	if ( strcasecmp(erule->p_group,"OUTPUT")==SAME) {
	    trule=createElementaryPolicyRule();
	    copyElementaryPolicyRule( trule , erule );

	    trule->src=NULL;
	    trule->dst=NULL;
	    trule->srv=NULL;
	    trule->p_action  = copyStr(temp_o_chain);
	    trule->t->state_new  = 0;
	    trule->final     = 1;
	    processElementaryPolicyRule(firewall,iface,rule,trule);
	}

	if ( strcasecmp(erule->p_group,"FORWARD")==SAME) {
	    trule=createElementaryPolicyRule();
	    copyElementaryPolicyRule( trule , erule );

	    trule->src=NULL;
	    trule->dst=NULL;
	    trule->srv=NULL;
	    trule->p_action  = copyStr(temp_f_chain);
	    trule->t->state_new  = 0;
	    trule->final     = 1;
	    processElementaryPolicyRule(firewall,iface,rule,trule);
	}

	erule->p_group      = copyStr( trule->p_action );
	erule->p_action     = "RETURN";

    } else {

	erule->p_action  = copyStr(rule_chain);

    }


    
/*
 *  The follwoing fragment handles the case when firewall object is source
 *  in the rule which goes into INPUT chain, or it is destination in
 *  the rule which goes into OUTPUT chain. We will go over all interfaces
 *  of the firewall object and generate multiple rules using address of
 *  each interface instead.
 *
 *  Exceptions: if chain is INPUT and firewall object is in
 *  destination, then we do not need to pick every interface. The same
 *  goes to the case with chain OUTPUT and firewall object in source
 */
    
    fs=fd=0;

    if ( cmpObjects(erule->src , firewall)==SAME ) fs=1;
    if ( cmpObjects(erule->dst , firewall)==SAME ) fd=1;

    if (fs==0 && fd==0) {
	parsePolicyRule( rule,erule);
	return;
    }

    
    if (fs==fd) {
	if ( strcasecmp(erule->p_group,"INPUT")==SAME  ) fd=0;
	if ( strcasecmp(erule->p_group,"OUTPUT")==SAME ) fs=0;
    }

    if ( (strcasecmp(erule->p_group,"INPUT")==SAME && fd ) ||
	 (strcasecmp(erule->p_group,"OUTPUT")==SAME && fs ) ) {

	parsePolicyRule( rule,erule);
	return;
    }

    if (fs || fd ) {

	ifaddr=getStr(iface,"address");
	on_loopback=(ifaddr && strcmp(ifaddr,"127.0.0.1")==SAME);
	
	for(c1=firewall->xmlChildrenNode; c1; c1=c1->next) {
	    if ( xmlIsBlankNode(c1) ) continue;
	    if (isElement(c1,"Interface")) {
		if ( isDynamicInterface(c1) ) continue;

		trule=createElementaryPolicyRule();
		copyElementaryPolicyRule(trule , erule);
		if (no_state) trule->t->state_new= 0;

		if (! on_loopback) {
		    ifaddr=getStr(c1,"address");
		    if (strcmp(ifaddr,"127.0.0.1")==SAME) continue;
		}
		if (fs) {
		    trule->src=c1;
		    parsePolicyRule( rule,trule);
		}
		if (fd) {
		    trule->dst=c1;
		    parsePolicyRule( rule,trule);
		}

	    }
	}


    } else 	
	parsePolicyRule( rule,erule);
    
}



void processPolicyRule(xmlNodePtr firewall,
		       xmlNodePtr iface,
		       xmlNodePtr rule,
		       const char *comment,
		       GSList     *subrules)
{
    GSList   *list_ptr1;
    elementaryRule *erule;
    int       i,o,f, srn;
    xmlNodePtr         c1;
    int       has_interfaces;

    i=o=f=0;


/*    dumpPolicyRule(stderr,doc,rule); */

    ready_subrules=NULL;
    new_chains=NULL;
    
    for(c1=firewall->xmlChildrenNode; c1; c1=c1->next) {
	if ( xmlIsBlankNode(c1) ) continue;
	if (isElement(c1,"Interface")) {
	    has_interfaces=1;
	    break;
	}
    }
    if (!has_interfaces) {
	fprintf(stderr, "Firewall does not have any interfaces\n");
	exit(1);
    }
    
    rulePrologue(firewall, iface, rule , g_slist_length(subrules) );

    srn=0;
    for (list_ptr1=subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	erule=(elementaryRule*)(list_ptr1->data);

	if (erule->num==-1) continue;

	erule->subrule_no=srn;
	if (no_state) erule->t->state_new= 0;

	processElementaryPolicyRule(firewall,iface,rule,erule);
	
	srn++;
    }
    
    ruleEpilogue(firewall, iface, rule , ready_subrules);

    if (! global_no_optimisation && ! turn_off_optimisation)
	optimisePolicyRules();

    printPolicyRule();
}


void parsePolicyRule(xmlNodePtr rule,elementaryRule *erule)
{
    xmlNodePtr  src,dst,srv;
    char        str[80];
    
    src=erule->src;
    dst=erule->dst;
    srv=erule->srv;


    if (erule->t->p_options) {
	erule->t->p_options=NULL;
    }

    
    if (src!=NULL) processSrc(src,erule->t);

    if (dst!=NULL) processDst(dst,erule->t);

    if (srv==NULL || isAny(srv) ) {
	storePolicyRule(erule);
	return;
    }

    processSrv(srv,erule->t);

    processTime(erule->time, str);
    erule->p_time=fwb_strdup(str);
    
    storePolicyRule(erule);
}


void storePolicyRule(elementaryRule *erule)
{
    ready_subrules=g_slist_append(ready_subrules , erule);

}

/*
 *  returns 1 if rule r1 "shades" rule r2, that is r1's destination
 *  is broader than r2's, or r1's source is broader than r2's
 *  Otherwise returns 0;
 */
int checkRuleShading(elementaryRule *r1,elementaryRule *r2)
{
    triplet *t1, *t2;
    int      f1,f2,f3,f4;
    int      res;
    
    f1=f2=f3=f4=0;
    res=0;
    
    if (cmpParameter(r1->p_group,r2->p_group)==SAME &&
	cmpParameter(r1->p_action,r2->p_action)==SAME &&
/*	cmpParameter(r1->p_direction,r2->p_direction)==SAME && */
	cmpParameter(r1->p_iface,r2->p_iface)==SAME &&
	cmpParameter(r1->p_time,r2->p_time)==SAME ) {

	t1= r1->t;
	t2= r2->t;
	
	if (t1->p_src==NULL) { t1->p_src="0/0"; f1=1; }
	if (t1->p_dst==NULL) { t1->p_dst="0/0"; f2=1; }
	if (t2->p_src==NULL) { t2->p_src="0/0"; f3=1; }
	if (t2->p_dst==NULL) { t2->p_dst="0/0"; f4=1; }
				
	if ( cmpParameter(t1->p_src ,t2->p_src)==SAME &&
	     cmpParameter(t1->p_sprt,t2->p_sprt)==SAME &&
	     cmpParameter(t1->p_dprt,t2->p_dprt)==SAME ) {
/* compare dst */

	    if (strcmp(t1->p_dst,"0/0")==SAME &&
		strcmp(t2->p_dst,"0/0")!=SAME )  res=1;
	}
	
	if ( cmpParameter(t1->p_dst, t2->p_dst)==SAME &&
	     cmpParameter(t1->p_sprt,t2->p_sprt)==SAME &&
	     cmpParameter(t1->p_dprt,t2->p_dprt)==SAME ) {
/* compare src */
	    
	    if (strcmp(t1->p_src,"0/0")==SAME &&
		strcmp(t2->p_src,"0/0")!=SAME )  res=1;
	}
    }

    if (f1) t1->p_src=NULL;
    if (f2) t1->p_dst=NULL;
    if (f3) t2->p_src=NULL;
    if (f4) t2->p_dst=NULL;

    return res;
}

void optimisePolicyRules()
{

    elementaryRule     *prptr1, *prptr2, *trule;
    GSList             *list_ptr1, *list_ptr2;
    int                 i,j;
    int                 srccnt, dstcnt;
    char                src_dst_rule_chain[64];
    int                 multiport_src_cntr, multiport_dst_cntr;

/*
 *  Functions checking rule equivalence and rule shading are very picky. Let's
 *  first make sure crucial rule parameters are compatible
 */
    
/*
 *  no need in destination in INPUT chain and source in OUTPUT chain
 *  restore direction using chain as a clue
 */
    for (list_ptr1=ready_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	prptr1=(elementaryRule*)(list_ptr1->data);
	if (prptr1->num==-1) continue;
	    

	if (prptr1->master_group &&
	    strcasecmp(prptr1->master_group,"INPUT")==SAME) {
		
	    prptr1->dst=NULL;
	    prptr1->t->p_dst=NULL;

	    if (prptr1->p_direction==NULL) prptr1->p_direction="inbound";
	}
	    
	if (prptr1->master_group &&
	    strcasecmp(prptr1->master_group,"OUTPUT")==SAME) {
		
	    prptr1->src=NULL;
	    prptr1->t->p_src=NULL;

	    if (prptr1->p_direction==NULL) prptr1->p_direction="outbound";
	}
    }

/*
 *  remove source and destination if they are "0/0"
 */
    for (list_ptr1=ready_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	prptr1=(elementaryRule*)(list_ptr1->data);
	if (prptr1->num==-1) continue;
	    
	if (prptr1->t->p_src!=NULL && strcmp(prptr1->t->p_src,"0/0")==SAME) {
	    prptr1->src=NULL;
	    prptr1->t->p_src=NULL;
	}

	if (prptr1->t->p_dst!=NULL && strcmp(prptr1->t->p_dst,"0/0")==SAME) {
	    prptr1->dst=NULL;
	    prptr1->t->p_dst=NULL;
	}
	
    }


/*
 *  check for identical rules and rule shading
 *
 */
    for (i=0,list_ptr1=ready_subrules;
	 list_ptr1!=NULL;
	 i++,list_ptr1=g_slist_next(list_ptr1)) {

	prptr1=(elementaryRule*)(list_ptr1->data);
	if (prptr1->num==-1) continue;

	for (j=0,list_ptr2=g_slist_next(list_ptr1);
	     list_ptr2!=NULL;
	     j++,list_ptr2=g_slist_next(list_ptr2)) {

	    prptr2=(elementaryRule*)(list_ptr2->data);
	    if (prptr2->num==-1) continue;

	    if (cmpPPolicyRules( prptr1,prptr2 )==SAME) {
		prptr2->num=-1;
		continue;
	    }
	    if (checkRuleShading(prptr1,prptr2)==1) {
		prptr2->num=-1;
		continue;
	    }

	    if (checkRuleShading(prptr2,prptr1)==1) {
		prptr1->num=-1;
		break;
	    }
	}
    }


    
/*
 *  Attempt to use multiport module
 */
    for (list_ptr1=ready_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	prptr1=(elementaryRule*)(list_ptr1->data);
	if (prptr1->num==-1) continue;

	if ( cmpParameter(prptr1->t->p_proto,"tcp")!=0 &&
	     cmpParameter(prptr1->t->p_proto,"udp")!=0 ) continue;

	multiport_src_cntr=multiport_dst_cntr=0;
    
	for (list_ptr2=g_slist_next(list_ptr1);
	     list_ptr2!=NULL;
	     list_ptr2=g_slist_next(list_ptr2)) {

	    prptr2=(elementaryRule*)(list_ptr2->data);
	    if (prptr2->num==-1) continue;

	    if ( cmpParameter(prptr1->p_iface,    prptr2->p_iface)==SAME &&
		 cmpParameter(prptr1->p_group,    prptr2->p_group)==SAME &&
		 cmpParameter(prptr1->p_action,   prptr2->p_action)==SAME &&
		 cmpParameter(prptr1->p_direction,prptr2->p_direction)==SAME &&
		 cmpParameter(prptr1->p_log,      prptr2->p_log)==SAME &&

		 cmpParameter(prptr1->t->p_proto,   prptr2->t->p_proto)==0 &&
		 cmpParameter(prptr1->t->p_options, prptr2->t->p_options)==0 &&
		 cmpParameter(prptr1->t->p_src,     prptr2->t->p_src)==0 &&
		 cmpParameter(prptr1->t->p_mac_src, prptr2->t->p_mac_src)==0 &&
		 cmpParameter(prptr1->t->p_dst,     prptr2->t->p_dst)==0 ) {

		if (cmpParameter(prptr1->t->p_sprt, prptr2->t->p_sprt)==0 &&
		    cmpParameter(prptr1->t->p_dprt, prptr2->t->p_dprt)!=0 &&
		    strchr(prptr1->t->p_dprt,':')==NULL &&
		    strchr(prptr2->t->p_dprt,':')==NULL ) {
/* source ports and _all_ other parameters are equal, but dest. ports
 * are different
 */
		    prptr1->multiport=MULTIPORT_DST;
		    prptr1->t->p_dprt=fwb_realloc(prptr1->t->p_dprt,
						  strlen(prptr1->t->p_dprt)+
						  strlen(prptr2->t->p_dprt)+2);
		    strcat(prptr1->t->p_dprt,",");
		    strcat(prptr1->t->p_dprt,prptr2->t->p_dprt);
		    prptr2->num=-1;	    

		    if ( ++multiport_dst_cntr>12 ) break;
		    continue;
		}
		if (cmpParameter(prptr1->t->p_sprt, prptr2->t->p_sprt)!=0 &&
		    cmpParameter(prptr1->t->p_dprt, prptr2->t->p_dprt)==0 &&
		    strchr(prptr1->t->p_sprt,':')==NULL &&
		    strchr(prptr2->t->p_sprt,':')==NULL ) {
/* destination ports and _all_ other parameters are equal, but source ports
 * are different
 */
		    prptr1->multiport=MULTIPORT_SRC;
		    prptr1->t->p_sprt=fwb_realloc(prptr1->t->p_sprt,
						  strlen(prptr1->t->p_sprt)+
						  strlen(prptr2->t->p_sprt)+2);
		    strcat(prptr1->t->p_sprt,",");
		    strcat(prptr1->t->p_sprt,prptr2->t->p_sprt);
		    prptr2->num=-1;	    

		    if ( ++multiport_src_cntr>12 ) break;
		    continue;
		}
		
	    }
	    

	}
    }

/*
 *  Optimize for source and destination
 */

    srcdstrule_num=0;
    
    for (list_ptr1=ready_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	prptr1=(elementaryRule*)(list_ptr1->data);
	if (prptr1->num==-1) continue;
	if (strncmp(prptr1->p_group +1,"SRCRULE",7)==SAME) continue;
	if (strncmp(prptr1->p_group +1,"DSTRULE",7)==SAME) continue;
	
/*
 *  first, we count how many rules use the same source or destination address
 */
	srccnt=dstcnt=0;
	for (list_ptr2=g_slist_next(list_ptr1);
	     list_ptr2!=NULL;
	     list_ptr2=g_slist_next(list_ptr2)) {

	    prptr2=(elementaryRule*)(list_ptr2->data);
	    if (prptr2->num==-1) continue;

	    if ( cmpParameter(prptr1->p_iface,    prptr2->p_iface)==SAME &&
		 cmpParameter(prptr1->p_group,    prptr2->p_group)==SAME &&
		 cmpParameter(prptr1->p_action,   prptr2->p_action)==SAME &&
		 cmpParameter(prptr1->p_direction,prptr2->p_direction)==SAME){

		if (prptr1->t->p_src && prptr2->t->p_src &&
		    cmpParameter(prptr1->t->p_src, prptr2->t->p_src)==0 ) {
/* the same source address */
		    srccnt++;
		    continue;
		}
		if (prptr1->t->p_dst && prptr2->t->p_dst &&
		    cmpParameter(prptr1->t->p_dst, prptr2->t->p_dst)==0 ) {
/* the same destination address */
		    dstcnt++;
		    continue;
		}
	    }
	}
	if (srccnt>1) {  /* the same source address appears more than twice */

	    trule=createElementaryPolicyRule();

	    sprintf(src_dst_rule_chain,"%cSRCRULE_%d_%d"  ,
		    prptr1->p_group[0],
		    prptr1->num , ++srcdstrule_num );

	    
	    if (prptr1->p_iface) trule->p_iface = fwb_strdup(prptr1->p_iface );
	    if (prptr1->p_group) trule->p_group = fwb_strdup(prptr1->p_group );
	    if (prptr1->p_direction)
		trule->p_direction = fwb_strdup(prptr1->p_direction);
	    trule->p_action    = fwb_strdup(src_dst_rule_chain);
	    trule->t->p_src    = fwb_strdup(prptr1->t->p_src);
	    trule->t->state_new    = prptr1->t->state_new;
	    
	    ready_subrules=g_slist_insert(ready_subrules, trule, 0);


	    for (list_ptr2=g_slist_next(list_ptr1);
		 list_ptr2!=NULL;
		 list_ptr2=g_slist_next(list_ptr2)) {

		prptr2=(elementaryRule*)(list_ptr2->data);
		if (prptr2->num==-1) continue;

		if ( cmpParameter(prptr1->p_iface,  prptr2->p_iface)==SAME &&
		     cmpParameter(prptr1->p_group,  prptr2->p_group)==SAME &&
		     cmpParameter(prptr1->p_action, prptr2->p_action)==SAME &&
		     cmpParameter(prptr1->p_direction,prptr2->p_direction)==SAME &&
		     cmpParameter(prptr1->t->p_src, prptr2->t->p_src)==0 ) {

		    prptr2->p_group = fwb_strdup( src_dst_rule_chain );
		}		
		
	    }
	    prptr1->p_group    = fwb_strdup(src_dst_rule_chain);

	    continue;

	}

	if (dstcnt>1) {  /* the same destination address appears more than twice */

	    trule=createElementaryPolicyRule();

	    sprintf(src_dst_rule_chain,"%cDSTRULE_%d_%d"  ,
		    prptr1->p_group[0],
		    prptr1->num , ++srcdstrule_num );

	    
	    if (prptr1->p_iface) trule->p_iface = fwb_strdup(prptr1->p_iface );
	    if (prptr1->p_group) trule->p_group = fwb_strdup(prptr1->p_group );
	    if (prptr1->p_direction)
		trule->p_direction = fwb_strdup(prptr1->p_direction);
	    trule->p_action    = fwb_strdup(src_dst_rule_chain);
	    trule->t->p_dst    = fwb_strdup(prptr1->t->p_dst);
	    trule->t->state_new    = prptr1->t->state_new;
	    
	    ready_subrules=g_slist_insert(ready_subrules, trule, 0);


	    for (list_ptr2=g_slist_next(list_ptr1);
		 list_ptr2!=NULL;
		 list_ptr2=g_slist_next(list_ptr2)) {

		prptr2=(elementaryRule*)(list_ptr2->data);
		if (prptr2->num==-1) continue;

		if ( cmpParameter(prptr1->p_iface,  prptr2->p_iface)==SAME &&
		     cmpParameter(prptr1->p_group,  prptr2->p_group)==SAME &&
		     cmpParameter(prptr1->p_action, prptr2->p_action)==SAME &&
		     cmpParameter(prptr1->p_direction,prptr2->p_direction)==SAME &&
		     cmpParameter(prptr1->t->p_dst, prptr2->t->p_dst)==0 ) {
		
		    prptr2->p_group = fwb_strdup( src_dst_rule_chain );
		}		
		
	    }
	    prptr1->p_group    = fwb_strdup(src_dst_rule_chain);
	}
	

	
    }


/*
 *  remove all rules in chain FORWARD if interface is loopback
 */
    for (list_ptr1=ready_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	prptr1=(elementaryRule*)(list_ptr1->data);
	if (prptr1->num==-1) continue;

	if (prptr1->p_iface!=NULL &&
	    strcmp(prptr1->p_iface,"lo")==SAME &&
	    strcmp(prptr1->p_group,"FORWARD")==SAME) prptr1->num=-1;
    }
}


void printPolicyRule()
{
    elementaryRule     *erule;
    GSList             *list_ptr1;
    
    for (list_ptr1=ready_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {
	
	erule=(elementaryRule*)(list_ptr1->data);
	if (erule->num==-1) continue;

	printElementaryRule( ofile, erule );
    }
}

gint _cmpChain(gconstpointer  a, gconstpointer  b)
{
    return strcmp( (char*)a , (char*)b );
}

/*
 * push names for all built-in chains into new_chains list
 */

void newChain(FILE *f,const char* chain)
{
    GSList             *lp;
    lp=g_slist_find_custom(new_chains, (gpointer)chain, _cmpChain );
    if (lp==NULL) {
	fprintf(f,"iptables -N %s\n",chain);
	new_chains=g_slist_append(new_chains, (gpointer)chain);
    }
}

void printElementaryRule(FILE *f, elementaryRule *erule )
{    
    if (strcmp(erule->p_group,"INPUT")!=SAME &&
	strcmp(erule->p_group,"OUTPUT")!=SAME &&
	strcmp(erule->p_group,"FORWARD")!=SAME ) newChain(f,erule->p_group);

/*
 * sometimes action has more than just predefined chain name
 * e.g. REJECT - it looks like "REJECT -p tcp --reject-with tcp-reset"
 */
    if (strncmp(erule->p_action,"INPUT",5)!=SAME &&
	strncmp(erule->p_action,"OUTPUT",6)!=SAME &&
	strncmp(erule->p_action,"FORWARD",7)!=SAME &&
	strncmp(erule->p_action,"RETURN",6)!=SAME &&
	strncmp(erule->p_action,"ACCEPT",6)!=SAME &&
	strncmp(erule->p_action,"DROP",4)!=SAME &&
	strncmp(erule->p_action,"REJECT",6)!=SAME &&
	strncmp(erule->p_action,"QUEUE",5)!=SAME &&
	strncmp(erule->p_action,"LOG",3)!=SAME ) newChain(f,erule->p_action);


    fprintf(f,"iptables -A %s ",erule->p_group);

	
    if (erule->p_direction!=NULL && erule->p_iface!=NULL) {
	if ( strcasecmp(erule->p_direction,"inbound")==0)
	    fprintf(f,"-i %s ",erule->p_iface);
	if ( strcasecmp(erule->p_direction,"outbound")==0)
	    fprintf(f,"-o %s ",erule->p_iface);
    }
	
    if ( erule->t->p_proto ) {
	fprintf(f," -p %s",erule->t->p_proto);
    }
    
    if ( erule->t->state_new ) 
	fprintf(f," -m state --state NEW ");

    if (strcmp( erule->p_group,"OUTPUT")!=SAME ) {
	if (erule->t->p_src!=NULL) fprintf(f," -s %s",erule->t->p_src);
	if (erule->t->p_mac_src!=NULL)
	    fprintf(f," -m mac --mac-source %s",erule->t->p_mac_src);
    }

    if ( erule->t->p_sprt && strlen(erule->t->p_sprt)!=0) {
	switch (erule->multiport) {
	case MULTIPORT_NONE:
	    fprintf(f," --source-port %s",erule->t->p_sprt);
	    break;
	case MULTIPORT_SRC:
	    fprintf(f," -m multiport --source-port %s",erule->t->p_sprt);
	    break;
	} 
    }
    
    if (strcmp( erule->p_group,"INPUT")!=SAME ) {
	if (erule->t->p_dst!=NULL) fprintf(f," -d %s",erule->t->p_dst);
    }

    if ( erule->t->p_dprt && strlen(erule->t->p_dprt)!=0) {
	switch (erule->multiport) {
	case MULTIPORT_NONE:
	    fprintf(f," --destination-port %s",erule->t->p_dprt);
	    break;
	case MULTIPORT_DST:
	    fprintf(f," -m multiport --destination-port %s",erule->t->p_dprt);
	    break;
	}
    }
	
    if ( erule->t->p_icmp_type && strcmp(erule->t->p_icmp_type,"-1")!=SAME)
	fprintf(f," --icmp-type %s", erule->t->p_icmp_type);

    
    if (erule->p_action!=NULL) {
	fprintf(f," -j %s",erule->p_action);
    }
	
    if (erule->t->p_options!=NULL)  {
	fprintf(f," %s",erule->t->p_options);
    }

    if (erule->p_time && strlen(erule->p_time)!=0)  {
	fprintf(f," %s",erule->p_time);
    }
    fprintf(f,"\n");

}


