// ---------------------------------------------------------------------
// $Id: GeneticAlgorithm.cc,v 1.37 2006/08/04 17:05:15 daaugusto Exp $
//
//   GeneticAlgorithm.cc (created on Tue Nov 08 01:08:35 BRT 2005)
// 
//   Genetic Algorithm File Fitter (gaffitter)
//
//   Copyright (C) 2005-2006 Douglas A. Augusto
// 
// This file is part of gaffitter.
// 
// gaffitter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at
// your option) any later version.
// 
// gaffitter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with gaffitter; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// ---------------------------------------------------------------------

#include "GeneticAlgorithm.hh"

#include <utility>

// ---------------------------------------------------------------------
GeneticAlgorithm::GeneticAlgorithm(vector<SizeName>& sn, Params& p): 
                  Optimizer(sn, p) 
{
   // This constructor is called just once. Thus, these parameters
   // remain "static" over all iterations. On the other hand,
   // Initialize() is called for each iteration.

   // sort files by size (reverse order), needed by the genetic operators
   sort(m_files.begin(), m_files.end(), SizeName::CmpSizeRev);

   // auto adjust if not explicitly declared

   // Selection pressure [default = 2]
   if (m_params.m_ga_sel_pressure <= 2) 
   {
      m_tournament_size = 2;
      Tournament = &GeneticAlgorithm::Tournament2;
   }
   else // maximum = pop size
   {
      m_tournament_size = m_params.m_ga_sel_pressure;
      Tournament = &GeneticAlgorithm::TournamentN;
   }

   // Population size and generations are proportionally to ln (natural
   // logarithmic) of input size (number of files).
   if (m_params.m_ga_pop_size <= m_tournament_size) m_params.m_ga_pop_size = 
       max(m_tournament_size+1, 
       static_cast<unsigned>(20*log((float) m_files.size()+1)));
    
   if (m_params.m_ga_num_gens <= 0) m_params.m_ga_num_gens = 
      static_cast<int>(50*log((float) m_files.size()+1));

   // ------ set params
   // resize population to m_ga_pop_size
   m_pop_size = m_params.m_ga_pop_size;
   m_num_gen  = m_params.m_ga_num_gens;
   m_prob_crossover = m_params.m_ga_cross_prob;
    

   // --- Population and Genetic Algorithm

   // reserve the correct size
   m_genomes.reserve(m_params.m_ga_pop_size);

   m_params.m_ga_seed = Random::Seed(m_params.m_ga_seed); 
}
// --------------------------------------------------------------------
GeneticAlgorithm::~GeneticAlgorithm()
{
   for(unsigned int i=0; i<m_genomes.size(); ++i) delete m_genomes[i];
   m_best = 0;
}

// --------------------------------------------------------------------
void
GeneticAlgorithm::Evolve()
{
   // Reset variables
   Reset();

   // Initialize the GeneticAlgorithm Optimizer
   Initialize();

   // ---- Evolving...
   // Initialize the GA population
   InitPopulation();

   if (m_params.m_verbose) cout << "> ";

   while (m_cur_gen++ < m_num_gen)
   { 
      if (m_params.m_verbose) cout << "." << flush;
      if (Generation()) break;
   }

   if (m_params.m_verbose) cout << " <" << endl << flush;

   // pick the best
   Params::Size_t best_score = numeric_limits<Params::Size_t>::max();

   for(unsigned int i=0; i<m_genomes.size(); ++i)
   {
      Params::Size_t tmp_score = Score(i);

      if (tmp_score < best_score) 
      { 
         m_best = &m_genomes[i]->m_chrom;
         best_score = tmp_score;
      }
   }
}

// --------------------------------------------------------------------
void
GeneticAlgorithm::Reset()
{
   for(unsigned int i=0; i<m_genomes.size(); ++i) delete m_genomes[i];
   m_best = 0;

   // set size to zero (useful for second iteration and so)
   m_genomes.clear();

   // reset generation counter
   m_cur_gen = 0;
}

// --------------------------------------------------------------------
void
GeneticAlgorithm::Initialize()
/* This function is called before each iteration. Useful when some
   parameters depends on input size (number of files). */
{
   // Automatic mutation is ~1/L (where L is the gnome's length)
   if (m_params.m_ga_mut_prob < 0.0  ||  m_params.m_ga_mut_prob > 1.0)
   {
        if (m_files.size() <= 10)
           m_prob_mutation = (1+log10((float) m_files.size()))
                                    /static_cast<float>(m_files.size());
        else // input size > 10
           m_prob_mutation = (1+log(10.0)/log((float) m_files.size()))
                                    /static_cast<float>(m_files.size());
   }
   else m_prob_mutation = m_params.m_ga_mut_prob;

   // print the header
   if (m_params.m_verbose) cout << *this;
}

// --------------------------------------------------------------------
void
GeneticAlgorithm::InitPopulation()
{
   // create the individuals

   // Probability between (p_min, p_max]
   const float p_min = min(1.0/m_files.size(),0.5); // maximum p_min = 0.5 (50%)
   const float p_max = 1.0 - p_min; // maximum p_max = 1.0 - p_min

   for(unsigned int j=0; j<m_pop_size; ++j)
   {
      float cur_prob = 
             NonUniformFiller((j+1)*(p_max - p_min)/m_pop_size + p_min);

      m_genomes.push_back(new Genome(m_files.size()));

      for (int i=0; i<m_genomes[j]->Length(); ++i) 
                   m_genomes[j]->Gene(i, Random::Probability(cur_prob));

      // evaluate
      m_genomes[j]->Diff(Diff(m_genomes[j]->m_chrom));

      LocalOptimizer(*m_genomes[j]);
   } 
   
   ++m_cur_gen; // the initial population is the first generation
}

// --------------------------------------------------------------------
float
GeneticAlgorithm::NonUniformFiller(float x) const
{
   const float p1 = 0.2;    // ends r'
   const float p2 = 0.8;    // begins r"
   const float angle = 0.1; // r inclination

   /*    ^
        1|                           (1,1)
         |                            /
         |                          /
         |      y2      r      y1 /  r"
   yo 0.5-       o_______------o/
         |  r'  /.             .
         |    /  .             .
         |  /    .             .
         |/      .             .
       0 ----------------|----------------> x
         |0      p1     0.5    p2       1
                        xo

       General Equations:

       - one point (yo,xo) and inclination (k):

         y = k (x - xo) + yo

       - two points (y2,x2;y1,x1):
       
             y2 - y1
         y = ------- (x-x1) + y1
             x2 - x1
   */

   if (x<=p1) // r'
   {
      /* r': 

             y2
       y = x --
             p1 
      */
      const float y2 = angle*(p1-0.5)+0.5;

      return x*y2/p1;
   }

   if (x>=p2) // r"
   {
      /* r":
       
           1-y1 
       y = ---- (x-p2) + y1
           1-p2
      */
      const float y1 = angle*(p2-0.5)+0.5;
      
      return (1.0-y1)/(1.0-p2)*(x-p2)+y1;
   }

   // r
   /* (p1<x<p2)
      r: y = k (x - xo) + yo */
   return angle*(x-0.5)+0.5;
}

// --------------------------------------------------------------------
void 
GeneticAlgorithm::LocalOptimizer(Genome& g) const
{
   if (Random::Probability(0.5))
   {
      int i = g.Length()-1;
      while (g.OverTarget()) { g.DelSize(i, m_files[i].Size()); --i; }
   }
   else
   {
      int i = 0;
      while (g.OverTarget()) { g.DelSize(i, m_files[i].Size()); ++i; }
   }

   if (g.UnderTarget())
   {
      for (int i=0; i<g.Length(); ++i) { g.AddSize(i, m_files[i].Size()); }
   }

}

// --------------------------------------------------------------------
void 
GeneticAlgorithm::DirectedMutation(Genome& g) const
{
   int r = Random::Int(0, g.Length()-1);

   if (g.OverTarget())
   {
      if (!g.Gene(r))
      {
         if (r<=g.Length()/2)  
            while (++r < g.Length() && !g.Gene(r)); 
         else 
            while (--r >= 0 && !g.Gene(r));

         if (r>=0 && r<g.Length())
         {
            g.Gene(r, false);
            g.Diff(g.Diff() + m_files[r].Size());
         }
      }
      else // equal 'true'
      {
         g.Gene(r, false);
         g.Diff(g.Diff() + m_files[r].Size());
      }
   }
   else
   {
      if (g.Gene(r))
      {
         if (r<=g.Length()/2)  
            while (++r < g.Length() && g.Gene(r)); 
         else 
            while (--r >= 0 && g.Gene(r));

         if (r>=0 && r<g.Length())
         {
            g.AddSize(r, m_files[r].Size());
         }
      }
      else // equal 'false'
      {
         g.AddSize(r, m_files[r].Size());
      }
   }
}

// --------------------------------------------------------------------
bool
GeneticAlgorithm::Generation()
{
   for(unsigned int i=0; i<m_genomes.size(); i+=2)
   {
      // select daddy and mommy and the losers (will be replaced by children)
      pair<int,int> dad = (this->*Tournament)(0, m_genomes.size()/2 - 1);
      pair<int,int> mom = 
           (this->*Tournament)(m_genomes.size()/2, m_genomes.size() - 1);

      // Reproduction (crossover and mutation)
      Reproduction(dad.first, mom.first, dad.second, mom.second);

      // Calculate the sum (evaluation)
      m_genomes[dad.second]->Diff(Diff(m_genomes[dad.second]->m_chrom));
      m_genomes[mom.second]->Diff(Diff(m_genomes[mom.second]->m_chrom));

      // ------  other genetic operators
      // Directed mutation
      DirectedMutation(*m_genomes[dad.second]);
      DirectedMutation(*m_genomes[mom.second]);

      // local optimizer and makes the individual feasible if needed
      LocalOptimizer(*m_genomes[dad.second]);
      LocalOptimizer(*m_genomes[mom.second]);

      // test if was found the perfect fit
      if (Score(dad.second) <= 0.0 || Score(mom.second) <= 0.0) return true;
   }

   // yet not found the perfect fit
   return false;
}

// --------------------------------------------------------------------
pair<int,int> 
GeneticAlgorithm::Tournament2(int from, int to) const
{
   int sel1 = Select(from,to);
   int sel2 = Select(from,to,sel1);

   // best and worse index, respectively
   if (Score(sel1) < Score(sel2))
      return make_pair(sel1,sel2);
   else
      return make_pair(sel2,sel1);
}

// --------------------------------------------------------------------
pair<int,int> 
GeneticAlgorithm::TournamentN(int from, int to) const
{
   vector<pair<Params::Size_t,int> > candidates;

   int sel = Select(from,to);
   candidates.push_back(make_pair(Score(sel), sel));

   for(unsigned int i=1; i<m_tournament_size; ++i)
   {
      sel = Select(from,to,sel);
      candidates.push_back(make_pair(Score(sel), sel));
   }

   // sort by score (smaller is better)
   sort(candidates.begin(), candidates.end());

   // best and worse index, respectively
   return make_pair(candidates.front().second, candidates.back().second);
}

// --------------------------------------------------------------------
void
GeneticAlgorithm::Reproduction(int dad, int mom, 
                               int child1, int child2) const
{
   if (Random::Probability(m_prob_crossover))
   {
      // mask uniform crossover
      for(int i=0; i<m_genomes[dad]->Length(); ++i)
      {
         if (Random::Probability(0.5))
         {
            m_genomes[child1]->Gene(i, CopyMutate(m_genomes[dad]->Gene(i)));
            m_genomes[child2]->Gene(i, CopyMutate(m_genomes[mom]->Gene(i)));
         }
         else
         {
            m_genomes[child1]->Gene(i, CopyMutate(m_genomes[mom]->Gene(i)));
            m_genomes[child2]->Gene(i, CopyMutate(m_genomes[dad]->Gene(i)));
         }
      }

   }
   else // just duplicate (with mutation)
   {
      for(int i=0; i<m_genomes[dad]->Length(); ++i)
      {
         // [no] switch mom/dad group for more diversity
         m_genomes[child1]->Gene(i, CopyMutate(m_genomes[dad]->Gene(i)));
         m_genomes[child2]->Gene(i, CopyMutate(m_genomes[mom]->Gene(i)));
      }
   }
}

// --------------------------------------------------------------------
ostream& 
GeneticAlgorithm::Write(ostream& s) const
{
   s << endl;
   s << "> -----------------------------------" << endl;
   s << "> Genetic Algorithm (steady-state)" << endl;
   s << "> -----------------------------------" << endl;
   s << "> Max number of generations: " << m_num_gen << endl;
   s << "> Population size: " << m_pop_size << endl;
   s << "> Crossover probability: " << m_prob_crossover << endl;
   s << "> Mutation probability: " << m_prob_mutation << endl;
   s << "> Tournament size: " << m_tournament_size << endl;
   s << "> Seed: " << m_params.m_ga_seed << endl;

   Optimizer::Write(s);

   s << flush << endl;

   return s;
}

// --------------------------------------------------------------------
