/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				calculate_psi.c

Purpose:
	Calculate dihedral angle psi.

Input:
	(1) Pointer to VectorS structure, with N atom coordinates for the
	    previous residue.
	(2) Pointer to VectorS structure, with CA atom coordinates.
	(3) Pointer to VectorS structure, with C atom coordinates.
	(4) Pointer to VectorS structure, with O atom coordinates.

Output:
	Return value.

Return value:
	(1) Dihedral angle psi, on success.
	(2) BADDIHEDANGLE on failure.

Notes:
	(1) This should help you to understand  phi and  psi definitions:
	    ........./................
	    | H * - * N              |
	    |        \    phi1 = 180 |
	    |         * CA           |
	    |        /    psi1 = 180 |
	    | O * - * C              |
	    |........\...............|
	    |       N * - * H        |
	    |        /    phi2 = 180 |
	    |    CA *                |
	    |        \    psi2 = 180 |
	    |       C * - * O        |
	    |......../...............|

========includes:============================================================*/

#include <stdio.h>

#include <math.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

void		VectorProduct_ (VectorS *, VectorS *, VectorS *);
double		AbsoluteValue_ (VectorS *);
double		ScalarProduct_ (VectorS *, VectorS *);

/*======calculate the psi angle:=============================================*/

double CalculatePsi_ (VectorS *N_vectorSP, VectorS *CA_vectorSP,
		      VectorS *C_vectorSP, VectorS *O_vectorSP)
{
VectorS		CA_N_vectorS, CA_C_vectorS, C_O_vectorS;
VectorS		u1S, u2S, v1S, v2S;
double		denom, ratio;
double		alpha, psi;


/* Prepare three auxiliary vectors: */
CA_N_vectorS.x = N_vectorSP->x - CA_vectorSP->x;
CA_N_vectorS.y = N_vectorSP->y - CA_vectorSP->y;
CA_N_vectorS.z = N_vectorSP->z - CA_vectorSP->z;
CA_C_vectorS.x = C_vectorSP->x - CA_vectorSP->x;
CA_C_vectorS.y = C_vectorSP->y - CA_vectorSP->y;
CA_C_vectorS.z = C_vectorSP->z - CA_vectorSP->z;
C_O_vectorS.x  = O_vectorSP->x - C_vectorSP->x;
C_O_vectorS.y  = O_vectorSP->y - C_vectorSP->y;
C_O_vectorS.z  = O_vectorSP->z - C_vectorSP->z;

/* Two vectors perpendicular to CA_C_vectorS, mutually orthog., */
/* one in the plane defined by  CA_C_vectorS and  CA_N_vectorS: */
VectorProduct_ (&u1S, &CA_N_vectorS, &CA_C_vectorS);
VectorProduct_ (&u2S, &u1S, &CA_C_vectorS);

/* Two vectors perpendicular to CA_C_vectorS, mutually orthog., */
/* one in the plane  defined by  CA_C_vectorS and  C_O_vectorS: */
VectorProduct_ (&v1S, &CA_C_vectorS, &C_O_vectorS);
VectorProduct_ (&v2S, &v1S, &CA_C_vectorS);

/* The angle alpha, which will be used to calculate phi: */
denom = AbsoluteValue_ (&v2S) * AbsoluteValue_ (&u2S);
if (denom == 0.0)
	{
	return BADDIHEDANGLE;
	}
ratio = ScalarProduct_ (&v2S, &u2S) / denom;
if ((ratio > 1.0) || (ratio < -1.0)) return BADDIHEDANGLE;
alpha = acos (ratio);

/* There are two possible solutions; the right one is resolved here: */
if (ScalarProduct_ (&v2S, &u1S) >= 0) psi = alpha;
else psi = -alpha;

/* Return the angle (in radians): */
return psi;
}

/*===========================================================================*/


