/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				catch.c

Purpose:
	Execute catch (take) command: catch the specified macromolecular
	complex and release all others.  Set the new default  caught  ID
	and refresh the control window.

Input:
	(1) Pointer to MolComplexS structure,  with macromol. complexes.
	(2) Number of macromolecular complexes.
	(3) Pointer to RuntimeS structure, with some runtime data.
	(4) Pointer to ConfigS structure, with configuration data.
	(5) Pointer to GUIS structure, with GUI data.
	(6) The string which contains the macromol. complex identifier.

Output:
	(1) The catch flag (catchF) set for each macromolecular complex.
	(2) Return value.

Return value:
	(1) Positive (command) code on success.
	(2) Negative (error) code on failure.

Notes:
	(1) This command takes only  one argument.  If you want to catch
	    more than one macromolecular complex, use command group.

========includes:============================================================*/

#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "commands.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

char		*ExtractToken_ (char *, int, char *, char *);
int		ControlRefresh_ (MolComplexS *, ConfigS *, GUIS *);

/*======execute catch (load) command:========================================*/

int Catch_ (MolComplexS *mol_complexSP, int mol_complexesN,
	    RuntimeS *runtimeSP,
	    ConfigS *configSP, GUIS *guiSP, char *stringP)
{
char		*remainderP;
char		tokenA[SHORTSTRINGSIZE];
int		complexID;
int		mol_complexI;
MolComplexS	*curr_mol_complexSP;
int		job_doneF = 0;

/* Extract the macromolecular complex identifier: */
remainderP = ExtractToken_ (tokenA, SHORTSTRINGSIZE, stringP, " \t\n");
if (!remainderP)
	{
	strcpy (runtimeSP->messageA, "Complex identifier missing!");
	runtimeSP->message_length = strlen (runtimeSP->messageA);
	return ERROR_NO_ID;
	}

/* Read the identifier (integer expected): */
if (sscanf (tokenA, "%d", &complexID) != 1)
	{
	strcpy (runtimeSP->messageA, "Bad macromolecular complex identifier!");
	runtimeSP->message_length = strlen (runtimeSP->messageA);
	return ERROR_BAD_ID;
	}

/* Set the catch flag for the specified complex: */
for (mol_complexI = 0; mol_complexI < mol_complexesN; mol_complexI++)
	{
	/** Prepare the pointer to the current macromolecular complex: **/
	curr_mol_complexSP = mol_complexSP + mol_complexI;

	/** Check the number of atoms;  for bad **/
	/** and discarded complexes it is zero: **/
	if (curr_mol_complexSP->atomsN == 0) continue;

	/** If macromol. complex is recognized, set the catch flag to one: **/
	if (curr_mol_complexSP->mol_complexID == complexID)
		{
		curr_mol_complexSP->catchF = 1;
		job_doneF = 1;
		}
	}

/* If complex was not found, return negative value: */
if (job_doneF == 0)
	{
	strcpy (runtimeSP->messageA, "Bad complex identifier (out of range)!");
	runtimeSP->message_length = strlen (runtimeSP->messageA);
	return ERROR_BAD_ID;
	}

/* If complex was found, set all other catch flags to zero: */
for (mol_complexI = 0; mol_complexI < mol_complexesN; mol_complexI++)
	{
	/** Prepare the pointer to the current macromolecular complex: **/
	curr_mol_complexSP = mol_complexSP + mol_complexI;

	/** For the specified complex,  do not reset the catch flag,  but **/
	/** use the complex index as the index of default caught complex: **/
	if (curr_mol_complexSP->mol_complexID == complexID)
		{
		runtimeSP->default_complexI = mol_complexI;
		continue;
		}

	/** For all other complexes, set the catch flag to zero: **/
	curr_mol_complexSP->catchF = 0;
	}

/* Refresh the control window: */
ControlRefresh_ (mol_complexSP + runtimeSP->default_complexI, configSP, guiSP);

/* Return positive value on success: */
return COMMAND_CATCH;
}

/*===========================================================================*/


