/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				check_con.c

Purpose:
	Check C-O...N angle, to check validity of hydrogen bond.

Input:
	(1) Pointer to VectorS structure, with O-C vector.
	(2) Pointer to AtomS structure,  with data about the first atom.
	(3) Pointer to AtomS structure, with data about the second atom.
	(4) Pointer to ConfigS structure.

Output:
	(1) Return value.

Return value:
	(1) Positive if angle fits into the hydrogen bond angular range.
	(2) Negative if angle is out of range.

========includes:============================================================*/

#include <stdio.h>

#include <math.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======check C-O...N angle:=================================================*/

int CheckCONAngle_ (VectorS *O_C_vectorSP,
		    AtomS *atom1SP, AtomS *atom2SP,
		    ConfigS *configSP)
{
VectorS		O_N_vectorS;
double		scalar_product;
double		angle;
double		abs_value1, abs_value2, denominator;

/* Prepare the O-N vector: */
O_N_vectorS.x = atom2SP->raw_atomS.x[0] - atom1SP->raw_atomS.x[0];
O_N_vectorS.y = atom2SP->raw_atomS.y    - atom1SP->raw_atomS.y;
O_N_vectorS.z = atom2SP->raw_atomS.z[0] - atom1SP->raw_atomS.z[0];

/* Calculate the scalar product: */
scalar_product = O_N_vectorS.x * O_C_vectorSP->x +
		 O_N_vectorS.y * O_C_vectorSP->y +
		 O_N_vectorS.z * O_C_vectorSP->z;

/* Check the scalar product - if it is zero, the angle is 90 degrees: */
if (scalar_product == 0.0)
	{
	/** Set angle: **/
	angle = 1.5707963;	/* In radians, of course! */

	/** Check it and return to caller: **/
	if      (angle < configSP->hydro_bond_angle_min) return -1;
	else if (angle > configSP->hydro_bond_angle_max) return -2;
	else return 1;
	}

/* If scalar product is  not equal to zero, */
/* prepare absolute values of both vectors: */
abs_value1 = sqrt (O_C_vectorSP->x * O_C_vectorSP->x +
		   O_C_vectorSP->y * O_C_vectorSP->y +
		   O_C_vectorSP->z * O_C_vectorSP->z);
abs_value2 = sqrt (O_N_vectorS.x * O_N_vectorS.x +
		   O_N_vectorS.y * O_N_vectorS.y +
		   O_N_vectorS.z * O_N_vectorS.z);

/* Calculate and check the angle: */
denominator = abs_value1 * abs_value2;
if (denominator == 0.0) angle = 0.0;
else angle = acos (scalar_product / denominator);
if      (angle < configSP->hydro_bond_angle_min) return -1;
else if (angle > configSP->hydro_bond_angle_max) return -2;
else return 1;

/* If this point is reached, the angle fits into the range: */
return 1;
}

/*===========================================================================*/


