/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				hydrophobicity.c

Purpose:
	This function  may be used to assign  the hydrophobicity to the
	specified residue or to retrieve minimal,  maximal,  average or
	threshold hydrophobicity value.
	
Input:
	(1) Pointer to residue name.
	(2) Hydrophobicity scale index.
	(3) Request index:  0 = return hydrophobicity,  1 = return min.
	    value, 2 = return max. value,  3 = return average value and
	    4 = return threshold value.

Output:
	Return value.

Return value:
	(1) Hydrophobicity of the specified residue, extreme hydrophob.
	    value,  average hydrophobicity or threshold hydrophobicity.
	    What will be returned is decided by the request index.

Notes:
	(1) The average hydrophobicity is assigned to unknown residues.

	(2) Threshold hydrophobicy  is used  to distinguish hydrophobic
	    and hydrophilic residues.  Residue which has hydrophobicity
	    value above the threshold should be treated as hydrophobic.
	    For Kyte-Doolittle scale and  White scale,  positive values
	    are assigned to hydrophobic residues and threshold value is
	    equal to zero. For normalized Eisenberg scale, threshold is
	    different from zero. This is caused by normalization.

	(3) References:

	    ===========================================================
	    | Scale           |  Reference                            |
	    ===========================================================
	    | S. White group  |  Biochemistry   35 (1996), p. 5109.   |
	    | Kyte-Doolitle   |  J. Mol. Biol. 157 (1982), p.  105.   |
	    | Eisenberg       |  J. Mol. Biol. 179 (1984), p.  125.   |
	    ===========================================================

========includes:============================================================*/

#include <stdio.h>

#include <string.h>

#include "defines.h"

/*======return hydrophobicity or requested reference value:==================*/

float Hydrophobicity_ (char *residue_nameP, int scaleI, int requestI)
{
float		hydrophobicity;
static char	amino_acidAA[23][4] =
		 {"ALA","ARG","ASN","ASP","CYS","GLN","GLU","GLY","HIS","ILE",
		  "LEU","LYS","MET","PHE","PRO","SER","THR","TRP","TYR","VAL",
		  "ASX","GLX","UNK"};
static float	hyphob_scaleAA[SCALESN][23] =
		{{-0.50,-1.81,-0.85,-3.64, 0.02,-0.77,-3.63,-1.15,-2.33, 1.12,
		   1.25,-2.80, 0.67, 1.71,-0.14,-0.46,-0.25, 2.09, 0.71, 0.46,
		  -2.25,-2.20,-0.52},  /* White & Wimley octanol scale */

		 { 1.80,-4.50,-3.50,-3.50, 2.50,-3.50,-3.50,-0.40,-3.20, 4.50,
		   3.80,-3.90, 1.90, 2.80,-1.60,-0.80,-0.70,-0.90,-1.30, 4.20,
		  -3.50,-3.50,-0.49},  /* Kyte & Doolittle hydropathy scale */

		 { 0.62,-2.53,-0.78,-0.90, 0.29,-0.85,-0.74, 0.48,-0.40, 1.38,
		   1.06,-1.50, 0.64, 1.19, 0.12,-0.18,-0.05, 0.81, 0.26, 1.08,
		  -0.84,-0.80, 0.00}}; /* Eisenberg normalized consensus sc. */
static float	min_hyphobA[SCALESN]       = {-3.64, -4.50, -2.53};
static float	max_hyphobA[SCALESN]       = { 2.09,  4.50,  1.38};
static float	average_hyphobA[SCALESN]   = {-0.52, -0.49,  0.00};
static float	threshold_hyphobA[SCALESN] = { 0.00,  0.00, -0.16};
int		i;

/* Check scale index: */
if (scaleI > SCALESN) return 0.0;

/* If minimal hydrophobicity is requested: */
if (requestI == 1) return min_hyphobA[scaleI];

/* If maximal hydrophobicity is requiested: */
else if (requestI == 2) return max_hyphobA[scaleI];

/* If average hydrophobicity is requested: */
else if (requestI == 3) return average_hyphobA[scaleI];

/* If threshold hydrophobicity is requested: */
else if (requestI == 4) return threshold_hyphobA[scaleI];

/* If this point is reached, hydrophobicity */
/* of  the specified residue  is requested. */

/* Use average value as default; it will */
/* be returned  if identification fails: */
hydrophobicity = average_hyphobA[scaleI];

/* Compare the purified residue name with 23 standard names: */
for (i = 0; i < 23; i++)
	{
	/* If amino acid is recognized: */
	if (strcmp (residue_nameP, amino_acidAA[i]) == 0)
		{
		hydrophobicity = hyphob_scaleAA[scaleI][i];
		break;
		}
	}

/* Return hydrophobicity value: */
return hydrophobicity;
}

/*===========================================================================*/


