/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <ctype.h>

#include "gdis.h"

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/*********************************************/
/* does a given string represent an element? */
/*********************************************/
#define DEBUG_ELEM_TYPE 0
gint elem_type(const gchar *input)
{
gint i, j, m, n;
gchar *elem1, *elem2;

/* zero length string matches something, so force it to return 0 */
if (!strlen(input))
  return(0);

/* duplicate for manipulation */
elem1 = g_strdup(input);

/* remove anything but alphabetic chars */
for (i=0 ; i<strlen(elem1) ; i++)
  if (!isalpha((int) *(elem1+i)))
    *(elem1+i) = ' ';

/* remove trailing/leading spaces */
g_strstrip(elem1);
m = strlen(elem1);

#if DEBUG_ELEM_TYPE
printf("Looking for [%s]...",elem1);
#endif

/* attempt to match atom type with database */
j=0;
/* FIXME - elim dependence on const */
for(i=0 ; i<sysenv.num_elements ; i++)
  {
/* duplicate for manipulation */
  elem2 = g_strdup(elements[i].symbol);
  g_strstrip(elem2);
  n = strlen(elem2);
/* NEW - only compare if stripped lengths match */
  if (n == m)
    if (g_strncasecmp(elem1, elem2, m) == 0)
      {
      j = i;
      break;
      }
  g_free(elem2);
  }

#if DEBUG_ELEM_TYPE
if (j)
  printf("found.\n");
else
  printf("not found.\n");
#endif

/* done */
g_free(elem1);
return(j);
}

/****************************/
/* match elem with database */
/* also cleans up elem      */
/****************************/
#define DEBUG_elem_seek 0
gint elem_seek(gint n, gint type, struct model_pak *data)
{
gint ix;
gchar *elem;

/* NB: assumes elem is already in uppercase */
if (type == SHELL)
  elem = (data->shells+n)->element;
else
  elem = (data->atoms+n)->element;

/* attempt to match atom type with database */
ix = elem_type(elem);

/* 0 is the default type */
if (!ix)
  printf("Warning: element [%s] not found.\n",elem);

/* assign data */
if (type == SHELL)
  {
  (data->shells+n)->atom_code = ix;
  (data->shells+n)->colour[0] = elements[ix].colour[0];
  (data->shells+n)->colour[1] = elements[ix].colour[1];
  (data->shells+n)->colour[2] = elements[ix].colour[2];
  }
else
  {
  (data->atoms+n)->atom_code = ix;
  (data->atoms+n)->colour[0] = elements[ix].colour[0];
  (data->atoms+n)->colour[1] = elements[ix].colour[1];
  (data->atoms+n)->colour[2] = elements[ix].colour[2];
  }

return(ix);
}

/*************************/
/* retrieve element data */
/*************************/
/* will return non-default data (if it exists) otherwise the default */
#define DEBUG_GET_ELEM 0
gint get_elem_data(gint code, struct elem_pak *elem,
                                struct model_pak *data)
{
gint i;

/* search the exceptions database */
#if DEBUG_GET_ELEM
printf("Searching non-default list for code = %d.\n",code);
#endif
for (i=data->num_elem ; i-- ; )
  {
/*
  if (code == (data->elements+i)->number)
*/
  if (code == i)
    {
    memcpy(elem, data->elements+i, sizeof(struct elem_pak));
    return(0);
    }
  }
#if DEBUG_GET_ELEM
printf("Not found.\n");
#endif

/* return from the standard database */
if (code >= 0 && code < sysenv.num_elements)
  {
#if DEBUG_GET_ELEM
printf("Retrieving default.\n");
#endif
  memcpy(elem, &elements[code], sizeof(struct elem_pak));
  return(0);
  }

/* done - no match */
#if DEBUG_GET_ELEM
printf("ERROR: bad element code.\n");
#endif
return(1);
}

/*******************************************/
/* put non-default element data in a model */
/*******************************************/
void put_elem_data(struct elem_pak *elem, struct model_pak *data)
{
gint i, ix, flag;

/* replace flag (as opposed to add new) */
ix=flag=0;
/* alloc if non-existent */
if (!data->num_elem)
  {
  data->elements = g_malloc(sizeof(struct elem_pak));
  data->num_elem++;
  flag++;
  }
else
  {
/* search for existing data */
  for (i=data->num_elem ; i-- ; )
    {
    if (elem->number == i)
      {
      flag++;
      ix = i;
      break;
      }
    }
  }

/* TODO - a linked list would be easier... */
/* replace or re-alloc & add? */
if (flag)
  memcpy(data->elements+ix, elem, sizeof(struct elem_pak));
else
  {
  data->elements = g_renew(struct elem_pak, data->elements, data->num_elem+1);
  memcpy(data->elements+data->num_elem, elem, sizeof(struct elem_pak));
  data->num_elem++;
  }
}

/****************************************/
/* transfer model specific element data */
/****************************************/
/* TODO - preserve what may already be there (if desired) */
gint copy_elem_data(struct model_pak *src, struct model_pak *dest)
{
/* free old */
if (dest->num_elem)
  g_free(dest->elements);

/* alloc and copy new */
dest->elements = g_malloc(src->num_elem * sizeof(struct elem_pak));
memcpy(dest->elements, src->elements, src->num_elem*sizeof(struct elem_pak));
dest->num_elem = src->num_elem;

return(0);
}

/**********************/
/* transfer gulp data */
/**********************/
gint copy_gulp_data(struct model_pak *src, struct model_pak *dest)
{
gint i;

/* free any old data in the destination model */
free_gulp_data(dest);

/* copy keyword stuff */
dest->gulp.run = src->gulp.run;
dest->gulp.free = src->gulp.free;
dest->gulp.zsisa = src->gulp.zsisa;
dest->gulp.compare = src->gulp.compare;
dest->gulp.ensemble = src->gulp.ensemble;
dest->gulp.coulomb = src->gulp.coulomb;

/* copy potentials data */
if (src->gulp.num_potentials)
  {
  dest->gulp.num_potentials = src->gulp.num_potentials;
  dest->gulp.pot = g_malloc(src->gulp.num_potentials*sizeof(gchar *));
  for (i=0 ; i<src->gulp.num_potentials ; i++)
    *(dest->gulp.pot+i) = g_strdup(*(src->gulp.pot+i));
  }
/* copy element data */
if (src->gulp.num_elem)
  {
  dest->gulp.num_elem = src->gulp.num_elem;
  dest->gulp.elem = g_malloc(src->gulp.num_elem*sizeof(gchar *));
  for (i=0 ; i<src->gulp.num_elem ; i++)
    *(dest->gulp.elem+i) = g_strdup(*(src->gulp.elem+i));
  }
/* copy species data */
if (src->gulp.num_species)
  {
  dest->gulp.num_species = src->gulp.num_species;
  dest->gulp.species = g_malloc(src->gulp.num_species*sizeof(gchar *));
  for (i=0 ; i<src->gulp.num_species ; i++)
    *(dest->gulp.species+i) = g_strdup(*(src->gulp.species+i));
  }

return(0);
}

gint copy_gulp_extra(struct model_pak *src, struct model_pak *dest)
{
/* copy unrecognized gulp options */
/* FIXME - if dest is 2D don't copy all this stuff, eg skip elastic consts */
if (src->gulp.extra)
  dest->gulp.extra = g_strdup(src->gulp.extra);
return(0);
}

/**********************************/
/* free memory used for gulp data */
/**********************************/
#define DEBUG_FREE_GULP_DATA 0
gint free_gulp_data(struct model_pak *data)
{
gint i;

#if DEBUG_FREE_GULP_DATA
printf("freeing gulp data...\n");
#endif
for (i=0 ; i<data->gulp.num_potentials ; i++)
  g_free(*(data->gulp.pot+i));
for (i=0 ; i<data->gulp.num_elem ; i++)
  g_free(*(data->gulp.elem+i));
for (i=0 ; i<data->gulp.num_species ; i++)
  g_free(*(data->gulp.species+i));

if (data->gulp.srcfile)
  g_free(data->gulp.srcfile);
if (data->gulp.minimiser)
  g_free(data->gulp.minimiser);
if (data->gulp.extra)
  g_free(data->gulp.extra);

#if DEBUG_FREE_GULP_DATA
printf("done.\n");
#endif
return(0);
}

/******************************************/
/* get a list of unique elements in model */
/******************************************/
gint construct_unique(struct model_pak *data)
{
gint i, j, n, flag;
gint *tmp;

/* scan for all unique atom codes */
n = 0;
tmp = g_malloc(data->num_atoms * sizeof(gint));
for (i=0 ; i<data->num_atoms ; i++)
  {
  flag=0;
  for (j=0 ; j<n ; j++)
    {
    if (*(tmp+j) == (data->atoms+i)->atom_code)
      {
      flag++;
      break;
      }
    }
  if (!flag)
    {
    *(tmp+n) = (data->atoms+i)->atom_code;
    n++;
    }
  }

/* transfer to the model struct */
if (data->num_unique)
  g_free(data->unique);
if (n)
  {
  data->unique = g_malloc(n*sizeof(gint));
  data->num_unique = n;
  memcpy(data->unique, tmp, n*sizeof(gint));
  }
return(n);
}

