/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef __sun
#include <sys/dirent.h>
#else
#include <sys/dir.h>
#endif

#include "gdis.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

#ifdef __sun
#include "scandir.h"
#endif

/**************************************************************/
/* correct numbers in binary files with reverse byte ordering */
/**************************************************************/
void swap_bytes(void *ptr, const gint size)
{
gint i,j;
gchar tmp;

/*
printf("start: ");
for (i=0 ; i<size ; i++)
  printf("%d ", *((char *)(ptr+i)));
printf("\n");
*/
j=size-1;
for (i=0 ; i<size/2 ; i++)
  {
  tmp = *((gchar *)(ptr+j));
  *((gchar *)(ptr+j)) = *((gchar *)(ptr+i));
  *((gchar *)(ptr+i)) = tmp;
  j--;
  }
/*
printf(" stop: ");
for (i=0 ; i<size ; i++)
  printf("%d ", *((char *)(ptr+i)));
printf("\n");
*/
}

/**************************************/
/* get a non trivial line from a file */
/**************************************/
gint fgetline(FILE *fp, gchar *line)
{
gint i, linlen;

for(;;)
  {
/* get a line */
  if (fgets(line, LINELEN/2, fp) == NULL)
    return(1);
  linlen = strlen(line);
/* only treated as data if not a comment and non empty */
  if (line[0] != '#' && linlen)
    {
/* ampersand concatenation */
/* TODO - extra var in fgetline() (eg mode) that requests this */
    for (i=linlen ; i-- ; )
      {
      if (line[i] == '&')
        {
        if (fgets(&line[linlen], LINELEN/2, fp) == NULL)
          return(1);
        break;
        }
      }
    break;
    }
  }

/* all clear */
return(0);
}

/*************************/
/* save cartesian coords */
/*************************/
#define DEBUG_SAVE_CART 0
void save_cart(struct model_pak *data)
{
gint i;
gfloat v[3];

if (data->fractional)
  {
#if DEBUG_SAVE_CART
printf("storing frac as cart.\n");
#endif
  for (i=0 ; i<data->num_atoms ; i++)
    {
    v[0] = (data->atoms+i)->x;
    v[1] = (data->atoms+i)->y;
    v[2] = (data->atoms+i)->z;
    vecmat(data->latmat, v);
    (data->atoms+i)->cx = v[0];
    (data->atoms+i)->cy = v[1];
    (data->atoms+i)->cz = v[2];
    }
  for (i=0 ; i<data->num_shells ; i++)
    {
    v[0] = (data->shells+i)->x;
    v[1] = (data->shells+i)->y;
    v[2] = (data->shells+i)->z;
    vecmat(data->latmat, v);
    (data->shells+i)->cx = v[0];
    (data->shells+i)->cy = v[1];
    (data->shells+i)->cz = v[2];
    }
  }
else
  {
#if DEBUG_SAVE_CART
printf("storing cart.\n");
#endif
  for (i=0 ; i<data->num_atoms ; i++)
    {
    (data->atoms+i)->cx = (data->atoms+i)->x;
    (data->atoms+i)->cy = (data->atoms+i)->y;
    (data->atoms+i)->cz = (data->atoms+i)->z;
    }
  for (i=0 ; i<data->num_shells ; i++)
    {
    (data->shells+i)->cx = (data->shells+i)->x;
    (data->shells+i)->cy = (data->shells+i)->y;
    (data->shells+i)->cz = (data->shells+i)->z;
    }
  }
}

/****************************/
/* restore cartesian coords */
/****************************/
void load_cart(struct model_pak *data)
{
gint i;

for (i=data->num_atoms ; i-- ; )
  {
  (data->atoms+i)->x = (data->atoms+i)->cx;
  (data->atoms+i)->y = (data->atoms+i)->cy;
  (data->atoms+i)->z = (data->atoms+i)->cz;
  }
for (i=data->num_shells ; i-- ; )
  {
  (data->shells+i)->x = (data->shells+i)->cx;
  (data->shells+i)->y = (data->shells+i)->cy;
  (data->shells+i)->z = (data->shells+i)->cz;
  }
}

/**********************************/
/* open and configure a data file */
/**********************************/
gint file_action(gchar *inp_file, gint file_mode, gint model)
{
gint id, ret;
struct model_pak *data;

id = model_type(inp_file);
if (id < 0)
  return(1);

/* setup appropriate file pointer/stream */
ret=0;
switch (file_mode)
  {
  case FILE_SAVE_AS:
    data = model_ptr(model, RECALL);
    g_return_val_if_fail(data != NULL,1);
/* overwrite the old filenames with the new */
    if (data)
      {
      strcpy(data->filename, inp_file);
      g_free(data->basename);
      data->basename = strdup_no_extension(g_basename(inp_file));
      }
  case FILE_SAVE:
/* checks */
    data = model_ptr(model, RECALL);
    g_return_val_if_fail(data != NULL,1);
/* special saves/supported types only */
    switch(id)
      {
      case MORPH:
        ret = write_gmf(inp_file, data);
        break;
      case GULP:
        ret = write_gulp(inp_file, data);
        break;
      case MARVIN:
        ret = write_marvin(inp_file, data);
        break;
      case XTL:
        ret = write_xtl(inp_file, data);
        break;
      case BIOSYM:
      case XYZ:
        ret = create_file(inp_file, id, data);
        break;
      default:
        printf("Unsupported file type.");    
        ret=1;
      }
    break;
  default:
    printf("Unknown file mode.");
    ret=1;
  }

/* done */
if (ret)
  show_text("Save failed.");    
else
  {
/* updates */
  new_tree_item(model, REPLACE);
  redraw_canvas(SINGLE);
  }
return(ret);
}

/*****************************/
/* Setup model for file read */
/*****************************/
#define DEBUG_LOAD_MODEL 0
gint load_model(gint model, gint type, gchar *inp_file)
{
gint i, status=0;
gchar line[LINELEN];
struct model_pak *data;
struct parse_pak parse_data;
FILE *fp;

/* can we open the file */
fp = fopen(inp_file,"r");
if (!fp)
  {
  show_text("Error: bad or missing filename.");
  return(1);
  }
/* memory assignment for the model (and initial assignment for its data) */
data = model_ptr(model, ASSIGN);
if (!data)
  return(1);
data->id = type;

/* get the appropriate parsing structure */
if (parse_ptr(&parse_data, type))
  {
  show_text("Error: unknown parse type requested.");
  sysenv.num_models--;
  return(1);
  }

/* TODO - all unique? */
status = load_data(fp, &parse_data, data);

/* FIXME - messy hack */
/* count the frames in BIOSYM files */
if (data->id == BIOSYM)
  {
/* already read in one frame */
  i=1;
  while(!fgetline(fp,line))
    if (g_strncasecmp("!DATE", line, 5) == 0)
      i++;
  data->num_frames = i;
  }
if (data->num_frames)
  {
  data->animation = TRUE;
#if DEBUG_LOAD_MODEL
  printf("Found %d frames.\n",data->num_frames);
#endif
  }

if (status)
  {
  fclose(fp);
#if DEBUG_LOAD_MODEL
  printf("Load failed, error code: %d\n",status);
#endif
/* FIXME - dodgy cleanup if status=6 ??? */
  model_ptr(model, RELEASE);
  display_shuffle();
  redraw_canvas(ALL);
  return(1);
  }

/* save details */
strcpy(data->filename, inp_file);
g_free(data->basename);
data->basename = strdup_no_extension(g_basename(inp_file));

/* configure for displaying */
data->mode = FREE;
sysenv.active = model;
data->axes_on = TRUE;
/* NB: CELL must be initialized b4 coords, for centering purposes */
/* non 90 axes shifting is done by update_coords(INITIAL) */
if (data->periodic)
  {
  data->cell_on = TRUE;
  data->axes_type = OTHER;
/* space group init here */
  update_shells(data);
  if (!data->sginfo.spacenum)
    {
#if DEBUG_LOAD_MODEL
printf("No Space Group found: assuming P1.\n");
#endif
    data->sginfo.spacenum=1;
    }
  if (genpos(data))
    printf("Error in Space Group lookup.\n");
  }

init_objs(INIT_COORDS, data);
/* store original cartesian coords */
save_cart(data);
if (data->periodic && !data->fractional)
    {
    latmat_fudge(data);
    init_objs(INIT_COORDS, data);
    }

calc_bonds(data);
calc_mols(data);

/* update any existing dialogue boxes */
new_tree_item(model, APPEND);

fclose (fp);
return(0);
}

/*******************/
/* data file parse */
/*******************/
#define DEBUG_LOAD 0
gint load_data(FILE *fp, struct parse_pak *parse_data, struct model_pak *data)
{
gchar line[LINELEN], key[LINELEN], buff[LINELEN];
gint i, j, start, type, region, status;
gint space_flag=0;
struct model_pak temp;

/* NEW - read from file into temp & copy back to data only on success */
/* copy data to temp */
memcpy(&temp, data, sizeof(struct model_pak));

/* skip any initial rubbish */
i=0;
while(i<parse_data->hdr_skip)
  {
/* EOF check */
  if (fgetline(fp,line))
    return(2);
/* flag if file has a PBC */
  sscanf(line,"%s",key); 
  if (g_strncasecmp(key,"PBC=ON",6) == 0)
    temp.periodic = 3;
  i++;
  }

#if DEBUG_LOAD
printf("Parsing header...\n");
#endif

switch(parse_data->id)
  {
/* if BIOSYM and periodic - start of coords is after PBC line (not !DATE) */
  case BIOSYM:
    if (temp.periodic)
      {
      i=0;
      while(g_strncasecmp("pbc ", line, 4) != 0 && !i)
        i = fgetline(fp,line);
/* EOF check */
      if (i)
        return(3);
      copy_items(buff,line,2,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[0]));
      copy_items(buff,line,3,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[1]));
      copy_items(buff,line,4,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[2]));
      copy_items(buff,line,5,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[3]));
      copy_items(buff,line,6,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[4]));
      copy_items(buff,line,7,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[5]));
/* degrees -> radians */
      temp.pbc[3] *= PI/180.0;
      temp.pbc[4] *= PI/180.0;
      temp.pbc[5] *= PI/180.0;
/* don't search for coord start string */
      parse_data->coord_start1_len = 0;
/* prevent the warning message (keyword corruption) being printed */
      strcpy(parse_data->coord_start1,"");
      }
    break;

  case XTL:
    temp.periodic = 3;
    temp.fractional = TRUE;
/* seek dimension keyword */
    while (!fgetline(fp,line))
      {
      if (g_strncasecmp("dimension", line, 9) == 0)
        {
        copy_items(buff,line,2,SINGLE);
        sscanf(buff,"%d",&temp.periodic);
        break;
        }
      }
    rewind(fp);
/* seek cell keyword */
    while (!fgetline(fp,line))
      {
      if (g_strncasecmp(parse_data->special_start, line,
                        parse_data->special_start_len) == 0)
        break;
      }
/* get next line (if possible) */
    if (fgetline(fp,line))
      return(4);
    copy_items(buff,line,1,SINGLE);
    sscanf(buff,"%f",&(temp.pbc[0]));
    copy_items(buff,line,2,SINGLE);
    sscanf(buff,"%f",&(temp.pbc[1]));
/* dimension dependent parsing */
    if (temp.periodic == 2)
      {
      copy_items(buff,line,3,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[5]));
      temp.pbc[2] = 0.0;
      temp.pbc[3] = 90.0;
      temp.pbc[4] = 90.0;
      }
    else
      {
      copy_items(buff,line,3,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[2]));
      copy_items(buff,line,4,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[3]));
      copy_items(buff,line,5,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[4]));
      copy_items(buff,line,6,SINGLE);
      sscanf(buff,"%f",&(temp.pbc[5]));
      }
/* degrees -> radians */
    temp.pbc[3] *= PI/180.0;
    temp.pbc[4] *= PI/180.0;
    temp.pbc[5] *= PI/180.0;

/* try to get symmetry info */
/* FIXME - yet another unholy file parsing mess */
    if (fgetline(fp,line))
      return(5);
    copy_items(buff,line,1,SINGLE);
    while(g_strncasecmp(buff,"symmetry",8) == 0 && !space_flag)
      {
      i=2;
      copy_items(buff,line,i,SINGLE);
      while(strlen(buff) && !space_flag)
        {
/* seek space group label */
        if (g_strncasecmp(buff,"label",5) == 0)
          {
          copy_items(buff,line,i+1,SINGLE);
          g_free(temp.sginfo.spacename);
          temp.sginfo.spacename = g_strdup_printf("%s",buff);

/* indicate that name should used in lookup */
          temp.sginfo.spacenum = -1;
          space_flag++;
          }
/* seek space group number */
/* TODO - origin qualifier (cellchoice) */
        if (g_strncasecmp(buff,"number",6) == 0 && temp.sginfo.spacenum > -1)
          {
          copy_items(buff,line,i+1,SINGLE);
/* do lookup via this number only if no label can be found */
          sscanf(buff,"%d",&temp.sginfo.spacenum);
          }
        i++;
        copy_items(buff,line,i,SINGLE);
        }
      if (!space_flag)
        {
        if (fgetline(fp,line))
          return(6);
        copy_items(buff,line,1,SINGLE);
        }
      }
    break;
  }

#if DEBUG_LOAD
printf("Searching for data...\n");
#endif

/* shouldn't happen any more ??? */
if (strlen(parse_data->coord_start1) != parse_data->coord_start1_len)
  printf("Warning: corrupted keyword [%s] (%d)\n",parse_data->coord_start1,
                                                  parse_data->coord_start1_len);
/* find start of coords */
i=0;
while(!i && parse_data->coord_start1_len)
  {
/* get lines */
  if (fgetline(fp,line))
    return(7);
/* get 1st item */
  copy_items(key,line,1,SINGLE);
/* keyword scan */
  if (strlen(key) < 3)
    continue;
  if (g_strncasecmp(key,parse_data->coord_start1,parse_data->coord_start1_len) == 0)
    {
    i=1;
    }
/* terminate on match2 (if there is one) - set any special flags */
  if (parse_data->coord_start2_len && 
      g_strncasecmp(key,parse_data->coord_start2,parse_data->coord_start2_len) == 0)
    {
    i=1;
    }
  }

#if DEBUG_LOAD
printf("found!\n");
#endif

/* marvin region parsing */
temp.region_empty[REGION1A] = TRUE;
temp.region_empty[REGION2A] = TRUE;
temp.region_empty[REGION1B] = TRUE;
temp.region_empty[REGION2B] = TRUE;
if (temp.id == MARVIN)
  {
/* get region type */
  copy_items(key,line,2,ALL);
  if (g_strncasecmp(key,"1 b",3) == 0 || g_strncasecmp(key,"1b",2) == 0)
    region = REGION1B;
  else if (g_strncasecmp(key,"2 a",3) == 0 || g_strncasecmp(key,"2a",2) == 0)
    region = REGION2A;
  else if (g_strncasecmp(key,"2 b",3) == 0 || g_strncasecmp(key,"2b",2) == 0)
    region = REGION2B;
  else
    region = REGION1A;      /* default */
/* we have data for some region */
  temp.region_empty[region] = FALSE;
  }
/* allows saving as mvn-r */
else
  {
  temp.region_empty[REGION1A] = FALSE;
  region = REGION1A;
  }

/* init for data reading */
i=j=0;                  /* atoms & shells counters */
status=0;               /* from now on an EOF is not an error */

/* MAIN - parse lines of data */
start=0;
while (!fgetline(fp,line))
  {
/* wait until 1st valid line is reached */
/* ie skip (but don't end!) on anything with less than 4 (atom x y z) items */
  copy_items(buff,line,4,SINGLE);
  if (!strlen(buff) && !start)
    continue;
  else
    start=1;

/* set up for keyword parse */
  copy_items(key,line,1,SINGLE);
/* test for a new start of data keyword */
  if (parse_data->coord_start1_len &&
      g_strncasecmp(key,parse_data->coord_start1,parse_data->coord_start1_len) == 0)
    {
/* do we have anything to do? */
    switch(temp.id)
      {
      case MARVIN:
        copy_items(key,line,2,ALL);
/* get region type */
        if (g_strncasecmp(key,"1 b",3) == 0 ||
            g_strncasecmp(key,"1b",2) == 0)
          region = REGION1B;
        else if (g_strncasecmp(key,"2 a",3) == 0 || 
                 g_strncasecmp(key,"2a",2) == 0)
          region = REGION2A;
        else if (g_strncasecmp(key,"2 b",3) == 0 || 
                 g_strncasecmp(key,"2b",2) == 0)
          region = REGION2B;
        else
          region = REGION1A;      /* default */
/* we have data for some region */
        temp.region_empty[region] = FALSE;
        break;
      }
/* ok, get next line (actual data) */
    if (fgetline(fp,line))
      break;
/* set up for keyword parse */
    copy_items(key,line,1,SINGLE);
    }

/* test for end keyword */
  if (g_strncasecmp(key,parse_data->coord_end1,parse_data->coord_end1_len) == 0)
    {
/* everything else breaks on this */
    if (temp.id != BIOSYM)
      break;
/* parse for things like end end */
    if (parse_data->coord_end2_len)
      {
      if (fgetline(fp,line))
        break;
      copy_items(key,line,1,SINGLE);
/* terminate on end end (biosym) */
      if (g_strncasecmp(key,parse_data->coord_end2,parse_data->coord_end2_len) == 0)
        break;
      }
    else
      break;
    }
/* end on anything with less than 4 (atom x y z) items */
/* unless MARVIN, since it may be a region changeover */
  copy_items(buff,line,4,SINGLE);
  if (!strlen(buff))
    {
    if (temp.id == MARVIN)
      continue;
    else
      break;
    }

/* if an atom type (ie core/shell) is specified - adjust data positions */
  type = 0;
  copy_items(buff,line,parse_data->x_pos,SINGLE);
  if (g_strncasecmp(buff,"core",4) == 0)
    type=1;
  if (g_strncasecmp(buff,"shel",4) == 0)
    type=2;
/* read appropriately */
  switch(type)
    {
    case 0:
    case 1:
/* new core */
      if (i >= temp.atom_limit)
        mem_grow(&temp, ATOM, 10);
/* setup */
      (temp.atoms+i)->status = NORMAL;
      (temp.atoms+i)->primary = TRUE;
      (temp.atoms+i)->orig = TRUE;
      (temp.atoms+i)->region = region;
      (temp.atoms+i)->offx = 0;
      (temp.atoms+i)->offy = 0;
/* NB: assumes if there is core/shell a_pos = 1 always */
      copy_items(buff,line,parse_data->a_pos,SINGLE);
/* read the atom type & corresponding coords */
      sscanf(buff,"%s",(temp.atoms+i)->element);
/* NB: assumes if there is core/shell l_pos = 1 always */
      copy_items(buff,line,parse_data->l_pos,SINGLE);
      sscanf(buff,"%s",(temp.atoms+i)->label);
/* get data */
      copy_items(buff,line,type+parse_data->x_pos,SINGLE);
      sscanf(buff,"%f",&((temp.atoms+i)->x));
      copy_items(buff,line,type+parse_data->y_pos,SINGLE);
      sscanf(buff,"%f",&((temp.atoms+i)->y));
      copy_items(buff,line,type+parse_data->z_pos,SINGLE);
      sscanf(buff,"%f",&((temp.atoms+i)->z));
/* occupancy */
      (temp.atoms+i)->sof = 1.0;
/* save additional data (if any) */
      if (parse_data->tail_pos)
        copy_items((temp.atoms+i)->tail,line,type+parse_data->tail_pos,ALL);
      else
        strcpy((temp.atoms+i)->tail,"\0");
/* this does setup */
      elem_seek(i, ATOM, &temp);

#if DEBUG_MORE
printf("%s %f %f %f [%d]\n", (temp.atoms+i)->label
                           , (temp.atoms+i)->x
                           , (temp.atoms+i)->y
                           , (temp.atoms+i)->z
                           , (temp.atoms+i)->atom_code);
#endif
      i++;
      break;
    case 2:
/* new shell */
      if (j >= temp.shell_limit)
        mem_grow(&temp, SHELL, 10);
/* setup the shell */
      (temp.shells+j)->status = NORMAL;
      (temp.shells+j)->primary = TRUE;
      (temp.shells+j)->orig = TRUE;
      (temp.shells+j)->region = region;
      (temp.shells+j)->offx = 0;
      (temp.shells+j)->offy = 0;
/* read the atom type & corresponding coords */
/* NB: assumes its always [atom] [core/shell] x y z */
      copy_items(buff,line,1,SINGLE);
      sscanf(buff,"%s",(temp.shells+j)->label);
      copy_items(buff,line,1,SINGLE);
      sscanf(buff,"%s",(temp.shells+j)->element);
      copy_items(buff,line,3,SINGLE);
      sscanf(buff,"%f",&((temp.shells+j)->x));
      copy_items(buff,line,4,SINGLE);
      sscanf(buff,"%f",&((temp.shells+j)->y));
      copy_items(buff,line,5,SINGLE);
      sscanf(buff,"%f",&((temp.shells+j)->z));
/* save additional data */
      if (parse_data->tail_pos)
        copy_items((temp.shells+j)->tail,line,type+parse_data->tail_pos,ALL);
      else
        strcpy((temp.shells+j)->tail,"\0");

#if DEBUG_MORE
printf("%s %f %f %f\n", (temp.shells+j)->element
                      , (temp.shells+j)->x
                      , (temp.shells+j)->y
                      , (temp.shells+j)->z);
#endif 

/* match shell type to database */      
      elem_seek(j, SHELL, &temp);
/* update counter */
      j++;
      break;
    }
  }

/* got them all */
temp.num_asym = temp.num_orig = temp.num_atoms = i;
temp.num_shells = j;

/* MARVIN - region highlighting */
if (temp.id == MARVIN)
  {
  j=0;
  for (i=REGION1A ; i<REGION2B ; i++)
    {
    if (!temp.region_empty[i] && !j)
      {
      temp.region[i] = TRUE;
      j++;
      }
    else
      temp.region[i] = FALSE;
    }
  }

#if DEBUG_LOAD
printf("model loaded: found %d atom(s) ",temp.num_atoms);
printf("and %d shell(s).\n",temp.num_shells);
printf("load_data() done.\n");
#endif

/* copy data to temp */
if (!status)
  memcpy(data, &temp, sizeof(struct model_pak));

return(status);
}

/********************/
/* BABEL conversion */
/********************/
gint babel_convert(gint model, gchar *inpfile)
{
gint status;
gchar *inp, *out, *outfile;
GString *txt; 

/* TODO - ask user what type of input it is, */
/* rather than an extension based guess */
switch (sysenv.babel_type)
  {
/* FIXME - what about the periodic data??? */
  case CSSR:
    inp = g_strdup("cssr");
    break;
  case FDAT:
    inp = g_strdup("fdat");
    break;
  case GAMOUT:
    inp = g_strdup("gamout");
    break;
  case MOPAC_OUT:
    inp = g_strdup("mopout");
    break;
  case PDB:
    inp = g_strdup("pdb");
    break;
  default:
    printf("babel_convert(): unknown file type.\n");
    return(1);
  }

/* set up new filename */
out = g_strdup("xyz");
txt = g_string_new(NULL);
g_string_assign(txt,inpfile);
g_string_append(txt,".xyz");
outfile = g_strdup(txt->str);
/* convert & load via standard routine */
printf("Converting with babel...\n");
g_string_sprintf(txt,"%s -i%s %s -o%s %s >& /dev/null",sysenv.babel_exe,
                                           inp, inpfile, out, outfile);
system(txt->str);
status = load_model(model, XYZ, outfile);
/* free & exit */
g_string_free(txt, TRUE);
g_free(inp);
g_free(out);
g_free(outfile);
return(status);
}

/*********************/
/* create a new file */
/*********************/
gint create_file(gchar *filename, gint type, struct model_pak *data)
{
FILE *fp;
gchar label[LABEL_SIZE], prefix[LABEL_SIZE], postfix[LINELEN];
gfloat x[3], s[3];
gint mol, m, n, i, j, flag;
time_t t1;

/* open the file */
fp = fopen(filename,"w");
if (!fp)
  {
  show_text("Error, bad filename!");
  return(1);
  }

/* print header */
switch(type)
  {
  case BIOSYM:
    show_text("Saving file in BIOSYM format!");
    fprintf(fp,"!BIOSYM archive 3\n");
    if (data->periodic)
      fprintf(fp,"PBC=ON\n");
    else
      fprintf(fp,"PBC=OFF\n");
    fprintf(fp,"Created by GDIS\n");
/* get & print the date (streamlined slightly by sxm) */
    t1 = time(NULL);
    fprintf(fp,"!DATE %s",ctime(&t1));
/* convert back to degrees!!! */
    if (data->periodic)
      fprintf(fp,"PBC   %7.5f  %7.5f  %7.5f  %7.5f  %7.5f  %7.5f (P1)\n"
                  ,data->pbc[0]
                  ,data->pbc[1]
                  ,data->pbc[2]
                  ,180.0*data->pbc[3]/PI
                  ,180.0*data->pbc[4]/PI
                  ,180.0*data->pbc[5]/PI);

/* TODO - if spacenum > 1 - only write asym unit */
/*
                  ,data->sginfo.spacename);
*/
    break;
  case XYZ:
    show_text("Saving file in XYZ format!");
    fprintf(fp,"%d\n",data->num_atoms);
    fprintf(fp,"XYZ\n");
    break;
/* all other types are not supported */
  default:
    show_text("Error: unsuported save type.");
    return(1);
  }


/* save via the number of molecules in the */
/* model - since this allows mol numbering */
mol=0;
for (n=0 ; n<data->num_mols ; n++)
  {
/* flag - atom written to file? (ie if UNDELETED) */
  flag=0;
/* m - number of atoms in molecule */
  for (m=0 ; m<(data->mols+n)->num_atoms ; m++)
    {
/* get the index for the mth atom in the nth molecule */
    i = *((data->mols+n)->atom+m);

/* setup the strings that go after the coords */
    if (data->id == BIOSYM)
      {
/* we should already have one */
      copy_items(label,(data->atoms+i)->tail, 1, SINGLE);
      copy_items(postfix,(data->atoms+i)->tail, 3, ALL);
      }
    else
      {
/* we need to create one */
      strcpy(label,"CORE");
      strcpy(postfix,"??      ?   0.000");
      put_item(postfix,(data->atoms+i)->element,2);
      }

/* write atom to file unless deleted */
    if (!((data->atoms+i)->status & DELETED))
      {
/* use the original label instead of the processed element symbol */
      g_snprintf(prefix,5,"%-4s",(data->atoms+i)->label);

/* get associated shell (if any) */
/* NOTE: no effect at present - how to save shells in BIOSYM format? */
      j=-1;
      if ((data->atoms+i)->has_shell)
        j = (data->atoms+i)->idx_shell;

/* paranoid test */
      if (!data->periodic && data->fractional)
        printf("Serious error in create_file().\n");
/* new scheme - evrything is cartesian after latmat mult */
        x[0] = (data->atoms+i)->x;
        x[1] = (data->atoms+i)->y;
        x[2] = (data->atoms+i)->z;
        vecmat(data->latmat, x);
/* shell */
        if (j >= 0)
          {
          s[0] = (data->shells+j)->x;
          s[1] = (data->shells+j)->y;
          s[2] = (data->shells+j)->z;
          vecmat(data->latmat, s);
          }

/* now write according to respective formats */
      switch(type)
        {
/* unique molecule numbering for BIOSYM files (>0) */
        case BIOSYM:
          fprintf(fp,"%4s  %14.9f %14.9f %14.9f %-4s %-6d %s\n",
                      prefix,x[0],x[1],x[2],label,mol+1,postfix);
          break;
        case XYZ:
          fprintf(fp,"%-7s    %14.9f  %14.9f  %14.9f\n",
                      (data->atoms+i)->element,x[0],x[1],x[2]);
          break;
        }
/* indicate we've written at least one atom */
      flag++;
      }
    }     /* for m */

/* if at least one atom in the molecule was written - inc mol numbering */
  if (flag)
    mol++;

/* do anything special at the end of each molecule? */
  switch(type)
    {
/* only write if at least one atom in molecule is undeleted */
    case BIOSYM:
      if (flag)
        fprintf(fp,"end\n");
      break;
    }
  }       /* for n */

/* create tail */
switch(type)
  {
  case BIOSYM:
    fprintf(fp,"end\n");
    break;
  }

fclose(fp);
return(0);
}

/***********/
/* globals */
/***********/
GtkWidget *path_list, *file_list, *curr_path, *file_name;
gint path_width=80, file_width=160;
extern int alphasort();
gint file_filter();
gint path_filter();

/**************************/
/* primary load file call */
/**************************/
void file_load()
{
gint i, flag, model, id, status;
gchar inp_file[FILELEN];
FILE *fp;
struct model_pak *data;

strcpy(inp_file, sysenv.cwd); 
strcat(inp_file, gtk_entry_get_text(GTK_ENTRY(file_name)));

/* get the model type */
if ((id = model_type(inp_file)) < 0)
  {
/* unrecognized - is it a directory? */
  set_path(inp_file);
  gtk_entry_set_text(GTK_ENTRY(file_name), "");
  return;
  }

/* get the new model number */
model = sysenv.num_models;
status = -1;
if (model < MAX_MODELS)
  {
/* create fp here, so errors in load routines don't have to clean up */
  fp = fopen(inp_file,"r");
  if (!fp)
    {
    show_text("Error: bad or missing filename.");
    return;
    }
/* cleaner branching to specific loading routines */
/* each routine does it's own alloc, so multiple models can be loaded */
  switch(id)
    {
    case CIF:
      status = load_cif(fp, model, inp_file);
      fclose(fp);
      break;
    case GULP:
      status = load_gulp(fp, model, inp_file);
      fclose(fp);
      break;
    case GULPOUT:
      status = load_gulp_output(fp, model, inp_file);
      fclose(fp);
      break;
    case MVNOUT:
      status = load_mvnout(fp, model, inp_file);
      fclose(fp);
      break;
    case MORPH:
      status = load_gmf(fp, model, inp_file);
      fclose(fp);
      break;
    case BIOSYM:
    case MARVIN:
    case XTL:
    case XYZ:
      fclose(fp);
      status = load_model(model, id, inp_file);
      break;
/* babel handled filetypes */
    case BABEL:
      fclose(fp);
/* TODO - get id from additional filtermenu */
      status = babel_convert(model, inp_file);
      break;
    default:
      printf("file_load() error, no handling routine for filetype.\n");
      fclose(fp);
    }

/* check for successful file load */
  if (status)
    printf("Load failed, error code: %d\n",status);
  else
    {
    data = model_ptr(model, RECALL);
/* surfaces are always conv by default */
    if (data->periodic == 2)
      data->gulp.method = CONV;
    show_text("New model loaded.");

/* NEW - gulp supercell hack */
flag=0;
for (i=0 ; i<3 ; i++)
  {
  if (data->gulp.super[i] > 1)
    {
    data->image_limit[2*i+1] = data->gulp.super[i];
    flag++;
    }
  }
if (flag)
  {
  update_images(data->number, 0, CREATE);
  make_p1(data);
/* refresh */
  init_objs(INIT_COORDS, data);
  calc_bonds(data);
  calc_mols(data);
  }

/* updates */
    update_geom_info();
    redraw_canvas(ALL);
/* highlight the new model */
    tree_select(sysenv.active);
    }
  }
else
  show_text("Maximum number of allowed models reached.");


/* finished - only now we destroy the file select widget */
close_dialog_type(FILE_LOAD);
}

/***************************************/
/* update contents of file load widget */
/***************************************/
#define DEBUG_UPDATE_FILE_PANE 0
void update_file_pane()
{
gint i,n,type;
gchar *name[1], full[FILELEN];
#ifdef __sun
struct dirent **files;
#else
struct direct **files;
#endif
struct stat buff;

name[0] = (gchar *) g_malloc(FILELEN*sizeof(gchar));

/* getting from this directory */
gtk_label_set_text(GTK_LABEL(curr_path), sysenv.cwd);

/* clear the CLists */
gtk_clist_clear((GtkCList *) path_list);
gtk_clist_clear((GtkCList *) file_list);

/* get directory listing */
n = scandir(sysenv.cwd, &files, 0, alphasort);

/* special case for a failure (is this always a no permission error?) */
if (n < 0)
  {
  show_text("scandir failed, perhaps you don't have access permission.");
/* stick in the .. so we can always climb out */
  strcpy(name[0], "..");
  gtk_clist_append((GtkCList *) path_list, &name[0]);
  }

for (i=0 ; i<n ; i++)
  {
/* copy to right pointer type */
  strcpy(name[0], files[i]->d_name);
/* stat the file (MUST have the full path) */
  strcpy(full, sysenv.cwd);
  strcat(full, files[i]->d_name);
  stat(full, &buff);

#if DEBUG_UPDATE_FILE_PANE
printf("[%s] : %d\n",full,buff.st_mode);
#endif

/* convert and check if directory */
  if (S_ISDIR(buff.st_mode))
    gtk_clist_append((GtkCList *) path_list, &name[0]);
  else
    {
/* is it a recognized type? */
    if ((type = model_type(files[i]->d_name)) < 0)
      continue;
/* display related types together */
    if (type == MVNOUT)
      type = MARVIN;
    if (type == GULPOUT)
      type = GULP;
/* add it to the list if filter allows */
/* separate babel from everything else */
    if (sysenv.file_type == BABEL || type == BABEL) 
      {
      if (type == sysenv.file_type) 
        gtk_clist_append((GtkCList *) file_list, &name[0]);
      }
    else
      if (type == sysenv.file_type || sysenv.file_type == DATA) 
        gtk_clist_append((GtkCList *) file_list, &name[0]);
    }
  }

/* done */
g_free(name[0]);
}

/***********************************/
/* change to a specified directory */
/***********************************/
#define DEBUG_SET_PATH 0
void set_path(gchar *text)
{
gint i, n;
gchar path[FILELEN], **buff;
struct stat info;

/* remove any \n's explicitly? */
g_strstrip(text);
buff = g_strsplit(text, "/", 0);

/* find number of tokens */
n=0;
while(*(buff+n))
  {
#if DEBUG_SET_PATH
printf("[%s] ", *(buff+n));
#endif
  n++;
  }
#if DEBUG_SET_PATH
printf("Found %d tokens\n", n);
#endif

/* do we go up a subdir? */
if (g_strncasecmp("..",*(buff+n-1),2) == 0)
  {
#if DEBUG_SET_PATH
printf("Going up a directory.\n");
#endif
/* form path from all but the last one */
  strcpy(path, "/");
  for (i=0 ; i<n-2; i++)
    {
    if (strlen(*(buff+i)))
      {
      strcat(path, *(buff+i));
      strcat(path, "/");
      }
    }
  g_strfreev(buff);
#if DEBUG_SET_PATH
printf("\n");
#endif
  }
else
  {
  strcpy(path, text);
  strcat(path, "/");
  }

/* TODO - test if path is valid */
stat(path, &info);
if (S_ISDIR(info.st_mode))
  {
  strcpy(sysenv.cwd, path);
#if DEBUG_SET_PATH
printf("Path changed to: [%s]\n", sysenv.cwd);
#endif
  }
else
  {
  printf("Invalid path: [%s]\n", path);
  }

/* get contents of new directory */
update_file_pane();
}

/*****************************/
/* CList handler - directory */
/*****************************/
void path_handler(GtkWidget *clist, gint row, gint column,
                  GdkEventButton *event, gpointer data)
{
gchar *text, path[FILELEN];

gtk_clist_get_text(GTK_CLIST(path_list), row, column, &text);

strcpy(path, sysenv.cwd);
strcat(path, text);

set_path(path);
}

/*****************************/
/* CList handler - file name */
/*****************************/
void file_handler(GtkWidget *clist, gint row, gint column,
                  GdkEventButton *event, gpointer data)
{
gchar *text;

gtk_clist_get_text(GTK_CLIST(file_list), row, column, &text);
gtk_entry_set_text(GTK_ENTRY (file_name), text);
}

/*******************/
/* filter handlers */
/*******************/
void change_filter(GtkWidget *w, gint type)
{
sysenv.file_type = type;
update_file_pane();
}
void change_babel_filter(GtkWidget *w, gint type)
{
sysenv.babel_type = type;
update_file_pane();
}

/****************************/
/* new filter item routines */
/****************************/
GtkWidget *make_filter_item(gchar *name, gpointer data)
{
GtkWidget *item;
 
item = gtk_menu_item_new_with_label(name);
gtk_signal_connect(GTK_OBJECT(item), "activate", change_filter, data);
gtk_widget_show(item);

return(item);
}

GtkWidget *make_babel_item(gchar *name, gpointer data)
{
GtkWidget *item;
 
item = gtk_menu_item_new_with_label(name);
gtk_signal_connect(GTK_OBJECT(item), "activate", change_babel_filter, data);
gtk_widget_show(item);

return(item);
}

/*************************************************/
/* customized load widget (with filetype filter) */
/*************************************************/
void load_widget()
{
gint id;
GtkWidget *swin, *hbox;
GtkWidget *filter, *filter_menu, *item;
GtkWidget *babel, *babel_menu;
GtkWidget *label, *button;
struct dialog_pak *fl_data;

/* get a dialog if possible */
if ((id = request_dialog(-1,FILE_LOAD)) < 0)
  return;
fl_data = &sysenv.dialog[id];

/* make and set up the dialog window */
fl_data->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW(fl_data->win), "Load file");
gtk_window_set_default_size(GTK_WINDOW(fl_data->win), 400, 350);
gtk_window_set_position(GTK_WINDOW(fl_data->win), GTK_WIN_POS_CENTER);
gtk_container_set_border_width(
    GTK_CONTAINER(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox)),10);
gtk_signal_connect(GTK_OBJECT(fl_data->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* TOP ROW - cwd printed */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox),hbox,FALSE,FALSE,0);
label = gtk_label_new("Current path: ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

curr_path = gtk_label_new("dummy");
gtk_box_pack_start(GTK_BOX(hbox), curr_path, FALSE, FALSE, 0);

/* SECOND ROW - sub directory & file listings */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox), hbox, TRUE, TRUE, 0);

/* PATH - scrolled model pane */
swin = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(hbox), swin, TRUE, TRUE, 0);
/* Create the CList */
path_list = gtk_clist_new(1);
/* event */
gtk_signal_connect(GTK_OBJECT(path_list), "select_row",
                   GTK_SIGNAL_FUNC(path_handler), NULL);

/* set some properties of the CList */
gtk_clist_set_column_width(GTK_CLIST(path_list), 0, path_width);
gtk_clist_set_column_title(GTK_CLIST(path_list), 0, "Tree");
gtk_clist_column_titles_show(GTK_CLIST(path_list));
gtk_clist_column_titles_passive(GTK_CLIST(path_list));
gtk_container_add(GTK_CONTAINER(swin), path_list);
gtk_container_set_border_width (GTK_CONTAINER (swin), 10);

/* FILE - scrolled model pane */
swin = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(hbox), swin, TRUE, TRUE, 0);
/* Create the CList */
file_list = gtk_clist_new(1);
/* event */
gtk_signal_connect(GTK_OBJECT(file_list), "select_row",
                   GTK_SIGNAL_FUNC(file_handler), NULL);

/* set some properties of the CList */
gtk_clist_set_column_width(GTK_CLIST(file_list), 0, file_width);
gtk_clist_set_column_title(GTK_CLIST(file_list), 0, "Files");
gtk_clist_column_titles_show(GTK_CLIST(file_list));
gtk_clist_column_titles_passive(GTK_CLIST(file_list));
gtk_container_add(GTK_CONTAINER(swin), file_list);
gtk_container_set_border_width (GTK_CONTAINER (swin), 10);

/* THIRD ROW - filename currently selected & file extension filter */
hbox = gtk_hbox_new(FALSE,5);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->vbox), hbox, FALSE, FALSE, 0);

/* filename */
file_name = gtk_entry_new();
gtk_box_pack_start(GTK_BOX (hbox), file_name, TRUE, TRUE, 0);
gtk_entry_set_editable(GTK_ENTRY (file_name), TRUE);
/* hook a <CR> event to the load action */
gtk_signal_connect(GTK_OBJECT (file_name), "activate", 
                   GTK_SIGNAL_FUNC (file_load), NULL);

/* file extension filter */
filter= gtk_option_menu_new();
gtk_widget_ref(filter);
gtk_box_pack_start(GTK_BOX (hbox), filter, FALSE, FALSE, 0);
filter_menu = gtk_menu_new();

/* NB: these should be sequential in the gdis.h enum{} */
item = make_filter_item ("all known", (gpointer *) DATA);
gtk_menu_append (GTK_MENU(filter_menu), item);

item = make_filter_item ("filterable", (gpointer *) BABEL);
gtk_menu_append (GTK_MENU (filter_menu), item);

item = make_filter_item ("biosym", (gpointer *) BIOSYM);
gtk_menu_append (GTK_MENU (filter_menu), item);

item = make_filter_item ("cif", (gpointer *) CIF);
gtk_menu_append (GTK_MENU (filter_menu), item);

item = make_filter_item ("gulp", (gpointer *) GULP);
gtk_menu_append (GTK_MENU (filter_menu), item);

item = make_filter_item ("marvin", (gpointer *) MARVIN);
gtk_menu_append (GTK_MENU (filter_menu), item);

item = make_filter_item ("morphology", (gpointer *) MORPH);
gtk_menu_append (GTK_MENU (filter_menu), item);

item = make_filter_item ("xtl", (gpointer *) XTL);
gtk_menu_append (GTK_MENU (filter_menu), item);

item = make_filter_item ("xyz", (gpointer *) XYZ);
gtk_menu_append (GTK_MENU (filter_menu), item);

gtk_option_menu_set_menu(GTK_OPTION_MENU (filter), filter_menu);

/* NB: if the above are not sequential, this will select the wrong item */
gtk_option_menu_set_history(GTK_OPTION_MENU(filter), sysenv.file_type - DATA);
gtk_menu_set_active(GTK_MENU(filter_menu), sysenv.file_type - DATA);

/* babel filter */
babel = gtk_option_menu_new();
gtk_widget_ref(babel);
gtk_box_pack_start(GTK_BOX (hbox), babel, FALSE, FALSE, 0);
babel_menu = gtk_menu_new();

item = make_babel_item("no filter", (gpointer *) NONBABEL);
gtk_menu_append(GTK_MENU(babel_menu), item);

item = make_babel_item("CSD cssr", (gpointer *) CSSR);
gtk_menu_append(GTK_MENU(babel_menu), item);

item = make_babel_item("CSD fdat", (gpointer *) FDAT);
gtk_menu_append(GTK_MENU(babel_menu), item);

item = make_babel_item("Gamess output", (gpointer *) GAMOUT);
gtk_menu_append(GTK_MENU(babel_menu), item);

item = make_babel_item("Mopac output", (gpointer *) MOPAC_OUT);
gtk_menu_append(GTK_MENU(babel_menu), item);

item = make_babel_item("PDB", (gpointer *) PDB);
gtk_menu_append(GTK_MENU(babel_menu), item);

gtk_option_menu_set_menu(GTK_OPTION_MENU (babel), babel_menu);
gtk_option_menu_set_history(GTK_OPTION_MENU(babel), sysenv.babel_type - NONBABEL);
gtk_menu_set_active(GTK_MENU(babel_menu), sysenv.babel_type - NONBABEL);

/* LAST ROW - terminating buttons */
hbox = gtk_hbox_new(FALSE,10);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fl_data->win)->action_area), hbox,
                                                     FALSE, TRUE, 0);

button = gtk_button_new_with_label ("     Ok    ");
gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (file_load), NULL);

button = gtk_button_new_with_label ("  Cancel  ");
gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT(button), "clicked",
                           GTK_SIGNAL_FUNC(close_dialog),
                          (gpointer) id);
/* all done */
gtk_widget_show_all(fl_data->win);
update_file_pane();
}

/********************/
/* save file action */
/********************/
/* no longer used */
void file_save()
{
struct model_pak *data;

if (sysenv.num_models)
  {
  data = model_ptr(sysenv.active, RECALL);
  if (!data)
    {
    printf("file_save(): failed to get model pointer!\n");
    return;
    }
  file_action(data->filename, FILE_SAVE, sysenv.active);
  }
}

/***********************/
/* save as file action */
/***********************/
void file_save_as(GtkWidget *w, GtkWidget *fs)
{
gchar *inp_file;

/* get selected name and check */
inp_file = (gchar *) gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs));
if (!strlen(inp_file))
  return;
/* paranoid check */
if (sysenv.num_models)
  file_action(inp_file, FILE_SAVE_AS, sysenv.active);
 
/* finished saving data - only now we destroy the file select widget */
gtk_widget_destroy(fs);
}

/******************/
/* file - save as */
/******************/
void save_as_widget()
{
GtkWidget *fs_widget;
struct model_pak *data;

/* return if nothing is loaded */
if (!sysenv.num_models)
  return;

fs_widget = gtk_file_selection_new ("File save as");
gtk_window_set_position(GTK_WINDOW(fs_widget), GTK_WIN_POS_CENTER);
gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(fs_widget));
 
/* setup events for the file selection widget */
gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (fs_widget)->ok_button),
                    "clicked", (GtkSignalFunc) file_save_as, fs_widget);

gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION 
                          (fs_widget)->cancel_button), "clicked",
                          (GtkSignalFunc) gtk_widget_destroy,
                           GTK_OBJECT (fs_widget));

/* set default filename to save as */
data = model_ptr(sysenv.active, RECALL);
gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs_widget), data->filename);
gtk_widget_show(fs_widget);
}

