/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "gdis.h"

/* main structure */
extern struct sysenv_pak sysenv;

/*************************/
/* multiply two matrices */
/*************************/
void matmat(gfloat *mat1, gfloat *mat2)
/* mat2 -> mat1*mat2 */
{
gint i;
gfloat tmp[9], *res, *ptr1, *ptr2;

/* init */
transpose(mat2);
res = tmp;
ptr1 = mat1;

/* mult - loop over rows of mat 1*/
for (i=0 ; i<3 ; i++)
  {
  ptr2 = mat2;

  *res = *(ptr1++) * (*(ptr2++));
  *res += *(ptr1++) * (*(ptr2++));
  *(res++) += *(ptr1--) * (*(ptr2++));
  ptr1--;

  *res = *(ptr1++) * (*(ptr2++));
  *res += *(ptr1++) * (*(ptr2++));
  *(res++) += *(ptr1--) * (*(ptr2++));
  ptr1--;

  *res = *(ptr1++) * (*(ptr2++));
  *res += *(ptr1++) * (*(ptr2++));
  *(res++) += *(ptr1++) * (*ptr2);
  }

/* copy result to mat2 */
memcpy(mat2,tmp,9*sizeof(gfloat));

return;
}

/****************************************/
/* transform a 3D vector by matrix mult */
/****************************************/
void vecmat(gfloat *mat, gfloat *vec)
{
gfloat x, y, z;
gfloat *mptr=mat, *vptr=vec;

/* init */
x = *(vptr++);
y = *(vptr++);
z = *(vptr--);
/* mult */
*(--vptr) *= *(mptr++);
*vptr += *(mptr++) * y;
*(vptr++) += *(mptr++) * z;
*vptr = *(mptr++) * x;
*vptr += *(mptr++) * y;
*(vptr++) += *(mptr++) * z;
*vptr = *(mptr++) * x;
*vptr += *(mptr++) * y;
*vptr += *(mptr) * z;
}

/************************************************************/
/* transform a 3D vector by matrix mult (transposed matrix) */
/************************************************************/
void vectmat(gfloat *mat, gfloat *vec)
{
gfloat x, y, z;
gfloat *mptr=mat, *vptr=vec;

/* init */
x = *(vptr++);
y = *(vptr++);
z = *(vptr--);
/* mult */
*(--vptr) *= *(mptr++);
*(++vptr) *= *(mptr++);
*(++vptr) *= *(mptr++);

vptr--;

*(--vptr) += *(mptr++) * y;
*(++vptr) += *(mptr++) * y;
*(++vptr) += *(mptr++) * y;

vptr--;

*(--vptr) += *(mptr++) * z;
*(++vptr) += *(mptr++) * z;
*(++vptr) += *(mptr) * z;
}

/**********************/
/* transpose a matrix */
/**********************/
void transpose(gfloat *mat)
{
gfloat tmp[9], *ptr=mat;

memcpy(tmp, mat, 9*sizeof(gfloat));
ptr++; 
*(ptr++) = tmp[3]; 
*(ptr++) = tmp[6]; 
*(ptr++) = tmp[1]; 
ptr++;
*(ptr++) = tmp[7]; 
*(ptr++) = tmp[2]; 
*ptr     = tmp[5]; 
}

/*******************/
/* Invert a matrix */
/*******************/
/* Gauss-Jordan */
#define DEBUG_INVERT 0
gint invmat(gfloat *mat, gint dim)
{
gint i,j,k,ix,ref,refrow,x,y;
gfloat mult, *new;
#if DEBUG_INVERT
gfloat *tmp;

tmp = (gfloat *) g_malloc(dim*dim*sizeof(gfloat));
memcpy(tmp,mat,dim*dim*sizeof(gfloat));
printf("determinant = %f\n", det(mat));
#endif

new = (gfloat *) g_malloc(dim*dim*sizeof(gfloat));

/* setup the unit matrix */
*(new+0) = 1.0;
*(new+1) = 0.0;
*(new+2) = 0.0;
*(new+3) = 0.0;
*(new+4) = 1.0;
*(new+5) = 0.0;
*(new+6) = 0.0;
*(new+7) = 0.0;
*(new+8) = 1.0;

#if DEBUG_INVERT
printf("*** Start ***\n");
printf("| %5.2f %5.2f %5.2f |",*mat,*(mat+1),*(mat+2));
printf("| %5.2f %5.2f %5.2f |\n",*new,*(new+1),*(new+2));
printf("| %5.2f %5.2f %5.2f |",*(mat+3),*(mat+4),*(mat+5));
printf("| %5.2f %5.2f %5.2f |\n",*(new+3),*(new+4),*(new+5));
printf("| %5.2f %5.2f %5.2f |",*(mat+6),*(mat+7),*(mat+8));
printf("| %5.2f %5.2f %5.2f |\n",*(new+6),*(new+7),*(new+8));
#endif

/*******************/
/* pass one - L->0 */
/*******************/

/* loop over the rows (down) */
for(i=1 ; i<dim ; i++)
  {
/* diagonal element to compare against to get the multiplier */
  ref = 0;
  refrow = 0;
/* loop along the row */
  for (j=0 ; j<i ; j++)
    {
    ix = i*dim+j;
/* calc multiplier to make ix 0 */
    if (*(mat+ref))
      mult = (gfloat) (*(mat+ix))/(gfloat) (*(mat+ref));
    else
      mult = 0.0; 
/* change the row */
    for (k=0 ; k<dim ; k++)
      {
      x = i*dim+k;
      y = refrow*dim+k;
      *(mat+x) -= mult*(*(mat+y));
      *(new+x) -= mult*(*(new+y));
      }
    
/* change ref  - slide down the diagonal */
    ref += dim+1;
    refrow++;
    }
  }

#if DEBUG_INVERT
printf("*** Pass 1 ***\n");
printf("| %5.2f %5.2f %5.2f |",*mat,*(mat+1),*(mat+2));
printf("| %5.2f %5.2f %5.2f |\n",*new,*(new+1),*(new+2));
printf("| %5.2f %5.2f %5.2f |",*(mat+3),*(mat+4),*(mat+5));
printf("| %5.2f %5.2f %5.2f |\n",*(new+3),*(new+4),*(new+5));
printf("| %5.2f %5.2f %5.2f |",*(mat+6),*(mat+7),*(mat+8));
printf("| %5.2f %5.2f %5.2f |\n",*(new+6),*(new+7),*(new+8));
#endif

/*******************/
/* pass two - U->0 */
/*******************/

/* loop over the rows (up) */
for(i=dim-2 ; i>=0 ; i--)
  {
/* diagonal element to compare against to get the multiplier */
  ref = dim*dim - 1;
  refrow = dim-1;
/* loop along the row */
  for (j=dim-1 ; j>i ; j--)
    {
    ix = i*dim+j;
/* calc multiplier to make ix 0 */
    if (*(mat+ref))
      mult = (gfloat) (*(mat+ix))/(gfloat) (*(mat+ref));
    else
      mult = 0.0;
/* change the row */
    for (k=0 ; k<dim ; k++)
      {
      x = i*dim+k;
      y = refrow*dim+k;
      *(mat+x) -= mult*(*(mat+y));
      *(new+x) -= mult*(*(new+y));
      }
    
/* change ref  - slide up the diagonal */
    ref -= dim+1;
    refrow--;
    }
  }

#if DEBUG_INVERT
printf("*** Pass 2 ***\n");
printf("| %5.2f %5.2f %5.2f |",*mat,*(mat+1),*(mat+2));
printf("| %5.2f %5.2f %5.2f |\n",*new,*(new+1),*(new+2));
printf("| %5.2f %5.2f %5.2f |",*(mat+3),*(mat+4),*(mat+5));
printf("| %5.2f %5.2f %5.2f |\n",*(new+3),*(new+4),*(new+5));
printf("| %5.2f %5.2f %5.2f |",*(mat+6),*(mat+7),*(mat+8));
printf("| %5.2f %5.2f %5.2f |\n",*(new+6),*(new+7),*(new+8));
#endif

/* set the diagonals to unity */
ref=0;
for(i=0 ; i<dim ; i++)
  {
  if (*(mat+ref))
    mult = 1.0/(*(mat+ref)); 
  else
    mult = 1.0;
/* start of the row */
  ix = i*dim;
  *(mat+ref) *= mult;
  for (j=0 ; j<dim ; j++)
    *(new+ix+j) *= mult;
/* slide down the diagonal */
  ref += dim+1;
  }

#if DEBUG_INVERT
printf("*** Completed ***\n");
printf("| %5.2f %5.2f %5.2f |",*mat,*(mat+1),*(mat+2));
printf("| %5.2f %5.2f %5.2f |\n",*new,*(new+1),*(new+2));
printf("| %5.2f %5.2f %5.2f |",*(mat+3),*(mat+4),*(mat+5));
printf("| %5.2f %5.2f %5.2f |\n",*(new+3),*(new+4),*(new+5));
printf("| %5.2f %5.2f %5.2f |",*(mat+6),*(mat+7),*(mat+8));
printf("| %5.2f %5.2f %5.2f |\n",*(new+6),*(new+7),*(new+8));
#endif

/* TODO - checks??? */
#if DEBUG_INVERT
matmat(new, tmp);
P3MAT("I: ", tmp);
g_free(tmp);
#endif

/* assign the inverse to the input matrix */
memcpy(mat,new,dim*dim*sizeof(gfloat));

g_free(new);
return(0);
}

/* TODO - replace the two below with 2D & 3D macros? */
/****************************/
/* general vector magnitude */
/****************************/
gfloat magnitude(gfloat *vec, gint dim)
{
gint i;
gfloat sum=0.0, *ptr=vec;

for (i=0 ; i<dim ; i++)
  sum += (*ptr) * (*ptr++);

return(sqrt(sum));
}

/********************************/
/* general vector normalization */
/********************************/
gint normalize(gfloat *vec, gint dim)
{
gint i;
gfloat len, *ptr=vec;

len = magnitude(vec, dim);
if (len < FRACTION_TOLERANCE)
  return(1);

for (i=0 ; i<dim ; i++)
  *ptr++ /= len;

return(0);
}

/*****************************/
/* vector intersection angle */
/*****************************/
/* TODO - eliminate the slow acos by returning the cos of the angle */
/*        instead; may not be possible for all calling routines... */
gfloat via(gfloat *vec1, gfloat *vec2, gint dim)
{
gint i;
gfloat lenprod, dot=0.0;
gfloat *ptr1=vec1, *ptr2=vec2;

for (i=0 ; i<dim ; i++)
  dot += (*ptr1++) * (*ptr2++);

lenprod = magnitude(vec1, dim) * magnitude(vec2, dim);

if (lenprod > FRACTION_TOLERANCE)
  return(acos(dot/lenprod));

return(0.0);
}

/*************************/
/* cross product for 3x3 */
/*************************/
void crossprod(gfloat *res, gfloat *vec1, gfloat *vec2)
{
*(res+0) = *(vec1+1) * *(vec2+2) - *(vec1+2) * *(vec2+1);
*(res+1) = *(vec1+2) * *(vec2)   - *(vec1)   * *(vec2+2);
*(res+2) = *(vec1)   * *(vec2+1) - *(vec1+1) * *(vec2);
}

/***********************/
/* determinant for 3x3 */
/***********************/
gfloat det(gfloat *mat)
{
gfloat vec1[3], vec2[3], vec3[3], tmp[3];

/* make vectors from columns */
VEC3SET(vec1, mat[0], mat[3], mat[6]);
VEC3SET(vec2, mat[1], mat[4], mat[7]);
VEC3SET(vec3, mat[2], mat[5], mat[8]);
/* get cross product of 2 & 3 */
crossprod(tmp, vec2, vec3);
/* return dot prod with 1 */
ARR3MUL(vec1, tmp);
return(vec1[0]+vec1[1]+vec1[2]);
}

/********************************************/
/* construct matrix for rotation about axis */
/********************************************/
/* NB: vec is assumed to point from the origin */
/* and is the normal for a mirror, and the axis for a rotation */
#define DEBUG_GET_MATRIX 0
void get_matrix(gfloat *mat, gint type, gdouble *vec, gint order) 
{
gfloat proj, len, angle;
gfloat mop[9];

#if DEBUG_GET_MATRIX
P3VEC("direction vector: ",vec);
#endif

switch(type)
  {
/* proper rotation axis */
  case PAXIS:
#if DEBUG_GET_MATRIX
printf("constructing for order %d axis...\n", order);
#endif
/* constants */
    len = VEC3MAG(vec);
    proj = sqrt(vec[1]*vec[1] + vec[2]*vec[2]);
    angle = 2.0*PI/order;

/* init the return matrix */
    VEC3SET(&mat[0], 1.0, 0.0, 0.0);
    VEC3SET(&mat[3], 0.0, 1.0, 0.0);
    VEC3SET(&mat[6], 0.0, 0.0, 1.0);

/* yz plane coincidence */
    VEC3SET(&mop[0], 1.0, 0.0, 0.0);
    VEC3SET(&mop[3], 0.0, vec[2]/proj, vec[1]/proj);
    VEC3SET(&mop[6], 0.0, -1.0*vec[1]/proj, vec[2]/proj);
    matmat(mop,mat);

/* z axis coincidence */
    VEC3SET(&mop[0], proj/len, 0.0, vec[0]/len);
    VEC3SET(&mop[3], 0.0, 1.0, 0.0);
    VEC3SET(&mop[6], -1.0*vec[0]/len, 0.0, proj/len);
    matmat(mop,mat);

/* rotation operation (about z) */
    VEC3SET(&mop[0], cos(angle), sin(angle), 0.0);
    VEC3SET(&mop[3], -1.0*sin(angle), cos(angle), 0.0);
    VEC3SET(&mop[6], 0.0, 0.0, 1.0);
    matmat(mop,mat);

/* z axis inverse */
    VEC3SET(&mop[0], proj/len, 0.0, -1.0*vec[0]/len);
    VEC3SET(&mop[3], 0.0, 1.0, 0.0);
    VEC3SET(&mop[6], vec[0]/len, 0.0, proj/len);
    matmat(mop,mat);

/* yz plane inverse */
    VEC3SET(&mop[0], 1.0, 0.0, 0.0);
    VEC3SET(&mop[3], 0.0, vec[2]/proj, -1.0*vec[1]/proj);
    VEC3SET(&mop[6], 0.0, vec[1]/proj, vec[2]/proj);
    matmat(mop,mat);
    break;

/* mirror plane */
  case PLANE:
#if DEBUG_GET_MATRIX
printf("NIY - constructing for mirror plane\n");
#endif
    break;

  default:;
  }
}

/******************************************/
/* construct the unit translation vectors */
/******************************************/
#define DEBUG_XLAT 0
#define MY_METHOD 1
void make_xlat(struct model_pak *data)
{
gfloat v1[3], v2[3], v3[3];
#if MY_METHOD
gfloat b1, b2, b3, c1, c2, c3;
#else
gfloat cosa, cosb, cosg, sing, trm1;
#endif

#ifdef TIMER
start_timer("make_xlat");
#endif

/* NEW - use a supplied latmat, rather than generating from pbc's */
/* NB: should be in gdis style colum vector format (gulp/marvin are in rows) */
if (data->construct_pbc)
  {
/* get lattice vector lengths */
  VEC3SET(v1, data->latmat[0], data->latmat[3], data->latmat[6]);
  VEC3SET(v2, data->latmat[1], data->latmat[4], data->latmat[7]);
  VEC3SET(v3, data->latmat[2], data->latmat[5], data->latmat[8]);
/* set lengths */
  data->pbc[0] = VEC3MAG(v1);
  data->pbc[1] = VEC3MAG(v2);
  if (data->periodic == 3)
    data->pbc[2] = VEC3MAG(v3);
  else
    data->pbc[2] = 0.0;
/* get cell angles */
  data->pbc[3] = via(v2,v3,data->periodic);
  data->pbc[4] = via(v1,v3,data->periodic);
  data->pbc[5] = via(v1,v2,data->periodic);
  }
else
  {
#if MY_METHOD
/* This basically works by using the cosine rule in conjunction */
/* with a few geometric constraints (eg normalized vectors) */
/* TODO - validity checks on the PBC */

/* compute the translation vector for b */
b1 = cos(data->pbc[5]);
b2 = sqrt(1.0 - b1*b1);
b3 = 0.0;              /* constrain a,b to remain on the x,y plane */
/* compute the translation vector for c */
c1 = cos(data->pbc[4]);
c2 = (2.0*cos(data->pbc[3]) + b1*b1 + b2*b2 - 2.0*b1*c1 - 1.0)/(2.0*b2);
/* the fabs shouldn't be necessary... */
c3 = sqrt(fabs(1.0 - c1*c1 - c2*c2));

/* assign in rows to make it easier to scale */
/* x & a are co-linear */
VEC3SET(&data->latmat[0], data->pbc[0], 0.0, 0.0);
VEC3SET(&data->latmat[3], b1, b2, b3);
VEC3SET(&data->latmat[6], c1, c2, c3);

/* vectors are in normalized (except for a) and in rows */
VEC3MUL(&data->latmat[3],data->pbc[1]);
/* if 2D have a 1 as the c vector */
if (data->periodic == 2)
  {
  VEC3SET(&data->latmat[6], 0.0, 0.0, 1.0);
  }
else
  {
  VEC3MUL(&data->latmat[6], data->pbc[2]);
  }
/* get vectors in cols */
transpose(data->latmat);

/* imported gulp/marvin latmat generation routine */
#else
cosa = cos(data->pbc[3]);
cosb = cos(data->pbc[4]);
cosg = cos(data->pbc[5]);
sing = sin(data->pbc[5]);
data->latmat[0] = 1.0;
data->latmat[1] = 0.0;
data->latmat[2] = 0.0;
data->latmat[5] = 0.0;
data->latmat[3] = cosg;
data->latmat[4] = sing;
data->latmat[6] = cosb;
data->latmat[7] = (cosa - cosg*cosb)/sing;
trm1 = data->latmat[7];
data->latmat[8] = sqrt(1.0 - cosb*cosb - trm1*trm1);

/* vectors are in normalized and in rows - so scale them up */
VEC3MUL(&data->latmat[0],data->pbc[0]);
VEC3MUL(&data->latmat[3],data->pbc[1]);

/* if 2D have a 1 as the c vector */
if (data->periodic == 2)
  {
  VEC3SET(&data->latmat[6], 0.0, 0.0, 1.0);
  }
else
  {
  VEC3MUL(&data->latmat[6], data->pbc[2]);
  }

/* get vectors in columns */
transpose(&data->latmat[0]);

#endif
  }

/* update dependents */
make_axes(data);
make_cell(data);

#if DEBUG_XLAT
printf("cell dimensions: [%6.2f %6.2f %6.2f] (%6.2f %6.2f %6.2f)\n",
       data->pbc[0], data->pbc[1], data->pbc[2],
       data->pbc[3], data->pbc[4], data->pbc[5]);
P3MAT("lattice matrix:",data->latmat);
#endif
#ifdef TIMER
stop_timer("make_xlat");
#endif
return;
}

/**************************************************************/
/* get reciprocal lattice vectors from direct lattive vectors */
/**************************************************************/
#define DEBUG_RLAT 0
void make_rlat(struct model_pak *data)
{
gfloat norm;
gfloat a[3], b[3], c[3];
gfloat axb[3], bxc[3], cxa[3];

#if DEBUG_RLAT
P3MAT("direct lattice matrix:",data->latmat);
#endif

/* extract direct lattice vectors */
a[0] = data->latmat[0];
a[1] = data->latmat[3];
a[2] = data->latmat[6];
b[0] = data->latmat[1];
b[1] = data->latmat[4];
b[2] = data->latmat[7];
c[0] = data->latmat[2];
c[1] = data->latmat[5];
c[2] = data->latmat[8];

/* calc intermediate products */
crossprod(bxc, b, c);
crossprod(cxa, c, a);
crossprod(axb, a, b);
norm = 1.0/(a[0]*bxc[0] + a[1]*bxc[1] + a[2]*bxc[2]);

/* create reciprocal lattice vectors */
ARR3SET(a, bxc);
ARR3SET(b, cxa);
ARR3SET(c, axb);
VEC3MUL(a, norm);
VEC3MUL(b, norm);
VEC3MUL(c, norm);

/* store */
data->rlatmat[0] = a[0];
data->rlatmat[3] = a[1];
data->rlatmat[6] = a[2];
data->rlatmat[1] = b[0];
data->rlatmat[4] = b[1];
data->rlatmat[7] = b[2];
data->rlatmat[2] = c[0];
data->rlatmat[5] = c[1];
data->rlatmat[8] = c[2];

#if DEBUG_RLAT
P3MAT("reciprocal lattice matrix:",data->rlatmat);
#endif
}

/******************************************/
/* convert cartesian coords to fractional */
/******************************************/
#define DEBUG_LATMAT_FUDGE 0
void latmat_fudge(struct model_pak *data)
{
gint i;
gfloat vec[3], mat[9];

#if DEBUG_LATMAT_FUDGE
P3MAT("latmat: ", data->latmat);
#endif

memcpy(mat, data->latmat, 9*sizeof(gfloat));
invmat(mat,3);

#if DEBUG_LATMAT_FUDGE
P3MAT("invmat: ", mat);
#endif

/* transform cores */
for (i=data->num_atoms ; i-- ; )
  {
  vec[0] = (data->atoms+i)->x; 
  vec[1] = (data->atoms+i)->y; 
  vec[2] = (data->atoms+i)->z; 
  vecmat(mat, vec);
  (data->atoms+i)->x = vec[0];
  (data->atoms+i)->y = vec[1];
  (data->atoms+i)->z = vec[2];
  }

/* transform shells */
for (i=data->num_shells ; i-- ; )
  {
  vec[0] = (data->shells+i)->x; 
  vec[1] = (data->shells+i)->y; 
  vec[2] = (data->shells+i)->z; 
  vecmat(mat, vec);
  (data->shells+i)->x = vec[0];
  (data->shells+i)->y = vec[1];
  (data->shells+i)->z = vec[2];
  }

/* morphology vertices */
for (i=data->num_vertices ; i-- ; )
  {
  vec[0] = (data->vertices+i)->x; 
  vec[1] = (data->vertices+i)->y; 
  vec[2] = (data->vertices+i)->z; 
  vecmat(mat, vec);
  (data->vertices+i)->x = vec[0];
  (data->vertices+i)->y = vec[1];
  (data->vertices+i)->z = vec[2];
  }
}

/****************************/
/* NEW - new lattice vector */
/****************************/
/* v - new orientation (may need projection if model is 2D) */
/* ix - which lattice vector to replace (0-2) */
#define DEBUG_NEW_LATVEC 0
void new_latvec(struct model_pak *data, gfloat *v, gint ix)
{
gint i, j, s1,s2, flag;
/* TODO - I think some precision related errors are occuring - use doubles? */
gfloat m, n, len, len1, len2, proj1, proj2, angle;
gfloat min_len, min_angle;
gfloat a[3], b[3], n1[3], n2[3], tmp1[3], tmp2[3];

/* checks */
g_return_if_fail(data != NULL);
g_return_if_fail(v != NULL);
if (ix < 0 || ix > 2)
  {
  printf("Bad index.\n");
  return;
  }

/* may need projection if model is 2D */
if (data->periodic == 2)
  {
/* check */
  if (ix == 2)
    {
    printf("Bad mapping.\n");
    return;
    }
/* get lattice vectors */
  VEC3SET(a, data->latmat[0], data->latmat[3], data->latmat[6]);
  VEC3SET(b, data->latmat[1], data->latmat[4], data->latmat[7]);
/* get surface normal */
  crossprod(n1, a, b);
/* normalize */
  len = VEC3MAG(n1);
  VEC3MUL(n1, 1.0/len);
/* project input vector onto the normal (ie dist to plane) */
  len = v[0]*n1[0] + v[1]*n1[1] + v[2]*n1[2];
/* get vector to plane */
  VEC3MUL(n1, len);
/* subtract vector to plane from input to get projection on plane */
  ARR3SUB(v, n1);
  }

/* which vector to map */
VEC3SET(n1, data->latmat[ix], data->latmat[ix+3], data->latmat[ix+6]);
len1 = data->pbc[ix];

/* another lattice vector for reference */
VEC3SET(n2, data->latmat[ix+1], data->latmat[ix+4], data->latmat[ix+7]);
len2 = data->pbc[ix+1];

/* get projection of v on a */
ARR3SET(tmp1, n1);
VEC3MUL(tmp1, 1.0/len1);
ARR3MUL(tmp1, v);
proj1 = tmp1[0] + tmp1[1] + tmp1[2];
if (proj1 < 0.0)
  s1 = -1;
else
  s1 = 1;

/* as we're altering a, this can't be 0 */
if (fabs(proj1) < FRACTION_TOLERANCE)
  {
  printf("Your vector won't span the space.\n");
  return;
  }

/* get projection of v on b */
ARR3SET(tmp2, n2);
VEC3MUL(tmp2, 1.0/len2);
ARR3MUL(tmp2, v);
proj2 = tmp2[0] + tmp2[1] + tmp2[2];
if (proj2 < 0.0)
  s2 = -1;
else
  s2 = 1;

/* setup for seek */
min_len = 100.0*len1*len2;
min_angle = 180.0;
m = n = 0.0;
flag = 0;
/* TODO - better/adjustable vector mutiple seeking */
for (j=0 ; j<10 ; j++)
  {
  for (i=1 ; i<10 ; i++)
    {
  ARR3SET(tmp1, n1);
  VEC3MUL(tmp1, (gfloat) i*s1);
  ARR3SET(tmp2, n2);
  VEC3MUL(tmp2, (gfloat) j*s2);
/* get i*n1 + j*n2 */
  ARR3ADD(tmp1, tmp2);

/* how close is it to the required otientation */
  angle = via(tmp1, v, data->periodic);  
/*
printf("angle difference for [%d] a + [%d] b : %f\n",i*s1,j*s2,angle);
*/
/* TODO - adjustable tolerance */
    if (angle < 0.1)
      {
      len = VEC3MAG(tmp1);
      if (len < min_len)
        {
        m = (gfloat) i*s1;
        n = (gfloat) j*s2;
        min_len = len;
        min_angle = angle;
        flag++;
        }
      }
    }
  }

if (!flag)
  {
  printf("Search failed.\n");
  return;
  }

#if DEBUG_NEW_LATVEC
printf("new lattice vector: %d(a) + %d(b), length %f, deviation %f\n",
                              (gint) m, (gint) n, min_len, min_angle);
#endif

/* NEW - update surface bulk energy */
data->gulp.sbulkenergy *= m*n;

/* generate new vector */
VEC3SET(n1, data->latmat[ix], data->latmat[ix+3], data->latmat[ix+6]);
VEC3SET(n2, data->latmat[ix+1], data->latmat[ix+4], data->latmat[ix+7]);

VEC3MUL(n1, m);
VEC3MUL(n2, n);

ARR3SET(v, n1);
ARR3ADD(v, n2);

/* generate enough images to fill the new surface cell */
/* FIXME - +ve dir min is 1, but -ve dir min is 0 */
if (s1 < 0.0)
  {
  data->image_limit[ix] = fabs(m)+1;
  data->image_limit[ix+1] = 1;
  }
else
  {
  data->image_limit[ix] = 0;
  data->image_limit[ix+1] = fabs(m)+1;
  }

if (s2 < 0.0)
  {
  data->image_limit[ix+2] = fabs(n)+1;
  data->image_limit[ix+3] = 1;
  }
else
  {
  data->image_limit[ix+2] = 0;
  data->image_limit[ix+3] = fabs(n)+1;
  }

update_images(data->number, 0, CREATE);

/* TODO - cope with perpendicular orientation */
/* TODO - ie if non 180/0 with the *other* vector */
if (0)
  {
  printf("Your vectors won't span the space/surface.\n");
  return;
  }

/* restore orig cartesian (help avoid precision errors) */
load_cart(data);

/* write new lattice vectors */
data->latmat[ix] = v[0];
data->latmat[ix+3] = v[1];
data->latmat[ix+6] = v[2];

/* then latmat fudge to get coords in new lattice space */
latmat_fudge(data);

/* reset images */
data->image_limit[0] = 0;
data->image_limit[1] = 1;
data->image_limit[2] = 0;
data->image_limit[3] = 1;
data->image_limit[4] = 0;
data->image_limit[5] = 1;
}

/***********************************************************/
/* transform a lattice vector with a specified orientation */
/***********************************************************/
/* orientation is indicated by the first 2 atoms in */
/* the current selection - better way? */
void transform_latvec(gint id)
{
gint i, j;
gfloat vec[3];
struct model_pak *data;

/* get model ptr assoc. with calling dialog */
data = model_ptr(sysenv.dialog[id].model, RECALL);
g_return_if_fail(data != NULL);

if (data->select_size < 2)
  {
  printf("Not enough atoms selected.\n");
  return;
  }

/* get selection atoms & clear */
i = *(data->select_list);
j = *(data->select_list+1);
clear_select();

/* get UNROTATED orientation as a CARTESIAN vector */
vec[0] = (data->atoms+j)->x - (data->atoms+i)->x;
vec[1] = (data->atoms+j)->y - (data->atoms+i)->y;
vec[2] = (data->atoms+j)->z - (data->atoms+i)->z;
vecmat(data->latmat, vec);

/* atom based calc */
pbc_constrain_atoms(data);
calc_bonds(data);
calc_mols(data);

/* TODO - make vector mapped alterable */
new_latvec(data, vec, 0);

/* setup the new pbc lengths */
data->construct_pbc = TRUE;
make_xlat(data);

/* mark existing atoms as belonging to the original cell */
make_p1(data);

/* delete atoms outside the new pbc */
pbc_constrain_atoms(data);

/* reinit & recalc connectivity */
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);

redraw_canvas(SINGLE);
}

/************************/
/* distinguish identity */
/************************/
gint is_identity(gfloat *mat)
{
if (*(mat+0) != 1.0)
  return(FALSE);
if (*(mat+4) != 1.0)
  return(FALSE);
if (*(mat+8) != 1.0)
  return(FALSE);

if (*(mat+1) != 0.0)
  return(FALSE);
if (*(mat+2) != 0.0)
  return(FALSE);
if (*(mat+3) != 0.0)
  return(FALSE);

if (*(mat+5) != 0.0)
  return(FALSE);
if (*(mat+6) != 0.0)
  return(FALSE);
if (*(mat+7) != 0.0)
  return(FALSE);

return(TRUE);
}

/*************************/
/* distinguish inversion */
/*************************/
gint is_inversion(gfloat *mat)
{
if (*(mat+0) != -1.0)
  return(FALSE);
if (*(mat+4) != -1.0)
  return(FALSE);
if (*(mat+8) != -1.0)
  return(FALSE);

if (*(mat+1) != 0.0)
  return(FALSE);
if (*(mat+2) != 0.0)
  return(FALSE);
if (*(mat+3) != 0.0)
  return(FALSE);

if (*(mat+5) != 0.0)
  return(FALSE);
if (*(mat+6) != 0.0)
  return(FALSE);
if (*(mat+7) != 0.0)
  return(FALSE);

return(TRUE);
}

/**************************************/
/* get the order of a symmetry matrix */
/**************************************/
gint order(gfloat *mat)
{
gint i;
gfloat m1[9], m2[9];

ARR3SET(&m1[0], (mat+0));
ARR3SET(&m1[3], (mat+3));
ARR3SET(&m1[6], (mat+6));
ARR3SET(&m2[0], (mat+0));
ARR3SET(&m2[3], (mat+3));
ARR3SET(&m2[6], (mat+6));

/* keep applying until get the identity */
i=1;
while(i<17)
  {
  if (is_identity(m2))
    return(i);
  matmat(m1, m2);
  i++;
  }

/* not a symmetry operator */
return(0);
}

