/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

/*******************/
/* data structures */
/*******************/

/************/
/* task pak */
/************/
struct task_pak
{
/* control */
gint pid;
gint status;
gchar *label;
/* main task and argument (run by the child) */
void (*primary) (gpointer *);
gpointer *ptr1;
/* cleanup task and argument (run by the parent) */
void (*cleanup) (gpointer *);
gpointer *ptr2;
};

/**************/
/* dialog pak */
/**************/
struct dialog_pak
{
gint active; 
gint model;       /* assoc. model */
gint type;        /* identifier */
GtkWidget *win;
GtkWidget *data;  /* TODO - make gpointer later */
};

/**********************************/
/* global unified rendering setup */
/**********************************/
struct render_pak
{
/* main setup */
gint type;
gint width;
gint height;
gint background;	/* TODO - make a colour, not a toggle */
gint camera[3];
gfloat vp_dist;         /* effectively the vanishing point for perspective proj */
gchar filename[FILELEN];
/* flags & control */
gint antialias;
gint shadowless;
gint axes;
gint atype;
gint wire_csurf;
gint wire_morph;
gint animate;
gint delay;
/* light */
gint num_lights;
struct light_pak *light;
gfloat ambience;
gfloat diffuse;
gfloat specular;
/* geometry */
gfloat ball_rad;
gfloat stick_rad;
gfloat line_thickness;
gfloat frame_thickness;
gfloat cpk_scale;
/* material */
gfloat ref_index;
gfloat transmit;
gfloat morph_colour[3];
gchar morph_finish[LINELEN];
};

/***********************/
/* top level structure */
/***********************/
struct sysenv_pak
{
/* current working directory */
gchar cwd[FILELEN];
/* filetype filters */
gint file_type;
gint babel_type;
/* database allocation size */
gint num_elements;
/* write at exit flag */
gint write_gdisrc;
/* home of gdisrc */
gchar gdisrc[FILELEN];
/* system font */
gchar font[LINELEN];
GdkFont *dfont;      /* drawing font */
gint dfontsize;
GtkWidget *rotframe;
gint rotframe_on;

/* external programs */
gint have_gulp;
gint have_povray;
gint have_convert;
gint have_viewer;
gint have_babel;
gchar *gulp_exe;
gchar *gulp_path;
gchar *povray_exe;
gchar *povray_path;
gchar *convert_exe;
gchar *convert_path;
gchar *viewer_exe;
gchar *viewer_path;
gchar *babel_exe;
gchar *babel_path;

/* number of loaded models */
gint num_models;
/* NEW - models allocation list (easier alloc ) */
GSList *mal;
/* NEW - number of added (model) subtree's */
gint num_trees;

/* a list of which models are displayed */
gint num_displayed;
gint displayed[MAX_DISPLAYED];
/* selection mode */
gint select_mode;
/* indicates which of the displayed models is 'active' */
gint active;
/* display flags for the drawing area */
gint show_names;
gint show_energy;

/* NEW - drawing */
GtkWidget *glarea;
GdkVisual *visual;
GdkColormap *colourmap;

/* screen dimensions */
gint x;
gint y;
gint width;
gint height;
/* size of colourmap */
gint depth;
/* number of subwindows */
gint subx;
gint suby;
/* pixel dimensions */
gint subwidth;
gint subheight;
/* smallest of the above two */
gint subscale;
/* fraction dimensions */
gfloat fsubwidth;
gfloat fsubheight;
/* list of subwindow centres */
gint subcenx[MAX_DISPLAYED];
gint subceny[MAX_DISPLAYED];
/* dialog list */
struct dialog_pak dialog[MAX_DIALOGS];
/* copy active select */
struct model_pak *select_source;
/* NEW - tasks handling */
gint num_tasks;
gint running_tasks;
gint max_running_tasks;
struct task_pak *task;
GtkWidget *task_label;
/* NEW - rendering */
struct render_pak render;
};

/********************************/
/* element properties structure */
/********************************/
/* TODO - atomic num, charge, ...? */
struct elem_pak
{
/* NB: big enough for string plus \0 */
gchar symbol[4];
gchar name[16];
gint number;         /* NB: keep! elem_pak is used for more than elem table */
gfloat weight;
gfloat cova;
gfloat vdw;
gint colour[3];
};

/*******************************/
/* model sub-structure (atoms) */
/*******************************/
struct atom_pak
{
/* main info */
gint atom_code;
gint status;
gint primary;          /* one of the initial atoms? */
gint sitecount;        /* repetitions in cell build */
gint orig;             /* original atom? (ie part of whole unit cell) */
/* bond info */
gint atom_bonds;
gint bond[MAX_BONDS];
gint molecule;         /* the molecule it's associated with */
/* associated shell (if any) */
gint has_shell;
gint idx_shell;
/* region type */
gint region;
/* coordinate data */
/* always the (calculated if fractional) initial cartesian coords loaded */
gfloat cx;
gfloat cy;
gfloat cz;
/* fractional if periodic, otherwise cartesian */
gfloat x;
gfloat y;
gfloat z;
/* rotated cartesian */
gfloat rx;
gfloat ry;
gfloat rz;
/* site occupancy factor */
gfloat sof;
/* display data */
gint colour[3];
gint px;
gint py;
gint offx;     /* offset (atom only, NOT global) */
gint offy;
/* file saving info */
gchar element[ELEM_LABEL_SIZE];
gchar label[LABEL_SIZE];
gchar tail[TAIL_SIZE];
};

/********************************/
/* model sub-structure (shells) */
/********************************/
struct shell_pak
{
gint atom_code;    /* atom type of the shell */
gint status;       /* normal, deleted, highlighted etc. */
gint primary;          /* NEW - one of the initial atoms? */
gint sitecount;        /* NEW - used to eliminate repetitions in cell build */
gint orig;             /* NEW - original atom? (ie part of whole unit cell) */
/* associated core (if any) */
gint has_core;
gint idx_core;
/* region type (marvin only) */
gint region;
/* coord data (see atom_pak) */
gfloat cx;
gfloat cy;
gfloat cz;
gfloat x;
gfloat y;
gfloat z;
gfloat rx;
gfloat ry;
gfloat rz;
/* display data */
gint colour[3];
gint px;
gint py;
gint offx;     /* offset (atom only, NOT global) */
gint offy;
/* file saving info */
gchar label[LABEL_SIZE];
gchar element[ELEM_LABEL_SIZE];
gchar tail[TAIL_SIZE];
};

/**********************/
/* general pixel pack */
/**********************/
struct pixel_pak
{
gint px;
gint py;
};

/*******************************/
/* model sub-structure (bonds) */
/*******************************/
struct bond_pak
{
/* single/double etc. */
gint type;
/* constituent atom indices */
gint atom1;
gint atom2;
/* pixel position of bond centre */
gint px;
gint py;
};

/***********************************/
/* model sub-structure (molecules) */
/***********************************/
struct mol_pak
{
gint num_atoms;
gint *atom;
gfloat centroid[3];
};

/**********************/
/* general coord pack */
/**********************/
struct coord_pak
{
/* orig coordinates */
gfloat x;
gfloat y;
gfloat z;
/* rotated coords */
gfloat rx;
gfloat ry;
gfloat rz;
/* pixel position */
gint px;
gint py;
};

/**********************/
/* general vector pak */
/**********************/
/* TODO - this will (eventually) replace the above */
struct vec_pak
{
/* orig coords in the lattice space */
gfloat x[3];
/* rotated coordinates in cartesian space */
gfloat rx[3];
/* projected pixel position */
gfloat px[2];
};

/***************************/
/* geometric notation pack */
/***************************/
struct geom_pak
{
gint type;       /* atom, bond, angle etc. */
gint *list;      /* atom index data */
gchar *text;    /* text */
};

/*****************************************************/
/* (will become) the general mol/contact surface pak */
/*****************************************************/
struct mysurf_pak
{
/* this may become dots per angs^2 */
gint shell_size;
gfloat prad;
gint num_points;
gint max_points;
gint *touch;
gint *px, *py;
gfloat *x, *y, *z; 
};

/********************/
/* vdW surface pack */
/********************/
struct molsurf_pak
{
gint grid;           /* grid dimensions */
gint rsf;            /* re-entrant surface flag (draw) */
gint ix[3];          /* variable index (indep, indep, dep) */
gfloat prad;         /* probe radius */
gfloat offset;       /* distance above to plot the surface */
gfloat min[3];       /* real grid min */
gfloat step[3];      /* real grid step */
/* connolly data */
gint *touch;         /* atom number at grid point */
gint *code;          /* atom code at grid point */
gfloat *h;           /* real surface at grid point */
gfloat *n[3];        /* normal vector */
gfloat *pa,*pb,*pc;  /* probe center */
gint *px,*py;        /* pixel position */
gfloat *rx,*ry,*rz;  /* orientated coords (for povray) */
};

/***********************************/
/* solvent accessible surface pack */
/***********************************/
struct solsurf_pak
{
gint grid;
gfloat prad;
gint num_points;
gfloat *px,*py,*pz;       /* valid points (probe center position) */
gint *num_touch;
gint **touch;             /* atoms touched at the valid points */
};

/***************************/
/* sphere tesselation pack */
/***************************/
struct sphere_pak
{
gint num_points;
gfloat *x;
gfloat *y;
gfloat *z;
};

/********************/
/* morphology packs */
/********************/
struct vertex_pak
{
/* orig coords */
gfloat x;
gfloat y;
gfloat z;
/* rotated coords */
gfloat rx;
gfloat ry;
gfloat rz;
/* screen coords */
gint px;
gint py;
/* neighbour vertices */
gint num_adj;
gint *adj;
};

struct plane_pak
{
/* INPUT (orientations) */
/* raw direction (should be miller, but use float just in case) */
gfloat m[3];
/* normalized direction */
gfloat x[3];
/* INPUT (distance data) */
gfloat len;
gfloat dhkl;
gfloat Eatt;
gfloat Esurf;

/* FIXME - this is merged with the old facet_pak and */
/* thus now contains some redundant data to be removed */

/* OUTPUT (hull) */
/* NEW - is this plane a visible facet? */
gint present;
gint num_points;
gint lim_points;
gint *points;
gint visible;
/* hkl label position */
gfloat rx, ry, rz;
gint px, py;
/* hkl label */
gint index[3];
/* unrotated normal */
gfloat norm[3];
};

/********************/
/* space group info */
/********************/
struct space_pak
{
gint spacenum;
gint lattice;
gint pointgroup;
gint cellchoice;
gchar *spacename;
gchar *latticename;
gint order;
/* signs & ordering of x,y,z general positions */
gfloat **matrix;
/* const offset to general positions */
gfloat **offset;

gpointer *raw;

};

/*****************/
/* symmetry info */
/*****************/
struct symop_pak
{
gint type;
gint order;
gfloat point[3];
gfloat norm[3];
gfloat matrix[9];
};

struct symmetry_pak
{
gint num_symops;
struct symop_pak *symops;
gint num_items;
gchar **items;
/* redundant? - pass around via attached data to callbacks? */
GtkWidget *summary;
GtkWidget *pg_entry;
gchar *pg_name;
};

/*************/
/* gulp data */
/*************/
struct gulp_pak
{
gint run;
/* conp/conv */
gint method;
gint optimiser;
gint free;
gint zsisa;
gint compare;
gint cycles;
gint ensemble;
gint coulomb;
gint maxcyc;
gint super[3];
gfloat energy;
gint no_esurf;   /* old gulp's can't handle sbulkenergy keyword */
gfloat esurf;
gchar *esurf_units;
gint no_eatt;    /* old gulp's can't handle dhkl keyword */
gfloat eatt;
gchar *eatt_units;
gfloat sbulkenergy;
gfloat sdipole;
gfloat gnorm;
gfloat temp;
/* widgets that need to be updated */
GtkWidget *energy_entry;
GtkWidget *esurf_entry;
GtkWidget *eatt_entry;
GtkWidget *sbe_entry;

gint num_potentials;
gint num_elem;
gint num_species;
gchar **pot;
gchar **elem;
gchar **species;
gchar *minimiser;
gchar *libfile;
gchar *srcfile;
gchar *extra;
/* filename control */
gchar *temp_file;
gchar *out_file;
gchar *dump_file;
gchar *lib_file;
gchar *trj_file;
};

/**********************/
/* surface generation */
/**********************/
struct surface_pak
{
gint model;
gint miller[3];
gfloat shift;
gfloat dspacing;
gint region[2];
};

/************************/
/* model data structure */
/************************/
struct model_pak
{
/* main info */
gint id;
gint number;
gint tree_number;
gint mode;

gint fractional;
gint periodic;
gint num_frames;
gint cur_frame;
gint has_sof;          /* gulp partial occ. */
gint sof_colourize;
gint construct_pbc;
gfloat rmax;           /* auto */
gfloat scale;          /* user */
GtkWidget *tree_label;
GtkWidget *tree_name;
/* ghosts */
gint num_ghosts;
gint *ghosts;
/* coord data */
gint num_atoms;        /* total (incl. periodic images etc.) */
gint num_orig;         /* whole cell */
gint num_asym;         /* asym cell */
gint num_ubonds;
gint num_mols;
gint num_shells;
gint num_geom;
/* NEW - bonds stored as linked list (makes it easier to add/delete) */
gint num_bonds;
GSList *bonds;
/* memory control parameters */
gint atom_limit;
gint ubond_limit;
gint mol_limit;
gint shell_limit;
gint geom_limit;
/* data pointers */
struct atom_pak *atoms;
struct bond_pak *ubonds;  /* for user modifications */
struct mol_pak *mols;
struct shell_pak *shells;
struct geom_pak *geom;
/* model geometric and display data */
gint offset[3];
gfloat angle[3];
gfloat da;
gfloat pbc[6];
gfloat centroid[3];
/* translation matrix/vectors */
gfloat latmat[9];
/* reciprocal lattice vectors */
gfloat rlatmat[9];
/* cummulative rotation matrix */
gfloat rotmat[9];
/* periodic image vectors */
gfloat piv_real[9];
gint piv_pix[6];
/* special object flags */
gint atom_labels, show_shells, axes_type;
gint axes_on,cell_on,mesh_on,box_on,csurf_on,asym_on,solsurf_on,molsurf_on;
/* special object data */
struct vec_pak xlat[6];
struct vec_pak axes[6];
struct vec_pak cell[8];
struct pixel_pak mesh;  /* px & py are the number of mesh pts */
struct pixel_pak box_select[2];
/* surface info */
struct molsurf_pak csurf;
struct mysurf_pak mysurf[2];
/* morphology data */
gint type;                      /* BFDH, etc. */
gint hpr;                       /* flag - hidden plane removal */
gint morph_label;               /* flag - hkl labelling */
gint num_vertices;
gint num_planes;                /* total number of faces */
gint num_facets;                /* number of present facets in plane list */
struct vertex_pak *vertices;
/* NEW - planes stored as linked list (makes it easier to add/delete) */
GSList *planes;
/* NEW */
GtkWidget *morph_sheet;

/* space group */
struct space_pak sginfo;
/* symmetry info */
struct symmetry_pak symmetry;
/* periodic image numbers - start/stop multiples */
gint num_images;
gint image_limit[6];
/* selection & highlight atom info */
gint select_limit;
gint select_size;
gint *select_list;
gint atom_info;
gint shell_info;
/* region display (marvin only) */
gint region[4];
gint region_empty[4];
/* file info */
gchar filename[FILELEN];
gchar gulplib[FILELEN];
gchar *basename;
/* gulp config */
struct gulp_pak gulp;
/* surface config */
struct surface_pak surface;
/* element data */
gint num_elem;
struct elem_pak *elements;
/* number of different elements */
gint num_unique;
gint *unique;
/* animation sequences */
gint animation;
gint delay;
FILE *afp;
};

/**********************************/
/* general file parsing structure */
/**********************************/
struct parse_pak
{
gint id;
gint hdr_skip;
gchar special_start[20];
gint special_start_len;
gchar coord_start1[20];
gint coord_start1_len;
gchar coord_start2[20];
gint coord_start2_len;
gint a_pos;
gint l_pos;
gint x_pos;
gint y_pos;
gint z_pos;
gint tail_pos;
gchar coord_end1[20];
gint coord_end1_len;
gchar coord_end2[20];
gint coord_end2_len;
};

/******************/
/* MDI structures */
/******************/
/* TODO - combine box_pak & cand_pak? */
struct box_pak
{
gint component;
gint x;
gint y;
gint z;
};

struct mdi_pak
{
gint box_dim;
gfloat latt_sep;
gint num_comp;
gint *comp_idx;
gint *comp_req;
gint *comp_done;
gint *array;
};

struct cand_pak
{
gint pos;
gint x;
gint y;
gint z;
};

/*********************/
/* POVray structures */
/*********************/
struct light_pak
{
gint colour;
gint x;
gint y;
gint z;
};

struct povray_pak
{
gint render_type;
gint width;
gint height;
gint background;   /* colour */
gint camera[3];
gint num_lights;
struct light_pak *light;
gint antialias;
gint shadowless;
gint animate;
gint atype;
gint axes;
gint delay;
gfloat ambience;
gfloat arad;       /* atom radius */
gfloat brad;       /* bond stick radius */
gfloat frad;       /* frame radius */
gfloat cpk_scale;  /* % scaling */
gchar filename[FILELEN];
gint wire_frame;
gfloat ref_index;
gfloat transmit;
gdouble morph_colour[3];
gchar morph_finish[LINELEN];
};

/******************/
/* cropping packs */
/******************/
struct crop_pak
{
gint x1;
gint y1;
gint x2;
gint y2;
gint win;
};

struct text_pak
{
gint x;
gint y;
gint win;
gchar *text;
};

struct arc_pak
{
gint win;
gint x;
gint y;
gint width;
gint height;
gint angle1;
gint angle2;
gint fill;
};

/*********************************/
/* widget convenience structures */
/*********************************/
struct callback_pak
{
gchar *label;
gint id;
};

/*************/
/* functions */
/*************/
void connect_events(void);

struct model_pak *model_ptr(gint, gint);

void get_eledata(void);
void put_elem_data(struct elem_pak *, struct model_pak *);
gint get_elem_data(gint, struct elem_pak *, struct model_pak *);
gint construct_unique(struct model_pak *);

void init_objs(gint, struct model_pak *);
void init_coords(struct model_pak *);

void calc_coords(gint, struct model_pak *);
void calc_bonds(struct model_pak *);
void update_bonds(struct model_pak *);
void calc_mols(struct model_pak *);

void make_axes(struct model_pak *);
void make_cell(struct model_pak *);
void make_mesh(struct model_pak *);
void make_xlat(struct model_pak *);
void make_rlat(struct model_pak *);

gint cent_coords(struct model_pak *);

gint mod_screen(gint, gint);
void redraw_canvas(gint);
void draw_line(gint, gint, gint, gint);
void rot(gfloat *, gfloat , gfloat , gfloat);

/* file related */
void swap_bytes(void *, const gint);

gint file_action(gchar *, gint, gint);
gint load_model(gint, gint, gchar *);
gint load_data(FILE *, struct parse_pak *, struct model_pak *);
gint create_file(gchar *, gint, struct model_pak *);

/* save to file */
gint write_gulp(gchar *, struct model_pak *);
gint write_gmf(gchar *, struct model_pak *);
gint write_marvin(char *, struct model_pak *);
gint write_xtl(char *, struct model_pak *);

/* load ie read and create new model */
gint load_cif(FILE *, gint , gchar *);
gint load_gulp(FILE *, gint, gchar *);
gint load_gulp_output(FILE *, gint, gchar *);
gint load_gulp_trj(FILE *, struct model_pak *, gint);
gint load_gmf(FILE *, gint, gchar *);
gint load_mvnout(FILE *, gint, gchar *);

/* just read in data */
gint read_gulp_output(FILE *, struct model_pak *);


void view_panel(void);
void toggle_cell(struct model_pak *);

void info_atom(gint, gint, struct model_pak *);
void info_bond(gint, gint, gint, struct model_pak *);
void info_dist(gint, gint, struct model_pak *);
void info_angle(gint, gint, struct model_pak *);
void info_torsion(gint, gint, struct model_pak *);

void render_setup_widget(void);
void mdi_setup_widget(void);
void mdi_model_create(gint);
void creation_widget(void);

void about(void);
void write_dat(struct model_pak *);
gint exec_gulp(gchar *, gchar *);

void exec_gulp_task(gpointer *);
void proc_gulp_task(gpointer *);

gint read_single_gulp(gchar *, struct model_pak *);
gint copy_gulp_data(struct model_pak *, struct model_pak *);
gint copy_gulp_extra(struct model_pak *, struct model_pak *);
gint free_gulp_data(struct model_pak *);
gint fgetline(FILE *, gchar *);
gint check_file(const gchar *, gint);

gfloat ran2(void);
gint fill(void);
gint make_pov(gint, gchar *);

void exec_pov_task(gpointer *);
void exec_img_task(gpointer *);

void run_pov(void);
void exec_pov(gchar *);
void switch_mode(gint);
void show_text(gchar *);
void colour_change(gint);
void display_properties(void);
void pbc_constrain_atoms(struct model_pak *);
void pbc_constrain_mols(struct model_pak *);
void update_box(gint, gint, struct model_pak *, gint);
void select_box(void);
void clear_select(void);
gint add_select(struct model_pak *, gint);
gint del_select(struct model_pak *, gint);
void new_colour(GtkWidget *, GtkColorSelection *);
void animate_setup_widget(void);
void animate(gint, gint);
gint get_frame(struct model_pak *data, gint);

void create_new_model(void);
void modify_model(void);
void delete_model(gint);
void delete_active(void);

void add_atom(gint, gint);
void move_atom(gint, gint, gint);
void delete_atom(gint, struct model_pak *);
void delete_atom_at(gint, gint, struct model_pak *);
void delete_mol(gint, gint, struct model_pak *);
void move_mol(gint, gint, gint);
void change_atom_at(gint, gint, struct model_pak *);

gint parse_ptr(struct parse_pak *, gint);

gint mem_check(struct model_pak *, gint);
gint mem_grow(struct model_pak *, gint , gint);
void mem_shrink(struct model_pak *);

gint invmat(gfloat *, gint);
void vecmat(gfloat *, gfloat *);
void vectmat(gfloat *, gfloat *);
void matmat(gfloat *, gfloat *);
void transpose(gfloat *);
void get_matrix(gfloat *, gint, gdouble *, gint);
void crossprod(gfloat *, gfloat *, gfloat *);
gint normalize(gfloat *, gint);
gfloat magnitude(gfloat *, gint);
gfloat via(gfloat *, gfloat *, gint);
gfloat det(gfloat *);
gint is_inversion(gfloat *);
gint is_identity(gfloat *);
gint order(gfloat *);

gint copy_items(gchar *, gchar *, gint, gint);
gint put_item(gchar *, gchar *, gint);
void capitals(gchar *, gchar *);
gchar **get_tokens(gchar *, gint);
gchar *get_token_pos(gchar *, gint);

void update_shells(struct model_pak *);
void template_model(struct model_pak *);

gint elem_seek(gint, gint, struct model_pak *);
gint seek_atom(gint, gint, struct model_pak *);
gint seek_shell(gint, gint, struct model_pak *);
gint elem_type(const gchar *);
gint copy_elem_data(struct model_pak *, struct model_pak *);

void pixel_coord_map(gint, gint, struct model_pak *, gint);

void gulp_widget(GtkWidget *, struct model_pak *);

void reset_view(GtkWidget *, gint);
void refresh_view(void);
void pick_model(gint);
void event_pick_model(GtkWidget *, struct model_pak *);

void new_tree_item(gint, gint);
void del_tree_item(gint);

void update_file_pane(void);
void load_widget(void);
void save_as_widget(void);
void file_save(void);

void colour_dialog(GtkWidget *, gint);
void save_apd(void);

gint geom_label(gint, struct model_pak *, gint *, gchar *);
void geom_info(void);
void update_geom_info(void);
void update_creator(void);
gfloat calc_dist(struct model_pak *, gint, gint);
gfloat calc_angle(struct model_pak *, gint, gint, gint);
gfloat calc_dihedral(struct model_pak *, gint, gint, gint, gint);

void set_rgb_colour(GdkGC *, gint *);

GSList *seek_bond(gint, gint, gint, struct model_pak *);

void user_bond(gint, gint, gint, struct model_pak *, gint);
void bond_overlay(struct model_pak *);

void pick_mol(gint, gint, struct model_pak *);

void surf_dialog(void);
void grid2cart(gfloat *, gfloat *);
void surf2_dialog(void);

void morph_widget(GtkWidget *, struct model_pak *);
gint morph_create(gchar *);
void qh_call(struct model_pak *);
gint ridge_visible(struct model_pak *, gint, gint);
gint facet_visible(struct model_pak *, struct plane_pak *);
gint facet_equiv(struct model_pak *, gint *, gint *);

void gperiodic_win(void);

void display_shuffle(void);
gint genpos(struct model_pak *);

gint exec_cdd(struct model_pak *);
void space_info(GtkWidget *w, struct model_pak *);
void sym_info(GtkWidget *, struct model_pak *);
void seek_sym(struct model_pak *);
void update_symlist(struct model_pak *);

gint request_dialog(gint, gint);
void close_dialog(gint);
void close_dialog_type(gint);
void event_close_dialog(GtkWidget *, gint);

gint write_gdisrc(void);
gint read_gdisrc(void);

gint model_type(const gchar *);

gint copy_core(struct model_pak *, struct model_pak *, gint);

gchar *strdup_no_extension(const gchar *);
gchar *strdup_no_path(const gchar *);
gchar *strdup_to_uppers(const gchar *);

void set_path(gchar *);

void surface_dialog(void);

gint generate_surface(struct model_pak *, struct model_pak *);

void update_images(gint , gint , gint);
void unfragment(struct model_pak *);
void complete_mols(gint);


void colsel_dialog(GtkWidget *, gpointer *);

gint event_render_modify(GtkWidget *, gpointer *); 

void check_geom(struct model_pak *);

gfloat str_to_float(gchar *);

void elem_colour_change(GtkWidget *, gpointer *);

void space_action(gint, gint);

void sof_colour(struct model_pak *);

void make_p1(struct model_pak *);

void tree_select(gint);

void print_atom(struct model_pak *, gint);
void print_shell(struct model_pak *, gint);

void latmat_fudge(struct model_pak *);

void init_rotmat(gfloat *);

/* marvin's timer */
void start_timer(char *);
void stop_timer(char *);
void print_times(void);

gint *get_keyword(gchar *, gint);
gint num_keys(void);

/* GTK events */
gint motion_notify_event(GtkWidget *, GdkEventMotion *);
gint button_press_event(GtkWidget *, GdkEventButton *);
gint button_release_event(GtkWidget *, GdkEventButton *);

/* open gl stuff */
void opengl_options(void);
void opengl_dialog(void);
void opengl_draw(GtkWidget *);

void gdk_draw(gint);

void overlay_models_dialog();

void delay(gint);

gint start_task(struct task_pak *);
gint update_tasks();
gint end_task(gint, gint);
gint kill_task();
void task_exit_handler(int);
void task_dialog(void);
void update_task_info();

gint exec(gchar *);
gint my_system(gchar *);

void transform_latvec(gint);
void new_latvec(struct model_pak *, gfloat *, gint);

void save_cart(struct model_pak *);
void load_cart(struct model_pak *);

void wipe_bond_list(struct model_pak *);

/* rdf stuff */
void create_rdfwindow(void);

void expose_canvas(void);

gint gl_expose_event(GtkWidget *, GdkEventExpose *);
gint gl_configure_event(GtkWidget *, GdkEventConfigure *);
gint gl_seek_atom(gint, gint, struct model_pak *);

gfloat tbl_sin(gfloat);
gfloat tbl_cos(gfloat);
void init_trig(void);

void exec_rdf_task(gpointer *);
void exec_rdf_cleanup(gpointer *);

