#!/usr/bin/perl

#  gdkxft - Provide Xft font support in gdk
#  Copyright (C) 2001 Josh Parsons
#  
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Library General Public
#  License as published by the Free Software Foundation; either
#  version 2 of the License, or (at your option) any later version.
#  
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
#  Library General Public License for more details.
#  
#  You should have received a copy of the GNU Library General Public
#  License along with this library; if not, write to the
#  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
#  Boston, MA 02111-1307, USA.
  
# need to output:
#   c source for real_gdk_xxx() funcs
#   header prototypes for real_gdk_xxx() funcs
#   definitions of MY_GDK_INIT_CHECK etc.

sub readgtkheader 
{
    my $hfile=shift || die "readgtkheader called without filename";
    my $modprefix=shift || ($module ."_");

    my $lastline="";

    open(HFILE,$hfile) || die "couldn't open $hfile";

    while(<HFILE>) {
	my $type;
	my $func;
	my @args;
	chomp;
	
	# three possibilities:
	# 1) line is of the form Type gdk_func (Type arg
	# 2) line is of the form gdk_func (Type arg
	# 3) line is of the form Type

	# 1) parse out type and funcname
	if(s/^((?:const\s+)?\w+\s*[*]*)\s+($modprefix\w+)\s*[\(]//) {
	    $type=$1;
	    $func=$2;
	}
	# 2) type must have been on previous line
	elsif(s/^($modprefix\w+)\s*[\(]//) {
	    $type=$lastline;
	    $func=$1;
	}
	# 3) might be a type. salt it away
	elsif(/^((?:const\s+)?\w+\s*[*]*)/) {
	    $lastline=$1;
	}

	# stop here if we have not found a function decl
	next unless ( $func && $type );

	# stash the info
	$functypes{$func} = $type;
	#print "[$type] [$func]\n";

	# parse out parameters
	my $comma;
	do {
	    my $argtype="";
	    my $arg="";
	    my $argsuf="";
	    $comma="";

	    # match Type arg[comma or close bracket]
	    if(s/^\s*((?:const\s+)?\w+\s*[*]*)\s+(\w+)\s*([\[\]]*)\s*([,\)])//) 
	    {
		$argtype=$1;
		$arg=$2;
		$argsuf=$3;
		   $comma=$4;
	    }
	    elsif(s/^\s*((?:const\s+)?\w+\s+[*]*)\s*(\w+)\s*([\[\]]*)\s*([,\)])//) 
	    {
		$argtype=$1;
		$arg=$2;
		$argsuf=$3;
		$comma=$4;
	    }
	    # match void
	    elsif(s/^\s*void\s*([,\)])//) {
		$comma=$1;
	    }
	    # blank line
	    elsif(/^\s*$/) {
	    }
	    # syntax error?
	    else {
		die "confusing argument syntax in $func near '$_'";
	    };
	    
	    # stash the info
	    if ($argtype && $arg) {
		push @args,{type => $argtype, 
			    name => $arg,
			    suffix => $argsuf};
		#print "  [$argtype] [$arg] [$argsuf] [$comma]\n";
	    };
	    
	    # move on if nothing left
	    if(/^\s*$/ and !($comma eq ")")) {
		$_=<HFILE> ||die;
		chomp;
	    };

	} until($comma eq ")");

	# stash the args
	$funcargs{$func}=\@args;
    }

    close(HFILE);
}

sub readmysource {
    my $file = shift || die "readmysource needs a file name";
    my $modprefix=shift || ($module ."_");

    $modprefix =~ tr/[a-z]/[A-Z]/;

    open(IN,$file) || die "couldn't open $file";
    while(<IN>) {
	if(/^MY_($modprefix\w+)/) {
	    my $func = $1;
	    $func =~ tr/[A-Z]/[a-z]/;
	    push @myfuncs,$func;
	};
    };
    close(IN);
}

sub print_args {
    my $func = shift || die;
    my $notype = shift;
    my $first = 1;
    foreach my $arghashref (@{$funcargs{$func}}) {
	my $type = $arghashref->{type};
	my $arg = $arghashref->{name};
	print "," unless $first;
	$first=0;
	print "$type " unless $notype;
	print "$arg";
    };
}

sub write_realfuncs_h {
    print "\n";
    foreach my $func (@myfuncs) {
	my $type = $functypes{$func};
	print "$type\nreal_$func(\n";
	print_args $func;
	print ");\n";
    }
}

sub write_realfuncfp_h {
    print "\n";
    foreach my $func (@myfuncs) {
	my $type = $functypes{$func};
	print "typedef $type\n(*real_${func}_t)(\n";
	print_args $func;
	print ");\n";
    }
}

sub write_realfuncmy_h {
    print "\n";
    foreach my $func (@myfuncs) {
	my $type = $functypes{$func};
	my $myname = "my_$func";

	$myname =~ tr/[a-z]/[A-Z]/;

	print "#define $myname $type ${func}(";
	print_args $func;
	print ")\n";
    }
}

sub write_realfuncreal_h {
    print "\n";
    foreach my $func (@myfuncs) {
	my $type = $functypes{$func};
	my $myname = "real_$func";

	$myname =~ tr/[a-z]/[A-Z]/;

	print "#define $myname ";
	#print "return " unless($type eq "void");
	print "real_${func}(";
	print_args($func,1);
	print ")\n";
    }
}

sub write_realfuncs_c {
    print "\n";

    print "#include <$modheader>\n";

    print <<HEADER;
    #include <unistd.h>
    #include <dlfcn.h>
    #include <glib.h>

    #include "realfuncs.h"

HEADER

    print "#define LIBNAME $modlibrary\n";
    print "\n";

    print <<LOOKUP;
    static void *lookup(const char *name) {
	void *fp;
	void *lib=dlopen(LIBNAME,RTLD_LAZY|RTLD_GLOBAL);
	if(lib==NULL) g_error("%s: could not dlopen %s (%s)",
			      PACKAGE,LIBNAME,
			      dlerror());
	fp=dlsym(lib,name);
	if(fp==NULL) g_error("%s: could not get symbol %s (%s)",
			     PACKAGE,name,
			     dlerror());
	dlclose(lib);
	/*g_message("%s: lookup(%s)",PACKAGE,name);*/
	return fp;
    }

LOOKUP

    foreach my $func (@myfuncs) {
	my $type = $functypes{$func};
	print "$type\nreal_$func(\n";
	print_args $func;
	print ")\n";
	
	print "{\n";
	print "static real_${func}_t fp=NULL;";
	print "if(fp==NULL) fp=lookup(\"$func\");\n";
	print "return " unless $type eq "void";
	print "fp(";

	my $first = 1;
	foreach my $arghashref (@{$funcargs{$func}}) {
	    my $type = $arghashref->{type};
	    my $arg = $arghashref->{name};
	    print ",\n" unless $first;
	    $first=0;
	    print "$arg";
	};

	print ");\n";
	print "}\n";
    }
}

$source= shift || die "usage: generate.pl <c source>";
$module="gdk";
$modlibrary="REAL_GDK_SO";
$modheader="gdk/gdk.h";

$indent = $ENV{INDENT};
$indent = "indent" unless $indent;

readgtkheader "echo '#include <$modheader>'| cpp -P `gtk-config --cflags`|";
readmysource $source;

open(OUT, "|$indent >realfuncs.h") ||die;
select(OUT);
print "/* automatically generated by generate.pl - do not edit */\n";
write_realfuncs_h;
write_realfuncfp_h;
write_realfuncmy_h;
write_realfuncreal_h;
close(OUT);

open(OUT, "|$indent >realfuncs.c") ||die;
select(OUT);
print "/* automatically generated by generate.pl - do not edit */\n";
write_realfuncs_c;
close(OUT);
