/*
** 2001-09-30 -	A rename command that creates numbered sequences. I want it when arranging
**		images from a digital camera, but it might be handy for other stuff too, I'm
**		sure.
**
**		This is slightly more complicated than you might think at first, since
**		it actually does _two_ renames: first all affected files are renamed
**		to sequential temporary names, and then in a second pass the final names
**		are set. This allows the final sequence to overlap the initial, which I
**		predict will be useful.
*/

#include "gentoo.h"

#include <ctype.h>

#include "dialog.h"
#include "dirpane.h"
#include "errors.h"
#include "fileutil.h"
#include "overwrite.h"

#include "cmd_renameseq.h"

/* ----------------------------------------------------------------------------------------- */

typedef enum { BASE_8, BASE_10, BASE_16L, BASE_16U } RSBase;

typedef struct {
	MainInfo	*min;

	guint32		start;		/* Index to start at. */
	RSBase		base;		/* The base we're using. */
	guint		precision;	/* Precision, i.e. number of digits. */
	GString		*head;		/* First part of name, before number. */
	GString		*tail;		/* End part, after number. */
} RenSeqInfo;

/* Widgets that are used to set the various parameters above, in dialog mode. */
typedef struct {
	RenSeqInfo	*rsi;
	const gchar	*name;		/* Name from first selected row, for Guess. */
	Dialog		*dlg;
	GtkWidget	*start;
	GtkWidget	*base;
	GtkWidget	*precision;
	GtkWidget	*head;
	GtkWidget	*tail;
	GtkWidget	*guess;
	GtkWidget	*preview;
} RenSeqDialog;

/* ----------------------------------------------------------------------------------------- */

/* 2001-09-30 -	Search for a "base" string that when combined with a numeral is unique
**		for all <num> files we are about to rename.
**		These temp files are in the same directory, which makes me think we can
**		"rely" (um) on rename() to be fairly quick, and also that no extra disk
**		space is required. Probably not bullet proof, but then again, what is?
*/
static const gchar * find_tmp_prefix(guint num)
{
	static gchar	buf[PATH_MAX], prefix[PATH_MAX];
	gboolean	found;
	guint		round = 0U, i;

	do
	{
		g_snprintf(prefix, sizeof prefix, "gt%d;%x*%u#", round, (guint) getpid(), (guint) time(NULL) % 1007);
		for(i = 0, found = FALSE; i < num && !found; i++)
		{
			g_snprintf(buf, sizeof buf, "%s%u", prefix, i);
			if(access(buf, F_OK) == 0)
				found = TRUE;
		}
		round++;
	} while(found && round < 100U);	/* Arbitrary numerical constant. */

	if(found)
		return NULL;

	return prefix;
}

static const gchar * compute_name(RenSeqInfo *rsi, guint index)
{
	const gchar	*fptr = NULL;
	static gchar	buf[PATH_MAX];

	switch(rsi->base)
	{
		case BASE_8:	fptr = "%s%0*o%s";	break;
		case BASE_10:	fptr = "%s%0*u%s";	break;
		case BASE_16L:	fptr = "%s%0*x%s";	break;
		case BASE_16U:	fptr = "%s%0*X%s";	break;
	}
	g_snprintf(buf, sizeof buf, fptr, rsi->head->str, rsi->precision, rsi->start + index, rsi->tail->str);

	return buf;
}

/* 2001-10-03 -	Restore the <index>'th temp name to its original form. Handy when
**		backing out of a failing operation.
*/
static gboolean restore_from_temp(const gchar *prefix, const GSList *sel, guint index)
{
	gchar	tmpname[PATH_MAX];

	g_snprintf(tmpname, sizeof tmpname, "%s%u", prefix, index);
	return rename(tmpname, DP_SEL_NAME(g_slist_nth((GSList *) sel, index))) == 0;
}

static gint do_renameseq(RenSeqInfo *rsi, DirPane *src)
{
	GSList		*slist, *iter;
	const gchar	*tmppfx;
	gchar		olddir[PATH_MAX];
	guint		num;

	/* A Rename in gentoo is never a move. Gotta keep'em separated. */
	if(strchr(rsi->head->str, G_DIR_SEPARATOR) != NULL ||
	   strchr(rsi->tail->str, G_DIR_SEPARATOR) != NULL)
		return 0;

	errno = 0;
	if(!fut_cd(src->dir.path, olddir, sizeof olddir))
		return FALSE;
	errno = 0;
	slist = dp_get_selection(src);
	num   = g_slist_length((GSList *) slist);
	if((tmppfx = find_tmp_prefix(num)) != NULL)
	{
		gchar	tmpname[PATH_MAX];
		guint	i;

		/* First, rename all target files to something temporary. */
		for(iter = slist, i = 0; iter != NULL; iter = g_slist_next(iter), i++)
		{
			g_snprintf(tmpname, sizeof tmpname, "%s%u", tmppfx, i);
			if(rename(DP_SEL_NAME(iter), tmpname) != 0)
				break;
			dp_unselect(src, DP_SEL_INDEX(src, iter));
		}

		if(i != num)	/* Couldn't temp-rename all files. Whine, and back out. */
		{
			guint	j;

			for(j = 0; j < i; j++)
				restore_from_temp(tmppfx, slist, j);
			errno = EINVAL;		/* Not exactly accurate, but... */
		}
		else
		{
			const gchar	*name;
			OvwRes		res;

			/* Start actual renaming, to final names. */
			ovw_overwrite_begin(rsi->min, _("\"%s\" Already Exists - Proceed With Rename?"), OVWF_NO_RECURSE_TEST);
			for(i = 0; i < num; i++)
			{
				g_snprintf(tmpname, sizeof tmpname, "%s%u", tmppfx, i);
				name = compute_name(rsi, i);
				res = ovw_overwrite_file(rsi->min, name, tmpname);
				if(res == OVW_SKIP)
				{
					restore_from_temp(tmppfx, slist, i);
					continue;
				}
				else if(res == OVW_CANCEL)
				{
					for(; i < num; i++)
					{
						if(!(restore_from_temp(tmppfx, slist, i)))
							break;
					}
					break;
				}
				if(rename(tmpname, name) != 0)
					break;
			}
			ovw_overwrite_end(rsi->min);
		}
	}
	dp_free_selection(slist);
	fut_cd(olddir, NULL, 0U);

	if(errno != 0)
		err_set(rsi->min, errno, "RenameSeq", NULL);

	return errno == 0;
}

static const gchar * base_format(RSBase base)
{
	switch(base)
	{
		case BASE_8:	return "%o";
		case BASE_10:	return "%d";
		case BASE_16L:	return "%x";
		case BASE_16U:	return "%X";
	}
	return NULL;
}

static void init_start(const RenSeqInfo *rsi, GtkWidget *entry)
{
	gchar	buf[64];

	g_snprintf(buf, sizeof buf, base_format(rsi->base), rsi->start);
	gtk_entry_set_text(GTK_ENTRY(entry), buf);
}

/* 2001-10-02 -	Crap day. Anyway, extract state from various widgets, and dump it into the
**		'rsi' field of <dlg>.
*/
static void rsi_update(RenSeqDialog *dlg)
{
	GtkWidget	*item;

	/* First, extract current base so we know how to parse the start index. */
	item = gtk_menu_get_active(GTK_MENU(gtk_option_menu_get_menu(GTK_OPTION_MENU(dlg->base))));
	dlg->rsi->base = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(item), "index"));
	gtk_widget_set_sensitive(dlg->guess, dlg->rsi->base < BASE_16L);

	if(sscanf(gtk_entry_get_text(GTK_ENTRY(dlg->start)), base_format(dlg->rsi->base), &dlg->rsi->start) != 1)
		dlg->rsi->start = 1;

	dlg->rsi->precision = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dlg->precision));

	g_string_assign(dlg->rsi->head, gtk_entry_get_text(GTK_ENTRY(dlg->head)));
	g_string_assign(dlg->rsi->tail, gtk_entry_get_text(GTK_ENTRY(dlg->tail)));
}

static void preview_update(RenSeqDialog *dlg)
{
	const gchar	*name;

	rsi_update(dlg);
	name = compute_name(dlg->rsi, 0U);
	gtk_entry_set_text(GTK_ENTRY(dlg->preview), name);
}

/* 2001-10-02 -	Joint widget-changed handler. Just call preview_update(). */
static void evt_something_changed(GtkWidget *wid, gpointer user)
{
	preview_update(user);
}

/* 2001-10-02 -	Inspect first name in selection, and attempt to set parameters
**		accordingly. Glitzy, sure, but I do like sugar in my software. :)
** BUG BUG BUG	Silently assumes base is decimal. Finding embedded hex numbers
**		is simply too tricky. Probably works in octal, though.
*/
static void evt_guess_clicked(GtkWidget *wid, gpointer user)
{
	RenSeqDialog	*dlg = user;
	GString		*head;
	const gchar	*ptr;
	guint		prec = 0;

	head = g_string_new("");
	for(ptr = dlg->name; *ptr && !isdigit((int) *ptr); ptr++)
		g_string_append_c(head, *ptr);
	if(isdigit((int) *ptr))	/* Did we find a digit? */
	{
		for(; *ptr && isdigit((int) *ptr); prec++, ptr++)
			;
		gtk_entry_set_text(GTK_ENTRY(dlg->tail), ptr);
	}
	else if((ptr = strchr(dlg->name, '.')) != NULL)
	{
		g_string_truncate(head, ptr - dlg->name);
		gtk_entry_set_text(GTK_ENTRY(dlg->tail), ptr);
	}
	gtk_entry_set_text(GTK_ENTRY(dlg->head), head->str);
	g_string_free(head, TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(dlg->precision), prec);
}

/* 2001-10-03 -	Well, here's the actual command entrypoint, like usual. */
int cmd_renameseq(MainInfo *min, DirPane *src, DirPane *dst, CmdArg *ca)
{
	static RenSeqInfo	rsi = { NULL };
	const gchar		*bname[] = { N_("Octal"), N_("Decimal"), N_("Hex (a-f)"), N_("Hex (A-F)") };
	GtkObject		*adj;
	GtkWidget		*vbox, *table, *label, *bmenu, *bmitem;
	RenSeqDialog		dlg;
	gint			i, ret;
	GSList			*sel;

	if((sel = dp_get_selection(src)) == NULL)
		return 0;
	dlg.name = DP_SEL_NAME(sel);
	dp_free_selection(sel);

	if(rsi.min == NULL)
	{
		rsi.min   = min;
		rsi.start = 1;
		rsi.base  = BASE_10;
		rsi.precision = 5;
		rsi.head = g_string_new("");
		rsi.tail = g_string_new("");
	}
	dlg.rsi = &rsi;

	vbox  = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new(_("This command renames all selected files\n"
				"into a numbered sequence. The controls\n"
				"below let you define how the names are\n"
				"formed."));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	table = gtk_table_new(4, 4, FALSE);
	label = gtk_label_new(_("Start At"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,  0,0,0,0);
	dlg.start = gtk_entry_new_with_max_length(10);
	init_start(&rsi, dlg.start);
	gtk_signal_connect(GTK_OBJECT(dlg.start), "changed", GTK_SIGNAL_FUNC(evt_something_changed), &dlg);
	gtk_table_attach(GTK_TABLE(table), dlg.start, 1, 4, 0, 1,  GTK_EXPAND|GTK_FILL,0,0,0);

	label = gtk_label_new(_("Base"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,  0,0,0,0);
	bmenu = gtk_menu_new();
	for(i = 0; i < sizeof bname / sizeof bname[0]; i++)
	{
		bmitem = gtk_menu_item_new_with_label(_(bname[i]));
		gtk_object_set_data(GTK_OBJECT(bmitem), "index", GINT_TO_POINTER(i));
		gtk_signal_connect(GTK_OBJECT(bmitem), "activate", GTK_SIGNAL_FUNC(evt_something_changed), &dlg);
		gtk_menu_append(GTK_MENU(bmenu), bmitem);
		gtk_widget_show(bmitem);
	}
	dlg.base = gtk_option_menu_new();
	gtk_option_menu_set_menu(GTK_OPTION_MENU(dlg.base), bmenu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(dlg.base), rsi.base);
	gtk_table_attach(GTK_TABLE(table), dlg.base, 1, 2, 1, 2,  GTK_EXPAND|GTK_FILL,0,0,0);

	label = gtk_label_new(_("Precision"));
	gtk_table_attach(GTK_TABLE(table), label, 2, 3, 1, 2,  0,0,0,0);
	adj   = gtk_adjustment_new(rsi.precision, 0.0, 10.0, 1.0, 1.0, 1.0);
	dlg.precision = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed", GTK_SIGNAL_FUNC(evt_something_changed), &dlg);
	gtk_table_attach(GTK_TABLE(table), dlg.precision, 3, 4, 1, 2,  GTK_EXPAND|GTK_FILL,0,0,0);

	label = gtk_label_new(_("Head"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,  0,0,0,0);
	dlg.head = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(dlg.head), rsi.head->str);
	gtk_signal_connect(GTK_OBJECT(dlg.head), "changed", GTK_SIGNAL_FUNC(evt_something_changed), &dlg);
	gtk_table_attach(GTK_TABLE(table), dlg.head, 1, 3, 2, 3,  GTK_EXPAND|GTK_FILL,0,0,0);

	label = gtk_label_new(_("Tail"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4,  0,0,0,0);
	dlg.tail = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(dlg.tail), rsi.tail->str);
	gtk_signal_connect(GTK_OBJECT(dlg.tail), "changed", GTK_SIGNAL_FUNC(evt_something_changed), &dlg);
	gtk_table_attach(GTK_TABLE(table), dlg.tail, 1, 3, 3, 4,  GTK_EXPAND|GTK_FILL,0,0,0);

	dlg.guess = gtk_button_new_with_label(_("Guess"));
	gtk_signal_connect(GTK_OBJECT(dlg.guess), "clicked", GTK_SIGNAL_FUNC(evt_guess_clicked), &dlg);
	gtk_table_attach(GTK_TABLE(table), dlg.guess, 3, 4, 2, 4,  0,GTK_EXPAND|GTK_FILL,0,0);

	label = gtk_label_new(_("Preview"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5,  0,0,0,0);
	dlg.preview = gtk_entry_new();
	gtk_editable_set_editable(GTK_EDITABLE(dlg.preview), FALSE);
	gtk_table_attach(GTK_TABLE(table), dlg.preview, 1, 4, 4, 5,  GTK_EXPAND|GTK_FILL,0,0,5);

	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

	preview_update(&dlg);

	dlg.dlg = dlg_dialog_sync_new(vbox, _("Sequential Rename"), _("OK|Cancel"));
	ret = dlg_dialog_sync_wait(dlg.dlg);

	if(ret == DLG_POSITIVE)
	{
		rsi_update(&dlg);
		err_clear(min);
		ret = do_renameseq(&rsi, src);
		dp_rescan(src);
		if(!ret)
			err_show(min);
	}
	dlg_dialog_sync_destroy(dlg.dlg);

	return ret == 0;
}
