/*
** 2002-07-22 -	Finally, gentoo has learned to use FAM (File Alteration Monitor) if available. FAM,
**		from SGI, provides an application with the ability to be notified when a file or
**		directory changes. We use this to trigger pane reloads, so gentoo knows what's going
**		on in the filesystem.
**
**		FAM support is entierly optional: if you don't want it, or you system doesn't have it,
**		it will be built as stubs that do nothing. Use --disable-fam to manually turn it off.
**
**		Btw, this module is named "gfam" because plain "fam.h" collided with <fam.h>. Weird.
*/

#include "gentoo.h"

#if defined HAVE_FAM
#include <fam.h>
#endif

#include "dialog.h"
#include "dirpane.h"
#include "miscutil.h"

#include "gfam.h"

/* If we don't have FAM, or it has been disabled, simply define the necessary functions
** as do-nothing stubs. Since the interface is (still?) pretty simple, it's easier and
** cleaner to do it in this all-at-once way, rather than scattering #if defined HAVE_FAMs
** all over the place. Perhaps the compiler can even optimize these stubs out?
*/
#if !defined HAVE_FAM

gboolean fam_initialize(MainInfo *min)
{
	return TRUE;
}

gboolean fam_is_active(void)
{
	return FALSE;
}

void fam_monitor(const DirPane *dp)
{
}

void fam_rescan_block(void)
{
}

void fam_rescan_unblock(void)
{
}

void fam_shutdown(MainInfo *min)
{
}

#else	/* If HAVE_FAM is actually defined, provide functional functions. */

/* ----------------------------------------------------------------------------------------- */

#define	MAGIC_OFFSET	16

static struct {
	gboolean	ok;
	MainInfo	*min;
	FAMConnection	conn;
	FAMRequest	req[2];
	guint		block_cnt;		/* If set, don't rescan. */
	guint		block_rescan;		/* Used to track what needs to be reloaded on unblock(). */
	guint		timeout[2];		/* Used to notify of pending refresh after rate limiting. */
} the_faminfo = { FALSE };

/* ----------------------------------------------------------------------------------------- */

/* 2003-09-30 -	Timeout handler to catch rate limited refreshes. Set when a refresh is denied
**		by the rate limiter, to do a final refresh since one is known to be needed.
*/
static gboolean evt_fam_timeout(gpointer data)
{
	guint index = GPOINTER_TO_UINT(data);

	if(the_faminfo.block_cnt)
		the_faminfo.block_rescan |= (1 << index);
	else
		dp_rescan(&the_faminfo.min->gui->pane[index]);

	the_faminfo.timeout[index] = 0U;
	return FALSE;
}

/* Refresh rate period is always in this range, in seconds. */
#define	LIMIT_MIN	0.3f
#define	LIMIT_MAX	2.0f

/* 2003-09-30 -	Rate limit pane rescans. Tracks elaped time since last rescan, and limit the frequency of
**		actual rescans performed. Returns TRUE if it did do the rescan, FALSE if not. The rate of
**		rescanning is dynamically altered based on how long it takes to do; aiming to keep the a
**		1:4 ratio between time to rescan and time between rescans.
*/
static gboolean rescan_rate_limited(guint index)
{
	static gfloat	limit = LIMIT_MIN;			/* Dynamically adjusted based on refresh time. */
	static GTimeVal stamp[2] = { { 0 }, { 0 } };
	static gfloat elapsed[2] = { 0.0f, 0.0f};		/* Time since last rescan allowed. */
	GTimeVal now;
	gboolean ret = TRUE;

	g_get_current_time(&now);
	if(stamp[index].tv_sec && stamp[index].tv_usec)
	{
		gfloat dt = msu_diff_timeval(stamp + index, &now);
		if(dt < limit)
		{
			elapsed[index] += dt;
			if(elapsed[index] < limit)	/* When blocking has reached limit, let a request through. */
			{
				if(the_faminfo.timeout[index])
					gtk_timeout_remove(the_faminfo.timeout[index]);
				the_faminfo.timeout[index] = gtk_timeout_add(1000.0f * limit, evt_fam_timeout, GUINT_TO_POINTER(index));
				ret = FALSE;
			}
		}
	}
	stamp[index] = now;

	if(ret)	/* Refresh needed; do it, measure how long it takes, and update rate. */
	{
		gfloat	dt;

		dp_rescan(&the_faminfo.min->gui->pane[index]);
		g_get_current_time(&now);
		dt = msu_diff_timeval(&stamp[index], &now);
		limit = 3.0f * dt;		/* 3 plus 1 is four. */
		if(limit < LIMIT_MIN)
			limit = LIMIT_MIN;
		else if(limit > LIMIT_MAX)
			limit = LIMIT_MAX;
		elapsed[index] = 0.0f;
		if(the_faminfo.timeout[index])
		{
			gtk_timeout_remove(the_faminfo.timeout[index]);
			the_faminfo.timeout[index] = 0U;
		}
	}
	return ret;
}

/* 2002-07-22 -	Rescan all panes whose index bit is set in <rescan>. Overkill. */
static void rescan_issue(guint rescan)
{
	guint	i, f = 0;

	for(i = 0; i < sizeof the_faminfo.req / sizeof *the_faminfo.req; i++)
	{
		if(rescan & (1 << i))
		{
			if(rescan_rate_limited(i))
			{
			/*	dp_rescan(&the_faminfo.min->gui->pane[i]);*/
				f |= the_faminfo.min->gui->cur_pane == &the_faminfo.min->gui->pane[i];
			}
		}
	}
	if(f)
		dp_show_stats(the_faminfo.min->gui->cur_pane);
}

/* 2002-07-22 -	GTK+ calls this whenever a FAM event is reported. We filter out the interesting
**		ones, and issue rescans on the affected panes, either now or buffered for later.
*/
static void evt_fam_monitor(gpointer data, gint source, GdkInputCondition cond)
{
	FAMEvent	evt;
	guint		rescan = 0U;

	while(FAMPending(&the_faminfo.conn))
	{
		if(FAMNextEvent(&the_faminfo.conn, &evt) == 1)	/* Returns -1 on error, annoyingly enough. */
		{
			if(evt.code < FAMAcknowledge)		/* Filter out events that wouldn't show up anyway. */
				rescan |= 1 << (evt.fr.reqnum - MAGIC_OFFSET);
		}
	}
	if(the_faminfo.block_cnt)
		the_faminfo.block_rescan |= rescan;	/* Non-destructive, important. */
	else if(rescan)
		rescan_issue(rescan);
}

/* 2002-07-22 -	Initialize FAM subsystem, by opening a connection and hooking up with GTK+'s event loop. */
gboolean fam_initialize(MainInfo *min)
{
	the_faminfo.min = min;
	if(FAMOpen2(&the_faminfo.conn, PACKAGE) == 0)
	{
		guint	i;

		for(i = 0; i < sizeof the_faminfo.req / sizeof *the_faminfo.req; i++)
			the_faminfo.req[i].reqnum = 0;

		/* Hook us into the main GTK+ event detection system. Simple. */
		gtk_input_add_full(FAMCONNECTION_GETFD(&the_faminfo.conn), GDK_INPUT_READ, evt_fam_monitor, NULL, NULL, NULL);
		the_faminfo.ok = TRUE;
		the_faminfo.block_cnt = 0;
	}
	else
		g_warning(_("FAM open failed, error %d--FAM will not be used"), FAMErrno);

	return the_faminfo.ok;
}

/* 2002-07-22 -	Just tell a caller if FAM is indeed in use. Says "FALSE" if built without FAM. */
gboolean fam_is_active(void)
{
	return the_faminfo.ok;
}

/* 2002-07-22 -	Start monitoring the directory of <dp>. Can't take path only, need to key it to pane index for rescan. */
void fam_monitor(const DirPane *dp)
{
	if(!the_faminfo.ok)
		return;
	if(the_faminfo.req[dp->index].reqnum > 0)
		FAMCancelMonitor(&the_faminfo.conn, &the_faminfo.req[dp->index]);
	the_faminfo.req[dp->index].reqnum = MAGIC_OFFSET + dp->index;
	if(FAMMonitorDirectory2(&the_faminfo.conn, dp->dir.path, &the_faminfo.req[dp->index]) != 0)
	{
		the_faminfo.req[dp->index].reqnum = 0;
		g_warning(_("Couldn't add FAM monitor on \"%s\", error %s (restart with --no-fam to go around, perhaps)"), dp->dir.path, FamErrlist[FAMErrno]);
	}
}

/* 2002-07-22 -	Caller is about to start doing stuff for which no rescans are allowed until unblock() is called.
**		This prevents dp_rescan() from reallocating the core buffers, and thus rendering the caller's
**		dp_get_selection() list to change meaning, which can be *very* devastating.
*/
void fam_rescan_block(void)
{
	if(the_faminfo.block_cnt == 0)
		the_faminfo.block_rescan = 0U;
	the_faminfo.block_cnt++;
}

/* 2002-07-22 -	Unblock rescans. If any FAM events occured during block, issue the rescans now. */
void fam_rescan_unblock(void)
{
	if(the_faminfo.block_cnt)
	{
		the_faminfo.block_cnt--;
		if(the_faminfo.block_cnt == 0U)
			rescan_issue(the_faminfo.block_rescan);
	}
	else
		g_warning("Non-balanced fam_rescan_unblock() call detected; underflow");
}

/* 2002-07-22 -	Shut down FAM system, by closing the connection. */
void fam_shutdown(MainInfo *min)
{
	if(the_faminfo.ok)
		FAMClose(&the_faminfo.conn);
}

#endif		/* HAVE_FAM */
