
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>

#include <glib/glist.h>

#include "config.h"
#include "libconf.h"
#include "getstream.h"

static struct config_s	*config;

/* Temporary variables for parsing */
static struct adapter_s	*adapter;
static struct stream_s	*stream;
static struct channel_s	*channel;
static struct sap_s	*sap;

static int cf_adapter_start(struct lc_centry *ce, struct lc_value *val) {
	adapter=calloc(1, sizeof(struct adapter_s));
	adapter->no=val->num;
	adapter->budgetmode=1;
	config->adapter=g_list_append(config->adapter, adapter);
	return 1;
}

static int cf_sap_scope(struct lc_centry *ce, struct lc_value *val)
	{ sap->scope=val->string; return 1; }
static int cf_sap_sap_group(struct lc_centry *ce, struct lc_value *val)
	{ sap->group=val->string; return 1; }
static int cf_sap_sap_port(struct lc_centry *ce, struct lc_value *val)
	{ sap->port=val->num; return 1; }
static int cf_sap_ttl(struct lc_centry *ce, struct lc_value *val)
	{ sap->ttl=val->num; return 1; }
static int cf_sap_playgroup(struct lc_centry *ce, struct lc_value *val)
	{ sap->playgroup=val->string; return 1; }
static int cf_sap_announce_host(struct lc_centry *ce, struct lc_value *val)
	{ sap->announcehost=val->string; return 1; }
static int cf_sap_announce_port(struct lc_centry *ce, struct lc_value *val)
	{ sap->announceport=val->num; return 1; }

static int cf_stream_remoteport(struct lc_centry *ce, struct lc_value *val)
	{ stream->remoteport=val->num; return 1; }
static int cf_stream_remoteaddr(struct lc_centry *ce, struct lc_value *val)
	{ stream->remoteaddr=val->string; return 1; }
static int cf_stream_rtpport(struct lc_centry *ce, struct lc_value *val)
	{ stream->rtpport=val->num; return 1; }
static int cf_stream_rtcpport(struct lc_centry *ce, struct lc_value *val)
	{ stream->rtcpport=val->num; return 1; }
static int cf_stream_ttl(struct lc_centry *ce, struct lc_value *val)
	{ stream->ttl=val->num; return 1; }
static int cf_stream_url(struct lc_centry *ce, struct lc_value *val)
	{ stream->url=val->string; return 1; }
static int cf_http_port(struct lc_centry *ce, struct lc_value *val)
	{ config->http_port=val->num; return 1; }
static int cf_stream_filename(struct lc_centry *ce, struct lc_value *val)
	{ stream->filename=val->string; return 1; }

static int cf_sap_start(struct lc_centry *ce, struct lc_value *val) {
	sap=calloc(1, sizeof(struct sap_s));
	stream->sap=sap;
	sap->stream=stream;

	/* Default values */
	sap->ttl=15;
	return 1;
}

static int cf_channel_start(struct lc_centry *ce, struct lc_value *val) {
	channel=calloc(1, sizeof(struct channel_s));
	channel->id=val->num;
	channel->adapter=adapter;
	adapter->channel=g_list_append(adapter->channel, channel);
	return 1;
}

static int cf_stream_start(struct lc_centry *ce, struct lc_value *val, int stype) {
	stream=calloc(1, sizeof(struct stream_s));
	stream->type=stype;
	stream->ttl=15;
	channel->stream=g_list_append(channel->stream, stream);
	return 1;
}

static int cf_stream_udp_start(struct lc_centry *ce, struct lc_value *val)
	{ return cf_stream_start(ce, val, STYPE_UDP); }
static int cf_stream_rtp_start(struct lc_centry *ce, struct lc_value *val)
	{ return cf_stream_start(ce, val, STYPE_RTP); }
static int cf_stream_rtcp_start(struct lc_centry *ce, struct lc_value *val)
	{ return cf_stream_start(ce, val, STYPE_RTCP); }
static int cf_stream_http_start(struct lc_centry *ce, struct lc_value *val)
	{ return cf_stream_start(ce, val, STYPE_HTTP); }
static int cf_stream_pipe_start(struct lc_centry *ce, struct lc_value *val)
	{ return cf_stream_start(ce, val, STYPE_PIPE); }

struct lc_ventry conf_sap[] = {
	{ "scope", 0, 1, LCV_STRING, 0, NULL, cf_sap_scope },
	{ "sap-group", 0, 1, LCV_IPV4ADDR, 0, NULL, cf_sap_sap_group },
	{ "sap-port", 0, 1, LCV_NUM, 0, NULL, cf_sap_sap_port },
	{ "announce-host", 0, 1, LCV_STRING, 0, NULL, cf_sap_announce_host },
	{ "announce-port", 0, 1, LCV_NUM, 0, NULL, cf_sap_announce_port },
	{ "ttl", 0, 1, LCV_NUM, 0, NULL, cf_sap_ttl },
	{ "playgroup", 0, 1, LCV_STRING, 0, NULL, cf_sap_playgroup },
	{ NULL, 0, 0, 0, 0, NULL },
};

struct lc_ventry conf_stream_udp[] = {
	{ "remote-address", 1, 1, LCV_IPADDR, 0, NULL, cf_stream_remoteaddr },
	{ "remote-port", 1, 1, LCV_NUM, 0, NULL, cf_stream_remoteport },
	{ "sap", 0, 1, LCV_NONE, 0, conf_sap, cf_sap_start },
	{ "ttl", 0, 1, LCV_NUM, 0, NULL, cf_stream_ttl },
	{ NULL, 0, 0, 0, 0, NULL, NULL },
};

struct lc_ventry conf_stream_rtp[] = {
	{ "remote-address", 1, 1, LCV_IPV4ADDR, 0, NULL, cf_stream_remoteaddr },
	{ "remote-port", 1, 1, LCV_NUM, 0, NULL, cf_stream_remoteport },
	{ "sap", 0, 1, LCV_NONE, 0, conf_sap, cf_sap_start },
	{ "ttl", 0, 1, LCV_NUM, 0, NULL, cf_stream_ttl },
	{ NULL, 0, 0, 0, 0, NULL },
};

struct lc_ventry conf_stream_rtcp[] = {
	{ "rtp-port", 1, 1, LCV_NUM, 0, NULL, cf_stream_rtpport },
	{ "rtcp-port", 1, 1, LCV_NUM, 0, NULL, cf_stream_rtcpport },
	{ "sap", 0, 1, LCV_NONE, 0, conf_sap, cf_sap_start },
	{ NULL, 0, 0, 0, 0, NULL },
};

struct lc_ventry conf_stream_http[] = {
	{ "url", 1, 1, LCV_STRING, 0, NULL, cf_stream_url },
	{ NULL, 0, 0, 0, 0, NULL },
};

struct lc_ventry conf_stream_pipe[] = {
	{ "filename", 1, 1, LCV_STRING, 0, NULL, cf_stream_filename },
	{ NULL, 0, 0, 0, 0, NULL },
};

static int cf_channel_name(struct lc_centry *ce, struct lc_value *val)
	{ channel->name=val->string; return 1; }

static int cf_channel_csa_key(struct lc_centry *ce, struct lc_value *val) {
	char		*eptr;
	uint64_t	key;
	int		i, l=strlen(val->string);

	key=strtoull(val->string, &eptr, 16);

	/*
	 * Was the string parsed ?
	 * Was the string a number until the end ?
	 * Was the string between 16 and 18 (0x) bytes long ?
	 *
	 */
	if (val->string == eptr || (eptr != NULL && eptr[0] != 0x0) || l<16 || l>18) {
		fprintf(stderr, "config: Invalid csa-key \"%s\" in line %d\n",
				val->string, ce->vline);
		return 0;
	}

	/* cpu_to_64be anyone ? */
	for(i=0;i<8;i++)
		channel->csakey[i]=(key>>(56-8*i))&0xff;

	channel->csat=csa_New();
	csa_SetCW(channel->csat, channel->csakey, channel->csakey);

	return 1;
};

static int cf_channel_csa_length(struct lc_centry *ce, struct lc_value *val) {
	if (val->num > TS_PACKET_SIZE) {
		fprintf(stderr, "config: Invalid csa length %ld in line %d. Needs to be between 0 and 188\n", 
			val->num, ce->vline);
		return 0;
	}
	channel->csalength=val->num;
	return 1;
}

static int cf_channel_csa(struct lc_centry *ce, struct lc_value *val)
	{ channel->csalength=TS_PACKET_SIZE; return 1; }

static struct lc_ventry conf_channel_csa[] = {
	{ "key", 0, 1, LCV_STRING, 0, NULL, cf_channel_csa_key },
	{ "length", 0, 1, LCV_STRING, 0, NULL, cf_channel_csa_length },
	{ NULL, 0, 0, 0, 0, NULL },
};

static struct lc_ventry conf_channel[] = {
	{ "name", 1, 1, LCV_STRING, 0, NULL, cf_channel_name },
	{ "csa", 0, 0, LCV_NONE, 0, conf_channel_csa, cf_channel_csa },
	{ "stream-udp", 0, 0, LCV_NONE, 0, conf_stream_udp, cf_stream_udp_start },
	{ "stream-rtp", 0, 0, LCV_NONE, 0, conf_stream_rtp, cf_stream_rtp_start },
	{ "stream-rtcp", 0, 0, LCV_NONE, 0, conf_stream_rtcp, cf_stream_rtcp_start },
	{ "stream-http", 0, 0, LCV_NONE, 0, conf_stream_http, cf_stream_http_start },
	{ "stream-pipe", 0, 0, LCV_NONE, 0, conf_stream_pipe, cf_stream_pipe_start },
	{ NULL, 0, 0, 0, 0, NULL },
};

static int cf_dvbs_trans_freq(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbs.t_freq=val->num; return 1; }
static int cf_dvbs_trans_pol(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbs.t_pol=val->string; return 1; }
static int cf_dvbs_trans_srate(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbs.t_srate=val->num; return 1; }
static int cf_dvbs_trans_diseqc(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbs.t_diseqc=val->num; return 1; }

static struct lc_ventry conf_dvbs_transponder[] = {
	{ "frequency", 1, 1, LCV_NUM, 0, NULL, cf_dvbs_trans_freq },
	{ "polarisation", 1, 1, LCV_STRING, 0, NULL, cf_dvbs_trans_pol },
	{ "symbol-rate", 1, 1, LCV_NUM, 0, NULL, cf_dvbs_trans_srate },
	{ "diseqc", 0, 1, LCV_NUM, 0, NULL, cf_dvbs_trans_diseqc },
	{ NULL, 0, 0, 0, 0, NULL },
};

static int cf_dvbs_lnb_lof1(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbs.lnb_lof1=val->num; return 1; }
static int cf_dvbs_lnb_lof2(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbs.lnb_lof2=val->num; return 1; }
static int cf_dvbs_lnb_slof(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbs.lnb_slof=val->num; return 1; }

static struct lc_ventry conf_dvbs_lnb[] = {
	{ "lof1", 1, 1, LCV_NUM, 0, NULL, cf_dvbs_lnb_lof1 },
	{ "lof2", 1, 1, LCV_NUM, 0, NULL, cf_dvbs_lnb_lof2 },
	{ "slof", 1, 1, LCV_NUM, 0, NULL, cf_dvbs_lnb_slof },
	{ NULL, 0, 0, 0, 0, NULL },
};

static struct lc_ventry conf_dvbs[] = {
	{ "lnb", 1, 1, LCV_NONE, 0, conf_dvbs_lnb, NULL },
	{ "transponder", 1, 1, LCV_NONE, 0, conf_dvbs_transponder, NULL },
	{ NULL, 0, 0, 0, 0, NULL },
};

static int cf_dvbt_bandwidth(struct lc_centry *ce, struct lc_value *val) {
	if (strcasecmp("auto", val->string) == 0) {
		adapter->dvbt.bandwidth=0;
	} else {
		int	bw=strtol(val->string, NULL, 10);
		if (bw != 6 && bw != 7 && bw != 8) {
			fprintf(stderr, "config: Illegal DVB-T bandwidth \"%s\" in line %d\n",
					val->string, ce->vline);
			return 0;
		}
		adapter->dvbt.bandwidth=bw;
	}
	return 1;
}

static int cf_dvbt_freq(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbt.freq=val->num; return 1; }

static int cf_dvbt_tmode(struct lc_centry *ce, struct lc_value *val) {
	if (strcasecmp("auto", val->string) == 0) {
		adapter->dvbt.tmode=0;
	} else {
		int	t=strtol(val->string, NULL, 10);
		if (t != 2 && t != 8) {
			fprintf(stderr, "config: Illegal DVB-T transmission-mode \"%s\" in line %d\n",
					val->string, ce->vline);
			return 0;
		}
		adapter->dvbt.tmode=t;
	}
	return 1;
}

static int cf_dvbt_modulation(struct lc_centry *ce, struct lc_value *val) {
	if (strcasecmp("auto", val->string) == 0) {
		adapter->dvbt.modulation=0;
	} else {
		int	m=strtol(val->string, NULL, 10);
		if (m != 16 && m != 32 && m != 64 && m != 128 && m != 256) {
			fprintf(stderr, "config: Illegal DVB-T modulation \"%s\" in line %d\n",
					val->string, ce->vline);
			return 0;
		}
		adapter->dvbt.modulation=m;
	}
	return 1;
}

static int cf_dvbt_guard(struct lc_centry *ce, struct lc_value *val) {
	if (strcasecmp("auto", val->string) == 0) {
		adapter->dvbt.guard=0;
	} else {
		int	gi=strtol(val->string, NULL, 10);
		if (gi != 4 && gi != 8 && gi != 16 && gi != 32) {
			fprintf(stderr, "config: Illegal DVB-T guard-interval \"%s\" in line %d\n",
					val->string, ce->vline);
			return 0;
		}
		adapter->dvbt.guard=gi;
	}
	return 1;
}

static int cf_dvbt_hierarchy(struct lc_centry *ce, struct lc_value *val) {
	if (strcasecmp("none", val->string) == 0)
		adapter->dvbt.hierarchy=-1;
	else if (strcasecmp("auto", val->string) == 0)
		adapter->dvbt.hierarchy=0;
	else {
		int	h=strtol(val->string, NULL, 0);

		if (h != 1 && h != 2 && h != 4) {
			fprintf(stderr, "config: Illegal DVB-T hierarchy %s in line %d\n",
					val->string, ce->vline);
			return 0;
		}

		adapter->dvbt.hierarchy=h;
	}
	return 1;
}

static struct lc_ventry conf_dvbt[] = {
	{ "bandwidth", 0, 1, LCV_STRING, 0, NULL, cf_dvbt_bandwidth },
	{ "frequency", 1, 1, LCV_NUM, 0, NULL, cf_dvbt_freq },
	{ "transmission-mode", 0, 1, LCV_STRING, 0, NULL, cf_dvbt_tmode },
	{ "modulation", 0, 1, LCV_STRING, 0, NULL, cf_dvbt_modulation },
	{ "guard-interval", 0, 1, LCV_STRING, 0, NULL, cf_dvbt_guard },
	{ "hierarchy", 0, 1, LCV_STRING, 0, NULL, cf_dvbt_hierarchy },
	{ NULL, 0, 0, 0, 0, NULL },
};

static int cf_dvbs(struct lc_centry *ce, struct lc_value *val)
	{ adapter->type=AT_DVBS; return 1; }
static int cf_dvbt(struct lc_centry *ce, struct lc_value *val)
	{ adapter->type=AT_DVBT; return 1; }
static int cf_dvbc(struct lc_centry *ce, struct lc_value *val)
	{ adapter->type=AT_DVBC; return 1; }

static int cf_dvbc_freq(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbc.freq=val->num; return 1; }
static int cf_dvbc_modulation(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbc.modulation=val->num; return 1; }
static int cf_dvbc_trans_srate(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbc.srate=val->num; return 1; }
static int cf_dvbc_fec(struct lc_centry *ce, struct lc_value *val)
	{ adapter->dvbc.fec=val->num; return 1; }

static struct lc_ventry conf_dvbc[] = {
	{ "frequency", 1, 1, LCV_NUM, 0, NULL, cf_dvbc_freq },
	{ "modulation", 0, 1, LCV_NUM, 0, NULL, cf_dvbc_modulation },
	{ "symbol-rate", 1, 1, LCV_NUM, 0, NULL, cf_dvbc_trans_srate },
	{ "fec", 0, 1, LCV_NUM, 0, NULL, cf_dvbc_fec },
	{ NULL, 0, 0, 0, 0, NULL },
};

static int cf_adapter_budget(struct lc_centry *ce, struct lc_value *val)
	{ adapter->budgetmode=val->num; return 1; }

static struct lc_ventry conf_adapter[] = {
	{ "budget-mode", 0, 0, LCV_BOOL, 0, NULL, cf_adapter_budget },
	{ "channel", 0, 0, LCV_NUM, LCO_UNIQ, conf_channel, cf_channel_start },
	{ "dvb-s", 0, 1, LCV_NONE, 0, conf_dvbs, cf_dvbs },
	{ "dvb-t", 0, 1, LCV_NONE, 0, conf_dvbt, cf_dvbt },
	{ "dvb-c", 0, 1, LCV_NONE, 0, conf_dvbc, cf_dvbc },
	{ NULL, 0, 0, 0, 0, NULL },
};


static struct lc_ventry conf_http[] = {
	{ "port", 1, 1, LCV_NUM, 0, NULL, cf_http_port },
};

static struct lc_ventry conf_main[] = {
	{ "adapter", 1, 1, LCV_NUM, LCO_UNIQ, conf_adapter, cf_adapter_start },
	{ "http", 1, 1, LCV_NONE, 0, conf_http, NULL },
	{ NULL, 0, 0, 0, 0, NULL },
};

struct config_s *readconfig(char *filename) {
	int			cfd;
	struct stat		sb;
	char			*ctext;
	struct lc_centry	*c;

	/* Base for config storage */
	config=calloc(1, sizeof(struct config_s));

	cfd=open(filename, O_RDONLY);
	if (cfd<0)
		return NULL;

	if (fstat(cfd, &sb)) {
		close(cfd);
		return NULL;
	}

	ctext=mmap(NULL, sb.st_size, PROT_READ, MAP_SHARED, cfd, 0);

	c=libconf_parse(ctext, sb.st_size);

	munmap(ctext, sb.st_size);
	close(cfd);

	if (!c)
		return NULL;

	if (!libconf_validate(c, conf_main)) {
		libconf_free(c);
		return NULL;
	}

	return config;
}
