/*  GFC-UI: GTK+ Foundation Classes (User Interface Library)
 *  Copyright (C) 2002-2004 The GFC Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/// @file gfc/gtk/togglebutton.hh
/// @brief A GtkToggleButton C++ wrapper interface.
///
/// Provides ToggleButton, a Button which will remain 'pressed-in' when clicked. Clicking again
/// will cause the toggle button to return to it's normal state. 

#ifndef GFC_GTK_TOGGLE_BUTTON_HH
#define GFC_GTK_TOGGLE_BUTTON_HH

#ifndef GFC_GTK_BUTTON_HH
#include <gfc/gtk/button.hh>
#endif

#ifndef __GTK_TOGGLE_BUTTON_H__
#include <gtk/gtktogglebutton.h>
#endif

namespace GFC {

namespace Gtk {

/// @class ToggleButton togglebutton.hh gfc/gtk/togglebutton.hh
/// @brief A GtkToggleButton C++ wrapper class.
///
/// A ToggleButton is a Button which will remain 'pressed-in' when clicked. Clicking again
/// will cause the toggle button to return to it's normal state. If you create an empty
/// toggle button it is advisable to pack a widget, (such as a Label and/or a Image), into
/// the toggle button container (see GtkButton for more information).
///
/// The state of a ToggleButton can be set specifically using set_active(), and retrieved 
/// using get_active(). To simply switch the state of a toggle button, use toggled.
 
class ToggleButton : public Button
{
	friend class G::Object;

	ToggleButton(const ToggleButton&);
	ToggleButton& operator=(const ToggleButton&);

protected:
/// @name Constructors
/// @{

	explicit ToggleButton(GtkToggleButton *toggle_button, bool owns_reference = false);
	///< Construct a new ToggleButton from an existing GtkToggleButton.
	///< @param toggle_button A pointer to a GtkToggleButton.
	///< @param owns_reference Set false if the initial reference count is floating, set true if it's not.
	///<
	///< <BR>The <EM>toggle_button</EM> can be a newly created GtkToggleButton or an existing
	///< GtkToggleButton (see G::Object::Object).
	
/// @}
/// @name Signal Prototypes
/// @{

	typedef G::Signal<void> ToggledSignalType;
	typedef G::SignalProxy<TypeInstance, ToggledSignalType> ToggledSignalProxy;
	static const ToggledSignalType toggled_signal;
	///< Toggled signal (see sig_destroy()). Calls a slot with the signature:
	///< @code
	///< void function();
	///< @endcode
	
/// @}

public:
/// @name Constructors
/// @{

	ToggleButton();
	///< Constructs a new toggle button. A widget should be packed into the button,
	///< such as a Label and/or Image.
	 
	explicit ToggleButton(const String& label, bool use_underline = false);
	///< Constructs a new toggle button with a text label. 
	///< @param label The text you want the Label to hold.
	///< @param use_underline Set <EM>true</EM> if label contains a mnemonic character.
	///<
	///< <BR>If characters in label are preceded by an underscore, they are underlined.
	///< If you need a literal underscore character in a label, use '__' (two underscores). 
	///< The first underlined character represents a keyboard accelerator called a mnemonic. 
	///< Pressing Alt and that key activates the button.
	
	virtual ~ToggleButton();
	///< Destructor.
	
/// @}
/// @name Accessors
/// @{

	GtkToggleButton* gtk_toggle_button() const;
	///< Get a pointer to the GtkToggleButton structure.
	
	operator GtkToggleButton* () const;
	///< Conversion operator; safely converts a ToggleButton to a GtkToggleButton pointer.

	bool get_active() const;
	///< Queries the toggle button and returns it's current state. 
	///< @return <EM>true</EM> if the toggle button is pressed in and <EM>false</EM> if it is raised. 

	bool get_mode() const; 
	///< Retrieves whether the button is displayed as a separate indicator and label (see set_mode()).
	///< @return <EM>true</EM> if the togglebutton is drawn as a separate indicator and label.
	
	bool get_inconsistent() const;
	///< Gets the value set by set_inconsistent().
	///< @return <EM>true</EM> if the button is displayed as inconsistent, <EM>false</EM> otherwise.
	 
/// @}
/// @name Methods
/// @{
	
	void set_mode(bool draw_indicator);
	///< Sets whether the button is displayed as a separate indicator and label.
	///< @param draw_indicator If <EM>true</EM>, draw the button as a separate indicator and label;
	///< if <EM>false</EM>, draw the button like a normal button. 
	///<	 
	///< <BR>You can call this method on a checkbutton or a radiobutton with <EM>draw_indicator</EM>
	///< set to false to make the button look like a normal button. This method only effects instances
	///< of classes like CheckButton and RadioButton that< derive from ToggleButton, not instances
	///< of ToggleButton itself.

	void set_active(bool is_active);
	///< Sets the status of the toggle button.
	///< @param is_active Either true or false.
	///<
	///< <BR>Set <EM>is_active</EM> true if you want the toggle button to be 'pressed in', and false
	///< to raise it. This action causes the <EM>toggled</EM> signal to be emitted.
	
	void set_inconsistent(bool setting);
	///< If the user has selected a range of elements (such as some text or spreadsheet cells) that
	///< are affected by a toggle button, and the current values in that range are inconsistent,
	///< you may want to display the toggle in an "in between" state. 
	///< @param setting <EM>true</EM> if the state is inconsistent.
	///<	
	///< <BR>This method turns on "in between" display. Normally you would turn off the 
	///< inconsistent state again if the user toggles the toggle button. This has to be
	///< done manually, set_inconsistent() only affects visual appearance, it doesn't 
	///< affect the semantics of the button.
	
	void toggled();
	///< Emits the toggled signal on the toggle button. 

/// @}
/// @name Signal Proxies
/// @{

	const ToggledSignalProxy sig_toggled();
	///< Connect to the toggled_signal; emitted whenever the toggle button's state is changed.
	
/// @}
};

} // namespace Gtk

} // namespace GFC

#include <gfc/gtk/inline/togglebutton.inl>

#endif // GFC_GTK_TOGGLE_BUTTON_HH

