// DOCVIEW.H : a "document/view"-style base class for all graphical user interfaces.

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef DOCVIEW_H
#define DOCVIEW_H

#ifdef ENABLE_GRAPHICS

class docview;

class color_mode;

class transparent_primitive;
class transparent_primitive_data;

class tpd_tri_3c;
class tpd_quad_4c;

/*################################################################################################*/

class dummy_object;		// objects.h
class smart_object;		// objects.h

class camera;			// camera.h
class light;			// camera.h

class transformer;		// camera.h

#include "model.h"
#include "views.h"

#include <vector>
using namespace std;

/*################################################################################################*/

#define SB_SIZE 512		// the OGL selection buffer size.

/**	The "##docview" class describes our "##document" (which is a molecular model 
	of some kind in our case) and supports different "##views" for the user to examine 
	and modify the "##document".
	
	If we use the "##document/view/controller"-taxonomy, "##docview" 
	is a "##document" and "##controller" class.
	
	If we use the "##model/view/controller"-taxonomy, "##docview" 
	is a "##model" and "##controller" class.
*/

// at some point in the future we might want to present different models (for example, a MM model
// and a QM model) in a single window. would that be enough if we just allow the Render() -function be
// separate from this class??? a "host" model could then call Render() for the "guest" models it has ?!?!?!?!

class docview : virtual public model_simple
{
	protected:
	
	char * project_path;		// may or may not exist...
	char * project_filename;	// should always exist...
	
	vector<camera *> camera_vector;
	vector<graphics_view *> graphics_view_vector;
	vector<plotting_view *> plotting_view_vector;
	
	/**	A static counter of display list IDs.
		In some targets there can be several GUI's running simultaneously,
		so we must make sure that the display list ID's are unique -> static.
	*/
	
	static iGLu list_counter;
	
	vector<transparent_primitive> tp_vector;
	
#ifdef ENABLE_TREELIST_VIEW

	// at the moment, we have only a single treelist_view object.
	// this could be always open?!?!? is it needed??? improve MDI first...
	
	treelist_view * tlv;
	
#endif	// ENABLE_TREELIST_VIEW

	public:
	
	enum rmode { Normal = 0, Transform1 = 1, Transform2 = 2 };
	
	graphics_class_factory * graphics_factory;
	
	vector<light *> light_vector;
	vector<smart_object *> object_vector;
	dummy_object * selected_object;
	
	iGLu * select_buffer;
	
	public:
	
	docview(ostream *, graphics_class_factory &);
	virtual ~docview(void);
	
	/**	Some functions to support transparent primitives... 
		External objects (planes, surfaces, etc) should use these.
	*/
	
	bool AddTP(void *, transparent_primitive &);
	void UpdateMPsForAllTPs(void *);
	void RemoveAllTPs(void *);
	
	// some file name functions...
	
	void SetProjectPath(const char *);
	void SetProjectFileName(const char *);
	
	void SetDefaultProjectFileName(void);
	
	void ParseProjectFileNameAndPath(const char *);
	
	const char * GetProjectFileName(bool);
	const char * GetFullProjectFileName(void);
	
	// some utility functions.
	
	virtual const char * GetType(void) = 0;
	virtual color_mode * GetDefaultColorMode(void) = 0;
	
	virtual void SelectAll(void) = 0;
	virtual void InvertSelection(void) = 0;
//	virtual void SelectNone(void) = 0;		// do this as a combination of "select all" and "invert selection"???

	// how output could co-operate in different targets??? these event_???() try that...
	// how output could co-operate in different targets??? these event_???() try that...
	// how output could co-operate in different targets??? these event_???() try that...
	
	virtual void event_SelectedObjectChanged(void) = 0;
	
	iGLu GetDisplayListIDs(iGLu);
	void DeleteDisplayLists(iGLu, iGLu);
	
	// some light-related functions:
	
	i32s IsLight(const dummy_object *);
	bool SelectLight(const dummy_object *);
	
	bool AddGlobalLight(light *);
	bool AddLocalLight(light *, camera *);
	bool RemoveLight(dummy_object *);
	
	iGLs CountGlobalLights(void);
	iGLs CountLocalLights(camera *);
	
	bool SetupLights(camera *);
	
	void SetGlobalLightNumbers(void);
	void SetLocalLightNumbers(camera *);
	
	// some object-related functions:
	
	i32s IsObject(const dummy_object *);
	bool SelectObject(const dummy_object *);
	
	void AddObject(smart_object *);
	bool RemoveObject(dummy_object *);
	
	// some camera/window functions:
	
	virtual fGL GetDefaultFocus(void) = 0;
	
	graphics_view * AddGraphicsView(bool = false);			// using a new camera...
	graphics_view * AddGraphicsView(camera *, bool = false);	// using an existing camera...
	
	bool IsThisLastGraphicsView(graphics_view *);
	bool RemoveGraphicsView(graphics_view *, bool = false);
	
#ifdef ENABLE_TREELIST_VIEW

	treelist_view * CreateTreeListView(void);
	bool DestroyTreeListView(void);
	
#endif	// ENABLE_TREELIST_VIEW

	void UpdateAllViews(void);
	virtual void UpdateAllWindowTitles(void);
	
	void UpdateAllGraphicsViews(bool = false);			// virtual
	void UpdateGraphicsViews(camera *, bool = false);
	void UpdateGraphicsView(graphics_view *, bool = false);
	
	void UpdatePlottingViews(void);
	
	// update for other views?!?!?!? or all views ?!?!?!?
	// update for other views?!?!?!? or all views ?!?!?!?
	// update for other views?!?!?!? or all views ?!?!?!?
	
	/**	Contains some OpenGL initialization commands 
		(color, material, light settings). 
		The OGL selection buffer should also be assigned here!!!
	*/
	
	void InitGL(void);
	
	/// This will contain the OpenGL rendering commands that draw the 3D-view...
	
	virtual void Render(graphics_view *, rmode) = 0;
	
	void RenderAllTPs(graphics_view *, rmode);
	
	/**	This should center the transformer object to the center of all
		selected atoms, and shift the center of all selected atoms to the origo.
	*/
	
	virtual void Center(transformer *) = 0;
	
	/**	This should get the transformation matrix from the transformer
		object and use it to transform all selected atoms/bonds/whatever.
	*/
	
	virtual void Transform(transformer *) = 0;
	
	virtual void DrawEvent(graphics_view *, vector<iGLu> &) = 0;
	virtual void EraseEvent(graphics_view *, vector<iGLu> &) = 0;
	virtual void SelectEvent(graphics_view *, vector<iGLu> &) = 0;
	virtual void MeasureEvent(graphics_view *, vector<iGLu> &) = 0;
	
	// some common operations are implemented here:
	
	void DoDeleteCurrentObject(void);
	
	void DoSwitchLocalLights(camera *, bool);
	void DoSwitchGlobalLights(camera *, bool);
};

float measure_len(float *, float *);
float measure_ang(float *, float *, float *);
float measure_tor(float *, float *, float *, float *);

/*################################################################################################*/

/// The "##color_mode" class is used to handle the color detemination details.

class color_mode
{
	protected:
	
// this is just an interface definition -> there is no relevant common data?!?!?
// this is just an interface definition -> there is no relevant common data?!?!?
// this is just an interface definition -> there is no relevant common data?!?!?

	public:
	
	color_mode(void) { }
	virtual ~color_mode(void) { }
	
// is this (void *)-method really the only possibility to give a virtual function different
// parameters without the "hiding problem"???????????????????

// should be [void *, i32s CRD_SET, fGL_a4] -> crd-set can be included in coloring???
// should be [void *, i32s CRD_SET, fGL_a4] -> crd-set can be included in coloring???
// should be [void *, i32s CRD_SET, fGL_a4] -> crd-set can be included in coloring???

	virtual void GetColor(const void *, fGL_a4 &, prefs *p = 0) = 0;
};

/*################################################################################################*/

/// An utility class for transparency effects...

class transparent_primitive
{
	private:
	
	fGL z_distance; void * owner;
	transparent_primitive_data * data;
	
	public:
	
	transparent_primitive(void);
	transparent_primitive(void *, transparent_primitive_data &);
	transparent_primitive(const transparent_primitive &);
	~transparent_primitive(void);
	
	bool TestOwner(void *) const;
	transparent_primitive_data * GetData(void) const;
	
	void UpdateDistance(const fGL *, const fGL *);
	
	bool operator<(const transparent_primitive &) const;
};

class transparent_primitive_data
{
	protected:
	
	fGL midpoint[3];
	
	public:
	
	transparent_primitive_data(void);
	virtual ~transparent_primitive_data(void);
	
	virtual void Render(void) = 0;
	virtual void UpdateMP(void) = 0;
	
	friend class transparent_primitive;
};

/// A triangle with 3 colors.

class tpd_tri_3c : public transparent_primitive_data
{
	protected:
	
	fGL * color[3];
	
	fGL * point[3];
	
	public:
	
	tpd_tri_3c(fGL *, fGL *, fGL *, fGL *, fGL *, fGL *);
	~tpd_tri_3c(void);
	
	void Render(void);		// virtual
	void UpdateMP(void);		// virtual
};

/// A quadrilateral with 4 colors.

class tpd_quad_4c : public transparent_primitive_data
{
	protected:
	
	fGL * color[4];
	
	fGL * point[4];
	
	public:
	
	tpd_quad_4c(fGL *, fGL *, fGL *, fGL *, fGL *, fGL *, fGL *, fGL *);
	~tpd_quad_4c(void);
	
	void Render(void);		// virtual
	void UpdateMP(void);		// virtual
};

/*################################################################################################*/

#endif	// ENABLE_GRAPHICS
#endif	// DOCVIEW_H

// eof
