// MM1ENG1.H : molecular mechanics force field and computation engine.

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef MM1ENG1_H
#define MM1ENG1_H

struct mm1_exp1_bt1;		// bond stretching
struct mm1_exp1_bt2;		// angle bending
struct mm1_exp1_bt3;		// torsion
// out of plane...

struct mm1_exp1_nbt1;		// nonbonded

class mm1_eng_exp1a;		// bt
class mm1_eng_exp1b1;		// nbt, non-periodic
class mm1_eng_exp1b2;		// nbt, exact periodic
class mm1_eng_exp1b3;		// nbt, minimum image model

class mm1_eng_exp1;
class mm1_eng_exp1_tst;
class mm1_eng_exp1_mim;

// how to calculate pressure in periodic boxes???
// how to calculate pressure in periodic boxes???
// how to calculate pressure in periodic boxes???

// ewald summation???
// ewald summation???
// ewald summation???

// what is the most reasonable objective or strategy here???
// there are some very advanced programs (ego? gromacs? others?!?!?!) to run the simulations efficiently.
// make simple implementations of these techniques here???

/*################################################################################################*/

class exp1_tables;		// mm1tab1.h

#include "mm1eng.h"

#include <vector>
using namespace std;

/*################################################################################################*/

struct mm1_exp1_bt1
{
	i32s atmi[2];
	
	f64 opt;
	f64 fc;
	
	i32s get_atmi(i32s index, bool dir)
	{
		return atmi[dir ? index : !index];
	}
};

struct mm1_exp1_bt2
{
	i32s atmi[3];
	
	i32s index1[2];
	bool dir1[2];
	
	f64 opt;
	f64 fc;
	
	i32s get_index(i32s index, bool dir)
	{
		return index1[dir ? index : !index];
	}
	
	bool get_dir(i32s index, bool dir)
	{
		return dir1[dir ? index : !index];
	}
};

struct mm1_exp1_bt3
{
	i32s atmi[4];
	
	i32s index2[2];
	i32s index1[4];
	bool dir1[4];
	
	f64 k; f64 s;
	
	bool constraint;
};

// here we use floats to reduce memory consumption...
// also seems to be slightly faster (at least in intel-machines) than doubles.

struct mm1_exp1_nbt1
{
	i32s atmi[2];
	
	float k1; float k2;
	float qq;
};

/*################################################################################################*/

/**	molecular mechanics; bonded part.

	Parameters for this experimental force field come mainly from the Tripos 5.2 
	parameter set. Some extra patameters (like the ones for 4- and 5-membered rings) 
	come from MMFF94 parameters and are only approximate. The system responsible for 
	atomic charges also come from a separate source...
	
	The following is just a useful set of references; not all are used here (at least yet):
	
	Clark M, Cramer III RD, Van Obdenbosch N : "##Validation of the General-Purpose Tripos 
	5.2 Force Field", J. Comp. Chem. 10, 982-1012, (1989)
	
	Bekker H, Berendsen HJC, van Gunsteren WF : "##Force and virial of Torsional-Angle-Dependent 
	Potentials", J. Comp. Chem. 16, 527-533, (1995)
	
	Halgren TA : "##Merck Molecular Force Field. I. Basis, Form, Scope, Parameterization, and 
	Performance of MMFF94", J. Comp. Chem. 17, 490-, (1996) [there are many parts in this paper]
	
	Tuzun RE, Noid DW, Sumpter BG : "##Efficient computation of potential energy first and second 
	derivatives for molecular dynamics, normal coordinate analysis, and molecular mechanics 
	calculations", Macromol. Theory Simul. 5, 771-778, (1996)
	
	Tuzun RE, Noid DW, Sumpter BG : "##Efficient treatment of Out-of-Plane Bend and Improper 
	Torsion Interactions in MM2, MM3, and MM4 Molecular Mechanics Calcultions", J. Comput. Chem. 
	18, 1804-1811, (1997)
	
	Lee SH, Palmo K, Krimm S : "New Out-of-Plane Angle and Bond Angle Internal Coordinates and 
	Related Potential Energy Functions for Molecular Mechanics and Dynamics Simulations", 
	J. Comp. Chem. 20, 1067-1084, (1999)
	
	THE FOLLOWING ARE JUST FUTURE PLANS:
	
	In the long run it would be best to build up a database of QM (and experimental, where 
	available) results, which could then be used to derive (and check!!!) the forcefield 
	parameters. This force field parameter stuff will probably always be inadequate/obsolete 
	somewhat, so when parameters are missing one could gather some extra information, add it 
	to the database, check how it matches with the existing old data, and derive the new parameters...
	
	QM results can be done using MPQC. large amount of molecules/conformations are needed. 
	How to generate the conformations??? Energies must be calculated. What about 1st derivatives??? 
	Or 2nd derivatives??? What is the strategy to derive the parameters?????
*/

class mm1_eng_exp1a : virtual public mm1_eng
{
	protected:
	
	mm1_bt1_data * bt1data;
	mm1_bt2_data * bt2data;
	
	vector<mm1_exp1_bt1> bt1_vector;
	vector<mm1_exp1_bt2> bt2_vector;
	vector<mm1_exp1_bt3> bt3_vector;
	
	public:
	
	mm1_eng_exp1a(mm1_mdl &);
	virtual ~mm1_eng_exp1a(void);
	
	i32s FindTorsion(mm1_atom *, mm1_atom *, mm1_atom *, mm1_atom *);	// virtual
	bool SetTorsionConstraint(i32s, f64, f64, bool);			// virtual
	
	protected:
	
	void ComputeBT1(i32s);		// virtual
	void ComputeBT2(i32s);		// virtual
	void ComputeBT3(i32s);		// virtual
	void ComputeBT4(i32s);		// virtual
};

/*################################################################################################*/

/// molecular mechanics; nonbonded part, non-periodic.

class mm1_eng_exp1b1 : virtual public mm1_eng
{
	protected:
	
	vector<mm1_exp1_nbt1> nbt1_vector;
	
	public:
	
	mm1_eng_exp1b1(mm1_mdl &);
	virtual ~mm1_eng_exp1b1(void);
	
	protected:
	
	void ComputeNBT1(i32s);		// virtual
};

/// molecular mechanics; nonbonded part, exact periodic.

class mm1_eng_exp1b2 : virtual public mm1_eng
{
	public:		// why public!?!?!?!?
	
	i32s layers;
	
	public:
	
	mm1_eng_exp1b2(mm1_mdl &);
	virtual ~mm1_eng_exp1b2(void);
	
	protected:
	
	void CompNBTerm(i32s, i32s, i32s, bool, f64 *, bool);
	
	void ComputeNBT1(i32s);		// virtual
};

// molecular mechanics; nonbonded part, minimum image model.

class mm1_eng_exp1b3 : public mm1_eng_pbc
{
	protected:
	
	vector<mm1_exp1_nbt1> nbt1_vector;
	
	f64 sw1; f64 sw2;
	f64 swA; f64 swB;
	
	f64 shft1;
	f64 shft2;
	
	public:		// why public!?!?!?!?
	
	f64 limit;
	bool update;
	
	public:
	
	mm1_eng_exp1b3(mm1_mdl &);
	virtual ~mm1_eng_exp1b3(void);
	
	protected:
	
	void UpdateTerms(void);
	
	void ComputeNBT1(i32s);		// virtual
};

/*################################################################################################*/

class mm1_eng_exp1 : public mm1_eng_exp1a, public mm1_eng_exp1b1
{
	public:
	
	mm1_eng_exp1(mm1_mdl & p1) : mm1_eng_exp1a(p1), mm1_eng_exp1b1(p1), mm1_eng(p1)
	{
	}
};

class mm1_eng_exp1_tst : public mm1_eng_exp1a, public mm1_eng_exp1b2
{
	public:
	
	mm1_eng_exp1_tst(mm1_mdl & p1) : mm1_eng_exp1a(p1), mm1_eng_exp1b2(p1), mm1_eng(p1)
	{
	}
};

class mm1_eng_exp1_mim : public mm1_eng_exp1a, public mm1_eng_exp1b3
{
	public:
	
	mm1_eng_exp1_mim(mm1_mdl & p1) : mm1_eng_exp1a(p1), mm1_eng_exp1b3(p1), mm1_eng(p1)
	{
	}
};

/*################################################################################################*/

#endif	// MM1ENG1_H

// eof
