// MM1DOCV.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm1docv.h"	// config.h is here -> we get ENABLE-macros here...

#ifdef ENABLE_GRAPHICS
#include "mm1alg.h"
#include "mm1eng1.h"
#include "mm1eng9.h"

#include "mm1rbn.h"

#include "plane.h"
#include "surface.h"

#include "color.h"
#include "views.h"

#include <fstream>
#include <strstream>
using namespace std;

/*################################################################################################*/

mm1_cm_element mm1_docv::cm_element = mm1_cm_element();
mm1_cm_state mm1_docv::cm_state = mm1_cm_state();

mm1_docv::mm1_docv(ostream * p1, graphics_class_factory & p2) :
	docview(p1, p2), mm1_mdl(p1, p2), model_simple(p1, p2)
{
}

mm1_docv::~mm1_docv(void)
{
}

fGL mm1_docv::GetDefaultFocus(void)
{
	return 2.0;
}

const char * mm1_docv::GetType(void)
{
	return "MM, all-atoms model";
}

color_mode * mm1_docv::GetDefaultColorMode(void)
{
	return & mm1_docv::cm_element;
}

void mm1_docv::SelectAll(void)
{
	if (selected_object != NULL)
	{
		selected_object = NULL;
		event_SelectedObjectChanged();
	}
	
	iter_mm1al it1 = atom_list.begin();
	while (it1 != atom_list.end()) (* it1++).selected = true;
	
	UpdateAllGraphicsViews();
}

void mm1_docv::InvertSelection(void)
{
	if (selected_object != NULL)
	{
		selected_object = NULL;
		event_SelectedObjectChanged();
	}
	
	iter_mm1al it1 = atom_list.begin();
	while (it1 != atom_list.end())
	{
		bool flag = (* it1).selected;
		(* it1++).selected = !flag;
	}
	
	UpdateAllGraphicsViews();
}

//void mm1_docv::SelectNone(void)
//{
//	if (selected_object != NULL)
//	{
//		selected_object = NULL;
//		event_SelectedObjectChanged();
//	}
//	
//	iter_mm1al it1 = atom_list.begin();
//	while (it1 != atom_list.end()) (* it1++).selected = false;
//	
//	UpdateAllGraphicsViews();
//}

bool mm1_docv::TestAtom(mm1_atom * ref, rmode rm)
{
	// what about the "visible"-flag???
	
	if (rm == Transform1 && ref->selected) return false;
	if (rm == Transform2 && !ref->selected) return false;
	
	return true;
}

bool mm1_docv::TestBond(mm1_bond * ref, rmode rm)
{
	// what about the "visible"-flag???
	
	if (rm == Transform1 && ref->atmr[0]->selected) return false;
	if (rm == Transform2 && !ref->atmr[0]->selected) return false;
	
	bool test = (ref->atmr[0]->selected != ref->atmr[1]->selected);
	if (rm != Normal && test) return false;
	
	return true;
}

void mm1_docv::SetColor(color_mode * cm, mm1_atom * ref)
{
	const fGL default_sel_color[3] = { 1.0, 0.0, 1.0 };
	fGL *select_color = model_prefs->ColorRGB("Graphics/SelectColor", default_sel_color);

	if (ref->selected) glColor3f(select_color[0], select_color[1], select_color[2]);
	else
	{
		fGL_a4 color;
		cm->GetColor(ref, color, model_prefs);
		glColor3fv(color);
	}

	delete [] select_color;
}

// it seems that there is something common for all models (like the sel-buffer) -> move to the docview?!?!
// it seems that there is something common for all models (like the sel-buffer) -> move to the docview?!?!
// it seems that there is something common for all models (like the sel-buffer) -> move to the docview?!?!

void mm1_docv::Render(graphics_view * gv, rmode rm)
{
	const fGL default_label_color[3] = { 0.0, 1.0, 1.0 };	// looks bad but won't fade easily into other colours...
	fGL *label_color = model_prefs->ColorRGB("Graphics/LabelColor", default_label_color);
	
	bool accum = gv->accumulate; if (rm != Normal) accum = false;
//if (accum) { glClear(GL_ACCUM_BUFFER_BIT); UpdateAccumValues(); }
//else if (rm != Transform2) glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	if (periodic && rm == Normal)
	{
		glLineWidth(1.0);
		glColor3f(1.0, 0.0, 1.0);
		glBegin(GL_LINES);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], +box_hdim[2]);
		glVertex3f(+box_hdim[0], -box_hdim[1], +box_hdim[2]);
		
		glVertex3f(-box_hdim[0], +box_hdim[1], +box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], -box_hdim[1], -box_hdim[2]);
		
		glVertex3f(-box_hdim[0], +box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], -box_hdim[2]);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(-box_hdim[0], -box_hdim[1], +box_hdim[2]);
		
		glVertex3f(-box_hdim[0], +box_hdim[1], -box_hdim[2]);
		glVertex3f(-box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glVertex3f(+box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], -box_hdim[1], +box_hdim[2]);
		
		glVertex3f(+box_hdim[0], +box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(-box_hdim[0], +box_hdim[1], -box_hdim[2]);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], +box_hdim[2]);
		glVertex3f(-box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glVertex3f(+box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], -box_hdim[2]);
		
		glVertex3f(+box_hdim[0], -box_hdim[1], +box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glEnd();
	}
	
	if (gv->enable_fog) glEnable(GL_FOG);
	
	i32s layers = 0;
	if (periodic && rm == Normal) layers = 1;
	
	for (i32s r1 = -layers;r1 < (layers + 1);r1++)
	{
		for (i32s r2 = -layers;r2 < (layers + 1);r2++)
		{
			for (i32s r3 = -layers;r3 < (layers + 1);r3++)
			{
				glPushMatrix();
				
				fGL trans1 = r1 * box_fdim[0];
				fGL trans2 = r2 * box_fdim[1];
				fGL trans3 = r3 * box_fdim[2];
				
				glTranslated(trans1, trans2, trans3);
				
				RenderScene(gv, rm, accum);
				
				glPopMatrix();
			}
		}
	}
	
	if (accum) glAccum(GL_RETURN, 1.0);
	else if (rm != Transform2) gv->cam->RenderObjects(gv);
	
	if (gv->label == LABEL_INDEX)
	{
		char string[32]; i32s tmp1 = 0;
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << tmp1++ << ends;
			
			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	else if (gv->label == LABEL_CHARGE)
	{
		char string[32];
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str.setf(ios::fixed | ios::showpos); str.precision(4); str << (* it1).charge << ends;

			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	else if (gv->label == LABEL_ELEMENT)
	{
		char string[32];
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << (* it1).el.GetSymbol() << ends;

			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	else if (gv->label == LABEL_ATOMTYPE)
	{
		char string[32];
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << "0x" << hex << (* it1).atmtp << ends;

			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	else if (gv->label == LABEL_BONDTYPE)
	{
		char string[32];
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1bl it1 = bond_list.begin();it1 != bond_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << (* it1).bt.GetSymbol1() << ends;

			fGL x = ((* it1).atmr[0]->crd_vector[0][0] + (* it1).atmr[1]->crd_vector[0][0]) / 2.0;
			fGL y = ((* it1).atmr[0]->crd_vector[0][1] + (* it1).atmr[1]->crd_vector[0][1]) / 2.0;
			fGL z = ((* it1).atmr[0]->crd_vector[0][2] + (* it1).atmr[1]->crd_vector[0][2]) / 2.0;
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	
	if (gv->enable_fog) glDisable(GL_FOG);
	
	// finally call this to handle transparency...
	// finally call this to handle transparency...
	// finally call this to handle transparency...
	
	RenderAllTPs(gv, rm);

	delete [] label_color;
}

void mm1_docv::RenderScene(graphics_view * gv, rmode rm, bool accum)
{
	for (i32u n1 = 0;n1 < cs_vector.size();n1++)
	{
		if (!GetCRDSetVisible(n1)) continue;
if (accum) glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);		// FIXME!!!

		if (gv->render == RENDER_WIREFRAME)
		{
			glPointSize(3.0); glLineWidth(1.0);
			for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)		// wireframe atoms
			{
				if (!TestAtom(& (* it1), rm)) continue;
				glPushName(GLNAME_MD_TYPE1); glPushName((i32u) & (* it1));
				
				glBegin(GL_POINTS);
				SetColor(gv->colormode, & (* it1));
				glVertex3fv((* it1).crd_vector[n1].data);
				glEnd();
				
				glPopName(); glPopName();
			}
			
			glEnable(GL_LINE_STIPPLE);
			for (iter_mm1bl it2 = bond_list.begin();it2 != bond_list.end();it2++)		// wireframe bonds
			{
				if (!TestBond(& (* it2), rm)) continue;
				
				switch ((* it2).bt.GetSymbol1())
				{
					case 'S': glLineStipple(1, 0xFFFF); break;
					case 'C': glLineStipple(1, 0x3FFF); break;
					case 'D': glLineStipple(1, 0x3F3F); break;
					case 'T': glLineStipple(1, 0x3333); break;
				}
				
				glBegin(GL_LINES);
				SetColor(gv->colormode, (* it2).atmr[0]);
				glVertex3fv((* it2).atmr[0]->crd_vector[n1].data);
				SetColor(gv->colormode, (* it2).atmr[1]);
				glVertex3fv((* it2).atmr[1]->crd_vector[n1].data);
				glEnd();
			}
			glDisable(GL_LINE_STIPPLE);
		}
		
		if (gv->render != RENDER_WIREFRAME && gv->render != RENDER_NOTHING)
		{
			glEnable(GL_LIGHTING);
			
			for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)		// atoms as spheres
			{
				if (!TestAtom(& (* it1), rm)) continue;
				
				SetColor(gv->colormode, & (* it1));
				
				float rad = 0.0; int res = 0;
				switch (gv->render)
				{
					case RENDER_BALL_AND_STICK:
					rad = model_prefs->Double("MM1Graphics/BallSize", 0.035);
					if (model_prefs->Boolean("MM1Graphics/BallVdWScale", false))
					    rad *= (* it1).el.GetVDWRadius() * 4.0;
					res = model_prefs->Value("MM1Graphics/BallResolution", 12);
					break;
					
					case RENDER_VAN_DER_WAALS:
					rad = (* it1).el.GetVDWRadius();
					res = model_prefs->Value("MM1Graphics/VdWResolution", 22);
					break;
					
					case RENDER_CYLINDERS:
					rad = model_prefs->Double("MM1Graphics/CylinderSize", 0.035);
					res = model_prefs->Value("MM1Graphics/CylinderResolution", 12);
					break;
				}
				
				glPushName(GLNAME_MD_TYPE1); glPushName((i32u) & (* it1));
				
				GLUquadricObj * qo = gluNewQuadric();
				gluQuadricDrawStyle(qo, (GLenum) GLU_FILL);
				
				glPushMatrix();
				glTranslated((* it1).crd_vector[n1][0], (* it1).crd_vector[n1][1], (* it1).crd_vector[n1][2]);
				gluSphere(qo, rad, res, res / 2);
				glPopMatrix();
				gluDeleteQuadric(qo);
				
				glPopName(); glPopName();
			}
			
			glDisable(GL_LIGHTING);
		}
		
		if (gv->render == RENDER_BALL_AND_STICK || gv->render == RENDER_CYLINDERS)
		{
			glEnable(GL_LIGHTING);
			
			for (iter_mm1bl it1 = bond_list.begin();it1 != bond_list.end();it1++)		// bonds as cylinders
			{
				if (!TestBond(& (* it1), rm)) continue;
				
				fGL vdwr[2] =
				{
					(* it1).atmr[0]->el.GetVDWRadius(),
					(* it1).atmr[1]->el.GetVDWRadius()
				};
				
				fGL vdwrsum = vdwr[0] + vdwr[1];
				
				for (i32s n2 = 0;n2 < 2;n2++)
				{
					fGL * crd1 = (* it1).atmr[n2]->crd_vector[n1].data;
					fGL * crd2 = (* it1).atmr[!n2]->crd_vector[n1].data;
					v3d<fGL> crt1 = v3d<fGL>(crd1);
					v3d<fGL> crt2 = v3d<fGL>(crd2);
					v3d<fGL> crt = crt2 - crt1;
					
					fGL pol[3]; crt2pol(crt.data, pol);
					
					SetColor(gv->colormode, (* it1).atmr[n2]);
					
					float trans, rad = 0.0; int res = 0;
					switch (gv->render)
					{
						case RENDER_BALL_AND_STICK:
						rad = model_prefs->Double("MM1Graphics/StickSize", 0.01);
						res = model_prefs->Value("MM1Graphics/StickResolution", 6);
						break;
						
						case RENDER_CYLINDERS:
						rad = model_prefs->Double("MM1Graphics/CylinderSize", 0.035);
						res = model_prefs->Value("MM1Graphics/CylinderResolution", 12);
						break;
					}
					
					glPushName(GLNAME_MD_TYPE1); glPushName((i32u) (* it1).atmr[n2]);
					
					GLUquadricObj * qo = gluNewQuadric();
					gluQuadricDrawStyle(qo, (GLenum) GLU_FILL);
					glPushMatrix();
					
					glTranslated(crd1[0], crd1[1], crd1[2]);
					
					glRotated(180.0 * pol[1] / M_PI, 0.0, 1.0, 0.0);
					glRotated(180.0 * pol[2] / M_PI, sin(-pol[1]), 0.0, cos(-pol[1]));
					
					// any chance to further define the orientation of, for example, double bonds???
					// one more rotation would be needed. but what is the axis, and how much to rotate???
					
					fGL length = crt.len() * vdwr[n2] / vdwrsum;
					
					if (gv->render == RENDER_BALL_AND_STICK)
					switch ((* it1).bt.GetValue())
					{
						case BONDTYPE_DOUBLE:
						trans = rad;
						rad = rad / 1.5;
						
						if (n2)
							glTranslated(0.0, trans, 0.0);
						else
							glTranslated(0.0, -trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);					
						if (n2)
							glTranslated(0.0, -2.0 * trans, 0.0);
						else
							glTranslated(0.0, 2.0 * trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);
						break;
						
						case BONDTYPE_CNJGTD:
						trans = rad;
						rad = rad / 1.5;
						
						if (n2)
							glTranslated(0.0, trans, 0.0);
						else
							glTranslated(0.0, -trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);
						if (n2)
							glTranslated(0.0, -2.0 * trans, 0.0);
						else
							glTranslated(0.0, 2.0 * trans, 0.0);
						
						glEnable(GL_LINE_STIPPLE);
						glLineStipple(1, 0x3F3F);
						gluQuadricDrawStyle(qo, (GLenum) GLU_LINE);
						gluCylinder(qo, rad, rad, length, res, 1);
						glDisable(GL_LINE_STIPPLE);
						break;
						
						case BONDTYPE_TRIPLE:
						trans = rad;
						rad = rad / 2.0;
						
						if (n2)
							glTranslated(0.0, trans, 0.0);
						else
							glTranslated(0.0, -trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);
						if (n2)
							glTranslated(0.0, -trans, 0.0);
						else
							glTranslated(0.0, trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);
						if (n2)
							glTranslated(0.0, -trans, 0.0);
						else
							glTranslated(0.0, trans, 0.0);
						gluCylinder(qo, rad, rad, length, res,1);
						break;
						
						default:
						gluCylinder(qo, rad, rad, length, res, 1);
					}
					else
						gluCylinder(qo, rad, rad, length, res, 1);
					
					glPopMatrix();
					gluDeleteQuadric(qo);
					
					glPopName(); glPopName();
				}
			}
			
			glDisable(GL_LIGHTING);
		}
		
		if (accum)
		{
			gv->cam->RenderObjects(gv);
			glAccum(GL_ACCUM, cs_vector[n1]->accum_value);
		}
	}
}

void mm1_docv::Center(transformer * p1)
{
	i32s sum = 0;
	p1->GetLocDataRW()->crd[0] = 0.0;
	p1->GetLocDataRW()->crd[1] = 0.0;
	p1->GetLocDataRW()->crd[2] = 0.0;
	
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			sum++;
			p1->GetLocDataRW()->crd[0] += (* it1).crd_vector[n1][0];
			p1->GetLocDataRW()->crd[1] += (* it1).crd_vector[n1][1];
			p1->GetLocDataRW()->crd[2] += (* it1).crd_vector[n1][2];
		}
	}
	
	if (!sum) return;
	
	p1->GetLocDataRW()->crd[0] /= (fGL) sum;
	p1->GetLocDataRW()->crd[1] /= (fGL) sum;
	p1->GetLocDataRW()->crd[2] /= (fGL) sum;
	
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			(* it1).crd_vector[n1][0] -= p1->GetLocData()->crd[0];
			(* it1).crd_vector[n1][1] -= p1->GetLocData()->crd[1];
			(* it1).crd_vector[n1][2] -= p1->GetLocData()->crd[2];
		}
	}
}

void mm1_docv::Transform(transformer * p1)
{
	fGL matrix[16]; p1->GetMatrix(matrix);
	
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			v3d<fGL> posv = v3d<fGL>((* it1).crd_vector[n1].data);
			TransformVector(posv, matrix);
			
			(* it1).crd_vector[n1][0] = posv[0];
			(* it1).crd_vector[n1][1] = posv[1];
			(* it1).crd_vector[n1][2] = posv[2];
		}
	}
}

void mm1_docv::DrawEvent(graphics_view * gv, vector<iGLu> & names)
{
	if (ogl_view::button == mouse_tool::Right) return;	// the right button is for popup menus...
	
	i32s mouse[2] =
	{
		gv->current_tool->latest_x,
		gv->current_tool->latest_y
	};
	
	if (ogl_view::state == mouse_tool::Down)
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[0] = (mm1_atom *) names[1];
		}
		else
		{
			fGL tmp1[3]; gv->GetCRD(mouse, tmp1);
			mm1_atom newatom(element::current_element, tmp1, cs_vector.size());
			AddAtom(newatom); draw_data[0] = & atom_list.back();
		}
	}
	else
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[1] = (mm1_atom *) names[1];
		}
		else
		{
			fGL tmp1[3]; gv->GetCRD(mouse, tmp1);
			mm1_atom newatom(element::current_element, tmp1, cs_vector.size());
			AddAtom(newatom); draw_data[1] = & atom_list.back();
		}
		
		// if different: update bondtype or add a new bond.
		// if not different: change atom to different element.
		
		if (draw_data[0] != draw_data[1])
		{
			mm1_bond newbond(draw_data[0], draw_data[1], bondtype::current_bondtype);
			iter_mm1bl it1 = find(bond_list.begin(), bond_list.end(), newbond);
			if (it1 != bond_list.end()) (* it1).bt = bondtype::current_bondtype;
			else AddBond(newbond);
		}
		else
		{
			draw_data[0]->el = element::current_element;
		}
		
		UpdateAllGraphicsViews();
	}
}

void mm1_docv::EraseEvent(graphics_view * gv, vector<iGLu> & names)
{
	if (ogl_view::button == mouse_tool::Right) return;	// the right button is for popup menus...
	
	if (ogl_view::state == mouse_tool::Down)
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[0] = (mm1_atom *) names[1];
		}
		else
		{
			draw_data[0] = NULL;
		}
	}
	else
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[1] = (mm1_atom *) names[1];
		}
		else
		{
			draw_data[1] = NULL;
		}
		
		if (!draw_data[0] || !draw_data[1]) return;
		
		// if different: try to find and remove a bond.
		// if not different: remove atom.
		
		if (draw_data[0] != draw_data[1])
		{
			mm1_bond tmpbond(draw_data[0], draw_data[1], bondtype::current_bondtype);
			iter_mm1bl it1 = find(bond_list.begin(), bond_list.end(), tmpbond);
			if (it1 != bond_list.end()) RemoveBond(it1); else return;
		}
		else
		{
			iter_mm1al it1 = find(atom_list.begin(), atom_list.end(), (* draw_data[0]));
			if (it1 != atom_list.end()) RemoveAtom(it1); else exit(EXIT_FAILURE);
		}
		
		UpdateAllGraphicsViews();
	}
}

void mm1_docv::SelectEvent(graphics_view *, vector<iGLu> & names)
{
	if (names[0] == GLNAME_MD_TYPE1)
	{
		mm1_atom * ref = (mm1_atom *) names[1];
		ref->selected = !ref->selected;
		UpdateAllGraphicsViews();
		
		//		cout << "atomtype is " << hex << ref->atmtp << dec << endl;
	}
}

void mm1_docv::MeasureEvent(graphics_view *, vector<iGLu> & names)
{
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	
	static mm1_atom *a1 = NULL;
	static mm1_atom *a2 = NULL;
	static mm1_atom *a3 = NULL;
// this function is quite similar in mm1_docv and qm1_docv. could this be shorter as a loop???

	if (names[0] == GLNAME_MD_TYPE1)
	{
		mm1_atom * ref = (mm1_atom *) names[1];
		ref->selected = !ref->selected;
		UpdateAllGraphicsViews();
		
		if (a1 == NULL)
		  {
		    a1 = ref;
		    str1 << "charge: " << ref->charge << endl << ends;
		    PrintToLog(mbuff1);
		  }
		else if (a1 != NULL && a2 == NULL)
		  {
		    if (a1 == ref) { a1->selected = false; a1 = NULL; return; }
		    
		    a2 = ref;
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    
		    float len = measure_len(p1, p2);
		    str1 << "distance: " << len << " nm" << endl << ends;
		    PrintToLog(mbuff1);
		  }
		else if (a1 != NULL && a2 != NULL && a3 == NULL)
		  {
		    if (a1 == ref) { a1->selected = false; a1 = a2; a2 = NULL; return; }
		    else if (a2 == ref) { a2->selected = false; a2 = NULL; return; }
		    
		    a3 = ref;
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    float * p3 = a3->crd_vector[0].data;
		    
		    float ang = measure_ang(p1, p2, p3);
		    str1 << "angle: " << ang << " deg" << endl << ends;
		    PrintToLog(mbuff1);
		  }
		else
		  {
		    if (a1 == ref) { a1->selected = false; a1 = a2; a2 = a3; a3 = NULL; return; }
		    else if (a2 == ref) { a2->selected = false; a2 = a3; a3 = NULL; return; }
		    else if (a3 == ref) { a3->selected = false; a3 = NULL; return; }
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    float * p3 = a3->crd_vector[0].data;
		    float * p4 = ref->crd_vector[0].data;
		    
		    float tor = measure_tor(p1, p2, p3, p4);
		    str1 << "torsion: " << tor << " deg " << endl << ends;
		    PrintToLog(mbuff1);
		    
		    a1->selected = false; a1 = NULL;
		    a2->selected = false; a2 = NULL;
		    a3->selected = false; a3 = NULL;
		    ref->selected = false;
		    
		    UpdateAllGraphicsViews();
		  }
	}
}

void mm1_docv::ProcessCommandString(graphics_view * gv, const char * command)
{
	char mbuff1[512];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Processing Command : " << command << endl << ends;
	PrintToLog(mbuff1);
	
	bool retval = ProcessCommonCommands(gv, command);
	if (retval == true) return;
	
	istrstream istr(command);
	char kw1[32]; istr >> kw1;
	
	// is it "help"??? if is, then append...
	
	if (!strcmp("help", kw1))
	{
		PrintToLog("> add plane <vf> <cf> <cscale> <dim> <res> <tp> <alpha> -- add a plane object.\n");
		PrintToLog(">   where: <vf> = value function : esp vdws dist unity\n");
		PrintToLog(">          <cf> = colour function : red green blue rb1 rb2\n");
		PrintToLog(">          <cscale> = scaling value for calculating the colours\n");
		PrintToLog(">          <dim> = dimension of the plane object (in nm units)\n");
		PrintToLog(">          <res> = resolution of the plane object\n");
		PrintToLog(">          <tp> = 0 or 1 telling if the object is transparent\n");
		PrintToLog(">          <alpha> = transparency alpha value\n");
		
		PrintToLog("> add volrend <vf> <cf> <cscale> <dim> <res> <alpha> -- add a volume-rendering object.\n");
		PrintToLog(">   where: <vf> = value function : esp vdws dist unity\n");
		PrintToLog(">          <cf> = colour function : red green blue rb1 rb2\n");
		PrintToLog(">          <cscale> = scaling value for calculating the colours\n");
		PrintToLog(">          <dim> = dimension of the plane object (in nm units)\n");
		PrintToLog(">          <res> = resolution of the plane object\n");
		PrintToLog(">          <alpha> = transparency alpha value\n");
		
		PrintToLog("> add surf1 <vf1> <vf2> <cf> <sscale> <cscale> <dim> <res> <solid> <tp> <alpha> -- add a single surface object.\n");
		PrintToLog(">   where: <vf1> = value function for calculating the surface : esp vdws dist unity\n");
		PrintToLog(">          <vf2> = value function for calculating the colours : esp vdws dist unity\n");
		PrintToLog(">          <cf> = colour function : red green blue rb1 rb2\n");
		PrintToLog(">          <sscale> = scaling value for calculating the surface\n");
		PrintToLog(">          <cscale> = scaling value for calculating the colours\n");
		PrintToLog(">          <dim> = dimension of the plane object (in nm units)\n");
		PrintToLog(">          <res> = resolution of the plane object\n");
		PrintToLog(">          <solid> = 0 or 1 telling if the object is solid\n");
		PrintToLog(">          <tp> = 0 or 1 telling if the object is transparent\n");
		PrintToLog(">          <alpha> = transparency alpha value\n");
		
		PrintToLog("> add surf2 <vf1> <vf2> <cf1> <cf2> <sscale1> <sscale2> <cscale> <dim> <res> <solid> <tp> <alpha> -- add a pair of surface objects.\n");
		PrintToLog(">   where: <vf1> = value function for calculating the surface : esp vdws dist unity\n");
		PrintToLog(">          <vf2> = value function for calculating the colours : esp vdws dist unity\n");
		PrintToLog(">          <cf1> = colour function for 1st surface : red green blue rb1 rb2\n");
		PrintToLog(">          <cf2> = colour function for 2nd surface : red green blue rb1 rb2\n");
		PrintToLog(">          <sscale1> = scaling value for calculating the surface for 1st surface\n");
		PrintToLog(">          <sscale2> = scaling value for calculating the surface for 2nd surface\n");
		PrintToLog(">          <cscale> = scaling value for calculating the colours\n");
		PrintToLog(">          <dim> = dimension of the plane object (in nm units)\n");
		PrintToLog(">          <res> = resolution of the plane object\n");
		PrintToLog(">          <solid> = 0 or 1 telling if the object is solid\n");
		PrintToLog(">          <tp> = 0 or 1 telling if the object is transparent\n");
		PrintToLog(">          <alpha> = transparency alpha value\n");
		
		PrintToLog("> random_search <cycles> <optsteps> -- perform a random conformational search.\n");
		PrintToLog("> systematic_search <divisions> <optsteps> -- perform a systematic conformational search.\n");
		PrintToLog("> montecarlo_search <init_cycles> <simul_cycles> <optsteps> -- perform a MonteCarlo search.\n");
		
		PrintToLog("> make_plot1 A B C D <div> <start_ang> <end_ang> <optsteps> (qmeng qmc) -- create a 1D energy vs. torsion plot.\n");
		PrintToLog("> make_plot2 A B C D <div> <start_ang> <end_ang> I J K L <div> <start_ang> <end_ang> <optsteps> (qmeng qmc) -- create a 2D energy vs. torsions plot.\n");
		
		PrintToLog("> build_amino <sequence> (helix/strand) -- amino acid sequence builder.\n");
		PrintToLog("> build_nucleic <sequence> -- nucleic acid sequence builder.\n");
		
		return;
	}
	
	// process other commands.
	// process other commands.
	// process other commands.
	
	if (!strcmp("add", kw1))
	{
		char kw2[32]; istr >> kw2;	// the 2nd keyword; type of the object to add.
		char object_name[128];
		
		if (!strcmp("plane", kw2))
		{
			char kw3[32]; istr >> kw3;	// vf
			char kw4[32]; istr >> kw4;	// cf
			char kw5[32]; istr >> kw5;	// cscale
			char kw6[32]; istr >> kw6;	// dim
			char kw7[32]; istr >> kw7;	// res
			char kw8[32]; istr >> kw8;	// tp
			char kw9[32]; istr >> kw9;	// alpha
			char ** endptr = NULL;
			
			cp_param cpp;
			cpp.docv = this; cpp.ref = (mm1_docv *) this;
			
			if (!strcmp(kw3, "esp")) cpp.vf = (ValueFunction *) mm1_GetESPValue;
			else if (!strcmp(kw3, "vdws")) cpp.vf = (ValueFunction *) mm1_GetVDWSValue;
			else if (!strcmp(kw3, "dist")) cpp.vf = (ValueFunction *) GetDistance;
			else if (!strcmp(kw3, "unity")) cpp.vf = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add plane : unknown value function.\n"); return; }
			
			if (!strcmp(kw4, "red")) cpp.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw4, "green")) cpp.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw4, "blue")) cpp.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw4, "rb1")) cpp.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw4, "rb2")) cpp.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add plane : unknown colour function.\n"); return; }
			
			f64 cscale = strtod(kw5, endptr);
			
			f64 dim = strtod(kw6, endptr);
			
			i32s res = strtol(kw7, endptr, 10);
			if (res < 2) res = 2;
			
			i32s tp = strtol(kw8, endptr, 10);
			if (tp < 0) tp = 0; if (tp > 1) tp = 1;
			
			f64 alpha = strtod(kw9, endptr);
			
			cpp.value = cscale; cpp.dim = dim; cpp.np = res;
			cpp.transparent = tp; cpp.alpha = alpha;
			
			ostrstream strN(object_name, sizeof(object_name));
			strN << kw3 << "-" << ends;
			
			AddObject(new color_plane_object(ol_static(), cpp, object_name));
			UpdateAllGraphicsViews();
			
			ostrstream strR(mbuff1, sizeof(mbuff1));
			strR << "Added a new object : plane (" << kw3 << " " << kw4 << ")." << endl << ends;
			PrintToLog(mbuff1);
			return;
		}
		
		if (!strcmp("volrend", kw2))
		{
			char kw3[32]; istr >> kw3;	// vf
			char kw4[32]; istr >> kw4;	// cf
			char kw5[32]; istr >> kw5;	// cscale
			char kw6[32]; istr >> kw6;	// dim
			char kw7[32]; istr >> kw7;	// res
			char kw8[32]; istr >> kw8;	// alpha
			char ** endptr = NULL;
			
			cp_param cpp;
			cpp.docv = this; cpp.ref = (mm1_docv *) this;
			
			if (!strcmp(kw3, "esp")) cpp.vf = (ValueFunction *) mm1_GetESPValue;
			else if (!strcmp(kw3, "vdws")) cpp.vf = (ValueFunction *) mm1_GetVDWSValue;
			else if (!strcmp(kw3, "dist")) cpp.vf = (ValueFunction *) GetDistance;
			else if (!strcmp(kw3, "unity")) cpp.vf = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add volrend : unknown value function.\n"); return; }
			
			if (!strcmp(kw4, "red")) cpp.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw4, "green")) cpp.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw4, "blue")) cpp.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw4, "rb1")) cpp.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw4, "rb2")) cpp.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add volrend : unknown colour function.\n"); return; }
			
			f64 cscale = strtod(kw5, endptr);
			
			f64 dim = strtod(kw6, endptr);
			
			i32s res = strtol(kw7, endptr, 10);
			if (res < 4) res = 4;
			
			f64 alpha = strtod(kw8, endptr);
			
			cpp.value = cscale; cpp.dim = dim; cpp.np = res;
			cpp.transparent = true; cpp.alpha = alpha;
			
			ostrstream strN(object_name, sizeof(object_name));
			strN << kw3 << "-" << ends;
			
			AddObject(new volume_rendering_object(ol_static(), cpp, res / 2, dim / 2.0, (* gv->cam), object_name));
			UpdateAllGraphicsViews();
			
			ostrstream strR(mbuff1, sizeof(mbuff1));
			strR << "Added a new object : volrend (" << kw3 << " " << kw4 << ")." << endl << ends;
			PrintToLog(mbuff1);
			return;
		}
		
		if (!strcmp("surf1", kw2))
		{
			char kw3[32]; istr >> kw3;	// vf1
			char kw4[32]; istr >> kw4;	// vf2
			char kw5[32]; istr >> kw5;	// cf
			char kw6[32]; istr >> kw6;	// sscale
			char kw7[32]; istr >> kw7;	// cscale
			char kw8[32]; istr >> kw8;	// dim
			char kw9[32]; istr >> kw9;	// res
			char kwA[32]; istr >> kwA;	// solid
			char kwB[32]; istr >> kwB;	// tp
			char kwC[32]; istr >> kwC;	// alpha
			char ** endptr = NULL;
			
			cs_param csp1; csp1.docv = this; csp1.ref = (mm1_docv *) this; csp1.next = NULL;
			
			if (!strcmp(kw3, "esp")) csp1.vf1 = (ValueFunction *) mm1_GetESPValue;
			else if (!strcmp(kw3, "vdws")) csp1.vf1 = (ValueFunction *) mm1_GetVDWSValue;
			else if (!strcmp(kw3, "dist")) csp1.vf1 = (ValueFunction *) GetDistance;
			else if (!strcmp(kw3, "unity")) csp1.vf1 = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add surf1 : unknown value function 1.\n"); return; }
			
			if (!strcmp(kw4, "esp")) csp1.vf2 = (ValueFunction *) mm1_GetESPValue;
			else if (!strcmp(kw4, "vdws")) csp1.vf2 = (ValueFunction *) mm1_GetVDWSValue;
			else if (!strcmp(kw4, "dist")) csp1.vf2 = (ValueFunction *) GetDistance;
			else if (!strcmp(kw4, "unity")) csp1.vf2 = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add surf1 : unknown value function 2.\n"); return; }
			
			if (!strcmp(kw5, "red")) csp1.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw5, "green")) csp1.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw5, "blue")) csp1.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw5, "rb1")) csp1.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw5, "rb2")) csp1.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add surf1 : unknown colour function.\n"); return; }
			
			f64 sscale = strtod(kw6, endptr);
			f64 cscale = strtod(kw7, endptr);
			
			f64 dim = strtod(kw8, endptr);
			
			i32s res = strtol(kw9, endptr, 10);
			if (res < 4) res = 4;
			
			i32s solid = strtol(kwA, endptr, 10);
			if (solid < 0) solid = 0; if (solid > 1) solid = 1;

			i32s tp = strtol(kwB, endptr, 10);
			if (tp < 0) tp = 0; if (tp > 1) tp = 1;
			
			f64 alpha = strtod(kwC, endptr);
			
			static fGL dim_arr[3];
			dim_arr[0] = dim_arr[1] = dim_arr[2] = dim;
			
			static i32s res_arr[3];
			res_arr[0] = res_arr[1] = res_arr[2] = res;
			
			csp1.svalue = sscale; csp1.cvalue = cscale;
			csp1.dim = dim_arr; csp1.np = res_arr; csp1.wireframe = !solid;
			csp1.transparent = tp; csp1.alpha = alpha; csp1.toler = fabs(1.0e-6 * sscale); csp1.maxc = 250;
			
			ostrstream strN(object_name, sizeof(object_name));
			strN << kw3 << "-" << kw4 << "-" << ends;
			
			AddObject(new color_surface_object(ol_static(), csp1, object_name));
			UpdateAllGraphicsViews();
			
			ostrstream strR(mbuff1, sizeof(mbuff1));
			strR << "Added a new object : surf1 (" << kw3 << " " << kw4 << " " << kw5 << ")." << endl << ends;
			PrintToLog(mbuff1);
			return;
		}
		
		if (!strcmp("surf2", kw2))
		{
			char kw3[32]; istr >> kw3;	// vf1
			char kw4[32]; istr >> kw4;	// vf2
			char kw5[32]; istr >> kw5;	// cf1
			char kw6[32]; istr >> kw6;	// cf2
			char kw7[32]; istr >> kw7;	// sscale1
			char kw8[32]; istr >> kw8;	// sscale2
			char kw9[32]; istr >> kw9;	// cscale
			char kwA[32]; istr >> kwA;	// dim
			char kwB[32]; istr >> kwB;	// res
			char kwC[32]; istr >> kwC;	// solid
			char kwD[32]; istr >> kwD;	// tp
			char kwE[32]; istr >> kwE;	// alpha
			char ** endptr = NULL;
			
			cs_param csp2a; cs_param csp2b;
			
			csp2a.docv = this; csp2a.ref = (mm1_docv *) this; csp2a.next = & csp2b;

			csp2b.docv = this; csp2b.ref = (mm1_docv *) this; csp2b.next = NULL;
			
			if (!strcmp(kw3, "esp")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) mm1_GetESPValue;
			else if (!strcmp(kw3, "vdws")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) mm1_GetVDWSValue;
			else if (!strcmp(kw3, "dist")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) GetDistance;
			else if (!strcmp(kw3, "unity")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add surf2 : unknown value function 1.\n"); return; }
			
			if (!strcmp(kw4, "esp")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) mm1_GetESPValue;
			else if (!strcmp(kw4, "vdws")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) mm1_GetVDWSValue;
			else if (!strcmp(kw4, "dist")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) GetDistance;
			else if (!strcmp(kw4, "unity")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add surf2 : unknown value function 2.\n"); return; }
			
			if (!strcmp(kw5, "red")) csp2a.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw5, "green")) csp2a.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw5, "blue")) csp2a.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw5, "rb1")) csp2a.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw5, "rb2")) csp2a.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add surf2 : unknown colour function 1.\n"); return; }
			
			if (!strcmp(kw6, "red")) csp2b.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw6, "green")) csp2b.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw6, "blue")) csp2b.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw6, "rb1")) csp2b.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw6, "rb2")) csp2b.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add surf2 : unknown colour function 2.\n"); return; }
			
			f64 sscale1 = strtod(kw7, endptr);
			f64 sscale2 = strtod(kw8, endptr);
			f64 cscale = strtod(kw9, endptr);
			
			f64 dim = strtod(kwA, endptr);
			
			i32s res = strtol(kwB, endptr, 10);
			if (res < 4) res = 4;
			
			i32s solid = strtol(kwC, endptr, 10);
			if (solid < 0) solid = 0; if (solid > 1) solid = 1;

			i32s tp = strtol(kwD, endptr, 10);
			if (tp < 0) tp = 0; if (tp > 1) tp = 1;
			
			f64 alpha = strtod(kwE, endptr);
			
			static fGL dim_arr[3];
			dim_arr[0] = dim_arr[1] = dim_arr[2] = dim;
			
			static i32s res_arr[3];
			res_arr[0] = res_arr[1] = res_arr[2] = res;
			
			csp2a.svalue = sscale1; csp2a.cvalue = cscale;
			csp2a.dim = dim_arr; csp2a.np = res_arr; csp2a.wireframe = !solid;
			csp2a.transparent = tp; csp2a.alpha = alpha; csp2a.toler = fabs(1.0e-6 * sscale1); csp2a.maxc = 250;
			
			csp2b.svalue = sscale2; csp2b.cvalue = cscale;
			csp2b.dim = dim_arr; csp2b.np = res_arr; csp2b.wireframe = !solid;
			csp2b.transparent = tp; csp2b.alpha = alpha; csp2b.toler = fabs(1.0e-6 * sscale2); csp2b.maxc = 250;
			
			ostrstream strN(object_name, sizeof(object_name));
			strN << kw3 << "-" << kw4 << "-" << ends;
			
			AddObject(new color_surface_object(ol_static(), csp2a, object_name));
			UpdateAllGraphicsViews();
			
			ostrstream strR(mbuff1, sizeof(mbuff1));
			strR << "Added a new object : surf2 (" << kw3 << " " << kw4 << " " << kw5 << " " << kw6 << ")." << endl << ends;
			PrintToLog(mbuff1);
			return;
		}
		
		PrintToLog("ERROR : could not process a command : \"add\".\n");
	}
	
	if (!strcmp("random_search", kw1))
	{
		char kw2[32]; istr >> kw2;	// the 2nd keyword; cycles.
		char kw3[32]; istr >> kw3;	// the 3rd keyword; optsteps.
		
		char ** endptr = NULL;
		i32s cycles = strtol(kw2, endptr, 10);
		i32s optsteps = strtol(kw3, endptr, 10);
		
		DoRandomSearch(cycles, optsteps);
		return;
	}

	if (!strcmp("systematic_search", kw1))
	{
		char kw2[32]; istr >> kw2;	// the 2nd keyword; divisions.
		char kw3[32]; istr >> kw3;	// the 3rd keyword; optsteps.
		
		char ** endptr = NULL;
		i32s divisions = strtol(kw2, endptr, 10);
		i32s optsteps = strtol(kw3, endptr, 10);
		
		DoSystematicSearch(divisions, optsteps);
		return;
	}

	if (!strcmp("montecarlo_search", kw1))
	{
		char kw2[32]; istr >> kw2;	// the 2nd keyword; n_init_steps.
		char kw3[32]; istr >> kw3;	// the 3rd keyword; n_simul_steps.
		char kw4[32]; istr >> kw4;	// the 4th keyword; optsteps.
		
		char ** endptr = NULL;
		i32s n_init_steps = strtol(kw2, endptr, 10);
		i32s n_simul_steps = strtol(kw3, endptr, 10);
		i32s optsteps = strtol(kw4, endptr, 10);
		
		DoMonteCarloSearch(n_init_steps, n_simul_steps, optsteps);
		return;
	}
	
	if (!strcmp("make_plot1", kw1))
	{
		char kw2[32]; istr >> kw2;	// A
		char kw3[32]; istr >> kw3;	// B
		char kw4[32]; istr >> kw4;	// C
		char kw5[32]; istr >> kw5;	// D
		char kw6[32]; istr >> kw6;	// div
		char kw7[32]; istr >> kw7;	// start_ang
		char kw8[32]; istr >> kw8;	// end_ang
		char kw9[32]; istr >> kw9;	// optsteps
		char kwA[32]; istr >> kwA;	// qmeng
		char kwB[32]; istr >> kwB;	// qmc
		char ** endptr = NULL;
		
		i32s ia = strtol(kw2, endptr, 10);
		i32s ib = strtol(kw3, endptr, 10);
		i32s ic = strtol(kw4, endptr, 10);
		i32s id = strtol(kw5, endptr, 10);
		i32s div1 = strtol(kw6, endptr, 10);
		fGL start1 = strtod(kw7, endptr);
		fGL end1 = strtod(kw8, endptr);
		
		i32s optsteps = strtol(kw9, endptr, 10);
		
		i32s qmeng = strtol(kwA, endptr, 10); if (!strlen(kwA)) qmeng = -1;
		i32s qmc = strtol(kwB, endptr, 10);
		
		DoEnergyPlot1D(ia, ib, ic, id, div1, start1, end1, optsteps, qmeng, qmc);
		return;
	}
	
	if (!strcmp("make_plot2", kw1))
	{
		char kw2[32]; istr >> kw2;	// A
		char kw3[32]; istr >> kw3;	// B
		char kw4[32]; istr >> kw4;	// C
		char kw5[32]; istr >> kw5;	// D
		char kw6[32]; istr >> kw6;	// div
		char kw7[32]; istr >> kw7;	// start_ang
		char kw8[32]; istr >> kw8;	// end_ang
		char kw9[32]; istr >> kw9;	// I
		char kwA[32]; istr >> kwA;	// J
		char kwB[32]; istr >> kwB;	// K
		char kwC[32]; istr >> kwC;	// L
		char kwD[32]; istr >> kwD;	// div
		char kwE[32]; istr >> kwE;	// start_ang
		char kwF[32]; istr >> kwF;	// end_ang
		char kwG[32]; istr >> kwG;	// optsteps
		char kwH[32]; istr >> kwH;	// qmeng
		char kwI[32]; istr >> kwI;	// qmc
		char ** endptr = NULL;
		
		i32s ia = strtol(kw2, endptr, 10);
		i32s ib = strtol(kw3, endptr, 10);
		i32s ic = strtol(kw4, endptr, 10);
		i32s id = strtol(kw5, endptr, 10);
		i32s div1 = strtol(kw6, endptr, 10);
		fGL start1 = strtod(kw7, endptr);
		fGL end1 = strtod(kw8, endptr);
		
		i32s ii = strtol(kw9, endptr, 10);
		i32s ij = strtol(kwA, endptr, 10);
		i32s ik = strtol(kwB, endptr, 10);
		i32s il = strtol(kwC, endptr, 10);
		i32s div2 = strtol(kwD, endptr, 10);
		fGL start2 = strtod(kwE, endptr);
		fGL end2 = strtod(kwF, endptr);
		
		i32s optsteps = strtol(kwG, endptr, 10);
		
		i32s qmeng = strtol(kwH, endptr, 10); if (!strlen(kwH)) qmeng = -1;
		i32s qmc = strtol(kwI, endptr, 10);
		
		DoEnergyPlot2D(ia, ib, ic, id, div1, start1, end1, ii, ij, ik, il, div2, start2, end2, optsteps, qmeng, qmc);
		return;
	}
	
	if (!strcmp("build_amino", kw1))
	{
		char kw2[4096]; istr >> kw2;	// sequence
		char kw3[32]; istr >> kw3;	// helix/sheet (optional)
		
		f64 aab_ah[] = { 302.0 * M_PI / 180.0, 313.0 * M_PI / 180.0, M_PI };
		f64 aab_bs[] = { 180.0 * M_PI / 180.0, 180.0 * M_PI / 180.0, M_PI };
		
		mm1_mdl::amino_builder.Build(this, kw2, kw3[0] == 'h' || kw3[0] == 'H' ? aab_ah : aab_bs);
		UpdateAllGraphicsViews();
		
		ostrstream strR(mbuff1, sizeof(mbuff1));
		strR << "built a sequence : " << kw2 << endl << ends;
		PrintToLog(mbuff1);
		return;
	}
	
	if (!strcmp("build_nucleic", kw1))
	{
		char kw2[4096]; istr >> kw2;	// sequence
		
		f64 nab[] =
		{
			44.6 * M_PI / 180.0, 261.0 * M_PI / 180.0, 320.8 * M_PI / 180.0, 208.6 * M_PI / 180.0,
			273.8 * M_PI / 180.0, 105.6 * M_PI / 180.0, 356.0 * M_PI / 180.0, 24.7 * M_PI / 180.0,
			88.7 * M_PI / 180.0, 264.6 * M_PI / 180.0
		};
		
		mm1_mdl::nucleic_builder.Build(this, kw2, nab);
		UpdateAllGraphicsViews();
		
		ostrstream strR(mbuff1, sizeof(mbuff1));
		strR << "built a sequence : " << kw2 << endl << ends;
		PrintToLog(mbuff1);
		return;
	}
	
	// error message...
	ostrstream strE(mbuff1, sizeof(mbuff1)); strE << "ERROR : Unknown command : " << command << endl << ends;
	PrintToLog(mbuff1); PrintToLog("The \"help\" command will give more information about command strings.\n");
}

void mm1_docv::DoEnergyPlot1D(i32s inda, i32s indb, i32s indc, i32s indd, i32s div1, fGL start1, fGL end1, i32s optsteps, i32s qmeng, i32s qmc)
{
	if (qmeng != -1)
	{
		char mbuff9[256]; strstream str9(mbuff9, sizeof(mbuff9));
		str9 << "QM method was requested: using " << qm1_mdl::engtab1[qmeng] << " method with charge = " << qmc << endl << ends;
		PrintToLog(mbuff9);
	}
	
	i32s molnum = 0; i32s in_crdset = 0;
	
	i32s atmi1[4] = { inda, indb, indc, indd };
	mm1_atom * atmr1[4]; f64 range1[2];
	range1[0] = M_PI * start1 / 180.0;
	range1[1] = M_PI * end1 / 180.0;
	
	for (i32s n1 = 0;n1 < 4;n1++)
	{
		iter_mm1al it1;
		
		it1 = FindAtomByIndex(atmi1[n1]);
		if (it1 == GetAtomsEnd()) { PrintToLog("ERROR : atom not found!\n"); return; }
		atmr1[n1] = & (* it1);
	}
	
	GatherGroups();		// for internal coordinates...
	mm1_intcrd * ic = new mm1_intcrd((* this), molnum, in_crdset);
	i32s ict1 = ic->FindTorsion(atmr1[1]->crd_vector[in_crdset].data, atmr1[2]->crd_vector[in_crdset].data);
if (ict1 < 0) { PrintToLog("ERROR : could not find ic for tor1.\n"); return; }
	
	v3d<fGL> v1a(atmr1[1]->crd_vector[in_crdset].data, atmr1[0]->crd_vector[in_crdset].data);
	v3d<fGL> v1b(atmr1[1]->crd_vector[in_crdset].data, atmr1[2]->crd_vector[in_crdset].data);
	v3d<fGL> v1c(atmr1[2]->crd_vector[in_crdset].data, atmr1[3]->crd_vector[in_crdset].data);
	f64 oldt1 = v1a.tor(v1b, v1c);
	
	mm1_eng * eng1 = CreateDefaultEngine();
	i32s fft1 = eng1->FindTorsion(atmr1[0], atmr1[1], atmr1[2], atmr1[3]);
if (ict1 < 0) { PrintToLog("ERROR : could not find tor-term for tor1.\n"); return; }

	mm1_eng * eng2 = CreateDefaultEngine();
	
	plot1d_view * plot = graphics_factory->ProducePlot1DView(this, PLOT_USERDATA_STRUCTURE, 0);
	
	f64 tor1 = range1[0];
	for (i32s s1 = 0;s1 < (div1 + 1);s1++)
	{
		ic->SetTorsion(ict1, tor1 - oldt1);
		ic->UpdateCartesian();
		
		CopyCRD(this, eng1, 0);		// lock_local_structure needs coordinates!!!
		eng1->SetTorsionConstraint(fft1, tor1, 5000.0, true);
		
		// optimize...
		
		mm1_geomopt * opt = new mm1_geomopt(eng1, 100, 0.025);		// optimal settings?!?!?
		
		for (i32s n1 = 0;n1 < optsteps;n1++)
		{
			opt->TakeCGStep(conjugate_gradient::Newton2An);
			cout << n1 << " " << opt->optval << " " << opt->optstp << endl;
			
			if (!(n1 % 50))
			{
				CopyCRD(eng1, this, 0); CenterCRDSet(0);
				UpdateAllGraphicsViews(true);
			}
		}
		
		CopyCRD(eng1, this, 0); CenterCRDSet(0);
		delete opt;
		
		// compute energy for final structure...
		
		f64 value;
		if (qmeng < 0)
		{
			CopyCRD(this, eng2, 0);
			eng2->Compute(0);
			
			value = eng2->energy;
		}
		else
		{
			qm1_mdl * mdl9 = new qm1_mdl(& cout, * console_class_factory::GetInstance());
			mdl9->aai_MakeCopy(this);
			
			mdl9->default_eng = qmeng; mdl9->SetTotalCharge(qmc);
			qm1_eng * eng9 = mdl9->CreateDefaultEngine();
			
			if (eng9 != NULL)
			{
				CopyCRD(mdl9, eng9, 0);
				eng9->Compute(0);
				
				value = eng9->energy;
				
				delete eng9;
				delete mdl9;
			}
			else
			{
				value = 0.0;	// if something went wrong...
				delete mdl9;	// ...we just use zero!!!
			}
		}
		
		// ...and add it to the plot.
		
		void * udata = mm1_convert_cset_to_plotting_udata(this, 0);
		plot->AddData(180.0 * tor1 / M_PI, value, udata);
		
		char mbuff1[256];
		strstream str1(mbuff1, sizeof(mbuff1));
		str1 << "tor = " << (180.0 * tor1 / M_PI) << " deg, energy = " << value << " kJ/mol." << endl << ends;
		PrintToLog(mbuff1);
		
		tor1 += (range1[1] - range1[0]) / (f64) div1;
	}
	
	plotting_view_vector.push_back(plot);
	plot->SetCenterAndScale();
	plot->Update();
	
	delete eng2;
	delete eng1;
	delete ic;
}

void mm1_docv::DoEnergyPlot2D(i32s inda, i32s indb, i32s indc, i32s indd, i32s div1, fGL start1, fGL end1, i32s indi, i32s indj, i32s indk, i32s indl, i32s div2, fGL start2, fGL end2, i32s optsteps, i32s qmeng, i32s qmc)
{
	if (qmeng != -1)
	{
		char mbuff9[256]; strstream str9(mbuff9, sizeof(mbuff9));
		str9 << "QM method was requested: using " << qm1_mdl::engtab1[qmeng] << " method with charge = " << qmc << endl << ends;
		PrintToLog(mbuff9);
	}
	
	i32s molnum = 0; i32s in_crdset = 0;
	
	i32s atmi1[4] = { inda, indb, indc, indd };
	mm1_atom * atmr1[4]; f64 range1[2];
	range1[0] = M_PI * start1 / 180.0;
	range1[1] = M_PI * end1 / 180.0;
	
	i32s atmi2[4] = { indi, indj, indk, indl };
	mm1_atom * atmr2[4]; f64 range2[2];
	range2[0] = M_PI * start2 / 180.0;
	range2[1] = M_PI * end2 / 180.0;
	
	for (i32s n1 = 0;n1 < 4;n1++)
	{
		iter_mm1al it1;
		
		it1 = FindAtomByIndex(atmi1[n1]);
		if (it1 == GetAtomsEnd()) { PrintToLog("ERROR : tor1 atom not found!\n"); return; }
		atmr1[n1] = & (* it1);
		
		it1 = FindAtomByIndex(atmi2[n1]);
		if (it1 == GetAtomsEnd()) { PrintToLog("ERROR : tor2 atom not found!\n"); return; }
		atmr2[n1] = & (* it1);
	}
	
	GatherGroups();		// for internal coordinates...
	mm1_intcrd * ic = new mm1_intcrd((* this), molnum, in_crdset);
	i32s ict1 = ic->FindTorsion(atmr1[1]->crd_vector[in_crdset].data, atmr1[2]->crd_vector[in_crdset].data);
if (ict1 < 0) { PrintToLog("ERROR : could not find ic for tor1.\n"); return; }
	i32s ict2 = ic->FindTorsion(atmr2[1]->crd_vector[in_crdset].data, atmr2[2]->crd_vector[in_crdset].data);
if (ict2 < 0) { PrintToLog("ERROR : could not find ic for tor2.\n"); return; }
	
	v3d<fGL> v1a(atmr1[1]->crd_vector[in_crdset].data, atmr1[0]->crd_vector[in_crdset].data);
	v3d<fGL> v1b(atmr1[1]->crd_vector[in_crdset].data, atmr1[2]->crd_vector[in_crdset].data);
	v3d<fGL> v1c(atmr1[2]->crd_vector[in_crdset].data, atmr1[3]->crd_vector[in_crdset].data);
	f64 oldt1 = v1a.tor(v1b, v1c);
	
	v3d<fGL> v2a(atmr2[1]->crd_vector[in_crdset].data, atmr2[0]->crd_vector[in_crdset].data);
	v3d<fGL> v2b(atmr2[1]->crd_vector[in_crdset].data, atmr2[2]->crd_vector[in_crdset].data);
	v3d<fGL> v2c(atmr2[2]->crd_vector[in_crdset].data, atmr2[3]->crd_vector[in_crdset].data);
	f64 oldt2 = v2a.tor(v2b, v2c);
	
	mm1_eng * eng1 = CreateDefaultEngine();
	i32s fft1 = eng1->FindTorsion(atmr1[0], atmr1[1], atmr1[2], atmr1[3]);
if (ict1 < 0) { PrintToLog("ERROR : could not find tor-term for tor1.\n"); return; }
	i32s fft2 = eng1->FindTorsion(atmr2[0], atmr2[1], atmr2[2], atmr2[3]);
if (ict2 < 0) { PrintToLog("ERROR : could not find tor-term for tor2.\n"); return; }

	mm1_eng * eng2 = CreateDefaultEngine();
	
	plot2d_view * plot = graphics_factory->ProducePlot2DView(this, PLOT_USERDATA_STRUCTURE, 0);
	
	f64 tor1 = range1[0];
	for (i32s s1 = 0;s1 < (div1 + 1);s1++)
	{
		f64 tor2 = range2[0];
		for (i32s s2 = 0;s2 < (div2 + 1);s2++)
		{
			ic->SetTorsion(ict1, tor1 - oldt1);
			ic->SetTorsion(ict2, tor2 - oldt2);
			ic->UpdateCartesian();
			
			CopyCRD(this, eng1, 0);		// lock_local_structure needs coordinates!!!
			eng1->SetTorsionConstraint(fft1, tor1, 5000.0, true);
			eng1->SetTorsionConstraint(fft2, tor2, 5000.0, true);
			
			// optimize...
			
			mm1_geomopt * opt = new mm1_geomopt(eng1, 100, 0.025);		// optimal settings?!?!?
			
			for (i32s n1 = 0;n1 < optsteps;n1++)
			{
				opt->TakeCGStep(conjugate_gradient::Newton2An);
				cout << n1 << " " << opt->optval << " " << opt->optstp << endl;
				
				if (!(n1 % 50))
				{
					CopyCRD(eng1, this, 0); CenterCRDSet(0);
					UpdateAllGraphicsViews(true);
				}
			}
			
			CopyCRD(eng1, this, 0); CenterCRDSet(0);
			delete opt;
			
			// compute energy for final structure...
			
			f64 value;
			if (qmeng < 0)
			{
				CopyCRD(this, eng2, 0);
				eng2->Compute(0);
				
				value = eng2->energy;
			}
			else
			{
				qm1_mdl * mdl9 = new qm1_mdl(& cout, * console_class_factory::GetInstance());
				mdl9->aai_MakeCopy(this);
				
				mdl9->default_eng = qmeng; mdl9->SetTotalCharge(qmc);
				qm1_eng * eng9 = mdl9->CreateDefaultEngine();
				
				if (eng9 != NULL)
				{
					CopyCRD(mdl9, eng9, 0);
					eng9->Compute(0);
					
					value = eng9->energy;
					
					delete eng9;
					delete mdl9;
				}
				else
				{
					value = 0.0;	// if something went wrong...
					delete mdl9;	// ...we just use zero!!!
				}
			}
			
			// ...and add it to the plot.
			
			void * udata = mm1_convert_cset_to_plotting_udata(this, 0);
			plot->AddData(180.0 * tor1 / M_PI, 180.0 * tor2 / M_PI, value, udata);
			
			char mbuff1[256];
			strstream str1(mbuff1, sizeof(mbuff1));
			str1 << "tor1 = " << (180.0 * tor1 / M_PI) << " deg, tor2 = " << (180.0 * tor2 / M_PI) << " deg, energy = " << value << " kJ/mol." << endl << ends;
			PrintToLog(mbuff1);
			
			tor2 += (range2[1] - range2[0]) / (f64) div2;
		}
		
		tor1 += (range1[1] - range1[0]) / (f64) div1;
	}
	
	plotting_view_vector.push_back(plot);
	plot->SetCenterAndScale();
	plot->Update();
	
	delete eng2;
	delete eng1;
	delete ic;
}

/*################################################################################################*/

void mm1_cm_element::GetColor(const void * p1, fGL_a4 & p2, prefs *p3)
{
	mm1_atom * ref = (mm1_atom *) p1;
	const fGL * color = ref->el.GetColor(p3);
	p2[0] = color[0]; p2[1] = color[1]; p2[2] = color[2];
}

void mm1_cm_state::GetColor(const void * p1, fGL_a4 & p2, prefs *p3)
{
//	mm1_atom * ref = (mm1_atom *) p1;
//	mm1_mdl * mdl = ref->mdl;		// CURRENTLY BROKEN...
	
	p2[0] = 0.0; p2[1] = 0.0; p2[2] = 1.0;		// loop
	
/*	if (mdl->chn_info == NULL) return;
	vector<mm1_chn_info *> & ci = * mdl->chn_info;
	
	if (ref->id[1] < 0 || ref->id[2] < 0) return;
	if (ci[ref->id[1]]->state == NULL) return;
	
	char state = ci[ref->id[1]]->state[ref->id[2]];
	
	switch (state)
	{
		case '4':
		p2[0] = 1.0; p2[1] = 0.0; p2[2] = 0.0;		// helix
		return;
		
		case 'S':
		p2[0] = 0.0; p2[1] = 1.0; p2[2] = 0.0;		// strand
		return;
	}	*/
}

/*################################################################################################*/

#endif	// ENABLE_GRAPHICS

// eof
