/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include "gimageview.h"

#include "dnd.h"
#include "fileutil.h"
#include "gfileutil.h"
#include "gtkutils.h"
#include "menu.h"
#include "prefs.h"
#include "thumbnail.h"
#include "thumbnail_view.h"


GtkTargetEntry dnd_types[] = {
   {"text/uri-list", 0, TARGET_URI_LIST},
};
const gint dnd_types_num = sizeof(dnd_types) / sizeof(dnd_types[0]);


GtkItemFactoryEntry dnd_file_popup_items [] =
{
   {N_("/Open in new tab"), NULL, menu_modal_cb, GDK_ACTION_PRIVATE, NULL},
   {N_("/---"),             NULL, NULL,          0,                  "<Separator>"},
   {N_("/Move"),            NULL, menu_modal_cb, GDK_ACTION_MOVE,    NULL},
   {N_("/Copy"),            NULL, menu_modal_cb, GDK_ACTION_COPY,    NULL},
   {N_("/Symbolic Link"),   NULL, menu_modal_cb, GDK_ACTION_LINK ,   NULL},
   {N_("/---"),             NULL, NULL,          0,                  "<Separator>"},
   {N_("/Cancel"),          NULL, NULL,          0,                  NULL},
   {NULL, NULL, NULL, 0, NULL},
};
 

/*
 *  dnd_get_file_list:
 *     @ convert string URI list to GList format.
 *
 *  string : file list (string format)
 *  Return : file list (GList format)
 */
GList *
dnd_get_file_list (const gchar *string)
{
   gchar *file;
   gchar *ptr, *uri;
   GList *list = NULL;
   gint len, pos = 0;

   uri = ptr = g_strdup (string);
   len = strlen (uri);

   while (*ptr && (pos < len)) {
      if (!strncmp(ptr, "file:", 5)) {
	 ptr += 5;
	 pos += 5;
      }

      file = ptr;

      while (*ptr != '\r' && *ptr != '\n' && *ptr != '\0') {
	 ptr++;
	 pos++;
      }
      *ptr++ = '\0';
      pos++;

      while (*ptr == '\r' || *ptr == '\n' || *ptr == '\0') {
	 ptr++;
	 pos++;
      }

      if (file && file[0] != '\r' && file[0] != '\n' && file[0] != '\0')
	 list = g_list_append (list, g_strdup(file));
   }

   g_free (uri);

   return list;
}


/*
 *  dnd_src_set:
 *     @
 *
 *  widget : widget to set DnD (source side).
 */
void
dnd_src_set (GtkWidget *widget, const GtkTargetEntry *entry, gint num)
{
   if (conf.dnd_enable_to_external)
      dnd_types[0].flags = 0;
   else
      dnd_types[0].flags = GTK_TARGET_SAME_APP;

   gtk_drag_source_set(widget,
		       GDK_BUTTON1_MASK | GDK_BUTTON2_MASK | GDK_BUTTON3_MASK,
		       entry, num,
		       GDK_ACTION_ASK  | GDK_ACTION_COPY
		       | GDK_ACTION_MOVE | GDK_ACTION_LINK);
}


/*
 *  dnd_dest_set:
 *     @
 *
 *  widget : widget to set DnD (destination side).
 */
void
dnd_dest_set (GtkWidget *widget, const GtkTargetEntry *entry, gint num)
{
   if (conf.dnd_enable_from_external)
      dnd_types[0].flags = 0;
   else
      dnd_types[0].flags = GTK_TARGET_SAME_APP;

   gtk_drag_dest_set(widget,
		     GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
		     entry, num,
		     GDK_ACTION_ASK  | GDK_ACTION_COPY
		     | GDK_ACTION_MOVE | GDK_ACTION_LINK);
}


/*
 *  dnd_file_operation:
 *     @ Open DnD context menu and do specified file operation when data
 *       received. This function will called by "drag_data_received" signal's
 *       callback function.
 *
 *  dest_dir : destination directory to move/copy/link specified files.
 *  context  : DnD context.
 *  seldata  : URI list (string format).
 *  time     : time when drag data received.
 *  tw       : Pointer to parent ThumbWindow.
 */
void
dnd_file_operation (const gchar *dest_dir, GdkDragContext *context,
		    GtkSelectionData *seldata, guint time, ThumbWindow *tw)
{
   GtkWidget *dnd_popup, *progress_win;
   GList *list, *node;
   ThumbView *tv;
   ConfirmType over_write = CONFIRM_ASK, cache_over_write = CONFIRM_YES_TO_ALL;
   gboolean dnd_success = TRUE, dnd_delete = FALSE, cancel = FALSE;
   gint n_menu_items, action, length, pos;
   gfloat progress;
   gchar *src_file, *src_cache, *endchr;
   gchar *dest_file, *dest_cache_dir = NULL;
   gchar message[BUF_SIZE];
   ThumbCacheType cache_type;

   g_return_if_fail (dest_dir && context && seldata);

   list = dnd_get_file_list (seldata->data);

   /* create popup menu */
   n_menu_items = sizeof (dnd_file_popup_items)
                     / sizeof (dnd_file_popup_items[0]) - 1;
   dnd_popup = menu_create_items (NULL, dnd_file_popup_items,
				  n_menu_items, "<DnDPop>",
				  NULL);

   /* popup menu */
   action = menu_popup_modal (dnd_popup, NULL, NULL, NULL, NULL);

   if (action == GDK_ACTION_PRIVATE) {
      open_images_dirs (list, tw, LOAD_CACHE, FALSE);
   } else {
      progress_win = gtkutil_create_progress_window ("File Operation", "...",
						     &cancel, 300, -1);
      gtk_grab_add (progress_win);

      /* do operation */
      length = g_list_length (list);
      node = list;
      while (node) {
	 src_file = node->data;

	 /* get cache file & dir */
	 src_cache = find_thumbcache (src_file, &cache_type);
	 if (src_cache) {
	    dest_file = g_strconcat (dest_dir, g_basename (src_file), NULL);
	    dest_cache_dir = thumbsupport_get_thumb_cache_path (dest_file, cache_type);
	    if (dest_cache_dir) {
	       endchr = strrchr (dest_cache_dir, '/');
	       if (endchr) {
		  *(endchr + 1) = '\0';
		  mkdirs (dest_cache_dir);
	       } else {
		  g_free (dest_cache_dir);
		  dest_cache_dir = NULL;
	       }
	    }
	    g_free (dest_file);
	 }

	 while (gtk_events_pending()) gtk_main_iteration();

	 pos = g_list_position (list, node);
	 progress = (gfloat) pos / (gfloat) length;

	 switch (action) {
	 case GDK_ACTION_MOVE:
	    g_snprintf (message, BUF_SIZE, _("Moving %s ..."), src_file);
	    gtkutil_progress_window_update (progress_win, _("Moving files"),
					    message, NULL, progress);
	    move_file (src_file, dest_dir, &over_write, TRUE);
	    if (src_cache && dest_cache_dir)
	       move_file (src_cache, dest_cache_dir, &cache_over_write, FALSE);
	    dnd_delete = TRUE;
	    break;
	 case GDK_ACTION_COPY:
	    g_snprintf (message, BUF_SIZE, _("Copying %s ..."), src_file);
	    gtkutil_progress_window_update (progress_win, _("Copying files"),
					    message, NULL, progress);
	    copy_file (src_file, dest_dir, &over_write, TRUE);
	    if (src_cache && dest_cache_dir)
	    copy_file (src_cache, dest_cache_dir, &cache_over_write, FALSE);
	    break;
	 case GDK_ACTION_LINK:
	    g_snprintf (message, BUF_SIZE, _("Creating Link %s ..."), src_file);
	    gtkutil_progress_window_update (progress_win, _("Creating Links"),
					    message, NULL, progress);
	    link_file (src_file, dest_dir, TRUE);
	    if (src_cache && dest_cache_dir)
	       link_file (src_cache, dest_cache_dir, FALSE);
	    break;
	 default:
	    dnd_success = FALSE;
	    break;
	 }

	 if (src_cache) {
	    g_free (src_cache);
	    src_cache = NULL;
	 }
	 if (dest_cache_dir) {
	    g_free (dest_cache_dir);
	    dest_cache_dir = NULL;
	 }

	 /* cancel */
	 if (cancel || (over_write == CONFIRM_CANCEL)) break;

	 /* reset to CONFIRM_ASK mode */
	 if (over_write != CONFIRM_YES_TO_ALL) {
	    over_write = CONFIRM_ASK;
	 }

	 node = g_list_next (node);
      }

      gtk_grab_remove (progress_win);
      gtk_widget_destroy (progress_win);
   }

   gtk_drag_finish(context, dnd_success, dnd_delete, time);

   /* update dest side file list */
   tv = thumbview_find_opened_dir (dest_dir);
   if (tv) {
      thumbview_refresh_list (tv);
   }

   g_list_foreach (list, (GFunc) g_free, NULL);
   g_list_free (list);
}
