/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "fileutil.h"
#include "gtkutils.h"
#include "icon.h"
#include "intl.h"

#ifndef BUF_SIZE
#define BUF_SIZE 4096
#endif

/* callback functions for confirm dialog */
static gint cb_dummy                 (GtkWidget   *button,
				      gpointer     data);
static void cb_confirm_yes           (GtkWidget   *button,
				      ConfirmType *type);
static void cb_confirm_yes_to_all    (GtkWidget   *button,
				      ConfirmType *type);
static void cb_confirm_no            (GtkWidget   *button,
				      ConfirmType *type);
static void cb_confirm_cancel        (GtkWidget   *button,
				      ConfirmType *type);

/* callback functions for message dialog */
static void cb_message_dialog_quit   (GtkWidget   *button,
				      gpointer     data);

/* callback functions for progress bar window */
static void cb_progress_win_cancel   (GtkWidget   *button,
				      gboolean    *cancel_pressed);

/* callback functions for text entry window */
static void cb_textpop_enter         (GtkWidget   *button,
				      gboolean    *ok_pressd);
static void cb_textpop_ok_button     (GtkWidget   *button,
				      gboolean    *ok_pressd);
static void cb_textpop_cancel_button (GtkWidget   *button,
				      gboolean    *ok_pressd);



/******************************************************************************
 *
 *   misc
 *
 ******************************************************************************/
/*
 *  create_toggle_button:
 *     @ Create toggle button widget.
 *
 *  label   : Label text for toggle button.
 *  def_val : Default value.
 *  Return  : Toggle button widget.
 */
GtkWidget *
create_check_button (const gchar *label_text, gboolean def_val,
		     gpointer func, gpointer data)
{
   GtkWidget *toggle;

   toggle = gtk_check_button_new_with_label (_(label_text));
   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(toggle), def_val);
   gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		       GTK_SIGNAL_FUNC (func), data);

   return toggle;
}



/******************************************************************************
 *
 *   Confirm Dialog Window
 *
 ******************************************************************************/
static gint
cb_dummy (GtkWidget *button, gpointer data)
{
   return TRUE;
}


static void
cb_confirm_yes (GtkWidget *button, ConfirmType *type)
{
   *type = CONFIRM_YES;
   gtk_main_quit ();
}


static void
cb_confirm_yes_to_all (GtkWidget *button, ConfirmType *type)
{
   *type = CONFIRM_YES_TO_ALL;
   gtk_main_quit ();
}


static void
cb_confirm_no (GtkWidget *button, ConfirmType *type)
{
   *type = CONFIRM_NO;
   gtk_main_quit ();
}


static void
cb_confirm_cancel (GtkWidget *button, ConfirmType *type)
{
   *type = CONFIRM_CANCEL;
   gtk_main_quit ();
}


ConfirmType
gtkutil_confirm_dialog (const gchar *title, const gchar *message,
			gboolean multiple)
{
   ConfirmType retval = CONFIRM_NO;
   GtkWidget *window;
   GtkWidget *vbox, *hbox, *button, *label;
   GtkWidget *icon;

   window = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (window), title); 
   gtk_window_set_default_size (GTK_WINDOW (window), 300, 120);
   /* gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG (window)->vbox), 5); */
   gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
   gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		       GTK_SIGNAL_FUNC (cb_dummy), NULL);
   gtk_widget_show (window);

   /* message area */
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), vbox, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 15);
   gtk_widget_show (vbox);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
   gtk_widget_show (hbox);

   /* icon */
   icon = icon_get_widget ("question");
   gtk_box_pack_start (GTK_BOX (hbox), icon, TRUE, TRUE, 0);
   gtk_widget_show (icon);

   /* message */
   label = gtk_label_new (message);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
   gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
   gtk_widget_show (label);

   /* buttons */
   button = gtk_button_new_with_label (_("Yes"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (cb_confirm_yes),
		       &retval);
   gtk_widget_show (button);
   gtk_widget_grab_focus (button);

   if (multiple) {
      button = gtk_button_new_with_label (_("Yes to All"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
			  button, TRUE, TRUE, 0);
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (cb_confirm_yes_to_all),
			  &retval);
      gtk_widget_show (button);
   }

   button = gtk_button_new_with_label (_("No"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (cb_confirm_no),
		       &retval);
   gtk_widget_show (button);

   if (multiple) {
      button = gtk_button_new_with_label (_("Cancel"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
			  button, TRUE, TRUE, 0);
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (cb_confirm_cancel),
			  &retval);
      gtk_widget_show (button);
   }

   gtk_grab_add (window);
   gtk_main ();
   gtk_grab_remove (window);
   gtk_widget_destroy (window);

   return retval;
}



/******************************************************************************
 *
 *   Message Dialog Window
 *
 ******************************************************************************/
static void
cb_message_dialog_quit (GtkWidget *button, gpointer data)
{
   gtk_main_quit ();
}


void
gtkutil_message_dialog (const gchar *title, const gchar *message)
{
   GtkWidget *window;
   GtkWidget *button, *label, *vbox, *hbox;
   GtkWidget *alert_icon;

   window = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (window), title); 
   gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
   gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		       GTK_SIGNAL_FUNC (cb_dummy), NULL);
   gtk_widget_show (window);

   /* message area */
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), vbox,
		       TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 15);
   gtk_widget_show (vbox);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
   gtk_widget_show (hbox);

   /* icon */
   alert_icon = icon_get_widget ("alert");
   gtk_box_pack_start (GTK_BOX (hbox), alert_icon, TRUE, TRUE, 0);
   gtk_widget_show (alert_icon);

   /* message */
   label = gtk_label_new (message);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
   gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
   gtk_widget_show (label);

   /* button */
   button = gtk_button_new_with_label (_("OK"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (cb_message_dialog_quit), NULL);
   gtk_widget_show (button);

   gtk_widget_grab_focus (button);

   gtk_grab_add (window);
   gtk_main ();
   gtk_grab_remove (window);
   gtk_widget_destroy (window);
}



/******************************************************************************
 *
 *   Progress Bar Window
 *
 ******************************************************************************/
static void
cb_progress_win_cancel (GtkWidget *button, gboolean *cancel_pressed)
{
   *cancel_pressed = TRUE;
}


void
gtkutil_progress_window_update (GtkWidget *window,
				gchar *title, gchar *message,
				gchar *progress_text, gfloat progress)
{
   GtkWidget *label;
   GtkWidget *progressbar;

   g_return_if_fail (window);

   label = gtk_object_get_data (GTK_OBJECT (window), "label");
   progressbar = gtk_object_get_data (GTK_OBJECT (window), "progressbar");

   g_return_if_fail (label && progressbar);

   if (title)
      gtk_window_set_title (GTK_WINDOW (window), _(title));
   if (message)
      gtk_label_set_text (GTK_LABEL (label), message);
   if (progress_text)
      gtk_progress_set_format_string(GTK_PROGRESS (progressbar),
				     progress_text);
   if (progress > 0.0 && progress < 1.0)
      gtk_progress_bar_update (GTK_PROGRESS_BAR (progressbar), progress);
}


GtkWidget *
gtkutil_create_progress_window (gchar *title, gchar *initial_message,
				gboolean *cancel_pressed,
				gint width, gint height)
{
   GtkWidget *window;
   GtkWidget *vbox;
   GtkWidget *label;
   GtkWidget *progressbar;
   GtkWidget *button;

   g_return_val_if_fail (title && initial_message && cancel_pressed, NULL);

   *cancel_pressed = FALSE;

   /* create dialog window */
   window = gtk_dialog_new ();
   gtk_container_border_width (GTK_CONTAINER (window), 3);
   gtk_window_set_title (GTK_WINDOW (window), title);
   gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
   gtk_window_set_default_size (GTK_WINDOW (window), width, height);
   gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		       GTK_SIGNAL_FUNC (cb_dummy), NULL);

   /* message area */
   vbox = gtk_vbox_new (FALSE, 5);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), vbox,
		       TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
   gtk_widget_show (vbox);

   /* label */
   label = gtk_label_new (initial_message);
   gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

   /* progress bar */
   progressbar = gtk_progress_bar_new();
   gtk_progress_set_show_text(GTK_PROGRESS(progressbar), TRUE);
   gtk_box_pack_start (GTK_BOX (vbox), progressbar, FALSE, FALSE, 0);

   /* cancel button */
   button = gtk_button_new_with_label (_("Cancel"));
   gtk_container_border_width (GTK_CONTAINER (button), 0);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), button,
		       TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(cb_progress_win_cancel), cancel_pressed);

   gtk_object_set_data (GTK_OBJECT (window), "label", label);
   gtk_object_set_data (GTK_OBJECT (window), "progressbar", progressbar);

   gtk_widget_show_all (window);

   return window;
}



/******************************************************************************
 *
 *   Text Entry Window
 *
 ******************************************************************************/
static void
cb_textpop_enter (GtkWidget *button, gboolean *ok_pressd)
{
   *ok_pressd = TRUE;
   gtk_main_quit ();
}


static void
cb_textpop_ok_button (GtkWidget *button, gboolean *ok_pressd)
{
   *ok_pressd = TRUE;
   gtk_main_quit ();
}


static void
cb_textpop_cancel_button (GtkWidget *button, gboolean *ok_pressd)
{
   *ok_pressd = FALSE;
   gtk_main_quit ();
}


gchar *
gtkutil_popup_textentry (const gchar *title, const gchar *label_text,
			 const gchar *entry_text)
{
   GtkWidget *window, *hbox, *vbox, *button, *label, *entry;
   gboolean ok_pressed = FALSE;
   gchar *str = NULL;

   /* dialog window */
   window = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (window), title); 
   gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
   gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		       GTK_SIGNAL_FUNC (cb_dummy), NULL);
   gtk_widget_show (window);

   /* main area */
   vbox = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), vbox, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
   gtk_widget_show (vbox);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
   gtk_widget_show (hbox);

   /* label */
   label = gtk_label_new (label_text);
   gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
   gtk_widget_show (label);

   /* entry */
   entry = gtk_entry_new();
   gtk_box_pack_start (GTK_BOX (hbox), entry, FALSE, FALSE, 0);
   gtk_signal_connect (GTK_OBJECT (entry), "activate",
		       GTK_SIGNAL_FUNC (cb_textpop_enter), &ok_pressed);
   if (entry_text)
      gtk_entry_set_text (GTK_ENTRY (entry), entry_text);
   gtk_widget_show (entry);

   gtk_widget_grab_focus (entry);

   /* button */
   button = gtk_button_new_with_label (_("OK"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(cb_textpop_ok_button), &ok_pressed);
   gtk_widget_show (button);

   button = gtk_button_new_with_label (_("Cancel"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), 
		       button, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(cb_textpop_cancel_button), &ok_pressed);
   gtk_widget_show (button);

   gtk_grab_add (window);
   gtk_main ();
   gtk_grab_remove (window);

   if (ok_pressed)
      str = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));

   gtk_widget_destroy (window);

   return str;
}


/******************************************************************************
 *
 *   Compare functions
 *
 ******************************************************************************/
gint
gtkutil_comp_spel (gconstpointer data1, gconstpointer data2)
{
   const gchar *str1 = data1;
   const gchar *str2 = data2;

   return strcmp (str1, str2);
}


gint
gtkutil_comp_casespel (gconstpointer data1, gconstpointer data2)
{
   const gchar *str1 = data1;
   const gchar *str2 = data2;

   return g_strcasecmp (str1, str2);
}
