/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gimageview.h"

#include "cursors.h"
#include "dnd.h"
#include "gimv_image.h"
#include "image_view.h"
#include "image_window.h"
#include "menu.h"
#include "prefs.h"


static void imageview_get_image_frame_size (ImageView       *iv,
					    gint            *width,
					    gint            *height);
static void imageview_calc_image_size      (ImageView       *iv);
static void imageview_rotate_render        (ImageView       *iv,
					    guint            action);

/* call back functions for reference popup menu */
static void cb_keep_aspect                 (ImageView   *iv,
					    guint        action,
					    GtkWidget   *widget);
static void cb_zoom                        (ImageView   *iv,
					    ImgZoomType  zoom,
					    GtkWidget   *widget);
static void cb_rotate                      (ImageView   *iv,
					    guint        action,
					    GtkWidget   *widget);
static void cb_toggle_buffer               (ImageView   *iv,
					    guint        action,
					    GtkWidget   *widget);

/* other call back functions */
static gint cb_image_configure             (GtkWidget         *widget,
					    GdkEventConfigure *event,
					    ImageView         *iv);
static gint cb_image_expose                (GtkWidget         *widget,
					    GdkEventExpose    *event,
					    ImageView         *iv);
static gint cb_image_button_press          (GtkWidget         *widget,
					    GdkEventButton    *event,
					    ImageView         *iv);
static gint cb_image_button_release        (GtkWidget         *widget, 
					    GdkEventButton    *event,
					    ImageView         *iv);
static gint cb_image_motion_notify         (GtkWidget         *widget, 
					    GdkEventMotion    *event,
					    ImageView         *iv);


/* reference menu items */
GtkItemFactoryEntry imageview_popup_items [] =
{
   {N_("/tear"),                 NULL, NULL,             0, "<Tearoff>"},
   {N_("/Zoom"),                 NULL, NULL,             0, "<Branch>"},
   {N_("/Rotate"),               NULL, NULL,             0, "<Branch>"},
   {N_("/---"),                  NULL, NULL,             0, "<Separator>"},
   {N_("/Memory Buffer"),        NULL, cb_toggle_buffer, 0, "<ToggleItem>"},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "Zoom" sub menu */
GtkItemFactoryEntry imageview_zoom_items [] =
{
   {N_("/tear"),               NULL,  NULL,            0,           "<Tearoff>"},
   {N_("/Zoom In"),            NULL,  cb_zoom,         ZOOM_IN,     NULL},
   {N_("/Zoom Out"),           NULL,  cb_zoom,         ZOOM_OUT,    NULL},
   {N_("/Fit to Widnow"),      NULL,  cb_zoom,         ZOOM_FIT,    NULL},
   {N_("/Keep aspect ratio"),  NULL,  cb_keep_aspect,  0,           "<ToggleItem>"},
   {N_("/---"),                NULL,  NULL,            0,           "<Separator>"},
   {N_("/10%"),                NULL,  cb_zoom,         ZOOM_10,     NULL},
   {N_("/25%"),                NULL,  cb_zoom,         ZOOM_25,     NULL},
   {N_("/50%"),                NULL,  cb_zoom,         ZOOM_50,     NULL},
   {N_("/75%"),                NULL,  cb_zoom,         ZOOM_75,     NULL},
   {N_("/100%"),               NULL,  cb_zoom,         ZOOM_100,    NULL},
   {N_("/125%"),               NULL,  cb_zoom,         ZOOM_125,    NULL},
   {N_("/150%"),               NULL,  cb_zoom,         ZOOM_150,    NULL},
   {N_("/175%"),               NULL,  cb_zoom,         ZOOM_175,    NULL},
   {N_("/200%"),               NULL,  cb_zoom,         ZOOM_200,    NULL},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "Rotate" sub menu */
GtkItemFactoryEntry imageview_rotate_items [] =
{
   {N_("/tear"),            NULL,  NULL,       0,           "<Tearoff>"},
   {N_("/Rotate 90 deg"),   NULL,  cb_rotate,  ROTATE_90,   NULL},
   {N_("/Rotate -90 deg"),  NULL,  cb_rotate,  ROTATE_270,  NULL},
   {N_("/Rotate 180 deg"),  NULL,  cb_rotate,  ROTATE_180,  NULL},
   {NULL, NULL, NULL, 0, NULL},
};


GList *ImageViewList;
GdkPixmap *buffer = NULL;


static void
allocate_draw_buffer (ImageView *iv)
{
   gint fwidth, fheight;

   if (!iv) return;

   if (buffer) gdk_pixmap_unref (buffer);
   imageview_get_image_frame_size (iv, &fwidth, &fheight);
   buffer = gdk_pixmap_new (iv->draw_area->window, fwidth, fheight, -1);
}


/*
 *  imageview_get_image_frame_size:
 *     @ Return frame (for drawing area) size.
 *
 *  iv     : Pointer to ImageView struct.
 *  width  : pointer to frame width  for return.
 *  height : pointer to frame height for return.
 */
static void
imageview_get_image_frame_size (ImageView *iv, gint *width, gint *height)
{
   *width = *height = 0;

   *width  = iv->draw_area->allocation.width;
   *height = iv->draw_area->allocation.height;
}


/*
 *  imageview_calc_image_size:
 *     @ calculate image size using specified scale and image frame size.
 *
 *  iv    : Pointer to ImageView struct.
 */
static void
imageview_calc_image_size (ImageView *iv)
{
   gint width, height;
   gfloat x_scale, y_scale;

   /* image scale */
   if (iv->rotate == 0 || iv->rotate == 2) {
      x_scale = iv->x_scale;
      y_scale = iv->y_scale;
   } else {
      x_scale = iv->y_scale;
      y_scale = iv->x_scale;
   }

   /* calculate image size */
   if (iv->fit_to_frame) {
      imageview_get_image_frame_size (iv, &width, &height);
      iv->x_scale = width  / (gfloat) gimv_image_width (iv->image) * 100;
      iv->y_scale = height / (gfloat) gimv_image_height (iv->image) * 100;

      if (iv->keep_aspect) {
	 if (iv->x_scale > iv->y_scale) {
	    iv->x_scale = iv->y_scale;
	    width = gimv_image_width (iv->image) * iv->x_scale / 100;
	 } else {
	    iv->y_scale = iv->x_scale;
	    height = gimv_image_height (iv->image) * iv->y_scale / 100;
	 }
      }

      iv->fit_to_frame = FALSE;
   } else {
      width  = gimv_image_width (iv->image)  * x_scale / 100;
      height = gimv_image_height (iv->image) * y_scale / 100;
   }

   /*
    * zoom out if image is too big & window is not scrollable in
    * fullscreen mode
    */
   /*
   win_width  = hadj->page_size;
   win_height = vadj->page_size;

   if (im->fullscreen && (width > win_width || height > win_height)) {
      gfloat width_ratio, height_ratio;

      width_ratio = (gdouble) width / hadj->page_size;
      height_ratio = (gdouble) height / vadj->page_size;
      if (width_ratio > height_ratio) {
	 width = win_width;
	 height = (gdouble) height / (gdouble) width_ratio;
      } else {
	 width = (gdouble) width / (gdouble) height_ratio;
	 height = win_height;
      }
   }
   */

   iv->width = width;
   iv->height = height;
}


/*
 *  imageview_rotate_render:
 *     @ Rotate image. Angle is relatively to old image.
 *
 *  iv    : Pointer to ImageView struct.
 *  angle : relative angle [1/90 deg]. Only 1, 2, 3 are recognized.
 */
static void
imageview_rotate_render (ImageView *iv, ImageAngle angle)
{
   switch (angle) {
   case ROTATE_90:
      iv->image = gimv_image_rotate_90 (iv->image, TRUE);
      break;
   case ROTATE_180:
      iv->image = gimv_image_rotate_180 (iv->image);
      break;
   case ROTATE_270:
      iv->image = gimv_image_rotate_90 (iv->image, FALSE);
      break;
   default:
      break;
   }
}


/******************************************************************************
 *
 *   Callback functions for reference popup menu.
 *
 ******************************************************************************/
/*
 *  cb_keep_aspect:
 *     @ Callback function for main menu item (/View/Zoom/Keep aspect ratio).
 *       Toggle keep aspect ratio or not when zoom image.
 *
 *  iv     : Pointer to ImageView struct.
 *  action :
 *  widget :
 */
static void
cb_keep_aspect (ImageView *iv, guint action, GtkWidget *widget)
{
   iv->keep_aspect = GTK_CHECK_MENU_ITEM(widget)->active;

   if (iv->aspect_cb_func)
      iv->aspect_cb_func (iv->aspect_cb_data);
}


/*
 *  cb_zoom:
 *     @ Callback function for main menu item (/View/Zoom/).
 *       Zoom image.
 *
 *  iv     : Pointer to ImageView struct.
 *  zoom   : Zoom type.
 *  widget :
 */
static void
cb_zoom (ImageView *iv, ImgZoomType zoom, GtkWidget *widget)
{
   imageview_zoom (iv, zoom, 0, 0);
}


/*
 *  cb_rotate:
 *     @ Callback function for main menu item (/View/Rotate/).
 *       Rotate image.
 *
 *  iv     : Pointer to ImageView struct.
 *  rotate : rotate angle (relativly) [1/90 deg]
 *  widget :
 */
static void
cb_rotate (ImageView *iv, ImageAngle rotate, GtkWidget *widget)
{
   guint angle;

   /* convert to absolute angle */
   angle = (iv->rotate + rotate) % 4;

   imageview_show_image (iv, angle);
}


/*
 *  cb_toggle_buffer:
 *     @ Callback function for popup menu item (/Memory Buffer).
 *       toggle Keep original image on memory or not.
 *
 *  iv     : Pointer to ImageView struct.
 *  action :
 *  widget :
 */
static void
cb_toggle_buffer (ImageView *iv, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      imageview_load_image_buf (iv);
      iv->buffer = TRUE;
   }
   else {
      iv->buffer = FALSE;
      imageview_free_image_buf (iv);
   }

   if (iv->buffer_cb_func)
      iv->buffer_cb_func (iv->buffer_cb_data);
}


static void
imageview_clicked (ImageView *iv, GdkEventButton *event, gpointer data)
{
   g_return_if_fail (iv);
   g_return_if_fail (event);

   if (event->button == 1) {
      if (event->state & GDK_SHIFT_MASK)
	 imageview_prev (iv);
      else
	 imageview_next (iv);
   } else if (event->button == 2) {
      imageview_prev (iv);
   }
}


static void
imageview_button4_pressed (ImageView *iv, gpointer data)   /* wheel up */
{
   g_return_if_fail (iv);

   imageview_prev (iv);
}


static void
imageview_button5_pressed (ImageView *iv, gpointer data)   /* wheel down */
{
   g_return_if_fail (iv);

   imageview_next (iv);
}


/******************************************************************************
 *
 *   other callback functions.
 *
 ******************************************************************************/
/*
 *  cb_image_configure:
 *
 *  widget :
 *  event  :
 *  iv     :
 */
static gint
cb_image_configure (GtkWidget *widget, GdkEventConfigure *event, ImageView *iv)
{
   gint fwidth, fheight;

   imageview_get_image_frame_size (iv, &fwidth, &fheight);

   if (fwidth < iv->width)
      iv->x_pos = 0;
   else
      iv->x_pos = (fwidth - iv->width) / 2;

   if (fheight < iv->height)
      iv->y_pos = 0;
   else
      iv->y_pos = (fheight - iv->height) / 2;

   imageview_draw_image (iv);

   return TRUE;
}


/*
 *  cb_image_expose:
 *     @ Callback function for Image (GtkDrawing Widget) expose event.
 *
 *  widget :
 *  event  :
 *  iv     : Pointer to ImageView struct.
 */
static gint
cb_image_expose (GtkWidget *widget, GdkEventExpose *event, ImageView *iv)
{
   if (iv->gdk_pixmap)
      imageview_draw_image (iv);

   return TRUE;
}


/*
 *  cb_image_button_press:
 *     @ Callback function for mouse button press on Image (GtkDrawing Widget).
 *
 *  widget :
 *  event  :
 *  iv     : Pointer to ImageView struct.
 */
static gint
cb_image_button_press (GtkWidget *widget, GdkEventButton *event,
		       ImageView *iv)
{
   GdkCursor *cursor;
   gint retval;

   iv->pressed = TRUE;
   iv->button  = event->button;

   gtk_widget_grab_focus (widget);

   if (iv->dragging)
      return FALSE;

   if(event->type == GDK_2BUTTON_PRESS && event->button == 1) {
      if (iv->d_button1_cb_func)
      iv->d_button1_cb_func (iv, iv->d_button1_cb_data);
   } else if (event->button == 1) {   /* scroll image */
      if (!iv->gdk_pixmap)
	 return FALSE;

      cursor = cursor_get (widget->window, CURSOR_HAND_CLOSED);
      retval = gdk_pointer_grab (widget->window, FALSE,
				 (GDK_POINTER_MOTION_MASK
				  | GDK_POINTER_MOTION_HINT_MASK
				  | GDK_BUTTON_RELEASE_MASK),
				 NULL, cursor, event->time);
      gdk_cursor_destroy (cursor);

      if (retval != 0)
	 return FALSE;

      iv->drag_startx = event->x - iv->x_pos;
      iv->drag_starty = event->y - iv->y_pos;

      allocate_draw_buffer (iv);

      return TRUE;
   } else if(event->type == GDK_BUTTON_PRESS && event->button == 2) {
      if (iv->button2_cb_func)
	 iv->button2_cb_func (iv, iv->button2_cb_data);
   } else if(event->type == GDK_BUTTON_PRESS && event->button == 3) {   /* popup menu */
      if (iv->imageview_popup)
	 gtk_menu_popup(GTK_MENU(iv->imageview_popup),
			NULL, NULL,
			NULL, NULL,
			event->button, event->time);
   } else if(event->type == GDK_BUTTON_PRESS && event->button == 4) {
      if (iv->button4_cb_func)
	 iv->button4_cb_func (iv, iv->button4_cb_data);
   } else if(event->type == GDK_BUTTON_PRESS && event->button == 5) {
      if (iv->button5_cb_func)
	 iv->button5_cb_func (iv, iv->button5_cb_data);
   }

   return FALSE;
}


/*
 *  cb_image_button_release:
 *     @ Callback function for mouse button release on Image
 *       (GtkDrawing Widget).
 *
 *  widget :
 *  event  :
 *  iv     : Pointer to ImageView struct.
 */
static gint
cb_image_button_release  (GtkWidget *widget, GdkEventButton *event,
			  ImageView *iv)
{
   /*
   if (event->button != 1)
      return FALSE;
   */

   if(iv->pressed && !iv->dragging) {
      if (iv->button_clicked_cb_func)
	 iv->button_clicked_cb_func (iv, event, iv->button_clicked_cb_data);
   }

   /*
   if (!im->dragging)
      gtk_signal_emit_by_name (GTK_OBJECT (im->event_box), "button_press_event");
   */

   if (buffer) {
      gdk_pixmap_unref (buffer);
      buffer = NULL;
   }

   iv->button   = 0;
   iv->pressed  = FALSE;
   iv->dragging = FALSE;
   gdk_pointer_ungrab (event->time);

   return TRUE;
}


/*
 *  cb_image_motion_notify:
 *     @ for image drag and scroll.
 *
 *  widget :
 *  event  :
 *  iv     : Pointer to ImageView struct.
 */
static gint
cb_image_motion_notify (GtkWidget *widget, GdkEventMotion *event,
			ImageView *iv)
{
   GdkModifierType mods;
   gint x, y;
   gint fwidth, fheight;

   if (!iv->pressed)
      return FALSE;

   iv->dragging = TRUE;

   /* scroll image */
   if (iv->button == 1) {
      if (event->is_hint)
	 gdk_window_get_pointer (widget->window, &x, &y, &mods);

      iv->x_pos = x - iv->drag_startx;
      iv->y_pos = y - iv->drag_starty;

      if (!conf.imgview_scroll_nolimit) {
	 imageview_get_image_frame_size (iv, &fwidth, &fheight);

	 if (iv->width <= fwidth) {
	    if (iv->x_pos < 0)
	       iv->x_pos = 0;
	    if (iv->x_pos > fwidth - iv->width)
	       iv->x_pos = fwidth - iv->width;
	 } else if (iv->x_pos < fwidth - iv->width) {
	    iv->x_pos = fwidth - iv->width;
	 } else if (iv->x_pos > 0) {
	    iv->x_pos = 0;
	 }

	 if (iv->height <= fheight) {
	    if (iv->y_pos < 0)
	       iv->y_pos = 0;
	    if (iv->y_pos > fheight - iv->height)
	       iv->y_pos = fheight - iv->height;
	 } else if (iv->y_pos < fheight - iv->height) {
	    iv->y_pos = fheight - iv->height;
	 } else if (iv->y_pos > 0) {
	    iv->y_pos = 0;
	 }
      }
   }

   imageview_draw_image (iv);

   return TRUE;
}


static void
cb_imageview_closed (GtkWidget *widget, ImageView *iv)
{
   ImageViewList = g_list_remove (ImageViewList, iv);

   if (iv->list_owner)
      imageview_remove_list (iv, iv->list_owner);

   imageview_free (iv);
}


static void
cb_drag_data_received (GtkWidget *widget,
		       GdkDragContext *context,
		       gint x, gint y,
		       GtkSelectionData *seldata,
		       guint info,
		       guint32 time,
		       gpointer data)
{
   ImageView *iv = data;
   OpenFiles *files;
   GList *filelist;
   gchar *tmpstr;

   g_return_if_fail (iv || widget);

   filelist = dnd_get_file_list (seldata->data);

   if (filelist) {
      imageview_change_image_file (iv, (gchar *) filelist->data);
      imageview_show_image (iv, iv->rotate);
      tmpstr = filelist->data;
      filelist = g_list_remove (filelist, tmpstr);
      g_free (tmpstr);

      if (filelist) {
	 files = files_loader_new ();
	 files->filelist = filelist;
	 open_image_files_in_image_view (files);
	 files->filelist = NULL;
	 files_loader_delete (files);
      }
   }

   g_list_foreach (filelist, (GFunc) g_free, NULL);
   g_list_free (filelist);
}



/******************************************************************************
 *
 *   Public functions.
 *
 ******************************************************************************/
/*
 *  imageview_create_zoom_menu:
 *     @ create referece "Zoom" sub menu.
 *
 *  window : Parent window.
 *  iv     :
 *  Return : GtkMenu widget.
 */
GtkWidget *
imageview_create_zoom_menu (GtkWidget *window, ImageView *iv)
{
   GtkWidget *menu;
   guint n_menu_items;

   n_menu_items = sizeof(imageview_zoom_items)
                     / sizeof(imageview_zoom_items[0]) - 1;
   menu = menu_create_items(window, imageview_zoom_items,
			    n_menu_items, "<ZoomSubMenu>", iv);
   menu_check_item_set_active (menu, "/Keep aspect ratio", iv->keep_aspect);

   return menu;
}


/*
 *  imageview_create_rotate_menu:
 *     @ create reference "Rotate" sub menu.
 *
 *  window : Parent window.
 *  iv     :
 *  Return : GtkMenu widget.
 */
GtkWidget *
imageview_create_rotate_menu (GtkWidget *window, ImageView *iv)
{
   GtkWidget *menu;
   guint n_menu_items;

   n_menu_items = sizeof(imageview_rotate_items)
                     / sizeof(imageview_rotate_items[0]) - 1;
   menu = menu_create_items(window, imageview_rotate_items,
			    n_menu_items, "<RotateSubMenu>", iv);

   return menu;
}


/*
 *  imageview_create_popup_menu:
 *     @ create reference popup menu.
 *
 *  window : Parent window.
 *  iv     :
 *  Return : GtkMenu widget.
 */
GtkWidget *
imageview_create_popup_menu (GtkWidget *window, ImageView *iv)
{
   guint n_menu_items;

   n_menu_items = sizeof(imageview_popup_items)
                     / sizeof(imageview_popup_items[0]) - 1;
   iv->imageview_popup = menu_create_items(window, imageview_popup_items,
					   n_menu_items, "<ViewSubMenu>", iv);

   iv->zoom_menu   = imageview_create_zoom_menu (window, iv);
   menu_set_submenu (iv->imageview_popup, "/Zoom",   iv->zoom_menu);

   iv->rotate_menu = imageview_create_rotate_menu (window, iv);
   menu_set_submenu (iv->imageview_popup, "/Rotate", iv->rotate_menu);

   menu_check_item_set_active (iv->imageview_popup, "/Memory Buffer", iv->buffer);

   return iv->imageview_popup;
}


/*
 *  imageview_free:
 *     @ free ImageView struct.
 *
 *  iv :
 */
void
imageview_free (ImageView *iv)
{
   if (!iv)
      return;
   gimv_image_free_pixmap_and_mask (iv->gdk_pixmap, iv->mask);
   iv->gdk_pixmap = NULL;
   iv->mask = NULL;

   if (iv->image)
      gimv_image_kill (iv->image);
   iv->image = NULL;

   if (iv->cursor)
      gdk_cursor_destroy (iv->cursor);

   /* decrease ref count of image info */
   if (iv->info)
      image_info_unref (iv->info);
   iv->info = NULL;

   g_free (iv);
}


/*
 *  imageview_free_image_buf:
 *     @ free image buffer.
 *
 *  iv : Pointer to ImageView struct.
 */
void
imageview_free_image_buf (ImageView *iv)
{
   if (!iv)
      return;

   if (iv->image && iv->buffer)
      return;

   gimv_image_kill (iv->image);
   iv->image = NULL;
}


/*
 *  imageview_load_image_buf:
 *     @ load image as GimvImage (GdkImlibImage, GdkPixbuf, ...)
 *       struct for rendering.
 *
 *  iv : Pointer to ImageView struct.
 */
void
imageview_load_image_buf (ImageView *iv)
{
   GimvImage *image;

   g_return_if_fail (iv);

   if (!iv->info) return;
   if (!g_list_find (ImageViewList, iv)) return;

   if (!iv->image && !iv->buffer) {
      image = gimv_image_load_file (iv->info->filename);
      if (!g_list_find (ImageViewList, iv)) {
	 gimv_image_kill (image);
	 return;
      } else {
	 iv->image = image;
      }

      imageview_rotate_render (iv, iv->rotate);
   }
}


/*
 *  imageview_change_image_file:
 *     @ Change to new image file.
 *
 *  iv      : Pointer to ImageView struct.
 *  newfile : new image file name.
 */
void
imageview_change_image_file (ImageView *iv, const gchar *newfile)
{
   GimvImage *image;

   g_return_if_fail (iv && newfile);

   if (!g_list_find (ImageViewList, iv)) return;

   /* free old image */
   if (iv->gdk_pixmap)
      gimv_image_free_pixmap_and_mask (iv->gdk_pixmap, iv->mask);
   iv->gdk_pixmap = NULL;
   iv->mask = NULL;

   if (iv->image)
      gimv_image_kill (iv->image);
   iv->image = NULL;

   /* free old image info */
   if (iv->info)
      image_info_unref (iv->info);

   /* reset fit_to_widnow */
   iv->fit_to_frame = conf.imgview_fit_image_to_win;

   /* allocate memory for new image info*/
   iv->info = image_info_get (newfile);

   /* load image */
   image = gimv_image_load_file ((gchar *) newfile);

   if (!g_list_find (ImageViewList, iv)) {
      gimv_image_kill (iv->image);
      return;
   } else {
      iv->image = image;
   }

   if (!iv->image) {
      g_print(_("Not an image (or unsupported) file: %s\n"), newfile);
      iv->image = NULL;

      /* clear image draw area */
      imageview_draw_image (iv);

      return;
   }

   /* reset ImageInfo struct data */
   image_info_set_data_from_image (iv->info, iv->image);

   return;
}


/*
 *  imageview_zoom:
 *     @ set image scale, render, and display it.
 *
 *  iv      :
 *  zoom    : zoom type (ZOOM_IN, ZOOM_OUT, ZOOM_10, ZOOM_25, ...)
 *  x_scale : not used yet
 *  y_scale : not used yet
 */
void
imageview_zoom (ImageView *iv, ImgZoomType zoom, gint x_scale, gint y_scale)
{
   switch (zoom) {
   case ZOOM_IN:
      if (iv->x_scale < IMG_MAX_SCALE && iv->y_scale < IMG_MAX_SCALE) {
	 iv->x_scale = iv->x_scale + IMG_MIN_SCALE;
	 iv->y_scale = iv->y_scale + IMG_MIN_SCALE;
      }
      break;
   case ZOOM_OUT:
      if (iv->x_scale > IMG_MIN_SCALE && iv->y_scale > IMG_MIN_SCALE) {
	 iv->x_scale = iv->x_scale - IMG_MIN_SCALE;
	 iv->y_scale = iv->y_scale - IMG_MIN_SCALE;
      }
      break;
   case ZOOM_FIT:
      iv->fit_to_frame = TRUE;
      break;
   case ZOOM_10:
      iv->x_scale = iv->y_scale =  10;
      break;
   case ZOOM_25:
      iv->x_scale = iv->y_scale =  25;
      break;
   case ZOOM_50:
      iv->x_scale = iv->y_scale =  50;
      break;
   case ZOOM_75:
      iv->x_scale = iv->y_scale =  75;
      break;
   case ZOOM_100:
      iv->x_scale = iv->y_scale = 100;
      break;
   case ZOOM_125:
      iv->x_scale = iv->y_scale = 125;
      break;
   case ZOOM_150:
      iv->x_scale = iv->y_scale = 150;
      break;
   case ZOOM_175:
      iv->x_scale = iv->y_scale = 175;
      break;
   case ZOOM_200:
      iv->x_scale = iv->y_scale = 200;
      break;	 
   case ZOOM_300:
      iv->x_scale = iv->y_scale = 300;
      break;	 
   default:
      break;
   }

   if (zoom != ZOOM_FIT)
      iv->fit_to_frame = FALSE;

   imageview_show_image (iv, iv->rotate);
}


/*
 *  imageview_draw_image:
 *     @ draw image using double buffer.
 *
 *  iv : Pointer to ImageView struct.
 */
void
imageview_draw_image (ImageView *iv)
{
   gboolean free = FALSE;

   /* allocate buffer */
   if (!buffer) {
      allocate_draw_buffer (iv);
      free = TRUE;
   }

   /* fill background by default bg color */
   gdk_draw_rectangle (buffer, iv->draw_area->style->bg_gc[GTK_WIDGET_STATE (iv->draw_area)],
		       TRUE, 0, 0, -1, -1);

   /* draw image to buffer */
   if (iv->gdk_pixmap) {
      if (iv->mask) {
	 gdk_gc_set_clip_mask (iv->draw_area->style->black_gc, iv->mask);
	 gdk_gc_set_clip_origin (iv->draw_area->style->black_gc, iv->x_pos, iv->y_pos);
      }

      gdk_draw_pixmap (buffer,
		       iv->draw_area->style->black_gc,
		       iv->gdk_pixmap, 0, 0, iv->x_pos, iv->y_pos, -1, -1);

      if (iv->mask) {
	 gdk_gc_set_clip_mask (iv->draw_area->style->black_gc, NULL);
	 gdk_gc_set_clip_origin (iv->draw_area->style->black_gc, 0, 0);
      }
   }

   /* draw from buffer to foreground */
   gdk_draw_pixmap (iv->draw_area->window,
		    iv->draw_area->style->fg_gc[GTK_WIDGET_STATE (iv->draw_area)],
		    buffer, 0, 0, 0, 0, -1, -1);

   /* free buffer */
   if (free) {
      gdk_pixmap_unref (buffer);
      buffer = NULL;
   }
}


/*
 *  imageview_show_image:
 *     @ render image and display it.
 *
 *  iv    : Pointer to ImageView struct.
 *  abgle : Absolute image angle from original image.
 */
void
imageview_show_image (ImageView *iv, ImageAngle angle)
{
   ImageAngle rotate_angle;
   gint width, height;

   while (gtk_events_pending ()) gtk_main_iteration ();

   imageview_load_image_buf (iv);

   if (!iv->image)
      goto func_end;

   /* rotate image */
   rotate_angle = (angle - iv->rotate) % 4;
   if (rotate_angle < 0)
      rotate_angle = rotate_angle + 4;

   iv->rotate = angle;
   imageview_rotate_render (iv, rotate_angle);

   /* set image size */
   imageview_calc_image_size (iv);

   /* image rendering */
   gimv_image_free_pixmap_and_mask (iv->gdk_pixmap, iv->mask);
   gimv_image_scale_get_pixmap (iv->image, iv->width, iv->height,
				&iv->gdk_pixmap, &iv->mask);

   /* reset image geometory */
   imageview_get_image_frame_size (iv, &width, &height);
   if (iv->width <= width)
      iv->x_pos = (width - iv->width) / 2;
   else
      iv->x_pos = 0;
   if (iv->height <= height)
      iv->y_pos = (height - iv->height) / 2;
   else
      iv->y_pos = 0;

   /* draw image */
   imageview_draw_image (iv);

   imageview_free_image_buf (iv);

   /* set cursor */
   if (!iv->cursor)
      iv->cursor = cursor_get (iv->draw_area->window, CURSOR_HAND_OPEN);
   gdk_window_set_cursor (iv->draw_area->window, iv->cursor);

 func_end:
   if (iv->render_cb_func)
      iv->render_cb_func (iv->render_cb_data);
}


void
imageview_set_list (ImageView       *iv,
		    GList           *list,
		    GList           *current,
		    gpointer         list_owner,
		    ImageViewNextFn  next_fn,
		    gpointer         next_fn_data,
		    ImageViewPrevFn  prev_fn,
		    gpointer         prev_fn_data,
		    ImageViewNthFn   nth_fn,
		    gpointer         nth_fn_data,
		    ImageViewRemoveListFn remove_list_fn,
		    gpointer              remove_list_fn_data)
{
   g_return_if_fail (iv);
   g_return_if_fail (list);
   g_return_if_fail (current);

   if (iv->list_owner)
      imageview_remove_list (iv, iv->list_owner);

   iv->image_list = list;
   if (!current)
      iv->image_list = list;
   else
      iv->image_current = current;

   iv->list_owner   = list_owner;

   iv->next_fn      = next_fn;
   iv->next_fn_data = next_fn_data;

   iv->prev_fn      = prev_fn;
   iv->prev_fn_data = prev_fn_data;

   iv->nth_fn       = nth_fn;
   iv->nth_fn_data  = nth_fn_data;

   iv->remove_list_fn      = remove_list_fn;
   iv->remove_list_fn_data = remove_list_fn_data;
}


void
imageview_remove_list (ImageView *iv, gpointer list_owner)
{
   g_return_if_fail (iv);

   if (iv->remove_list_fn && iv->list_owner == list_owner)
      iv->remove_list_fn (iv, iv->list_owner, iv->remove_list_fn_data);

   iv->image_list    = NULL;
   iv->image_current = NULL;
   iv->list_owner    = NULL;

   iv->next_fn       = NULL;
   iv->next_fn_data  = NULL;
   iv->prev_fn       = NULL;
   iv->prev_fn_data  = NULL;
   iv->nth_fn        = NULL;
   iv->nth_fn_data   = NULL;

   iv->remove_list_fn      = NULL;
   iv->remove_list_fn_data = NULL;
}


void
imageview_next (ImageView *iv)
{
   g_return_if_fail (iv);
   if (!iv->next_fn) return;

   iv->image_current = iv->next_fn (iv, iv->list_owner,
				    iv->image_current, iv->next_fn_data);
}


void
imageview_prev (ImageView *iv)
{
   g_return_if_fail (iv);
   if (!iv->prev_fn) return;

   iv->image_current = iv->prev_fn (iv, iv->list_owner,
				    iv->image_current, iv->prev_fn_data);
}


void
imageview_nth (ImageView *iv, guint nth)
{
   g_return_if_fail (iv);
   if (!iv->prev_fn) return;

   iv->image_current = iv->nth_fn (iv, iv->image_list, nth, iv->nth_fn_data);
}


/*
 *  imageview_create:
 *     @ create new image view module.
 *
 *  filename  : File name for open. If this value is NULL, create blank canvas.
 *  Return    : Pointer to created ImageView struct.
 */
ImageView *
imageview_create (const gchar *filename)
{
   ImageView *iv;
   GimvImage *image = NULL;

   while (gtk_events_pending()) gtk_main_iteration();

   /* allocate memory */
   iv = g_new0 (ImageView, 1);
   if (!iv)
      return NULL;

   if (filename) {
      iv->info = image_info_get (filename);
      if (!iv->info) {
	 g_free (iv);
	 return NULL;
      }
   } else {
      iv->info = NULL;
   }

   /* initialize struct data based on config */
   iv->fit_to_frame  = conf.imgview_fit_image_to_win;
   iv->x_scale       = conf.imgview_scale;
   iv->y_scale       = conf.imgview_scale;
   iv->keep_aspect   = conf.imgview_keep_aspect;
   iv->rotate        = 0;
   iv->buffer        = conf.imgview_buffer;

   iv->cursor          = NULL;
   iv->imageview_popup = NULL;
   iv->zoom_menu       = NULL;
   iv->rotate_menu     = NULL;

   iv->render_cb_func = iv->render_cb_data = NULL;
   iv->aspect_cb_func = iv->aspect_cb_data = NULL;
   iv->buffer_cb_func = iv->buffer_cb_data = NULL;

   iv->button_clicked_cb_func = imageview_clicked;
   iv->button_clicked_cb_data = NULL;
   iv->d_button1_cb_func      = NULL;
   iv->d_button1_cb_func      = NULL;
   iv->button2_cb_func        = NULL;
   iv->button2_cb_data        = NULL;
   iv->button4_cb_func        = imageview_button4_pressed;
   iv->button4_cb_data        = NULL;
   iv->button5_cb_func        = imageview_button5_pressed;
   iv->button5_cb_data        = NULL;

   iv->image_list    = NULL;
   iv->image_current = NULL;

   iv->next_fn = iv->next_fn_data = NULL;
   iv->prev_fn = iv->prev_fn_data = NULL;
   iv->nth_fn  = iv->nth_fn_data  = NULL;

   /* create imageview container */
   iv->event_box = gtk_event_box_new ();

   iv->draw_area = gtk_drawing_area_new ();
   gtk_container_add (GTK_CONTAINER (iv->event_box), iv->draw_area);
   gtk_widget_show (iv->draw_area);

   /* set signals */
   gtk_signal_connect (GTK_OBJECT (iv->draw_area), "configure_event",
		       GTK_SIGNAL_FUNC (cb_image_configure), iv);

   gtk_signal_connect (GTK_OBJECT (iv->draw_area), "expose_event",
		       GTK_SIGNAL_FUNC (cb_image_expose), iv);

   gtk_signal_connect (GTK_OBJECT (iv->event_box), "button_press_event",
		       GTK_SIGNAL_FUNC (cb_image_button_press), iv);

   gtk_signal_connect (GTK_OBJECT (iv->event_box), "button_release_event",
		       GTK_SIGNAL_FUNC (cb_image_button_release), iv);

   gtk_signal_connect (GTK_OBJECT (iv->event_box), "motion_notify_event",
		       GTK_SIGNAL_FUNC (cb_image_motion_notify), iv);

   gtk_signal_connect (GTK_OBJECT (iv->event_box), "destroy",
		       GTK_SIGNAL_FUNC (cb_imageview_closed), iv);

   /* for droping file list */
   gtk_signal_connect(GTK_OBJECT (iv->event_box), "drag_data_received",
		      GTK_SIGNAL_FUNC (cb_drag_data_received), iv);
   dnd_dest_set (iv->event_box, dnd_types, dnd_types_num);

   /* load image */
   if (filename) {
      image = gimv_image_load_file ((gchar *) filename);
      if (image) {
	 iv->image = image;
	 image_info_set_data_from_image (iv->info, iv->image);
      } else {
	 if (conf.disp_filename_stdout)
	    g_print(_("Not an image (or unsupported) file: %s\n"), filename);
      }
   }

   ImageViewList = g_list_append (ImageViewList, iv);

   return iv;
}
