/* GWMHints
|  A plugin for GKrellM for setting Gnome Window Manager Hints
|
|  Copyright (C) 2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define PACKAGE	"gkrellm-gnome"

#include <gkrellm/gkrellm.h>

/* Need GKrellM version 1.0.4 or better!
*/
#if (GKRELLM_VERSION_MAJOR<1) || (GKRELLM_VERSION_MAJOR==1 \
	&& GKRELLM_VERSION_MINOR==0 && GKRELLM_VERSION_REV<4)
#error This plugin requires GKrellM version >= 1.0.4
#endif

#include <gnome.h>

#define	GKRELLM_GNOME_VERSION_MAJOR		0
#define	GKRELLM_GNOME_VERSION_MINOR		1


#define CONFIG_NAME				"GKrellM Gnome"
#define PLUGIN_CONFIG_KEYWORD	"gkrellm-gnome"

GnomeWinHints	current_hints;
GnomeWinState	current_state;

gboolean		skip_task_bar,
				skip_win_list,
				skip_focus,
				sticky,
				layer_on_top;

GnomeClient		*client;


/* From http://developer.gnome.org/doc/tutorials/gnome-libs/
		using-the-libgnomeui-library.html#AEN1364
*/
static gint
save_yourself(GnomeClient *client, gint phase, GnomeRestartStyle save_style,
			gint shutdown, GnomeInteractStyle interact_style,
			gint fast, gpointer client_data)
	{
	gchar	*prefix	= gnome_client_get_config_prefix(client);
	gchar	*argv[]	= { "rm", "-r", NULL, NULL, NULL, NULL, NULL, NULL, NULL};
	gint	argc;

	/* Save the state using gnome-config stuff.  Probably dont' need to do
	|  this at all since I don't save any data through gnome with
	|  gnome_config_set_int("Section/Key",some_value);
	|  I save all my own stuff with gkrellm_save_all().
	*/
	gnome_config_push_prefix(prefix);
	/* ... gnome_config_set_int()	*/
	gnome_config_pop_prefix();
	gnome_config_sync();

	/* Here is the real SM code. We set the argv to the parameters needed
	|  to restart/discard the session that we've just saved and call the
	|  gnome_session_set_*_command to tell the session manager.
	*/
	argv[2] = gnome_config_get_real_path(prefix);
	gnome_client_set_discard_command(client, 3, argv);

	/* Set commands to clone and restart this application.  Note that we
	|  use the same values for both -- the session management code will
	|  automatically add whatever magic option is required to set the
	|  session id on startup.  The client_data was set to the command used
	|  to start this app when the save_yourself handler was connected.
	*/
	argv[0] = (char*) client_data;
	argc = gkrellm_get_restart_options(&argv[1],
				sizeof(argv) / sizeof(gchar *) - 1);
	gnome_client_set_clone_command(client, argc + 1, argv);
	gnome_client_set_restart_command(client, argc + 1, argv);

	gkrellm_save_all();

	return TRUE;
	}

static void
client_die(GnomeClient *client, gpointer client_data)
	{
	gtk_exit(0);
	}

static void
create_gkrellm_gnome(GtkWidget *vbox, gint first_create)
	{
	GnomeWinHints	hints;
	GtkWidget		*top_win;
	gchar			**argv;
	static gboolean	gnome_initialized;	/* avoid enable/disable cycling */

	if (gnome_initialized)
		return;
	argv = gkrellm_get_sm_argv();
	gnome_init("gkrellm", "1.0.3", gkrellm_get_sm_argc(), argv);
	gnome_win_hints_init();
	gnome_initialized = TRUE;

	top_win = gkrellm_get_top_window();
	hints = 0;
	if (skip_task_bar)
		hints |= WIN_HINTS_SKIP_TASKBAR;
	if (skip_win_list)
		hints |= WIN_HINTS_SKIP_WINLIST;				
	if (skip_focus)
		hints |= WIN_HINTS_SKIP_FOCUS;
	if (hints)
		gnome_win_hints_set_hints(GTK_WIDGET(top_win), hints);
	current_hints = hints;

	if (layer_on_top)
		gnome_win_hints_set_layer(GTK_WIDGET(top_win), WIN_LAYER_ONTOP);

	if (sticky)
		gnome_win_hints_set_state(GTK_WIDGET(top_win), WIN_STATE_STICKY);

	client = gnome_master_client ();
	gtk_signal_connect(GTK_OBJECT(client), "save_yourself",
			GTK_SIGNAL_FUNC(save_yourself), (gpointer) argv[0]);
	gtk_signal_connect(GTK_OBJECT(client), "die",
			GTK_SIGNAL_FUNC(client_die), NULL);
	}

static void
save_gkrellm_gnome_config(FILE *f)
	{
	fprintf(f, "%s skip_task_bar %d\n", PLUGIN_CONFIG_KEYWORD, skip_task_bar);
	fprintf(f, "%s skip_win_list %d\n", PLUGIN_CONFIG_KEYWORD, skip_win_list);
	fprintf(f, "%s skip_focus %d\n", PLUGIN_CONFIG_KEYWORD, skip_focus);
	fprintf(f, "%s sticky %d\n", PLUGIN_CONFIG_KEYWORD, sticky);
	fprintf(f, "%s layer_on_top %d\n", PLUGIN_CONFIG_KEYWORD, layer_on_top);
	}

static void
load_gkrellm_gnome_config(gchar *arg)
	{
	gchar	config[64], item[256];
	gint	n;

	n = sscanf(arg, "%s %[^\n]", config, item);
	if (n == 2)
		{
		n = 0;
		if (strcmp(config, "skip_task_bar") == 0)
			sscanf(item, "%d\n", &skip_task_bar);
		else if (strcmp(config, "skip_win_list") == 0)
			sscanf(item, "%d\n", &skip_win_list);
		else if (strcmp(config, "skip_focus") == 0)
			sscanf(item, "%d\n", &skip_focus);
		else if (strcmp(config, "sticky") == 0)
			sscanf(item, "%d\n", &sticky);
		else if (strcmp(config, "layer_on_top") == 0)
			sscanf(item, "%d\n", &layer_on_top);
		}
	}



/* --------------------------------------------------------------------- */
static GtkWidget    *skip_task_bar_button,
					*skip_win_list_button,
					*skip_focus_button,
					*sticky_button,
					*layer_on_top_button;
 
static void
apply_gkrellm_gnome_config()
    {
	GnomeWinHints	hints;
	GnomeWinState	state;
	GtkWidget		*top_win;
	gint			enabled;

	hints = 0;
	state = 0;

	top_win = gkrellm_get_top_window();
	skip_task_bar = GTK_TOGGLE_BUTTON(skip_task_bar_button)->active;
	if (skip_task_bar)
		hints |= WIN_HINTS_SKIP_TASKBAR;
	skip_win_list = GTK_TOGGLE_BUTTON(skip_win_list_button)->active;
	if (skip_win_list)
		hints |= WIN_HINTS_SKIP_WINLIST;				
	skip_focus = GTK_TOGGLE_BUTTON(skip_focus_button)->active;
	if (skip_focus)
		hints |= WIN_HINTS_SKIP_FOCUS;
	if (hints != current_hints)
		gnome_win_hints_set_hints(GTK_WIDGET(top_win), hints);
	current_hints = hints;

	enabled = GTK_TOGGLE_BUTTON(layer_on_top_button)->active;
	if (enabled != layer_on_top)
		gnome_win_hints_set_layer(GTK_WIDGET(top_win),
				enabled ? WIN_LAYER_ONTOP : WIN_LAYER_NORMAL);
	layer_on_top = enabled;

	sticky = GTK_TOGGLE_BUTTON(sticky_button)->active;
	if (sticky)
		state = WIN_STATE_STICKY;
	if (state != current_state)
		gnome_win_hints_set_state(GTK_WIDGET(top_win), state);
	current_state = state;
	}

static gchar	*gkrellm_gnome_info_text[] =
{
N_("<b>Hints for Gnome Compliant Window Managers\n"),
"\n",
N_("With this plugin you can set various hints which your window manager\n"
"will recognize if it is Gnome compliant.  While some of these may be easily\n"
"set with your window manager, this plugin provides a convenient grouping\n"
"of window and Gnome desktop manager functions.\n"),
"\n",
N_("It's best to think of these as startup hints because some of them may be\n"
"recognized by your window or desktop manager only when GKrellM is started.\n"),
"\n",
N_("<b>Gnome Session\n"),
"\n",
N_("<i>Enabling\n"),
N_("\tIf gnome-session is running and this plugin is enabled, GKrellM will\n"
"\tconnect to the session manager so that it can be saved as part of your\n"
"\tdesktop setup when you save a session.  Then when you next login,\n"
"\tGKrellM will be automatically started.\n"),
"\n",
N_("<i>Disabling\n"),
N_("\tIf you disable this plugin, then when GKrellM is started it will\n"
"\tno longer connect to the session manager.  But if you have previously\n"
"\tsaved a GKrellM session, then the session manager will still start\n"
"\tGKrellM and it will have an unknown state.  So, you should either\n"
"\tremove GKrellM from the session manager properties or save another\n"
"\tsession without GKrellM running.\n")
};

static void
create_gkrellm_gnome_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs;
	GtkWidget	*frame;
	GtkWidget	*vbox, *vbox1;
	GtkWidget	*scrolled;
	GtkWidget	*text;
	GtkWidget	*label;
	gchar		*buf;
	gint		i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Setup Tab */
	vbox = gkrellm_create_tab(tabs, _("Hints"));

	frame = gtk_frame_new(_("Window Manager Hints"));
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
	vbox1 = gtk_vbox_new (FALSE, 2);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);
	gkrellm_check_button(vbox1, &layer_on_top_button, layer_on_top, FALSE, 0,
			_("Always on top layer"));
	gkrellm_check_button(vbox1, &sticky_button, sticky, FALSE, 0,
			_("Make sticky (visible on all desktops)"));
	gkrellm_check_button(vbox1, &skip_win_list_button, skip_win_list, FALSE, 0,
			_("Don't include in the window list"));
	gkrellm_check_button(vbox1, &skip_focus_button, skip_focus, FALSE, 0,
			_("Skip focus when doing \"alt-tab\""));

	frame = gtk_frame_new(_("Gnome Task List Hints"));
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
	vbox1 = gtk_vbox_new (FALSE, 2);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);
	gkrellm_check_button(vbox1, &skip_task_bar_button, skip_task_bar, FALSE, 0,
			_("Don't put on the task list (or bar)"));

/* --Info Tab */
	vbox = gkrellm_create_tab(tabs, _("Info"));
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	for (i = 0; i < sizeof(gkrellm_gnome_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(gkrellm_gnome_info_text[i]));
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);
 
/* --About Tab */
	vbox = gkrellm_create_tab(tabs, _("About"));
	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

	buf = g_strdup_printf(_("GKrellM-Gnome %d.%d\n\n"
				"Copyright (c) 2001 by Bill Wilson\n"
				"bill@gkrellm.net\n"
				"http://gkrellm.net\n\n"
				"Released under the GNU Public License"),
				GKRELLM_GNOME_VERSION_MAJOR, GKRELLM_GNOME_VERSION_MINOR);
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	}


static Monitor  gkrellm_gnome_mon =
    {
	N_(CONFIG_NAME),			/* Name, for config tab.    */
	0,							/* Id,  0 if a plugin       */
    create_gkrellm_gnome, 		/* The create_plugin() function */
    NULL,						/* The update_plugin() function */
    create_gkrellm_gnome_tab,	/* The create_plugin_tab() config function */
    apply_gkrellm_gnome_config,	/* The apply_plugin_config() function      */

    save_gkrellm_gnome_config,	/* The save_plugin_config() function  */
    load_gkrellm_gnome_config,	/* The load_plugin_config() function  */
    PLUGIN_CONFIG_KEYWORD,		/* config keyword                     */

    NULL,           /* Undefined 2  */
    NULL,           /* Undefined 1  */
    NULL,           /* private		*/

    MON_CLOCK,		/* Insert plugin before this monitor.       */
    NULL,			/* Handle if a plugin, filled in by GKrellM */
    NULL			/* path if a plugin, filled in by GKrellM   */
	};

Monitor *
init_plugin(void)
    {
	gkrellm_gnome_mon.name = _(gkrellm_gnome_mon.name);
    return &gkrellm_gnome_mon;
	}
