/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*******************************************************
 * FTW implementation that your system shouldn't lack. *
 *******************************************************/

#include <dirent.h>             /* DIR, opendir(), readdir(), closedir() */
#include <stdlib.h>             /* realloc(), malloc(), free() */
#include <string.h>             /* strlen() */
#include <stdio.h>              /* sprintf() */
#include "ftw.h"

/* List of processed dirs. */
static struct {
    dev_t st_dev;
    ino_t st_ino;
} *list;
static int list_length;         /* Its length. */

static int ftw_rec(const char *filename, ftw_func func);

static void append_to_list(dev_t dev, ino_t ino)
{
    list = realloc(list, (list_length + 1) * sizeof(*list));
    list[list_length].st_dev = dev;
    list[list_length].st_ino = ino;
    list_length++;
}

/* Return 1 if the dir is already is in the list. */
static int is_in_list(dev_t dev, ino_t ino)
{
    int i;

    for (i = 0; i < list_length; i++) {
        if (list[i].st_ino == ino && list[i].st_dev == dev)
            return 1;
    }
    return 0;
}

/* Returns 0 if path is neither ".." nor "." to avoid infinite recursion. */
static int path_ok(const char *path)
{
    if (path == NULL)
        return -1;

    if (path[0] == '.') {
        if ((path[1] == '\0') || (path[1] == '.' && path[2] == '\0'))
            return -1;
    }
    return 0;
}

static int ftw_dir(const char *dirname, ftw_func func)
{
    struct dirent *dir_ent;
    DIR *dir;
    int dir_len;
    char *name;
    struct stat st;
    int ret = 0;

    stat(dirname, &st);
    append_to_list(st.st_dev, st.st_ino);

    /* 2 = (1 for '/') + (1 for '\0') */
    dir_len = strlen(dirname) + 2;

    dir = opendir(dirname);
    if (dir == NULL)
        /* Item is a directory which can't be read. */
        return (*func) (dirname, &st, FTW_DNR);

    dir_ent = readdir(dir);

    while (dir_ent != NULL && ret == 0) {
        if (path_ok(dir_ent->d_name) == 0) {
            name = malloc((dir_len + strlen(dir_ent->d_name)) * sizeof(char));
            sprintf(name, "%s/%s", dirname, dir_ent->d_name);
            ret = ftw_rec(name, func);
            free(name);
        }
        dir_ent = readdir(dir);
    }
    closedir(dir);

    return ret;
}

static int ftw_rec(const char *filename, ftw_func func)
{
    struct stat st;

    lstat(filename, &st);
    if (S_ISLNK(st.st_mode)) {
        /* We have a symbolic link. */
        stat(filename, &st);
        if (S_ISDIR(st.st_mode) && is_in_list(st.st_dev, st.st_ino)) {
            /*
             * We already traversed this directory so
             * adding it may cause infinite recursion.
             */
            return 0;
        }
    }

    if (S_ISDIR(st.st_mode)) {
        ftw_dir(filename, func);
        return (*func) (filename, &st, FTW_D);
    }

    return (*func) (filename, &st, FTW_F);
}

int ftw(const char *filename, ftw_func func, int unused)
{
    struct stat st;
    int ret;

    if (lstat(filename, &st) == -1) {
        (*func) (filename, &st, FTW_NS);
        return -1;
    }

    list = NULL;
    list_length = 0;

    ret = ftw_rec(filename, func);

    free(list);
    return ret;
}
