#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <errno.h>

#include "interface.h"
#include "support.h"

#include "mfsk.h"
#include "wf.h"
#include "snd.h"
#include "fft.h"
#include "macro.h"
#include "prefs.h"
#include "ptt.h"

GtkWidget *appwindow;
GtkWidget *PopupMenu;

gboolean AFCon = FALSE;
gboolean TXon = FALSE;
gboolean SQUELCHon = FALSE;
gboolean WFpause = FALSE;

static struct mfsktx *tx = NULL;
static struct mfskrx *rx = NULL;

static gint sound_tag;


static void msgbox(gchar *type, const gchar *buf)
{
	GtkWidget *dialog;

	dialog = gnome_message_box_new(buf, type, GNOME_STOCK_BUTTON_OK, NULL);
	gnome_dialog_run(GNOME_DIALOG (dialog));
}

void errmsg(const gchar *fmt, ...)
{
	va_list args;
	char buf[1024];

	va_start(args, fmt);
	vsnprintf(buf, sizeof(buf), fmt, args);
	buf[sizeof(buf) - 1] = 0;
	va_end(args);

	msgbox(GNOME_MESSAGE_BOX_ERROR, buf);
}

static void statusmsg(const char *fmt, ...)
{
	va_list args;
	char buf[1024];
	GnomeAppBar *appbar;

	va_start(args, fmt);
	vsnprintf(buf, sizeof(buf), fmt, args);
	buf[sizeof(buf) - 1] = 0;
	va_end(args);

	appbar = GNOME_APPBAR(lookup_widget(appwindow, "appbar1"));
	gnome_appbar_set_status(appbar, buf);
}

static gint timeout_cb(gpointer unused)
{
	/* if TX is on or the engine is stopped do nothing */
	if (WFpause == TRUE || TXon == TRUE)
		return TRUE;

	wf_run();

	return TRUE;
}

static void write_cb(gpointer data, gint fd, GdkInputCondition cond)
{
	gfloat *buf;

	buf = txprocess(tx);

	if (tx->state == STATE_FINISH)
		statusmsg("Flushing: %d", tx->queuelen - tx->queueptr);
	else
		statusmsg("Transmitting");

	if (buf) {
		if (sound_write(buf, SymbolLen) < 0) {
			errmsg("%s", sound_error());
			stop_trx(TRUE);
			return;
		}
	} else {
		stop_trx(TRUE);
		start_rx();
	}
}

void start_tx(void)
{
	int fd;

	/* TX on already... */
	if (TXon)
		return;

	/* these might have been left uninitialized from previous tx */
	tx->queuelen = 0;
	tx->queueptr = 0;
	tx->bitstate = 0;
	tx->encstate = 0;
	tx->stopflag = 0;
	tx->state = STATE_PREAMBLE;

	if ((fd = sound_open_for_write()) < 0) {
		errmsg("sound_open_for_write: %s", sound_error());
		stop_trx(TRUE);
		return;
	}

	TXon = TRUE;
	set_ptt(1);

	sound_tag = gdk_input_add(fd, GDK_INPUT_WRITE, write_cb, NULL);
}

static void read_cb(gpointer data, gint fd, GdkInputCondition cond)
{
	gfloat buf[WF_BLOCKLEN];
	gint len;

	len = sound_read(buf, WF_BLOCKLEN);

	if (len < 0) {
		if (errno != EAGAIN) {
			errmsg("%s", sound_error());
			stop_trx(TRUE);
		}
		return;
	}

	wf_setdata(buf, len);

	rxprocess(rx, buf, len);

	statusmsg("Receiving");
}

void start_rx(void)
{
	int fd;

	/* if TX is on wait for write_cb() to turn it off */
	if (TXon)
		return;

	if ((fd = sound_open_for_read()) < 0) {
		errmsg("sound_open_for_read: %s", sound_error());
		stop_trx(TRUE);
		return;
	}

	/* A bug in gdk? Without this the input doesn't seem to work.. */
	read_cb(NULL, 0, 0);

	sound_tag = gdk_input_add(fd, GDK_INPUT_READ, read_cb, NULL);
}

void stop_trx(gboolean abort)
{
	if (TXon == TRUE && abort == FALSE) {
		txstop(tx);
		return;
	}

	sound_close();

	set_ptt(0);
	TXon = FALSE;

	gdk_input_remove(sound_tag);
}

void trx_set_freq(gfloat freq)
{
	GtkSpinButton *spin;
	Waterfall *wfall;
	gint x1, x2;

	if (freq < WF_STARTFREQ)
		freq = WF_STARTFREQ;

	if (freq > WF_STOPFREQ - (NumTones * ToneSpacing))
		freq = WF_STOPFREQ - (NumTones * ToneSpacing);

	if (rx != NULL)
		rx->freq = freq;

	if (tx != NULL)
		tx->freq = freq;

	spin = GTK_SPIN_BUTTON(lookup_widget(appwindow, "freqspinbutton"));
	gtk_spin_button_set_value(spin, freq);

	x1 = WF_FTOX(freq);
	x2 = WF_FTOX(freq + (NumTones - 1) * ToneSpacing);

	wfall = WATERFALL(lookup_widget(appwindow, "waterfall"));
	waterfall_setmarker2(wfall, x1, x2);
}

void start_tune(void)
{
	tx->tune = 1;
	push_button("txbutton");
}

void stop_tune(void)
{
	tx->tune = 0;
	push_button("rxbutton");
}

void update_metric_dial(gfloat metric)
{
	GtkDial *dial;

	dial = GTK_DIAL(lookup_widget(appwindow, "dial1"));
	gtk_dial_set_value(dial, metric);
}

int main(int argc, char *argv[])
{
	GtkText *text;
	GtkStyle *style;
	GtkToggleButton *button;

	gnome_init("gmfsk", VERSION, argc, argv);

	if ((rx = initrx()) == NULL) {
		g_print("initrx failed!\n");
		return 1;
	}

	if ((tx = inittx()) == NULL) {
		g_print("inittx failed!\n");
		return 1;
	}

	appwindow = create_appwindow();
	PopupMenu = create_popupmenu();

	load_config();
	macroconfig_load();

	init_ptt(prefs.pttdev, prefs.pttinv);
	init_wf();

	/* draw the tx window "cursor" */
	text = GTK_TEXT(lookup_widget(appwindow, "txtext"));
	gtk_text_insert(text, NULL, NULL, NULL, "_", -1);

	/* set tx window font and bg color */
	style = gtk_widget_get_default_style();
	style = gtk_style_copy(style);
	style->font = txfont;
	style->base[GTK_STATE_NORMAL] = prefs.tx_win_color;
	gtk_widget_set_style(GTK_WIDGET(text), style);

	/* set word wrap on rx window */
	text = GTK_TEXT(lookup_widget(appwindow, "rxtext"));
	gtk_text_set_word_wrap(text, TRUE);

	/* set rx window bg color */
	style = gtk_widget_get_default_style();
	style = gtk_style_copy(style);
	style->base[GTK_STATE_NORMAL] = prefs.rx_win_color;
	gtk_widget_set_style(GTK_WIDGET(text), style);

	/* set AFC and SQUELCH */
	button = GTK_TOGGLE_BUTTON(lookup_widget(appwindow, "afcbutton"));
	gtk_toggle_button_set_active(button, prefs.afc);
	button = GTK_TOGGLE_BUTTON(lookup_widget(appwindow, "squelchbutton"));
	gtk_toggle_button_set_active(button, prefs.squelch);

	/* start receiving */
	start_rx();
	trx_set_freq(1000.0);

	/* peridic task - draw waterfall */
	gtk_timeout_add(100, timeout_cb, NULL);

	/* show the main window */
	gtk_widget_show(appwindow);

	/* create the bitshape scope */
	bitshape_init();

	/* let GTK do it's job */
	gtk_main();
	return 0;
}

