/*
 * main.c
 * Contains the actual hw functions and main operations
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "interface.h"
#include "support.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/perm.h>
#include <sys/mman.h>
#include <unistd.h>
#include <math.h>
#include <errno.h>
#include "profile.h"
#include "hwcode.h"


gint verbosity = 0;
gint advanced_settings = 0;
gint load_settings = 0;
gint automatic_DCC = 1;
GtkWidget *mainWindow = NULL;
GtkWidget *opt3Window = NULL;
GtkWidget *ramWindow = NULL;
static char* loadfile = NULL;

void dumpOptions (struct profile *pf);
extern int do_open (gchar* filename);
extern void buildProfile (GtkWidget* widget, struct profile *pf);
extern void on_chkDCC_toggled (GtkToggleButton *togglebutton, gpointer user_data);

struct poptOption options[] = {
   {"load", 
    'l', 
    POPT_ARG_STRING, 
    &loadfile, 
    0, 
    N_("Load settings from FILE and exit"), 
    N_("FILE")
   }, 
   {NULL}
};


struct profile presets[] = {
    {"G400 BIOS", 0x09, 0x3C, 0x10, 0x6007C120, 0x01003000, 0x0190A419},
    {"G400", 0x13, 0x7A, 0x08, 0x50044120, 0x01003000, 0x0190A421},
    {"G400 Win98", 0x02, 0x1B, 0x18, 0x5053C120, 0x01003000, 0x019B8419},
    {"G400Max", 0x02, 0x1B, 0x18, 0x5053C120, 0x01003000, 0x019B8419},
    {"G400Max Win98", 0x05, 0x42, 0x18, 0x50574120, 0x01003000, 0x019B8419},
};

int presetCount = (sizeof(presets) / sizeof(presets[0]));

/**********************************************************/

void
storeProfile(struct profile *p, long addr)
{
   struct g400_option* opt;
   struct g400_option3* opt3;
   BYTE stat = 0;
   WORD gclksel, mclksel, wclksel;
   opt = (struct g400_option*) &p->opt1;
   opt3 = (struct g400_option3*) &p->opt3;

   /* Disable system clock */
   opt->sysclkdis = 1;
   outl_pci(0x40, p->opt1);

   /* Store original clock sources */
   gclksel = opt3->gclksel;
   mclksel = opt3->mclksel;
   wclksel = opt3->wclksel;
   
   /* Select PCI clock as the source for clocks */
   opt3->gclksel = 0;
   opt3->mclksel = 0;
   opt3->wclksel = 0;
   outl_pci(0x54, p->opt3);

   /* Enable system clock */
   opt->sysclkdis = 0;
   outl_pci(0x40, p->opt1);

   /* Reprogram registers */
   out_dac(0x2c, p->m);
   out_dac(0x2d, p->n);
   out_dac(0x2e, p->pllp);
   outl_pci(0x50, p->opt2);
   outl_pci(0x54, p->opt3);

   /* Wait until SysPLL becomes stable */
   stat = in_dac(0x2f);
   while ( stat != 64 )
   {
      stat = in_dac(0x2f);
   }

   /* Disable system clock */
   opt->sysclkdis = 1;
   outl_pci(0x40, p->opt1);

   /* Select the new source for clocks */
   opt3->gclksel = gclksel;
   opt3->mclksel = mclksel;
   opt3->wclksel = wclksel;
   outl_pci(0x54, p->opt3);

   /* Enable system clock */
   opt->sysclkdis = 0;
   outl_pci(0x40, p->opt1);

   /* ALL DONE */
}

/**********************************************************/
void
getProfile(struct profile *p, long addr)
{

   p->m = in_dac(0x2c);
   p->n = in_dac(0x2d);
   p->pllp = in_dac(0x2e);
   p->opt1 = inl_pci(0x40);
   p->opt2 = inl_pci(0x50);
   p->opt3 = inl_pci(0x54);
}

/**********************************************************/

/*
WORD
getMgabase(long addr)
{
   WORD bs;
   struct mgabase* base = (struct mgabase*) &bs;

   bs = inl_pci(0x14);
   printf("base: 0x%02lx\n", bs);

   printf("mgabase1: 0x%02x\n", base->mgabase1);
   return( base->mgabase1 );
}
*/

/**********************************************************/
/*
DWORD
getMemrdbk(long addr)
{
   WORD mgabase1;
   WORD *MappedAddr;
   WORD memrd;

   mgabase1 = getMgabase( addr );
   printf("phys: 0x%02x\n", mgabase1);

   (void*) MappedAddr = mmap(0 , (WORD) MGABS1LMT, PROT_READ, MAP_SHARED | MAP_ANON, 0, (off_t) mgabase1);

   if ( (int) MappedAddr < 0 ) {
      printf("Memory mapping failed! %i\n", errno);
      return 0;
   }

   msync( MappedAddr, (WORD) MGABS1LMT, MS_SYNC);

   printf("linear: %p\n", MappedAddr);

   memrd = MappedAddr[0x1e44/4];
   
   if (munmap (MappedAddr, MGABS1LMT) < 0) {
      printf("Memory unmapping failed! %i\n", errno);
   }

   printf("memrd: 0x%02x\n", memrd);

   return memrd;
}
*/
/**********************************************************/

/*
void storeMemrdbk(struct memrdbk memrd, long addr)
{
   WORD base, rd, *pmemrd = (WORD*) &memrd;
   DWORD* MappedAddr;

   base = getMgabase( addr );

   printf("phys: 0x%02x\n", base);

   (void*) MappedAddr = mmap(0 , (WORD) MGABS1LMT, PROT_WRITE, MAP_PRIVATE | MAP_ANON, 0, (off_t) base);

   if ( (int) MappedAddr < 0 ) {
      printf("Memory mapping failed! %i\n", errno);
      return;
   }

   msync( MappedAddr, (WORD) MGABS1LMT, MAP_PRIVATE | MAP_ANON);
   printf("linear: %p\n", MappedAddr);

   MappedAddr[0x1e44] = pmemrd;

   if (munmap (MappedAddr, MGABS1LMT) < 0) {
      printf("Memory unmapping failed! %i\n", errno);
   }
}
*/

/**********************************************************/

/*
void
storeMemctl(struct memctlwtst* mctl, long addr)
{



}
*/

/**********************************************************/
int
cmpProfile(struct profile *p, long addr)
{
    BYTE tm,tn,tp;
    WORD topt1,topt2,topt3;

    tm = in_dac(0x2c);
    tn = in_dac(0x2d);
    tp = in_dac(0x2e);
    topt1 = inl_pci(0x40);
    topt2 = inl_pci(0x50);
    topt3 = inl_pci(0x54);

    return (tm == p->m &&
	    tn == p->n &&
	    tp == p->pllp &&
	    topt1 == p->opt1 &&
	    topt2 == p->opt2 &&
	    topt3 == p->opt3);
}

/***********************************************************/

int
cmpProfile2(struct profile *p, struct profile *p2)
{
   return (p2->m == p->m &&
	   p2->n == p->n &&
	   p2->pllp == p->pllp &&
	   p2->opt1 == p->opt1 &&
	   p2->opt2 == p->opt2 &&
	   p2->opt3 == p->opt3);
}

/************************************************************/

unsigned long
findCard(int cardIndex)
{
    int i;
    unsigned long d;
    unsigned long addr = 0x81000000;

    i = 0;
    for(addr -= 0x800; addr >= 0x80000000; addr -= 0x800) {
        outl( 0xCF8, addr );
        d = inl( 0xCFC );
        if( (d & 0xFFFF) == 0x102B ) {
	    if(verbosity > 0)
	        printf( "Found card at %lx\n", addr);
            if(i == cardIndex)
       	        break;
	    i++;
        }
    }

    if(addr < 0x80000000)
	return 0;
    return addr;
}

/************************************************************/
void
updateFields ( GtkWidget *widget, struct profile *pf )
{
   GtkEntry *tmpEntry;
   GtkWidget *tmpWidget;
   char *tmpstr, *preset;
   unsigned long addr = 0x81000000;
   int cardID;
   struct syspllp *pllp;

   pllp = (struct syspllp*) &pf->pllp;

   tmpstr = malloc(100);

   tmpWidget = lookup_widget( widget, "spnCard" );
   cardID = gtk_spin_button_get_value_as_int ( (GtkSpinButton*) tmpWidget );
   
   tmpWidget = lookup_widget( widget, "entryType" );
   if ( (addr = findCard(cardID)) == 0 ) {
      gtk_entry_set_text( (GtkEntry*) tmpWidget, "None");
      tmpWidget = lookup_widget( (GtkWidget*) widget, "entryAddr" );
      gtk_entry_set_text( (GtkEntry*) tmpWidget, "0x00000000" );

      tmpWidget = lookup_widget( (GtkWidget*) widget, "cmbPreset");
      gtk_widget_set_sensitive( tmpWidget, FALSE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "entryClk");
      gtk_widget_set_sensitive( tmpWidget, FALSE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "chkAdvanced");
      gtk_widget_set_sensitive( tmpWidget, FALSE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "frmAdvanced");
      gtk_widget_set_sensitive( tmpWidget, FALSE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "button3");
      gtk_widget_set_sensitive( tmpWidget, FALSE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "btnO3");
      gtk_widget_set_sensitive( tmpWidget, FALSE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "entryO3");
      gtk_widget_set_sensitive( tmpWidget, FALSE );
   } else {
      gtk_entry_set_text( (GtkEntry*) tmpWidget,"Matrox MGA-G400 AGP");
      sprintf(tmpstr, "0x%lx", addr);
      tmpWidget = lookup_widget( (GtkWidget*) widget, "entryAddr");
      gtk_entry_set_text( (GtkEntry*) tmpWidget, tmpstr );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "cmbPreset");
      gtk_widget_set_sensitive( tmpWidget, TRUE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "entryPreset");
      preset = gtk_entry_get_text( (GtkEntry*) tmpWidget);
      if ( strcasecmp(preset, "Custom") == 0)
      {
	 tmpWidget = lookup_widget( (GtkWidget*) widget, "entryClk");
	 gtk_widget_set_sensitive( tmpWidget, TRUE );
      }
      tmpWidget = lookup_widget( (GtkWidget*) widget, "chkAdvanced");
      gtk_widget_set_sensitive( tmpWidget, TRUE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "button3");
      gtk_widget_set_sensitive( tmpWidget, TRUE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "btnO3");
      gtk_widget_set_sensitive( tmpWidget, TRUE );
      tmpWidget = lookup_widget( (GtkWidget*) widget, "entryO3");
      gtk_widget_set_sensitive( tmpWidget, TRUE );
      if ( advanced_settings )
      {
	 tmpWidget = lookup_widget( (GtkWidget*) widget, "frmAdvanced");
	 gtk_widget_set_sensitive( tmpWidget, TRUE );
      }
   }
   
   tmpEntry = (GtkEntry*) lookup_widget( widget, "entryM" );
   sprintf(tmpstr, "0x%02x", pf->m);
   gtk_entry_set_text( tmpEntry, tmpstr );

   tmpEntry = (GtkEntry*) lookup_widget( widget, "entryN" );
   sprintf(tmpstr, "0x%02x", pf->n);
   gtk_entry_set_text( tmpEntry, tmpstr );

   tmpEntry = (GtkEntry*) lookup_widget( widget, "entryP" );
   sprintf(tmpstr, "0x%02x", pllp->p);
   gtk_entry_set_text( tmpEntry, tmpstr );

   tmpEntry = (GtkEntry*) lookup_widget( widget, "entryS" );
   sprintf(tmpstr, "0x%02x", pllp->s);
   gtk_entry_set_text( tmpEntry, tmpstr );

   tmpEntry = (GtkEntry*) lookup_widget( widget, "entryO1" );
   sprintf(tmpstr, "0x%02x", pf->opt1);
   gtk_entry_set_text( tmpEntry, tmpstr );

   tmpEntry = (GtkEntry*) lookup_widget( widget, "entryO2" );
   sprintf(tmpstr, "0x%02x", pf->opt2);
   gtk_entry_set_text( tmpEntry, tmpstr );

   tmpEntry = (GtkEntry*) lookup_widget( widget, "entryO3" );
   sprintf(tmpstr, "0x%02x", pf->opt3);
   gtk_entry_set_text( tmpEntry, tmpstr );

   free(tmpstr);
}

/***********************************************************/
void
checkIfPreset( GtkWidget* widget, struct profile *pf )
{
   GtkWidget *tmpWidget;
   int i;

   tmpWidget = lookup_widget( widget, "entryPreset" );
   for (i = 0; i < presetCount; i++)
   {
      if (cmpProfile2(&presets[i],pf))
      {
	 gtk_entry_set_text( (GtkEntry*) tmpWidget, presets[i].name );
	 return;
      }
   }
   gtk_entry_set_text( (GtkEntry*) tmpWidget, "Custom" );
}

/**********************************************************/
void
calcClock( GtkWidget* widget, struct profile *pf )
{
   BYTE m, n, p;
   float clk;
   gchar* tmpstr;
   GtkWidget *tmpWidget;

   tmpstr = g_malloc0(10);

   m = pf->m;
   n = pf->n;
   p = pf->pllp >> 5;

   clk = (double) ((n + 1) * 27.00000L) / ((m + 1)*(p + 1));

   sprintf(tmpstr, "%.2f", clk);
   tmpWidget = lookup_widget( widget, "entryClk" );
   gtk_entry_set_text((GtkEntry*) tmpWidget, tmpstr);
   g_free(tmpstr);
}

/***********************************************************/
double
getSyspll( void )
{
   GtkWidget *tmpWidget;
   double clk;
   gchar* tmpstr;
   
   tmpWidget = lookup_widget( mainWindow, "entryClk" );
   tmpstr = gtk_entry_get_text( (GtkEntry*) tmpWidget );
   clk = atof( tmpstr );
   return clk;
}

/***********************************************************/
double
clkMatch(double dsrFreq, struct profile *pf)
{
	
   int i, n = 1, m = 1, p = (int) pow(2,0)-1;
   BYTE bestM, bestN, bestP, p2, s2;
   double calcFreq, nrsFreq, fvcoFreq, Diff, smlstDiff;
   struct syspllp *pllp;

   pllp = (struct syspllp*) &pf->pllp;

   calcFreq = (((double) n + 1.0) / (((double) m + 1.0)
				     * ((double) p + 1.0))) * 27.00000L;
   Diff     = fabs(dsrFreq - calcFreq);
   smlstDiff = Diff;
   bestM = m;
   bestN = n;
   bestP = p;
   nrsFreq = calcFreq;

   for (i = 0; i < 4; i++) {
      p = (int) pow(2,i) - 1;
      for (m=1; m<32; m++) {
	 for (n=1; n<128; n++) {
	    calcFreq = (((double) n + 1.0) / (((double) m + 1.0)
					      * ((double) p + 1.0))) * 27.00000L;
	    Diff = fabs(dsrFreq - calcFreq);
	    if (Diff < smlstDiff) {
	       smlstDiff = Diff;
	       nrsFreq = calcFreq;
	       bestM = m;
	       bestN = n;
	       bestP = p;
	    }
	 }
      }
   }
   
   pf->m = bestM;
   pf->n = bestN;
   pllp->p = bestP;
   
   fvcoFreq = (double) ((bestN + 1) * 27.00000L) / (bestM + 1);
   
   if (fvcoFreq < 110) pllp->s = 0;
   else if (fvcoFreq < 170) pllp->s = 1;
   else if (fvcoFreq < 240) pllp->s = 2;
   else pllp->s = 3;

   s2 = pllp->s << 3;
   p2 = pllp->p << 5;
   pf->pllp = p2 | s2;

   return nrsFreq;
}

/***********************************************************************/
void
printMessage( GtkWidget *widget, const char* message, gchar* color )
{
   GtkWidget *tmpWidget;
   GdkColor gcolor;

   gdk_color_parse(color, &gcolor);
   tmpWidget = lookup_widget( widget, "Intro" );

   gtk_text_insert( (GtkText*) tmpWidget, NULL, &gcolor, NULL, message, -1 );
}

/************************************************************************/
void
printProfile( GtkWidget *widget, struct profile *pf )
{
   struct syspllp *pllp;
   gchar* tmpstr;
   tmpstr = g_malloc0(255);

   pllp = (struct syspllp*) &pf->pllp;

   sprintf( tmpstr, "M: 0x%02x (%i)\n", pf->m, pf->m );
   printMessage( widget, tmpstr, "black" );
   sprintf( tmpstr, "N: 0x%02x (%i)\n", pf->n, pf->n );
   printMessage( widget, tmpstr, "black" );
   sprintf( tmpstr, "P: 0x%02x (%i)\n", pllp->p, pllp->p);
   printMessage( widget, tmpstr, "black" );
   sprintf( tmpstr, "S: 0x%02x (%i)\n", pllp->s, pllp->s);
   printMessage( widget, tmpstr, "black" );
   sprintf( tmpstr, "PLLP: 0x%02x (%i)\n", pf->pllp, pf->pllp);
   printMessage( widget, tmpstr, "black" );
   printMessage( widget, "=================\n", "red");
   g_free(tmpstr);
   dumpOptions(pf);
}

/************************************************************/
void
printStatus (GtkWidget* widget, const gchar *text )
{
   GtkWidget *tmpWidget;
   tmpWidget = lookup_widget( widget, "barStatus" );

   gnome_appbar_set_status( (GnomeAppBar*) tmpWidget, text );
}

/***********************************************************/
void
dumpOptions (struct profile *pf)
{
   struct g400_option* opt1;
   struct g400_option2* opt2;
   struct g400_option3* opt3;

   opt1 = (struct g400_option*) &pf->opt1;
   printf("\nOPTION 1\n");
   printf("sysclkdis: \t%i\n", opt1->sysclkdis);
   printf("syspllpdn: \t%i\n", opt1->syspllpdn);
   printf("pllsel: \t%i\n", opt1->pllsel);
   printf("memconfig: \t%i\n", opt1->memconfig);
   printf("hardpwmsk: \t%i\n", opt1->hardpwmsk);
   printf("rfhcnt: \t%i\n", opt1->rfhcnt);
   printf("enhmemacc: \t%i\n", opt1->enhmemacc);
   printf("nohireq: \t%i\n", opt1->nohireq);
   printf("noretry: \t%i\n", opt1->noretry);
   printf("biosen: \t%i\n", opt1->biosen);
   printf("powerpc: \t%i\n", opt1->powerpc);
   printf("----------------\n");

   opt2 = (struct g400_option2*) &pf->opt2;
   printf("\nOPTION 2\n");
   printf("eepromwt: \t%i\n", opt2->eepromwt);
   printf("mod2clkp: \t%i\n", opt2->mod2clkp);
   printf("modclkp: \t%i\n", opt2->modclkp);
   printf("codclksl: \t%i\n", opt2->codclksl);
   printf("codprediv: \t%i\n", opt2->codprediv);
   printf("codpstdiv: \t%i\n", opt2->codpstdiv);
   printf("----------------\n");

   opt3 = (struct g400_option3*) &pf->opt3;
   printf("\nOPTION 3\n");
   printf("gclksel: \t%i\n", opt3->gclksel);
   printf("gclkdiv: \t%i\n", opt3->gclkdiv);
   printf("gclkdcyc: \t%i\n", opt3->gclkdcyc);
   printf("mclksel: \t%i\n", opt3->mclksel);
   printf("mcldiv: \t%i\n", opt3->mcldiv);
   printf("mclkdcyc: \t%i\n", opt3->mclkdcyc);
   printf("wclksel: \t%i\n", opt3->wclksel);
   printf("wclkdiv: \t%i\n", opt3->wclkdiv);
   printf("wclkdcyc: \t%i\n", opt3->wclkdcyc);
   printf("----------------\n");
}

/***********************************************************/   
int
main (int argc, char *argv[])
{
  GtkWidget *tmpWidget;
  GList *items = NULL;
  int i, cardID, status;
  long unsigned addr;
  struct profile pf;
  gchar *tmpstr, *filename;

  if(iopl(3)) {
     fprintf(stderr, "gMGAclock has to be run as root!\n\n");
     exit(1);
  }  

  tmpstr = g_malloc(200);
  gnome_init_with_popt_table ("gmgaclock", VERSION, argc, argv, options, 0, NULL);
 
  mainWindow = create_gmgaclock ();
  opt3Window = create_winO3();

  /* Right align help menuitem */
  tmpWidget = lookup_widget( mainWindow, "help1" );
  gtk_menu_item_right_justify( (GtkMenuItem*) tmpWidget );

  /* Fetch current settings */
  tmpWidget = lookup_widget( mainWindow, "spnCard" );
  cardID = gtk_spin_button_get_value_as_int ( (GtkSpinButton*) tmpWidget );
  
  addr = findCard(cardID); 
  
  getProfile(&pf, addr);
  updateFields( mainWindow, &pf );

  /* Add presets to the combolist */
  tmpWidget = lookup_widget( mainWindow, "cmbPreset" );
  items = g_list_append( items, "Custom" );
  for (i = 0; i < presetCount; i++)
  {
     items = g_list_append( items, presets[i].name );
  }
  gtk_combo_set_popdown_strings ( (GtkCombo*) tmpWidget, items);

  /* Check if current settings match a preset */
  checkIfPreset(mainWindow, &pf);

  /* Check autoload */
  load_settings = gnome_config_get_int("/gmgaclock/settings/load_settings");

  /* Load settings from a file */
  if (load_settings || (loadfile != NULL)) {
     if( loadfile == NULL ) {
	filename = gnome_util_prepend_user_home(".gmgaclockrc");
     } else {
	filename = loadfile;
     }
     status = do_open( filename );
     buildProfile(mainWindow, &pf);
     storeProfile(&pf, addr);

     /* Commandline? */
     if (loadfile != NULL) {
	if (status == 1) {
	   printf("gMGAclock: Settings loaded from %s\n", filename);
	} else {
	   printf("gMGAclock: Failed to load settings from %s\n", filename);
	}
	exit(0);
     }

     tmpWidget = lookup_widget(mainWindow, "load_settings_on_startup1");
     gtk_check_menu_item_set_active( (GtkCheckMenuItem*) tmpWidget, TRUE );
  }

  /* Disable apply */
  tmpWidget = lookup_widget( mainWindow, "button3" );
  gtk_widget_set_sensitive( tmpWidget, FALSE );

  /* Check automatic DCC */
  automatic_DCC = gnome_config_get_int("/gmgaclock/settings/automatic_DCC");
  tmpWidget = lookup_widget( opt3Window, "chkDCC" );

  if (automatic_DCC) {
     gtk_toggle_button_set_active( (GtkToggleButton*) tmpWidget, TRUE );
  } else {
     gtk_toggle_button_set_active( (GtkToggleButton*) tmpWidget, FALSE );
  }
  on_chkDCC_toggled ((GtkToggleButton*) tmpWidget, NULL);

  if (verbosity) {
     sprintf( tmpstr, "addr: 0x%lx\n", addr );
     printMessage(mainWindow, "Current settings:\n", "red");
     printMessage(mainWindow, tmpstr, "black");
     printProfile(mainWindow, &pf);
  }	

  /* Print welcome msg */
  printStatus( mainWindow, "Welcome to gMGAclock!" );

  gtk_widget_show (mainWindow);
  gtk_main ();
  g_free(tmpstr);
  return 0;
}

