/* commands.c - gnapster command handling */

#include <stdio.h>
#include <stdlib.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <netinet/in.h>

#include <signal.h>

#include "gnapster.h"

#include "chan.h"
#include "xtext.h"
#include "themes.h"

#include "commands.h"
#include "regexp.h"

extern GnapsterMain *gmain;
extern UserInfo user_info;

extern GList *ignore_list;
extern GList *signore_list;
extern GList *hooks;
extern GList *nick_history;

STab *st = NULL;
ConnInfo *ci = NULL;

#define CHECK_ARGS() if (!args) { invalid_args(command); return; }
#define INVALID_ARGS() { invalid_args(command); return; }

GList *acmds = NULL;
GList *ucmds = NULL;

/*Command cmds[] = {
   { "desc",            NULL,           "desc",                 "Generic Commands" },
   { "/help",		do_help,	"[command]", 		"Shows help for Gnapster commands" },
   { "/version",	do_version,	"", 			"Requests server information" },
   { "/sv",		do_sv,		"",			"Displays your client version to the current tab" },
   { "/clear:/cls",	do_clear,	"",			"Clears the current tab" },
   { "/load",           do_load,        "<theme>",              "Attemps to load the specified theme" },
   { "/reload",         do_reload,      "",                     "Reloads the current text theme" },
   { "/stab",           do_stab,        "[-r]",                 "New server tab" },   
   
   { "desc",            NULL,           "desc",                 "Channel Commands" },
   { "/topic:/t",	do_topic,	"<topic>",		"Changes the current channels topic" },
   { "/me",		do_me,		"<action>",		"Sends an action style message to the current channel" },
   { "/join:/j",	do_join,	"<channel>",		"Joins a Napster channel" },
   { "/part:/p",	do_part,	"",			"Leaves a Napster channel" },
   { "/say",            do_sendtext,    "<msg>",                "Outputs msg to the current tab" },
   
   { "desc",            NULL,           "desc",                 "Chat Commands" },
   { "/whois:/wi",	do_whois,	"[user]",		"Requests server whois information for the user" },
   { "/msg:/m",	        do_msg,		"<user> <message>", 	"Send a private message to another user" },
   { "/query:/q",	do_query, 	"[user]",		"Insert another tab with a private conversation to a user in it" },
   { "/ignore:/ig",	do_ignore,	"[user]", 		"Place the user on your ignore list" },
   { "/unignore:/uig",	do_unignore,	"<user>",		"Removes a user from ignore" },
   
   { "desc",            NULL,           "desc",                 "Debugging" },
   { "/raw",		do_raw,		"<message number> [message]",	"For developer use only!" },
   { "/debug",          do_debug,       "",                     "Enable debugging messages" },
   { "/stats",		do_stats, 	"",			"Displays server stats for opennap servers" },
   { "/hook",           do_hook,        "<hook_name> [arg1 [arg2 [argn...]]]", "Display a theme hook using the supplied arguments (for theme creation purposes)" },
   
   { "desc",            NULL,           "desc",                 "Admin Commands" },
   { "/admin",          do_admin,       "<admin command> [command args]", "Main admin command control" },

   { "/wallop:/wall",	do_wallop,	"<message>", 		"Send a wallop message to all admins/moderators" },
   { "/global",		do_global,	"<message",		"Send a global message to all users" },
   { "/op",		do_op,		"<user>",		"Sets the specified user as operator of the current channel" },
   { "/deop:/dop", 	do_deop,	"<user>",		"Removes the specified user as an operator of the current channel" },
   { "/nuke",           do_nuke,        "<user>",               "Deletes a users account" },
   { "/kill",		do_kill,	"<user>",		"Disconnects a user with a lower status than you" },
   { "/kick:/k",	do_kick,	"<user> [reason]",	"Kicks the user from the current channel" },
   { "/cban",		do_cban,	"<user|ip> [reason]",	"Bans the user from the current channel" },
   { "/cunban", 	do_cunban,	"<user|ip>",		"Unbans the user from the current channel" },
   { "/cbans",		do_cbans,	"",			"Lists the current channel bans" },
   { "/cunban_all",	do_cunban_all,	"",			"Removes all channel bans on the current channel" },
   { "/ban",		do_ban,		"<user|ip> [reason]",	"Attempts to place a server ban on the user" },
   { "/tban",           do_tban,        "<user|ip> <timeout>",  "Sets a timed ban in hours" },
   { "/unban",		do_unban,	"<user|ip>",		"Requests that the user be unbanned" },
   { "/bans",           do_bans,        "[regexp]",             "Shows the server ban list" },
   { "/muzzle",		do_muzzle,	"<user> [reason]",	"Removes a users ability to send public messages" },
   { "/unmuzzle",	do_unmuzzle,	"<user>",		"Restores a users ability to send public messages" },
   { "/mode",		do_mode,	"<channel> [mode]",	"Sets a channels mode" },
   { "/setport",	do_setport,	"<user> <port>",	"Attempts to change a users data port" },
   { "/userlevel:/ul",	do_userlevel,	"<user> <level>",	"Changes a users status level" },
   { "/setline",	do_setline,	"<user> <speed>",	"Changes a users line speed" },
   { "/links",		do_links,	"",			"Shows opennap server links" },
   { "/signore:/sig",	do_signore,	"<match string>",	"Ignores server messages that match the string (wildcards accepted)" },
   { "/unsignore:/usig",do_unsignore,	"<index>",		"Removes a server ignore from the list (to get the index number type /signore without arguments" },
   { "/sigsave",	do_sigsave,	"",			"Saves the server ignore list to disk and reloads when Gnapster runs again" },
   { "/chanlevel",	do_chanlevel, 	"<channel> <level>",	"Sets the minimum user level required to join the channel" },
   { "/usermode:/um",	do_usermode,	"[modes]",		"Sets your usermode which controls what type of messages you will see from the server" },
   { NULL,		NULL,		NULL },
};*/

void invalid_args(char *command) {
   if (!command)
     return;
   
   hook_text_insert(st, CURR, SYSTEM, "invalid_args", "%s",
		    command + 1);
}

void setup_command_stab(STab *stab) {
   st = stab;
   ci = st ? st->ci : NULL;
}

int handle_command(STab *stab, char *command, char *args) {
   GList *iter_ptr;
   LinkedCommand *cmd;
   
   d_assert_return(stab != NULL, 0);
   
   d_assert_return(command != NULL, 0);
   d_assert_return(*command == '/', 0);
   d_assert_return(command[1], 0);
   
   setup_command_stab(stab);
   
   /* push past the '/' */
   command++;
   
   ITER_LIST(ucmds) {
      LIST_DATA(cmd);
      
      if (!cmd->cmd && !cmd->func)
	continue;
      
      if (j_strcmp(cmd->command, command))
	continue;
      
      if (cmd->cmd > 0)
	exec_linked_cmd(cmd, args);
      else
	exec_cmd(cmd, args);
      
      return 1;
   }
   
   setup_command_stab(NULL);
   
   return 0;
}

/* for backwards compat */
void exec_cmd(LinkedCommand *cmd, char *args) {
   char *cmdbuf;
   
   d_msprintf(&cmdbuf, "/%s", cmd->command);
   (cmd->func) (cmdbuf, args);
   d_free(cmdbuf);
}

/*int handle_command(STab *stab, char *command, char *args) {
   int i = 0, x;
   char *ptr, *hld, *cmd[2];

   st = stab;
   ci = stab->ci;
   
   while(cmds[i].command) {
      ptr = d_strdup(cmds[i].command);
      x = 0;
      
      while((cmd[x] = next_arg_full(ptr, &ptr, ':'))) {
	 if (!g_strcasecmp(command, cmd[x++])) {
	    x = -1;
	    break;
	 }
      }
      
      NA_RESET();
      
      d_free(cmd[0]);
      if (x == -1) 
	break;
      i++;
   }

   if (cmds[i].func == NULL)
     return 0;
   
   if (!command)
     return 0;
   
   hld = d_strdup(args);
   
   cmds[i].func (command, args);
   
   d_free(hld);

   st = NULL;
   ci = NULL;
   
   return 1;
}*/

void dispatch_text(ChannelInfo *chan, int page_num, char *text) {
   hook_text_insert(st, page_num, MESSAGE, "public_msg_self", "%s\4%s",
		    ci->account->user, text);
   napster_send(ci->sock,
		(chan->user) ? NAPSTER_USER_MESSAGE : NAPSTER_CHAN_MESSAGE,
		"%s %s", chan->channel_name, text);
}

COMMAND(do_sendtext) {
   ChannelInfo *chan;
   char *ptr, buf[181], *nptr;
   int page_num, i, last_space = 0, buf_pos = 0;
   
   page_num = gnapster_get_pagenum(st);
   if (page_num <= 1) 
     return;
   
   chan = g_list_nth_data(st->ct->channel_list, 
			  page_num - 2);
   
   ptr = args;
   
   if (!chan->user)
     nptr = nick_completion(ptr, st, GTK_CLIST(chan->clist));
   else
     nptr = d_strdup(ptr);
   
   for(i=0; nptr[i]; i++) {
      if (nptr[i] == ' ')
	last_space = buf_pos;
      if (nptr[i] == '\n') {
	 buf[buf_pos] = 0;
	 buf_pos = 0;

	 dispatch_text(chan, page_num, buf);
      } else if (buf_pos > 170) {
	 i -= (buf_pos - last_space);
	 buf[last_space] = 0;
	 buf_pos = 0;

	 dispatch_text(chan, page_num, buf);
      } else
	buf[buf_pos++] = nptr[i];
   }
   
   if (buf_pos) {
      buf[buf_pos] = 0;
      dispatch_text(chan, page_num, buf);
   }
   
   d_free(nptr);
   d_free(args);
}

COMMAND(do_xfer_stats) {
   char *msg;
   unsigned long d, u, td, tu;
   float d_kps, u_kps, td_kps, tu_kps;
   
   get_xfer_stats(NULL, &d, &u, &td, &tu, &d_kps, &u_kps, &td_kps, &tu_kps);
   
   d_msprintf(&msg, "Outgoing (%lu) Avg: %.02fK/s (%.02fK/s over %.01fM)",
	      tu, u_kps, tu_kps, gmain->out);
   
   do_sendtext(NULL, msg);
}

COMMAND(do_help) {
   LinkedCommand *cmdptr;
   GList *iter_ptr, *list;
   int i, pn;
   
   pn = gnapster_get_pagenum(st);
   d_assert(pn >= 0);
   
   if (args && *args == 'a')
     list = acmds;
   else
     list = ucmds;

   if (args) {
      ITER_LIST(list) {
	 LIST_DATA(cmdptr);
	 
	 if (!cmdptr->command)
	   continue;
	 
	 if (j_strcmp(args, cmdptr->command))
	   continue;
	 
	 hook_text_insert(st, pn, SYSTEM, "general_message",
			  "/%s %s - %s", cmdptr->command, cmdptr->syntax,
			  cmdptr->help ? cmdptr->help : "No help available");
	 
	 return;
      }
      
      hook_text_insert(st, pn, SYSTEM, "general_message",
		       "Help for command %s not found\n", args);
      
      return;
   }   
   
   i = 0;
   ITER_LIST(list) {
      LIST_DATA(cmdptr);
      
      if (!cmdptr->cmd && !cmdptr->func) { /* cmd group */
	 gnapster_text_insert(st, pn, MESSAGE, "\n");
	 if (((i % 3) > 0))
	   gnapster_text_insert(st, pn, MESSAGE, "\n");
	 
	 hook_text_insert(st, pn, SYSTEM, "general_message", "%s\n",
			  cmdptr->command);
	 
	 i = 0;
	 
	 continue;
      }
      
      /* regular command listing */
      hook_text_insert(st, pn, ((i % 3) == 0) ? SYSTEM : MESSAGE,
		       "help_list", "%-12s", cmdptr->command);
      
      if (((i % 3) == 2))
	gnapster_text_insert(st, pn, MESSAGE, "\n");
      
      i++;
   }
   
   gnapster_text_insert(st, pn, MESSAGE, "\n");
}

/*COMMAND(do_help) {
   int i = 0, x;
   Command *cmd_ptr;
   char *slasharg, *ptr, *hld, *cmd[2];
   
   if (!args) {
      int page_num =
	gnapster_get_pagenum(st);
      
      hook_text_insert(st, page_num, SYSTEM, "command_list", NULL);
      
      for(cmd_ptr=cmds, i=0; cmd_ptr->command; cmd_ptr++) {
	 if (*(cmd_ptr->command) != '/') {
	    gnapster_text_insert(st, page_num, MESSAGE, "\n\n");
	    hook_text_insert(st, page_num, SYSTEM, "general_message", "%s\n",
				 cmd_ptr->help);
	    i = 0;
	    continue;
	 }
	 
	 ptr = d_strdup(cmd_ptr->command);
	 hld = ptr;
	 
	 hook_text_insert(st, page_num, ((i % 3) == 0) ? SYSTEM : MESSAGE, 
			  "help_list", "%-12s", next_arg_full(hld, &hld, ':'));
	 
	 if (((i % 3) == 2) || !((cmd_ptr + 1)->command))
	   gnapster_text_insert(st, page_num, MESSAGE, "\n");

	 d_free(ptr);
	 
	 i++;
      }
      
      NA_RESET();
      
      hook_text_insert(st, page_num, SYSTEM, "help_message", NULL);
      
      return;
   }
   
   d_msprintf(&slasharg, "/%s", next_arg(args, &args));
   
   while(cmds[i].command) {
      ptr = d_strdup(cmds[i].command);
      hld = ptr;
      
      x = 0;
      
      while((cmd[x] = next_arg_full(hld, &hld, ':'))) {
	 if (!g_strcasecmp(slasharg, cmd[x++])) {
	    x = -1;
	    break;
	 }
      }
      
      d_free(ptr);
      
      if (x == -1) 
	break;
      
      i++;
   }
   
   NA_RESET();

   if (cmds[i].func)
     gnapster_text_insert(st, CURR, SYSTEM, "%s %s - %s\n",
			  slasharg, cmds[i].syntax, cmds[i].help);
   
   d_free(slasharg);
}*/

/*COMMAND(do_version) {
   napster_send(ci->sock, NAPSTER_SERVER_VERSION, NULL);
}

COMMAND(do_wallop) {
   CHECK_ARGS();
   
   hook_text_insert(st, CURR, MESSAGE, "wallop_self", "%s\4%s",
		    ci->account->user, args);
   napster_send(ci->sock, NAPSTER_ADMIN_MESSAGE, "%s",
		args);
}

COMMAND(do_global) {
   CHECK_ARGS();
   
   hook_text_insert(st, CURR, MESSAGE, "global_self", "%s\4%s",
		    ci->account->user, args);
   napster_send(ci->sock, NAPSTER_GLOBAL_MESSAGE, "%s",
		args);
}

COMMAND(do_topic) {
   char *chan;

   chan = get_current_chan();
   if (!chan) 
     return;

   if (args)
     napster_send(ci->sock, NAPSTER_CHAN_TOPIC, "%s %s",
		  chan, args);
   else
     napster_send(ci->sock, NAPSTER_CHAN_TOPIC, "%s",
		  chan);
}*/

COMMAND(do_clear) {
   ChannelInfo *chan;
   GtkWidget *text;
   
   chan = g_list_nth_data(st->ct->channel_list, 
			  gnapster_get_pagenum(st) - 2);
   if (chan)
     text = chan->text;
   else 
     text = st->ct->text;
   
   gtk_xtext_remove_lines(GTK_XTEXT(text), -1, TRUE);
}

COMMAND(do_sv) {
   int page_num;
   ChannelInfo *chan;
   
   page_num = gnapster_get_pagenum(st);
   
   if (args) {
      char *user;
      
      user = next_arg(args, &args);
      
      NA_RESET();
      
      hook_text_insert(st, CURR, MESSAGE, "privmsg_out", "%s\4%s",
		       user, gnapster_sv());
      napster_send(ci->sock, NAPSTER_USER_MESSAGE, "%s %s",
		   user, gnapster_sv());
      return;
   }
   
   if (page_num <= 1) 
     return;
   
   chan = g_list_nth_data(st->ct->channel_list,
			  page_num - 2);
   if (!chan) 
     return;

   hook_text_insert(st, page_num, MESSAGE, "public_msg_self", "%s\4%s",
		    ci->account->user, gnapster_sv());
   napster_send(ci->sock,
		(chan->user) ? NAPSTER_USER_MESSAGE : NAPSTER_CHAN_MESSAGE,
		"%s %s", chan->channel_name, gnapster_sv());
}

/*COMMAND(do_whois) {
   char *user;
   
   user = next_arg(args, &args);
   if (!user)
     user = ci->account->user;
   
   NA_RESET();
   
   hook_text_insert(st, CURR, SYSTEM, "whois_ack", "%s",
		    user);
   
   napster_send(ci->sock, NAPSTER_WHOIS_REQUEST, user);
}*/

COMMAND(do_msg) {
   char *user, *message;
   
   user = next_arg(args, &args);
   message = last_arg(args, &args);
   
   NA_RESET();
   
   if (!message)
     INVALID_ARGS();
   
   nick_history = add_nick_history(nick_history, user);
   
   hook_text_insert(st, CURR, MESSAGE, "privmsg_out", "%s\4%s",
		    user, message);
   napster_send(ci->sock, NAPSTER_USER_MESSAGE, "%s %s",
		user, message);
}

COMMAND(do_me) {
   char *chan;

   CHECK_ARGS();
   
   chan = get_current_chan();
   if (!chan)
     return;
   
   hook_text_insert(st, CURR, SYSTEM, "public_action_self", "%s\4%s",
		    ci->account->user, args);
   napster_send(ci->sock, NAPSTER_ACTION, "%s \"%s\"",
		chan, args);
}

COMMAND(do_query) {
   if (!st)
     st = get_current_stab();
   
   if (!args) {
      ChannelInfo *data;
      
      data = g_list_nth_data(st->ct->channel_list,
			     gnapster_get_pagenum(st) - 2);
      if (!data)
	return;
      
      if (data->user)
	remove_console_tab(st, NULL, 1);
      else
	hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
			 "current tab is not a user query");
   } else {
      char *user;
      
      user = next_arg(args, &args);
      
      NA_RESET();
      
      real_query(st, 1, user);
   }
   
   st = NULL;
}

COMMAND(do_ignore) {
   if (!args) {
      GList *ptr;

      ptr = ignore_list;
      
      if (ptr) 
	hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
			 "ignore list");
      else
	hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
			 "no one on your ignore list");

      while(ptr) {
	 gnapster_text_insert(st, CURR, MESSAGE, "     %s\n",
			      ptr->data);
	 
	 ptr = ptr->next;
      }
   } else {
      char *user;
      user = next_arg(args, &args);
      
      NA_RESET();
      
      if (user_ignored(user)) {
	 hook_text_insert(st, CURR, SYSTEM, "general_warning", "%s %s",
			  user, "is already on ignore");
      } else if (!strcmp(user, ci->account->user)) {
	 hook_text_insert(st, CURR, SYSTEM_N, "general_error", "%s",
			  "you cannot ignore yourself");
      } else {
	 hook_text_insert(st, CURR, SYSTEM, "general_message", "%s %s",
			  "now ignoring", user);

	 ignore_list = g_list_append(ignore_list, d_strdup(user));
      }
   }
}

COMMAND(do_unignore) {
   GList *link;
   char *user;
   
   CHECK_ARGS();
   
   user = next_arg(args, &args);
   
   NA_RESET();
   
   link = ignore_list;
   while(link) {
      if (!link->data) 
	break;
      if (!strcmp(link->data, user))
	break;
      link = link->next;
   }
   if (link) {
      hook_text_insert(st, CURR, SYSTEM, "general_message", "removing %s %s",
		       user, "from the ignore list");

      ignore_list = g_list_remove_link(ignore_list, link);
      
      d_free(link->data);
   } else {
      hook_text_insert(st, CURR, SYSTEM, "general_message", "%s %s",
		       user, "is not in the ignore list");
   }
}

COMMAND(do_join) {
   char *chan;
   
   CHECK_ARGS();
   
   chan = next_arg(args, &args);
   
   NA_RESET();
   
   hook_text_insert(st, CURR, SYSTEM, "join_ack", "%s",
		    chan);
   napster_send(ci->sock, NAPSTER_JOIN_CHAN, chan);
}

COMMAND(do_part) {
   if (!remove_console_tab(st, NULL, 0))
     hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
		      "you are not in a channel tab");
}

COMMAND(do_raw) {
   int srvmsg;
   char *message; 

   if (!args) {
      d_summary();
      return;
   }
   
   CHECK_ARGS();
   
   srvmsg = my_atoi(next_arg(args, &args));
   message = last_arg(args, &args);
   
   hook_text_insert(st, CURR, SYSTEM, "raw_msg", "%i\4%s",
		    srvmsg, message);
   napster_send(ci->sock, srvmsg, "%s", message);
}

COMMAND(do_kill) {
   char *user, *reason;
   
   user = next_arg(args, &args);
   reason = last_arg(args, &args);
   
   NA_RESET();
   
   if (!user)
     INVALID_ARGS();
   
   if (reason) 
     napster_send(ci->sock, NAPSTER_KILL_USER, "%s \"%s\"",
		  user, reason);
   else
     napster_send(ci->sock, NAPSTER_KILL_USER, "%s",
		  user);   
}

COMMAND(do_kick) {
   char *chan;
   char *user, *reason;
   
   user = next_arg(args, &args);
   reason = last_arg(args, &args);
   
   NA_RESET();
   
   if (!user)
     INVALID_ARGS();

   chan = get_current_chan();
   if (!chan)
     return;
   
   if (reason) 
     napster_send(ci->sock, NAPSTER_KICK, "%s %s \"%s\"",
		  chan, user, reason);
   else
     napster_send(ci->sock, NAPSTER_KICK, "%s %s",
		  chan, user);
}

COMMAND(do_ban) {
   char *user, *reason;
   
   CHECK_ARGS();
   
   user = next_arg(args, &args);
   reason = last_arg(args, &args);
   
   NA_RESET();
   
   if (reason)
     napster_send(ci->sock, NAPSTER_BAN_USER, "%s \"%s\"",
		  user, reason);
   else
     napster_send(ci->sock, NAPSTER_BAN_USER, "%s",
		  user);
}

COMMAND(do_tban) {
   char *user, *t;
   unsigned long timeout;
   
   CHECK_ARGS();
   
   user = next_arg(args, &args);
   t = next_arg(args, &args);
   
   NA_RESET();
   
   if (!t)
     return;
   
   convert(t, "%lu", &timeout);
   
   timeout *= 60;
   
   napster_send(ci->sock, NAPSTER_BAN_USER, "%s \"\" %lu\n",
		timeout);
}

COMMAND(do_unban) {
   char *user;
   
   CHECK_ARGS();
   
   user = next_arg(args, &args);
   
   NA_RESET();
   
   napster_send(ci->sock, NAPSTER_UNBAN_USER, "%s",
		user);
}

COMMAND(do_bans) {
   ci->ban_exp = d_strdup(next_arg(args, &args));
   
   hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
		    "retrieving ban list");
   napster_send(ci->sock, NAPSTER_BAN_LIST, NULL);
}

COMMAND(do_muzzle) {
   char *user, *reason;
   
   CHECK_ARGS();
   
   user = next_arg(args, &args);
   reason = last_arg(args, &args);
   
   NA_RESET();
   
   if (reason)
     napster_send(ci->sock, NAPSTER_MUZZLE_USER, "%s \"%s\"",
		  user, reason);
   else
     napster_send(ci->sock, NAPSTER_MUZZLE_USER, "%s",
		  user);
}

COMMAND(do_unmuzzle) {
   char *user;
   
   CHECK_ARGS();
   
   user = next_arg(args, &args);
   
   NA_RESET();
   
   napster_send(ci->sock, NAPSTER_UNMUZZLE_USER, "%s",
		user);
}

COMMAND(do_mode) {
   char *chan, *mode;
   
   CHECK_ARGS();
   
   chan = next_arg(args, &args);
   mode = last_arg(args, &args);
   
   NA_RESET();

   if (mode) 
     napster_send(ci->sock, NAPSTER_ChannelInfo_MODE, "%s %s", chan, mode);
   else
     napster_send(ci->sock, NAPSTER_ChannelInfo_MODE, "%s", chan);
}

COMMAND(do_setport) {
   char *user;
   int port;
   
   user = next_arg(args, &args);
   port = my_atoi(next_arg(args,  &args));
   
   NA_RESET();
   
   if (port < 0)
     INVALID_ARGS();
   
   napster_send(ci->sock, NAPSTER_REQUEST_PORTCHANGE, "%s %i",
		user, port);
}

COMMAND(do_userlevel) {
   char *user, *ul;
   
   user = next_arg(args, &args);
   ul = next_arg(args, &args);
   
   NA_RESET();
   
   if (!ul)
     INVALID_ARGS();
   
   napster_send(ci->sock, NAPSTER_CHANGE_USERLEVEL, "%s %s",
		user, ul);
}

COMMAND(do_setline) {
   char *user, *line;
   extern char **conn_table;
   int i;
   
   user = next_arg(args, &args);
   line = next_arg(args, &args);
   
   NA_RESET();
   
   if (!line)
     INVALID_ARGS();
   
   for(i=0; i<10; i++)
     if (!strcmp(line, conn_table[i]))
       break;
     
   if (i == 11)
     return;
   
   napster_send(ci->sock, NAPSTER_SET_LINESPEED, "%s %i",
		user, i);
}

COMMAND(do_links) {
   napster_send(ci->sock, NAPSTER_LINKS, NULL);
}

COMMAND(do_signore) {
   if (!args) {
      GList *ptr;
      int i = 1, page_num;
      
      page_num = gnapster_get_pagenum(st);
      ptr = signore_list;
      if (!ptr) {
	 hook_text_insert(st, page_num, SYSTEM, "general_message", "%s",
			  "your server ignore list is empty");
	 return;
      }
      
      hook_text_insert(st, page_num, SYSTEM, "general_message", "%s",
		       "ignoring server messages matching: ");
      
      while(ptr) {
	 gnapster_text_insert(st, page_num, MESSAGE,
			      "    %-3i %s\n", i++, (char *)ptr->data);
	 ptr = ptr->next;
      }
      
      return;
   }
   
   signore_list = g_list_append(signore_list, d_strdup(args));
   
   hook_text_insert(st, CURR, SYSTEM, "general_message", "added %s %s",
		    args, "to your server ignore list");
}

COMMAND(do_unsignore) {
   GList *ptr;
   int n;
   
   CHECK_ARGS();
   
   n = my_atoi(next_arg(args, &args));
   
   NA_RESET();
   
   if (n < 0)
     INVALID_ARGS();

   ptr = g_list_nth(signore_list, n - 1);
   if (!ptr)
     return;
   
   hook_text_insert(st, CURR, SYSTEM, "general_message", "removing %s %s",
		    ptr->data, "from your server ignore list");

   signore_list = g_list_remove_link(signore_list, ptr);
   d_free(ptr->data);
}

COMMAND(do_sigsave) {
   GList *ptr;
   char *str, *buf;
   int page_num;

   page_num = gnapster_get_pagenum(st);

   buf = d_strdup("");
   
   for(ptr=signore_list; ptr; ptr=ptr->next) {
      str = ptr->data;
      if (!str)
	continue;
      
      d_strexp(&buf, "%s%s\4", buf, str);
   }
   
   if (!buf || !(*buf))
     return;
   
   hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
		    "saving signore list");
   
   j_config_set_string("/gnapster/Options/signore", buf);
   j_config_sync();
   
   d_free(buf);
}

COMMAND(do_stats) {
   hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
		    "sending server stats request");
   napster_send(ci->sock, NAPSTER_SERVER_STATS, NULL);
}

COMMAND(do_chanlevel) {
   char *chan, *level;
   
   chan = next_arg(args, &args);
   level = next_arg(args, &args);
   
   NA_RESET();
   
   if (!level)
     INVALID_ARGS();
   
   napster_send(ci->sock, NAPSTER_ChannelInfo_LEVEL, "%s %s",
		chan, level);
}

COMMAND(do_usermode) {
   CHECK_ARGS();
   
   napster_send(ci->sock, NAPSTER_SET_USER_MODE, "%s",
		args);
}

COMMAND(do_op) {
   char *chan, *user;
   
   user = next_arg(args, &args);
   
   NA_RESET();
   
   if (!user)
     INVALID_ARGS();
   
   chan = get_current_chan();
   if (!chan)
     return;
   
   napster_send(ci->sock, NAPSTER_SET_ChannelInfo_OP, "%s %s",
		chan, user);
}

COMMAND(do_deop) {
   char *chan, *user;
   
   user = next_arg(args, &args);
   
   NA_RESET();
   
   if (!user)
     INVALID_ARGS();

   chan = get_current_chan();
   if (!chan)
     return;
   
   napster_send(ci->sock, NAPSTER_REMOVE_ChannelInfo_OP, "%s %s",
		chan, user);
}

COMMAND(do_cban) {
   char *chan, *user, *reason;
   
   user = next_arg(args, &args);
   reason = last_arg(args, &args);
   
   NA_RESET();
   
   if (!user)
     INVALID_ARGS();
   
   chan = get_current_chan();
   if (!chan)
     return;
   
   if (reason)
     napster_send(ci->sock, NAPSTER_CBAN, "%s %s \"%s\"",
		  chan, user, reason);
   else
     napster_send(ci->sock, NAPSTER_CBAN, "%s %s",
		  chan, user);
}

COMMAND(do_cunban) {
   char *chan, *user;
   
   user = next_arg(args, &args);
   
   NA_RESET();
   
   if (!user)
     INVALID_ARGS();
   
   chan = get_current_chan();
   if (!chan)
     return;

   napster_send(ci->sock, NAPSTER_CUNBAN, "%s %s",
		chan, user);
}

COMMAND(do_cbans) {
   char *chan;
   
   chan = get_current_chan();
   if (!chan)
     return;
   
   hook_text_insert(st, CURR, SYSTEM, "general_message", "%s %s",
		    "retrieving channel ban list for", chan);
   napster_send(ci->sock, NAPSTER_CBAN_LIST, "%s",
		chan);
}

COMMAND(do_cunban_all) {
   char *chan;
   
   chan = get_current_chan();
   if (!chan)
     return;
   
   napster_send(ci->sock, NAPSTER_CBAN_CLEAR, "%s",
		chan);
}

COMMAND(do_load) {
   char *path, *theme, *ttheme;
   
   theme = next_arg(args, &args);
   
   NA_RESET();
   
   if (!theme)
     INVALID_ARGS();
   
   ttheme = NULL;
   
   if (!strstr(theme, ".theme"))
     d_msprintf(&ttheme, "%s.theme", theme);
   
   if (!ttheme)
     ttheme = d_strdup(theme);
   
   path = local_path("themes", ttheme);
   
   if (file_exists(path))
     j_config_set_string("/gnapster/Display/theme", path);
   else {
      d_free(path);
      path = global_path("themes", ttheme);
      if (file_exists(path))
	j_config_set_string("/gnapster/Display/theme", path);
      else
	hook_text_insert(st, CURR, SYSTEM_N, "general_error", 
			 "%s %s", ttheme, "not found");
   }
   
   j_config_sync();
   
   d_free(path);
   
   d_free(ttheme);
   
   destroy_theme_hook(hooks);
   load_theme();
}

COMMAND(do_reload) {
   destroy_theme_hook(hooks);
   load_theme();
}

COMMAND(do_nuke) {
   char *user;
   
   user = next_arg(args, &args);
   
   NA_RESET();
   
   if (!user)
     INVALID_ARGS();
   
   napster_send(ci->sock, NAPSTER_NUKE_USER, "%s",
		user);   
}

COMMAND(do_hook) {
   char *hook, *arg, *arg_out;
   
   hook = next_arg(args, &args);
   
   NA_RESET();

   if (!hook)
     INVALID_ARGS();
   
   arg_out = d_strdup("");
   
   /* convert the ' ' delimited arguments into 0x04's for hook_text_insert */
   while((arg = next_arg(args, &args)))
     d_strexp(&arg_out, "%s%s\4", arg_out, arg);
   
   NA_RESET();
   
   gnapster_text_insert(st, CURR, SYSTEM, "%s: ", hook);
   hook_text_insert(st, CURR, MESSAGE, hook, "%s", arg_out);
   
   d_free(arg_out);
}

COMMAND(do_stab) {
   if (args &&
       !strcmp(args, "-r")) {
      remove_server_tab(st);
      
      return;
   }
   
   append_server_tab();
}

COMMAND(do_debug) {
   if (debug(ci)) {
      hook_text_insert(st, CURR, SYSTEM, "general_message", "%s %i",
		       "disabling debugging on", ci->sock);
      ci->state &= ~DEBUG_MASK;
   } else {
      hook_text_insert(st, CURR, SYSTEM, "general_message", "%s %i",
		       "enabling debugging on", ci->sock);
      ci->state |= DEBUG_MASK;
   }
}

/*void do_connect_finish(void *data, unsigned int ip) {
   DSPACK();
   Server *srv;
   struct in_addr in;
   
   d_assert(data != NULL);
   
   SPACK(srv);
   
   if (!ip) {
      hook_text_insert(stb, CURR, SYSTEM, "general_error", "%s",
		       "failure resolving server hostname");
      return;
   }
   
   in.s_addr = ip;
      
   stb->ci->ip = d_strdup(inet_ntoa(in));
   if (!stb->ci->ip) {
      hook_text_insert(stb, CURR, SYSTEM, "general_error", "%s",
		       "failure resolving server hostname");
      return;
   }
   
   stb->ci->port = srv->port;
   stb->ci->server = d_strdup(srv->desc);
   
   connect_cb(stb);
}*/

/*COMMAND(do_connect) {
   char *host;
   unsigned short port;
   
   host = next_arg_full(args, &args, ':');
   if (!ip)
     INVALID_ARGS();
   
   convert(next_arg(args, &args), "%hu", &port);
   
   if (na_err)
     port = 8888;
   
   nb_gethostbyname(do_connect_finish, host, st);
}*/

void exec_linked_cmd(LinkedCommand *cmd, char *args) {
   char *out;
   GList *iter_ptr;
   CommandArg *argptr, arg;
   char *na;
   int na_int;
   int argcount = 0;

   out = d_strdup("");

   ITER_LIST(cmd->args) {
      LIST_DATA(argptr);
      
      arg = (CommandArg)argptr;
      
      if (arg == STRING || arg == INT)
	na = next_arg(args, &args);
      else if (arg & LAST)
	na = last_arg(args, &args);
      
      na = d_strdup(na);
      
      if (arg == INT) {
	 convert(na, "%i", &na_int);
	 d_strexp(&na, "%i", na_int);
      }
      
      if (!na)
	continue;
      
      if (arg & QUOTED)
	d_strexp(&na, "\"%s\"", na);
      
      d_strexp(&out, "%s%s ", out, na);

      argcount++;
      
      d_free(na);
   }
   
   rem_trail(out, ' ');
   
   if (argcount < cmd->min_args || (cmd->max_args > 0 && argcount > cmd->max_args)) {
      hook_text_insert(st, CURR, SYSTEM, "general_error", 
		       "wrong number of args for %s (min_args = %i, max_args = %i)\n", 
		       cmd->command, cmd->min_args, cmd->max_args);
      d_free(out);
      return;
   }

   /* extend the string again w/ the special funcs */
   if (cmd->link_func) {
      if ((cmd->link_func) (cmd, &out) < 0) {
	 hook_text_insert(st, CURR, SYSTEM, "general_error", 
			  "unknown error while executing command %s\n", 
			  cmd->command);
	 d_free(out);
	 return;
      }
   }

   if (!*(out)) {
      d_free(out);
      out = NULL;
   }
   
   rem_trail(out, ' ');
   
   napster_send(ci->sock, cmd->cmd, out);
   
   d_free(out);
}

COMMAND(do_admin) {
   char *group;
   GList *ptr;
   LinkedCommand *cmd;
   
   group = next_arg(args, &args);
   if (!group)
     return;
   
   for(ptr=acmds; ptr; ptr=ptr->next) {
      cmd = ptr->data;
      if (!cmd)
	continue;
      
      if (!strcmp(cmd->command, group)) {
	 exec_linked_cmd(cmd, args);
	 return;
      }
   }
}

void new_linked_copy(GList **tbl, char *group, LinkedCommand *orig) {
   LinkedCommand *cmd;
   
   d_assert(group != NULL);
   
   cmd = d_malloc(sizeof(LinkedCommand));
   
   cmd->command = d_strdup(group);
   
   cmd->cmd = orig->cmd;
   
   cmd->min_args = orig->min_args;
   cmd->max_args = orig->max_args;
   
   /* if later use calls for this, set it */
   cmd->func = orig->func;
   
   cmd->link_func = orig->link_func;
   
   cmd->args = orig->args;
   
   *tbl = g_list_append(*tbl, cmd);
}

void new_linked_cmd(GList **tbl, char *group, unsigned short command, int min_args, int max_args, void *special, ...) {
   LinkedCommand *cmd;
   va_list args;
   CommandArg *arg;
   
   d_assert(group != NULL);
   
   cmd = d_malloc(sizeof(LinkedCommand));
   
   cmd->command = d_strdup(group);
   
   /* napster command */
   cmd->cmd = command;
   
   cmd->min_args = min_args;
   cmd->max_args = max_args;
   
   /* if we need any extension func, set it here */
   cmd->link_func = special;

   cmd->args = NULL;
   va_start(args, special);
   for(;;) {
      arg = va_arg(args, CommandArg *);
      if (!arg)
	break;
      
      cmd->args = g_list_append(cmd->args, arg);
   }
   va_end(args);
   
   *tbl = g_list_append(*tbl, cmd);
}

void new_cmd_group(GList **tbl, char *group) {
   LinkedCommand *cmd;
   
   d_assert(group != NULL);
   
   cmd = d_malloc(sizeof(LinkedCommand));
   
   cmd->command = d_strdup(group);
   
   *tbl = g_list_append(*tbl, cmd);
}

void new_cmd(GList **tbl, char *command, void *func, char *args, char *desc) {
   LinkedCommand *cmd;
   
   cmd = d_malloc(sizeof(LinkedCommand));
   
   cmd->command = d_strdup(command);
   
   cmd->func = func;
   
   cmd->syntax = d_strdup(args);
   cmd->help = d_strdup(desc);
   
   *tbl = g_list_append(*tbl, cmd);
}

void new_cmd_alias(GList **tbl, char *orig, ...) {
   GList *iter_ptr;
   LinkedCommand *cmd;
   char *cmdname;
   va_list args;

   /* aliases work by locating the original command and duplicating it under 
    * different command names */
   ITER_LIST(*tbl) {
      LIST_DATA(cmd);
      
      if (strcmp(cmd->command, orig))
	continue;
      
      va_start(args, orig);
      for(;;) {
	 cmdname = va_arg(args, char *);
	 if (!cmdname)
	   break;
	 
	 if (cmd->cmd > 0)
	   new_linked_copy(tbl, cmdname, cmd);
	 else
	   new_cmd(tbl, cmdname, cmd->func, cmd->syntax, cmd->help);	 
      }
      va_end(args);
      
      return;
   }
}

void hostcb(unsigned int ip) {
   struct in_addr a;
   
   a.s_addr = ip;
   
   printf("%u -> %s\n", ip, inet_ntoa(a));
}

COMMAND(do_dns) {
   nb_gethostbyname(hostcb, "jasta.gotlinux.org");
}

void command_setup() {
   new_cmd_group(&ucmds, "Generic Commands");
   new_cmd(&ucmds, "help", do_help, "[command]", "Shows help for Gnapster commands");
   new_linked_cmd(&ucmds, "version", CMDS_SERVERVERSION, 0, 0, NULL, NULL);
   new_cmd(&ucmds, "sv", do_sv, "[user]", "Shows version information");
   new_cmd(&ucmds, "clear", do_clear, "", "Clear the current text widget");
   new_cmd_alias(&ucmds, "clear", "cls", NULL);
   new_cmd(&ucmds, "load", do_load, "<theme>", "Loads the given theme");
   new_cmd(&ucmds, "reload", do_reload, "", "Reloads the current theme");
/*   new_cmd(&ucmds, "connect", do_connect, "<server>[:port]", "Manually connects to a server");
   new_cmd_alias(&ucmds, "connect", "server", NULL);
   new_cmd(&ucmds, "disconnect", do_disconnect, "", "Disconnects from the current server");
   new_cmd_alias(&ucmds, "disconnect", "disco", NULL);*/
   new_cmd(&ucmds, "signore", do_signore, "[wildmatch string]", "Ignores a server message that matches [wildmatch string]");
   new_cmd_alias(&ucmds, "signore", "sig", NULL);
   new_cmd(&ucmds, "unsignore", do_unsignore, "<signore #>", "Removes a signore");
   new_cmd_alias(&ucmds, "unsignore", "usig", NULL);
   new_cmd(&ucmds, "sigsave", do_sigsave, "", "Saves your current signore setup");

   new_cmd_group(&ucmds, "Channel Commands");
   new_linked_cmd(&ucmds, "topic", CMDS_TOPIC, 0, -1, prepend_chan, LAST, NULL);
   new_cmd_alias(&ucmds, "topic", "t", NULL);
   new_linked_cmd(&ucmds, "me", CMDS_SENDME, 1, -1, cmd_ack, LAST, NULL);
   new_linked_cmd(&ucmds, "join", CMDS_JOIN, 1, 1, cmd_ack, STRING, NULL);
   new_cmd_alias(&ucmds, "join", "j", NULL);
/*   new_linked_cmd(&ucmds, "part", CMDS_PART, 0, 1, prepend_chan, STRING, NULL);*/
   new_cmd(&ucmds, "part", do_part, "", "Parts the current channel");
   new_cmd_alias(&ucmds, "part", "p", NULL);
   new_cmd(&ucmds, "say", do_sendtext, "<text>", "Sends text to the current channel");
   new_linked_cmd(&ucmds, "op", CMDS_CREATEOP, 1, 3, prepend_chan, LAST, NULL);
   new_linked_cmd(&ucmds, "deop", CMDS_DELETEOP, 1, 3, prepend_chan, LAST, NULL);
   new_linked_cmd(&ucmds, "cban", CMDS_CBAN, 1, -1, prepend_chan, STRING, LAST | QUOTED, NULL);
   new_linked_cmd(&ucmds, "cunban", CMDS_CUNBAN, 1, -1, prepend_chan, STRING, LAST | QUOTED);
   new_linked_cmd(&ucmds, "cbanlist", CMDS_CBANLIST, 0, 0, prepend_chan, NULL);
   new_cmd_alias(&ucmds, "cbanlist", "cbans", NULL);
   new_linked_cmd(&ucmds, "cbanclear", CMDS_CBANCLEAR, 0, 0, prepend_chan, NULL);
   new_linked_cmd(&ucmds, "kick", CMDS_KICK, 1, -1, prepend_chan, STRING, LAST | QUOTED, NULL);
   new_cmd_alias(&ucmds, "kick", "k", NULL);
   new_linked_cmd(&ucmds, "kickban", CMDS_KICK, 1, -1, kickban, STRING, LAST | QUOTED, NULL);
   new_cmd_alias(&ucmds, "kickban", "kb", NULL);
   new_linked_cmd(&ucmds, "opsay", CMDS_OPWALL, 1, -1, prepend_chan, LAST, NULL);
   new_cmd_alias(&ucmds, "opsay", "ops", NULL);
   new_linked_cmd(&ucmds, "cmuzzle", CMDS_CHANNELMUZZLE, 1, -1, prepend_chan, STRING, LAST | QUOTED, NULL);
   new_linked_cmd(&ucmds, "cunmuzzle", CMDS_CHANNELUNMUZZLE, 1, -1, prepend_chan, STRING, LAST | QUOTED, NULL);

   new_cmd_group(&ucmds, "Chat Commands");
   new_linked_cmd(&ucmds, "whois", CMDS_WHOIS, 0, 1, cmd_ack, STRING, NULL);
   new_cmd_alias(&ucmds, "whois", "w", "wi", NULL);
/*   new_linked_cmd(&ucmds, "msg", CMDS_SENDMSG, 2, -1, cmd_ack, STRING, LAST, NULL);*/
   new_cmd(&ucmds, "msg", do_msg, "<user> <message>", "Sends a user a private message");
   new_cmd_alias(&ucmds, "msg", "m", NULL);
   new_cmd(&ucmds, "query", do_query, "<user>", "Open a chat session with a user");
   new_cmd_alias(&ucmds, "query", "q", NULL);
   new_linked_cmd(&ucmds, "ignore", CMDS_IGNOREADD, 1, 1, cmd_ack, STRING, NULL);
   new_cmd_alias(&ucmds, "ignore", "ig", NULL);
   new_linked_cmd(&ucmds, "ignores", CMDS_IGNORELIST, 0, 0, cmd_ack, NULL);
   new_linked_cmd(&ucmds, "unignore", CMDS_IGNOREREMOVE, 1, 1, cmd_ack, STRING, NULL);
   new_cmd_alias(&ucmds, "unignore", "uig", "unig", NULL);
   new_linked_cmd(&ucmds, "mode", CMDS_CHANNELMODE, 0, -1, prepend_chan, LAST, NULL);
   new_cmd(&ucmds, "xfer", do_xfer_stats, "", "Displays transfer statistics to the current channel");
   
   new_cmd_group(&ucmds, "Debugging");
   new_cmd(&ucmds, "raw", do_raw, "<numeric> [data]", "Sends a raw numeric");
   new_cmd(&ucmds, "debug", do_debug, "", "Enable debugging on the current socket");
   new_cmd(&ucmds, "hook", do_hook, "<hook_name> [arg1 [arg2 [argn...]]]", "Test a given theme hook with supplied data");
   
   new_cmd_group(&ucmds, "Admin Commands");
/*   new_cmd(&ucmds, "admin", do_admin, "", "");*/
   new_cmd_alias(&ucmds, "admin", "a", NULL);
/*   new_linked_cmd(&ucmds, "links", CMDS_SERVERLINKS, 0, 0, NULL, NULL);
   new_linked_cmd(&ucmds, "tban", CMDS_BANUSER, 2, -1, tban, STRING, STRING, LAST, NULL);
   new_linked_cmd(&ucmds, "wall", CMDS_OPSAY, 1, -1, cmd_ack, LAST, NULL);*/
   
   
/*   new_admin_cmd("killserver", CMDS_SERVERKILL, 1, -1, NULL, NULL);
   new_admin_cmd("banuser", CMDS_BANUSER, 1, -1, NULL, NULL);
   new_admin_cmd("connect", CMDS_SERVERLINK, 2, -1, NULL, NULL);
   new_admin_cmd("disconnect", CMDS_SERVERUNLINK, 1, -1, NULL, NULL);
   new_admin_cmd("config", CMDS_SETCONFIG, 1, -1, NULL, NULL);
   new_admin_cmd("unnukeuser", CMDS_UNNUKEUSER, 1, -1, NULL, NULL);
   new_admin_cmd("removeserver", CMDS_SERVERREMOVE, 1, -1, NULL, NULL);
   new_admin_cmd("version", CMDS_SERVERVERSION, 0, 0, NULL, NULL);
   new_admin_cmd("reload", CMDS_RELOADCONFIG, 1, 1, NULL, NULL);*/
   
   new_linked_cmd(&ucmds, "muzzle", CMDS_MUZZLE, 1, -1, NULL, LAST, NULL);
   new_linked_cmd(&ucmds, "unmuzzle", CMDS_UNMUZZLE, 1, -1, NULL, LAST, NULL);
   new_linked_cmd(&ucmds, "setuserlevel", CMDS_SETUSERLEVEL, 2, 2, NULL, STRING, STRING, NULL);
   new_linked_cmd(&ucmds, "setlinespeed", CMDS_SETLINESPEED, 2, 2, NULL, STRING, STRING, NULL);
   new_linked_cmd(&ucmds, "setdataport", CMDR_SETDATAPORT, 2, 2, NULL, STRING, STRING, NULL);
   new_linked_cmd(&ucmds, "wallop", CMDS_OPSAY, 1, -1, cmd_ack, LAST, NULL);
   new_cmd_alias(&ucmds, "wallop", "wall", NULL);
   new_linked_cmd(&ucmds, "announce", CMDR_ANNOUNCE, 1, -1, cmd_ack, LAST, NULL);
   new_cmd_alias(&ucmds, "announce", "global", NULL);
   new_linked_cmd(&ucmds, "op", CMDS_CREATEOP, 1, 3, NULL, LAST, NULL);
   new_linked_cmd(&ucmds, "deop", CMDS_DELETEOP, 1, 3, NULL, LAST, NULL);
   new_cmd_alias(&ucmds, "deop", "dop", NULL);
   new_linked_cmd(&ucmds, "nukeuser", CMDS_NUKEUSER, 1, 3, NULL, LAST, NULL);
   new_cmd_alias(&ucmds, "nukeuser", "nuke", NULL);
   new_linked_cmd(&ucmds, "unnukeuser", CMDS_UNNUKEUSER, 1, 1, NULL, STRING, NULL);
   new_cmd_alias(&ucmds, "unnukeuser", "unnuke", NULL);
   new_linked_cmd(&ucmds, "links", CMDS_SERVERLINKS, 0, 0, NULL, NULL);
   new_linked_cmd(&ucmds, "ban", CMDS_BANUSER, 1, -1, NULL, STRING, LAST | QUOTED, NULL);
   new_linked_cmd(&ucmds, "unban", CMDS_UNBANUSER, 1, -1, NULL, STRING, LAST | QUOTED, NULL);
   new_linked_cmd(&ucmds, "kill", CMDS_KILLUSER, 1, -1, NULL, STRING, LAST | QUOTED, NULL);
/*   new_linked_cmd(&ucmds, "kickban", CMDS_KICK, 1, -1, kickban, STRING, LAST | QUOTED);*/
   new_cmd(&ucmds, "banlist", do_bans, "[regexp match]", "Lists global server bans");
/*   new_cmd(&ucmds, "tban", do_tban, "<user|ip> <timeout>", "Temporarily sets a global ban in hours");*/
}

LINKED_COMMAND(tban) {
   char *ptr;
   char *user, *timeout, *reason;
   
   ptr = *args;
   
   user = next_arg(ptr, &ptr);
   timeout = next_arg(ptr, &ptr);
   reason = last_arg(ptr, &ptr);
   
   d_strexp(args, "%s \"%s\" %s", user, reason ? reason : "", timeout);
   
   return 0;
}

LINKED_COMMAND(kickban) {
   prepend_chan(cmd, args);
   
   napster_send(ci->sock, CMDS_CBAN, "%s", *args);
   
   return 0;
}

LINKED_COMMAND(prepend_chan) {
   char *chan;
   
   if (**args == '#')
     return 1;
   
   chan = get_current_chan();
   if (!chan)
     return -1;
   
   if (**args)
     d_strexp(args, "%s %s", chan, *args);
   else
     d_strexp(args, "%s", chan);
   
   return 0;
}

LINKED_COMMAND(prepend_nick) {
   if (**args)
     return 1;
   
   d_strexp(args, "%s", st->ci->account->user);
   
   return 0;
}

LINKED_COMMAND(cmd_ack) {
   char *str;
   
   switch(cmd->cmd) {
    case CMDS_JOIN:
      hook_text_insert(st, CURR, SYSTEM, "join_ack", "%s",
		       *args);
      break;
    case CMDS_WHOIS:
      if (!(**args))
	prepend_nick(cmd, args);
      
      hook_text_insert(st, CURR, SYSTEM, "whois_ack", "%s",
		       *args);
      break;
    case CMDS_SENDMSG:
      str = translate_args(*args, 2);
      hook_text_insert(st, CURR, MESSAGE, "privmsg_out", "%s",
		       str);
      d_free(str);
      break;
    case CMDS_SENDME:      
      hook_text_insert(st, CURR, SYSTEM, "public_action_self", "%s\4%s",
		       st->ci->account->user, *args);

      d_strexp(args, "\"%s\"", *args);

      prepend_chan(cmd, args);
      break;
    case CMDS_IGNOREADD:
      hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
		       "Adding user to ignore list...");
      break;
    case CMDS_IGNOREREMOVE:
      hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
		       "Removing user from ignore list...");
      break;
    case CMDS_IGNORELIST:
      hook_text_insert(st, CURR, SYSTEM, "general_message", "%s",
		       "Ignore list:");
      break;
    case CMDS_OPSAY:
      hook_text_insert(st, CURR, MESSAGE, "wallop_self", "%s\4%s",
		       st->ci->account->user, *args);
      break;
    default:
      printf("failed to ack %s, %hu\n", cmd->command, cmd->cmd);
      break;
   }
   
   return 0;
}
