/* gnapster.c - controls main loop and various run-time GUI changes */

#include <stdio.h>

#include <signal.h>

#include <sys/utsname.h>
#include <ctype.h>

#include "gnapster.h"

#include "chan.h"
#include "resume.h"
#include "xtext.h"
#include "themes.h"
#include "servers.h"
#include "commands.h"

#include "timestamp.h"

#include "../pixmaps/speedgreen.xpm"
#include "../pixmaps/speedyellow.xpm"
#include "../pixmaps/speedred.xpm"
#include "../pixmaps/speedgray.xpm"

/* TabTable */
GList *ttbl = NULL;

GnapsterMain *gmain;

PropDlg prop_dlg;
BuildDlg build_dlg;
BrowseDlg browse_dlg;
AddDlg add_dlg;
EditDlg edit_dlg;
ListChanDlg list_chan_dlg;
ResumeDlg resume_dlg;
RefreshDlg refresh_dlg;

UserInfo user_info;

int na_err = 0;

int building;
GList *ignore_list = NULL;
GList *signore_list = NULL;
GList *hooks = NULL;
GdkFont *xtext_font = NULL;

GdkPixmap *speedgreen = NULL, *speedyellow = NULL, *speedred = NULL, *speedgray = NULL;
GdkBitmap *speedgreenb = NULL, *speedyellowb = NULL, *speedredb = NULL, *speedgrayb = NULL;

ConfigTable conf_table[] = {
  /* Key                        Gnapster key		Default Value */
   { "name_fix", 		NAME_FIX,		1 },
   { "line_speed_switch", 	LINE_SPEED_SWITCH,	0 },
   { "line_speed_cmp", 		LINE_SPEED_CMP,		0 },
   { "line_speed", 		LINE_SPEED,		0 },
   { "bitrate_switch", 		BITRATE_SWITCH,		0 },
   { "bitrate_cmp", 		BITRATE_CMP,		0 },
   { "bitrate", 		BITRATE,		0 },
   { "max_uploads_switch", 	MAX_UPLOADS_SWITCH,	1 },
   { "max_uploads", 		MAX_UPLOADS,		5 },
   { "reject_ul", 		REJECT_UL,		0 },
   { "firewall", 		FIREWALL,		0 },
   { "max_peruser_uploads_switch", MAX_PERUSER_UPLOADS_SWITCH,	0 },
   { "max_peruser_uploads",	MAX_PERUSER_UPLOADS,	2 },
   { "autoquery",		AUTOQUERY,		1 },
   { "max_downloads_switch",	MAX_DOWNLOADS_SWITCH,	0 },
   { "max_downloads",		MAX_DOWNLOADS,		4 },
   { "max_peruser_downloads_switch", MAX_PERUSER_DOWNLOADS_SWITCH, 1 },
   { "max_peruser_downloads",	MAX_PERUSER_DOWNLOADS,	1 },
   { "hide_users",		HIDE_USERS,		0 },
   { "logging",                 LOGGING,                1 },
   { "hide_joinpart",           HIDE_JOINPART,          0 },
   { "socks_firewall",          SOCKS_FIREWALL,         0 },
   { "socks4",                  SOCKS4,                 0 },
   { "socks5",                  SOCKS5,                 1 },
   { "tint",                    TINT,                   0 },
   { "tint_red",                TINT_RED,               180 },
   { "tint_green",              TINT_GREEN,             180 },
   { "tint_blue",               TINT_BLUE,              180 },
   { "autocancel",              AUTOCANCEL,             0 },
   { "search_local",            SEARCH_LOCAL,           0 },
   { "list_refresh",            LIST_REFRESH,           1 },
   { "broadcast",               BROADCAST,              0 },
   { "show_stabs",              SHOW_STABS,             1 },
   { "disable_ul",              DISABLE_UL,		0 },
   { "exec_dl_optn",            EXEC_DL_OPTN,           0 },
   { "bw_down",                 BW_DOWN,                0 },
   { "bw_up",                   BW_UP,                  0 },
   { "timestamp",               TIMESTAMP_EV,           0 },
   { NULL, 		       -1,		       -1 },
};

/* this function is really hackish ... */
Transfer *real_download_guess(STab **st, char *user, char *file, int *row) {
   Transfer *t;
   
/*   FORSTABS(*st = stab; if ((t = real_download(stab, user, file, row))) return t);*/
/*   *st = stab;*/
   if ((t = real_download(NULL, user, file, row))) {
      *st = t->st;
      return t;
   }
      
   /* hmm, this is bad ;) */
   *st = NULL;
   return NULL;
}

int real_upload_guess(STab **st, char *user, char *file) {
   FORSTABS(*st = stab; if (is_upload(stab, user, file)) return 1);
   
   *st = NULL;
   
   return 0;
}

Transfer *real_download(STab *stab, char *user, char *file, int *row) {
   Transfer *ret;
   GList *ptr;
   GtkCList *clist;
   int i = 0;
   
   clist = GTK_CLIST(gmain->dt->clist);
   
   for(ptr=clist->row_list; ptr; ptr=ptr->next, i++) {
      ret = gtk_clist_get_row_data(clist, i);
      
      if (!strcmp(ret->file, file) &&
	  !strcmp(ret->user, user)) {
	 if (row)
	   *row = i;
	 
	 return ret;
      }
   }
   
   return NULL;
}


char *gnapster_sv() {
   static char ver[256];
   struct utsname uts;
   
   if (uname(&uts) < 0)
     return NAP_VERSION;
   
   g_snprintf(ver, sizeof(ver), "%s [ %s ] / %s %s %s",
	      NAP_VERSION, TIMESTAMP, uts.nodename, uts.sysname, uts.release);
   
   return ver;
}

int gnapster_key_lookup(char *s) {
   int i = 0;
   
   if (!s) return -1;
   
   while(conf_table[i].conf_str) {
      if (!strcmp(conf_table[i].conf_str, s))
	return i;
      i++;
   }
   
   return -1;
}

void config_combo_widget(GtkWidget *w, char *key) {
   WidgetProperty *prop;
   char *full_conf;
   GtkWidget *widget;
   int conf;
   
   widget = GTK_IS_SPIN_BUTTON(w) ? w : GTK_COMBO(w)->entry;
   
   conf = gnapster_key_lookup(key);
   if (conf < 0)
     fprintf(stderr, "Warning, key '%s' not found!\n", key);
   
   if (GTK_IS_SPIN_BUTTON(w))
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), user_info.conf[conf]);

   prop = d_new(W_PROP);
   
   d_msprintf(&full_conf, "/gnapster/Options/%s", key);
   prop->key = conf;
   prop->conf = full_conf;
   
   gtk_object_set_data_full(GTK_OBJECT(widget), "prop", prop, gnapster_destroy);
 
/*   if (stab)
     stab->pwidgets = g_list_prepend(stab->pwidgets, widget);*/
   
   gtk_signal_connect(GTK_OBJECT(widget), "changed",
		      GTK_SIGNAL_FUNC(combo_changed), NULL);
}

void config_toggle_widget(GtkWidget *w, char *key, ...) {
   WidgetProperty *prop;
   GList *sensitive = NULL;
   char *full_conf;
   GtkWidget *widget;
   int conf;
   va_list args;
   
   conf = gnapster_key_lookup(key);
   if (conf < 0)
     fprintf(stderr, "Warning, key '%s' not found!\n", key);
   
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), user_info.conf[conf]);
   
   prop = d_new(W_PROP);
   
   d_msprintf(&full_conf, "/gnapster/Options/%s", key);
   prop->key = conf;
   prop->conf = full_conf;
   
   va_start(args, key);
   
   for(;;) {
      widget = va_arg(args, GtkWidget *);
      if (!widget) 
	break;
      
      gtk_widget_set_sensitive(GTK_WIDGET(widget), user_info.conf[conf]);
      
      sensitive = g_list_append(sensitive, widget);
   }
   
   va_end(args);
   
   prop->sensitive = sensitive;
   
   gtk_object_set_data_full(GTK_OBJECT(w), "prop", prop, gnapster_destroy);
   
/*   if (stab)
     stab->pwidgets = g_list_prepend(stab->pwidgets, w);*/
   
   gtk_signal_connect(GTK_OBJECT(w), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &user_info.conf[conf]);
}

int upload_accepted(STab *st, char *user, char *file) {
   char *tf;
   int uploads, downloads;
   
   d_assert_return(user, 0);
   d_assert_return(file, 0);
   
   if (!sharing(st->ci))
     return 0;
   
   tf = trunc_file(file);
   
   downloads = CLIST_LENGTH(gmain->dt->clist);
   uploads = CLIST_LENGTH(gmain->ut->clist);
   
   if (user_info.conf[MAX_UPLOADS_SWITCH] &&
       uploads >= user_info.conf[MAX_UPLOADS]) {
      hook_text_insert(st, CONSOLE, SYSTEM_N, "reject_upload", "%s\4%s",
		       user, tf);
      napster_send(st->ci->sock, NAPSTER_SENDLIMIT,
		   "%s \"%s\" %i", user, file,
		   user_info.conf[MAX_UPLOADS]);

      return 0;
   }
   
   if (user_info.conf[REJECT_UL] && downloads) {
      hook_text_insert(st, CONSOLE, SYSTEM, "reject_upload", "%s\4%s",
		       user, tf);
      napster_send(st->ci->sock, NAPSTER_SENDLIMIT,
		   "%s \"%s\" 1", user, file);

      return 0;
   }
   
   if (user_info.conf[MAX_PERUSER_UPLOADS_SWITCH] &&
       napster_user_uploads(st, user) >= user_info.conf[MAX_PERUSER_UPLOADS]) {
      hook_text_insert(st, CONSOLE, SYSTEM, "reject_upload", "%s\4%s",
		       user, tf);
      napster_send(st->ci->sock, NAPSTER_SENDLIMIT,
		   "%s \"%s\" %i", user, file,
		   user_info.conf[MAX_PERUSER_UPLOADS]);
      
      return 0;
   }
   
   return 1;
}

int verify_file_shared(char *sfile) {
   FILE *f;
   GList *felem, *ptr;
   FileEntry *fent;
   char *fptr, *mtime, *file;
   char *dptr;
   int x = 0;
   
   if (!sfile)
     return 0;
   
   f = open_shared("r");
   if (!f)
     return 0;

   felem = read_file(f);
   
   fclose(f);
   
   for(ptr=felem; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      dptr = fent->lptr;
      
      mtime = next_arg_first(dptr, &dptr, &fptr);
      file = next_arg(dptr, &dptr);
      
      NA_ERR_HANDLE_CONTINUE();
      
      if (!strcmp(DOSPATH(file), sfile))
	x = 1;
      
      if (x)
	break;
   }
   
   read_file_free(felem);
   
   return x;
}

FILE *open_file(Transfer *download, char *file) {
   FILE *f;
   char *fn = NULL, *basefile, *ext;
   int i = 0;
   
   basefile = d_strdup(file);
   
   ext = strrchr(basefile, '.');
   if (!ext)
     return NULL;
   
   *ext++ = 0;

   d_msprintf(&fn, "%s/%s", user_info.download_dir, file);
   conv_file(fn);
   
   if (file_exists(fn))
     for(i=1; i <= 20; i++) {
	d_msprintf(&fn, "%s/%s.%i.%s", 
		   user_info.download_dir, basefile, i, ext);
	conv_file(fn);

	if (!file_exists(fn))
	  break;

	d_free(fn);
     }
   
   d_free(basefile);
   
   f = fopen(fn, "w");
   if (!f)
     return NULL;
   
   download->fpath = fn;
   if (i)
     hook_text_insert(download->st, CONSOLE, SYSTEM, "download_newfile", 
		      "%s\4%s\4%s", download->user, download->trunc_file, fn);
   
   return f;
}

void *download_dup(Transfer *download) {
   Transfer *temp;
   
   if (!download)
     return NULL;

   temp = d_new(TRANSFER);
   transfer_init(temp);
   
   temp->file = d_strdup(download->file);
   temp->trunc_file = trunc_file(temp->file);
   temp->user = d_strdup(download->user);
   temp->id = true_id(download->id);
   temp->num = download->num;
   temp->kbps = download->kbps;
   temp->freq = download->freq;
   temp->size = download->size;
   temp->total = download->total;
   temp->connection = download->connection;
   temp->st = download->st;
   
   temp->resume_path = (download->resume_path) ? 
     d_strdup(download->resume_path) : NULL;
   
   return temp;
}

void *search_dup(Search *search, int type) {
   if (!search) 
     return NULL;
   
   if (type == N_SEARCH) {
      Search *temp;
      
      temp = d_new(N_SEARCH);
      
      temp->file = d_strdup(search->file);
      temp->trunc_file = trunc_file(temp->file);
      temp->user = d_strdup(search->user);
      temp->id = true_id(search->id);
      temp->num = search->num;
      temp->size = search->size;
      temp->kbps = search->kbps;
      temp->freq = search->freq;
      temp->connection = search->connection;
      temp->ip = search->ip;
      temp->st = search->st;
      
      return temp;
   } else if (type == TRANSFER) {
      Transfer *temp;
      
      temp = d_new(TRANSFER);
      transfer_init(temp);

      temp->file = d_strdup(search->file);
      temp->trunc_file = trunc_file(temp->file);
      temp->user = d_strdup(search->user);
      temp->id = true_id(search->id);
      temp->num = search->num;
      temp->size = search->size;
      temp->kbps = search->kbps;
      temp->freq = search->freq;
      temp->connection = search->connection;
      temp->st = search->st;
      
      temp->resume_path = (search->resume_path) ? 
	d_strdup(search->resume_path) : NULL;
      
      return temp;
   }
   return NULL;
}

void construct_path(GtkWidget *w, Search *orig, Transfer *out) {
   GtkCTreeNode *node, *node_ptr;
   GList *paths, *ptr;
   char *dir, *path;

   /* we use the original to locate the node */
   node = gtk_ctree_find_by_row_data(GTK_CTREE(w), NULL, orig);
   
   paths = NULL;
   path = NULL;
   
   /* now, we need to work backwards to build the directory path */
   for(node_ptr=node; node_ptr; node_ptr=GTK_CTREE_ROW(node_ptr)->parent) {
      dir = GTK_CELL_PIXTEXT(GTK_CTREE_ROW(node_ptr)->row.cell[0])->text;

      paths = g_list_prepend(paths, dir);
   }
   
   if (!paths)
     return;
   
   /* username */
   out->user = d_strdup(paths->data);
   
   /* the first entry *SHOULD* be the username */
   for(ptr=paths->next; ptr; ptr=ptr->next) {
      dir = ptr->data;
      if (!dir)
	continue;
      
      d_strexp(&path, "%s\\%s", path ? path : "", dir);
   }
      
   g_list_free(paths);
   
   if (!path)
     return;
   
   if (out->file)
     d_strexp(&out->file, "%s", path + 1);
   else
     out->file = d_strdup(path);
   
   d_free(path);
   
   out->trunc_file = trunc_file(out->file);
}

Transfer *gnapster_parse_download(STab *stab, char *s, int alt) {
   char *file, *user, *id;
   int port, connection;
   unsigned long int ip = 0;
   Transfer *temp;
   
   user = next_arg(s, &s);
   convert(next_arg(s, &s), "%lu", &ip);
   convert(next_arg(s, &s), "%i", &port);
   file = next_arg(s, &s);
   id = next_arg(s, &s);
   convert(next_arg(s, &s), "%i", &connection);
   
   NA_ERR_HANDLE(NULL);
   
   if (strchr(id, '-'))
     id = next_arg_full(id, &id, '-');
   
   NA_RESET();
   
   if (port == 0 && user_info.conf[FIREWALL] && !alt) {
      Transfer *real;
      
      hook_text_insert(stab, CONSOLE, SYSTEM, "general_message", "unable to receive from %s, %s",
		       user, "user cannot accept connections");
      
      real = real_download(stab, user, file, NULL);
      if (real)
	download_close(real, TRANSFER_ERROR);
      
      return NULL;
   }
   
   temp = d_new(TRANSFER);
   transfer_init(temp);
   
   temp->file = d_strdup(file);
   temp->trunc_file = trunc_file(temp->file);
   temp->user = d_strdup(user);
   temp->id = true_id(id);
   temp->fpath = NULL;
   temp->connection = connection;
   temp->port = port;
   temp->ip = BSWAP32(ip);
   temp->st = stab;
   
   return temp;
}

Search *gnapster_parse_search(STab *st, char *s) {
   char *file, *user, *id;
   int num, size, kbps, freq, connection;
   unsigned long int ip = 0;
   Search *temp;

   file = next_arg(s, &s);
   id = next_arg(s, &s);
   size = my_atoi(next_arg(s, &s));
   kbps = my_atoi(next_arg(s, &s));
   freq = my_atoi(next_arg(s, &s));
   num = my_atoi(next_arg(s, &s));
   user = next_arg(s, &s);
   convert(next_arg(s, &s), "%lu", &ip);
   connection = my_atoi(next_arg(s, &s));
   
   NA_ERR_HANDLE(NULL);

/*   trunc_file = strrchr(file, '\\');
   if (!trunc_file) 
     return NULL;
   
   trunc_file++;*/
   
   temp = d_new(N_SEARCH);

   temp->file = d_strdup(file);
   temp->trunc_file = trunc_file(temp->file);
   temp->user = d_strdup(user);
   temp->id = true_id(id);
   temp->num = num;
   temp->size = size;
   temp->kbps = kbps;
   temp->freq = freq;
   temp->connection = connection;
   temp->ip = ip;
   temp->st = st;
   
   return temp;
}

Search *gnapster_parse_resume(char *s) {
   char *file, *user, *id, *port;
   int size, connection;
   unsigned long int ip = 0;
   Search *temp;

   user = next_arg(s, &s);
   convert(next_arg(s, &s), "%lu", &ip);
   port = next_arg(s, &s);
   file = next_arg(s, &s);
   id = next_arg(s, &s);
   size = my_atoi(next_arg(s, &s));
   connection = my_atoi(next_arg(s, &s));
   
   NA_ERR_HANDLE(NULL);
   
/*   trunc_file = strrchr(file, '\\');
   if (!trunc_file) 
     return NULL;
   
   trunc_file++;*/
   
   temp = d_new(N_SEARCH);

   temp->file = d_strdup(file);
   temp->trunc_file = trunc_file(temp->file);
   temp->user = d_strdup(user);
   temp->id = true_id(id);
   temp->size = size;
   temp->connection = connection;
   temp->ip = ip;
   
   return temp;
}

void check_resume(Transfer *t) {
   FILE *f;
   GList *ptr, *felem;
   FileEntry *fent;
   char *dptr, *file, *checksum;
   size_t size;
   
   if (!t || t->resume_path)
     return;
   
   /* if t->size matches any other size currently downloading, return */
   
   f = open_resume("r");
   if (!f)
     return;
   
   felem = read_file(f);
   
   fclose(f);
   
   for(ptr=felem; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      dptr = fent->lptr;
      
      file = next_arg(dptr, &dptr);
      next_arg(dptr, &dptr);
      checksum = next_arg_full(dptr, &dptr, '-');
      convert(next_arg(dptr, &dptr), "%lu", &size);
      
      NA_ERR_HANDLE_CONTINUE();
      
      if (t->size == size && match_checksum(t->id, checksum))
	t->resume_path = d_strdup(file);
      else
	continue;
      
      break;
   }
   
   if (t->resume_path) {
      if (!file_exists(t->resume_path)) {
	 d_free(t->resume_path);
	 t->resume_path = NULL;
      }
   }
   
   read_file_free(felem);
}

Search *gnapster_parse_browse(STab *stab, char *s, char **next, int opennap) {
   char *user = NULL, *file, *id;
   int size, kbps, freq, num;
   Search *temp;

   if (!opennap)
     user = next_arg(s, &s);
   
   file = next_arg(s, &s);
   id = next_arg(s, &s);
   size = my_atoi(next_arg(s, &s));
   kbps = my_atoi(next_arg(s, &s));
   freq = my_atoi(next_arg(s, &s));
   num = my_atoi(next_arg(s, &s));
   
   NA_ERR_HANDLE(NULL);
   
   if (next)
     *next = s;

/*   if (!opennap) {
      trunc_file = strrchr(file, '\\');
      if (!trunc_file) 
	return NULL;
      
      trunc_file++;
   }*/
   
   temp = d_new(N_SEARCH);
   
   temp->file = d_strdup(file);
   temp->trunc_file = (opennap) ? 
     temp->file : trunc_file(temp->file);
   temp->user = user ? d_strdup(user) : NULL;
   temp->id = true_id(id);
   temp->num = num;
   temp->size = size;
   temp->kbps = kbps;
   temp->freq = freq;
   temp->st = stab;
   
   return temp;
}

int gnapster_update_appbar() {
   float val;
   
   val = gtk_progress_get_value(GTK_PROGRESS(gmain->progress));
   
   val += 0.5;
   
   if (val > 100) 
     val = 0;
   
   gtk_progress_set_value(GTK_PROGRESS(gmain->progress), val);
   
   return 1;
}

void update_status(int x, STab *stab, char *s, int i) {
   ConnInfo *ci;
   
   if (!stab)
     return;
   
   ci = stab->ci;

   if (s || x)
     ci->status = d_free(ci->status);
   
   if (s)
     ci->status = d_strdup(s);
   else if (x) {
      d_msprintf(&ci->status, "Connected (%s:%i):",
		 ci->ip, ci->port);
      
      if (ci->users)
	d_strexp(&ci->status, "%s %lu songs, %hu users, %hu gig%s",
		 ci->status, ci->songs, ci->users, ci->gigs,
		 (ci->gigs == 1) ? "" : "s");
   }
   
   if (i >= 0)
     ci->astatus = i;
   
   if (x && get_current_stab() != stab)
     return;
   
   /* if we are trying to change the current stab, lets change
    * the status now */
#ifdef HAVE_GNOME
   gnome_appbar_pop(GNOME_APPBAR(gmain->appbar));
   gnome_appbar_push(GNOME_APPBAR(gmain->appbar), ci->status ? ci->status : "");
#else
   gtk_statusbar_pop(GTK_STATUSBAR(gmain->appbar), 1);
   gtk_statusbar_push(GTK_STATUSBAR(gmain->appbar), 1, ci->status ? ci->status : "");
#endif /* HAVE_GNOME */
   
   if (ci->astatus < 0)
     return;
   
   TIMEOUT_REMOVE(gmain->appbar_timeout);
   
   if (ci->astatus) {
      gmain->appbar_timeout = 
	TIMEOUT_ADD(50, gnapster_update_appbar, NULL);
   }

   gtk_progress_set_activity_mode(GTK_PROGRESS(gmain->progress), ci->astatus);
   gtk_progress_set_value(GTK_PROGRESS(gmain->progress), 0);
}

void change_title(STab *stab) {
   char *t;
   
   if (!stab)
     return;

   if (stab->ci->server)
     d_msprintf(&t, "Gnapster [ %s ]", stab->ci->server);
   else
     d_msprintf(&t, "Gnapster");
   
   gtk_window_set_title(GTK_WINDOW(gmain->window), t);
   
   d_free(t);
}

void change_status(STab *stab) {
   if (stab)
     update_status(0, stab, NULL, -1);
}

void highlight(GtkWidget *notebook, GtkWidget *vbox) {
   GtkWidget *ptr, *vptr, *label, *notebook_page;
   int curr_page;
   
   vptr = NULL;
   
   label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook),
				      vbox);
   
   curr_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
   notebook_page = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), 
					     curr_page);
   
   if (notebook_page != GTK_WIDGET(vbox))
     gtk_widget_set_style(label, gmain->highlight);
   
   for(ptr=notebook->parent; ptr; ptr=ptr->parent) {
      if (GTK_IS_NOTEBOOK(ptr)) {
	 highlight(ptr, vptr);
	 break;
      }
      
      if (GTK_IS_VBOX(ptr))
	vptr = ptr;
      
      if (!GTK_IS_NOTEBOOK(ptr) && GTK_IS_VBOX(ptr) && GTK_IS_HBOX(ptr))
	break;
   }
}

int user_lookup(STab *stab, char *user) {
   GList *ptr;
   int i = 0;
   
   for(ptr=stab->ct->channel_list; ptr; ptr=ptr->next, i++) {
      if (!g_strcasecmp(((ChannelInfo *)ptr->data)->channel_name, user) &&
	  ((ChannelInfo *)ptr->data)->user)
	return (i + 2);
   }
   
   return -1;
}

int gnapster_get_pagenum(STab *stab) {
   int page_num;
   
   page_num = 
     gtk_notebook_get_current_page(GTK_NOTEBOOK(stab->ct->notebook));
   
   if (page_num < 0)
     return -1;
   
   return (page_num) ? (page_num + 1) : 0;
}

int gnapster_clist_append(GtkWidget *clist, GtkWidget *vbox, void *data, ...) {
   va_list args;
   GList *ap = NULL;
   char *s;
   int ll, row = -1;

   va_start(args, data);
   while(1) {
      s = va_arg(args, char *);
      if (!s)
	break;
      ap = g_list_append(ap, s);
   }
   va_end(args);
   
   ll = g_list_length(ap);
   
     {
	char *add_id[ll];
	GList *ptr;
	int x = 0;
	
	ptr = ap;
	while(ptr) {
	   add_id[x++] = (ptr->data) ? ptr->data : "???";
	   ptr = ptr->next;
	}

	row = gtk_clist_append(GTK_CLIST(clist), add_id);
   
	if (data)
	  gtk_clist_set_row_data_full(GTK_CLIST(clist), row, data,
				      (GtkDestroyNotify)gnapster_destroy);
     }

   if (vbox)
     highlight(vbox->parent, vbox);
   
   return row;
}

void *gnapster_get_selection_data(GtkWidget *clist, int *row) {
   void *temp;
   GList *selection;
   
   selection = GTK_CLIST(clist)->selection;
   if (!selection) return NULL;

   temp = gtk_clist_get_row_data(GTK_CLIST(clist), INT(selection->data));
   if (row)
     *row = INT(selection->data);
   
   return temp;
}

void *gnapster_get_ctree_selection_data(GtkWidget *ctree, int *row) {
   void *temp;
   GList *selection;
   
   selection = GTK_CLIST(ctree)->selection;
   if (!selection)
     return NULL;
   
   temp = gtk_ctree_node_get_row_data(GTK_CTREE(ctree), selection->data);
   
   return temp;
}

void gnapster_get_styles(GtkWidget *widget) {
   gmain->normal = gtk_widget_get_style(widget);
   
   gmain->highlight = gtk_style_new();
   gmain->highlight->font = gmain->normal->font;
   
   if (same_color(gmain->normal->fg[GTK_STATE_NORMAL],
		  gmain->normal->fg[GTK_STATE_SELECTED])) {
      gmain->highlight->fg[GTK_STATE_NORMAL].pixel = 
	gmain->normal->fg[GTK_STATE_SELECTED].pixel;
      
      gmain->highlight->fg[GTK_STATE_NORMAL].red = 65535;
      gmain->highlight->fg[GTK_STATE_NORMAL].green = 0;
      gmain->highlight->fg[GTK_STATE_NORMAL].blue = 0;
   } else
     gmain->highlight->fg[GTK_STATE_NORMAL] = gmain->normal->fg[GTK_STATE_SELECTED];
}

void gnapster_config_set_int(int defaults) {
   char *ts;
   int i, val;

   for(i=0; conf_table[i].conf_str; i++) {
      d_msprintf(&ts, "/gnapster/Options/%s", conf_table[i].conf_str);
      
      val = (defaults) ? conf_table[i].conf_val : user_info.conf[i];
      j_config_set_int(ts, val); 
   }
   
   j_config_sync();
}

void gnapster_config_get_int() {
   char *ts;
   int i;
   
   if (!j_config_has_section("/gnapster/Options")) 
     return;
   
   for(i=0; conf_table[i].conf_str; i++) {
      d_msprintf(&ts, "/gnapster/Options/%s=%i",
		 conf_table[i].conf_str, conf_table[i].conf_val);
      
      user_info.conf[i] = j_config_get_int(ts);
      
      d_free(ts);
   }
}

int verify_shared() {
   FILE *f;
   char *ptr, *str, buf[256];
   int args = 0;

   f = open_shared("r");
   /* Return 1 to just ignore it */
   if (!f) 
     return 1;
   
   /* Read the first line only */
   fgets(buf, sizeof(buf) - 1, f);
   fclose(f);
   str = buf;
   do {
      ptr = next_arg(str, &str);
      NA_RESET();
      
      args++;
   } while(ptr);
   
   if (args <= 7) 
     return 0;
   
   return 1;
}

void load_signore() {
   char *signore;
   char *ptr = NULL, *freeptr = NULL;
   int i = 0;
   
   signore = d_config_get_string("/gnapster/Options/signore");
   if (!signore)
     return;
   
   while((ptr = next_arg_full(signore, &signore, '\4'))) {
      if (i == 0)
	freeptr = ptr;
      
      signore_list = g_list_append(signore_list, d_strdup(ptr));
      
      i++;
   }
   
   NA_RESET();
   
   d_free(freeptr);
}

int idle_startup_cb(gpointer data) {
   if (!strcmp(data, "connect")) {
      /* connect and focus the first stab */
      connect_cb(ttbl->data);
      gtk_notebook_set_page(GTK_NOTEBOOK(gmain->srv_notebook), 0);
   } else if (!strcmp(data, "props")) {
      preferences_cb();
   }
   
   return 0;
}

void load_pixmaps() {
   GtkStyle *style;
   
   if (speedgreen != NULL)
     return;
   
   style = gtk_widget_get_style(gmain->window);
   speedgreen = gdk_pixmap_create_from_xpm_d(gmain->window->window,
					     &speedgreenb, &style->bg[GTK_STATE_NORMAL],
					     speedgreen_xpm);
   speedyellow = gdk_pixmap_create_from_xpm_d(gmain->window->window,
					     &speedyellowb, &style->bg[GTK_STATE_NORMAL],
					     speedyellow_xpm);
   speedred = gdk_pixmap_create_from_xpm_d(gmain->window->window,
					     &speedredb, &style->bg[GTK_STATE_NORMAL],
					     speedred_xpm);
   speedgray = gdk_pixmap_create_from_xpm_d(gmain->window->window,
					     &speedgrayb, &style->bg[GTK_STATE_NORMAL],
					     speedgray_xpm);   
}

void set_window_geometry() {
   int w, h;
   
   w = j_config_get_int("/gnapster/Geometry/w=690");
   h = j_config_get_int("/gnapster/Geometry/h=355");
   
   gtk_widget_set_usize(GTK_WIDGET(gmain->window), w, h);
}

void gnapster_142to150() {
   FILE *f;
   char *u, *p, *exec_dl;
   
   f = open_local_path("accounts", NULL, "r");
   if (!f) {
      u = d_config_get_string("/gnapster/User/user");
      p = d_config_get_string("/gnapster/User/pass");
   
      if (u && p) {
	 f = open_local_path("accounts", NULL, "w");
	 fchmod(fileno(f), 0600);
	 fprintf(f, "%s:%s 1\n", u, p);
      }
      
      d_free(u);
      d_free(p);
   }
   
   if (f)
     fclose(f);
   
   f = open_stabs("r");
   if (!f) {
      char *old_ip;
      unsigned short old_port;
      
      u = d_config_get_string("/gnapster/User/user");
      old_ip = d_config_get_string("/gnapster/Options/server");
      old_port = j_config_get_int("/gnapster/Options/server_port");
      
      if (u && old_ip && old_port > 0) {
	 f = open_stabs("w");
	 fprintf(f, "%s:%hu \"%s\" %s\n", old_ip, old_port, old_ip, u);
      }
      
      d_free(old_ip);
      d_free(u);
   }
   
   if (f)
     fclose(f);
   
   exec_dl = d_config_get_string("/gnapster/User/exec_dl");
   if (!exec_dl)
     j_config_set_string("/gnapster/User/exec_dl", "xmms -e");
   
   d_free(exec_dl);
}

int main(int argc, char **argv) {
   int ret;
   
#ifdef HAVE_GNOME
   gnome_init(PACKAGE, VERSION, argc, argv);
#else
   gtk_init(&argc, &argv);
#endif /* HAVE_GNOME */

   /* set client start time */
   gmain = d_new(GMAIN);
   gmain->clistart = time(NULL);
   
   /* register file extensions for share types */
   register_formats();
   
   /* log all these hooks */
   log_add_hooks(NULL, "general_error", "join_ack", "server_msg",
		 "global", "global_self", "privmsg", "privmsg_out",
		 "cancel_download", "timeout_download", "download_error",
		 "download_resume", "download_complete", "accept_upload", 
		 "reject_upload", "cancel_upload", "timeout_upload", 
		 "upload_complete", 
		 NULL);

   /* int-based user info data */
   memset(&user_info, 0, sizeof(UserInfo));
   user_info.conf[TERMINATION] = -1;

   /* handle int based config stuff */
   create_priv_dir();
   gnapster_config_get_int();
   
   /* deprecated -- does nothing */
   unlink_log();
   
   /* hackish way to force this exists in the config */
   user_info.default_font = d_config_get_string("/gnapster/xtext/font");
   if (!user_info.default_font) {
      user_info.default_font =
	d_strdup("-b&h-lucidatypewriter-medium-r-normal-*-*-120-*-*-m-*-*-*");
      j_config_set_string("/gnapster/xtext/font", user_info.default_font);
   }

   xtext_font = load_font();
   
   /* create the main dialog */
   create_gnapster_main();
   generate_colors(gmain->window);
   set_window_geometry();
   
   gmain->st = create_search_tab(gmain->srv_notebook);
   gmain->bt = create_browse_tab(gmain->srv_notebook);
   gmain->dt = create_download_tab(gmain->srv_notebook);
   gmain->ut = create_upload_tab(gmain->srv_notebook);

   /* create the initial tab */
   append_server_tab();
   gtk_widget_show_all(gmain->window);

   gnapster_get_styles(gmain->window);

   load_pixmaps();

   building = 0;
   
   load_signore();
   
   /* handle the default theme */
   load_theme();
   
   /* force old 1.4.x based data to be converted to 1.5.x ... 
    * allows backwards compatability */
   gnapster_142to150();
   
   /* initialize console commands */
   command_setup();
   
   ret = get_user_info();

   /* preferences stuff */
   if (ret) {
      update_status(1, ttbl->data, _("Auto-connecting..."), -1);
      
      /* server loading */
      load_stabs();
      
      gnapster_config_get_int();
   }

   /* don't allow port to be set if firewall is set */
   gmain->port = (user_info.conf[FIREWALL]) ?
     0 : j_config_get_int("/gnapster/Options/port=6699");
   
   gmain->bind_input = -1;
   
   signal(SIGPIPE, SIG_IGN);
   
   /* if the username exists in the config, auto-connect, else,
    * open up the preferences */
   gtk_idle_add(idle_startup_cb, ret ? "connect" : "props");
   
   setup_sigchld();

   gtk_main();
   
   d_summary();
   
   return 0;
}
