/******************************************************************************\
 gnofin/record.c   $Revision: 1.14.2.1 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "common.h"
#include <math.h>
#include "record.h"
#include "data-if.h"


/******************************************************************************
 * Record info
 */

guint
record_info_copy (RecordInfo *dest, const RecordInfo *src, guint mask)
{
  trace ("");
  g_return_val_if_fail (src, 0);
  g_return_val_if_fail (dest, 0);
  g_return_val_if_fail (!(mask & ~RECORD_ALL_WRITABLE_FIELDS), 0);

  if (mask == 0)
    mask = RECORD_ALL_WRITABLE_FIELDS;

  if (mask & RECORD_FIELD_TYPE)
    dest->type = src->type;
  
  if (mask & RECORD_FIELD_DATE)
    dest->date = src->date;
  
  if (mask & RECORD_FIELD_NUMBER)
    dest->number = src->number;

  if (mask & RECORD_FIELD_LINKED_ACC_NAME)
    dest->linked_acc_name = g_strdup (src->linked_acc_name);
  
  if (mask & RECORD_FIELD_CATEGORY)
    dest->category = g_strdup (src->category);

  if (mask & RECORD_FIELD_PAYEE)
    dest->payee = g_strdup (src->payee);

  if (mask & RECORD_FIELD_MEMO)
    dest->memo = g_strdup (src->memo);

  if (mask & RECORD_FIELD_CLEARED)
    dest->cleared = src->cleared;

  if (mask & RECORD_FIELD_AMOUNT)
    dest->amount = src->amount;

  if (mask & RECORD_FIELD_EXCHANGE_RATE)
    dest->exchange_rate = src->exchange_rate;

  return mask;
}

void
record_info_clear (RecordInfo *info, guint mask)
{
  trace ("");
  g_return_if_fail (info);

  if (mask == 0)
    mask = RECORD_ALL_WRITABLE_FIELDS;
  
  if (mask & RECORD_FIELD_CATEGORY)
    g_free (info->category);

  if (mask & RECORD_FIELD_PAYEE)
    g_free (info->payee);

  if (mask & RECORD_FIELD_MEMO)
    g_free (info->memo);

  if (mask & RECORD_FIELD_LINKED_ACC_NAME)
    g_free (info->linked_acc_name);
}

gboolean
record_info_diff (const RecordInfo *a, const RecordInfo *b, guint mask)
{
  trace ("");
  g_return_val_if_fail (a, TRUE);
  g_return_val_if_fail (b, TRUE);
  g_return_val_if_fail (mask, TRUE);

  if (mask & RECORD_FIELD_TYPE)
    if (a->type != b->type) return TRUE;
  
  if (mask & RECORD_FIELD_DATE)
    if (g_date_compare ((GDate *) &a->date, (GDate *) &b->date)) return TRUE;
  
  if (mask & RECORD_FIELD_NUMBER)
    if (a->number != b->number) return TRUE;

  if (mask & RECORD_FIELD_LINKED_ACC)
    if (a->linked_acc != b->linked_acc) return TRUE;
  
  if (mask & RECORD_FIELD_LINKED_REC)
    if (a->linked_rec != b->linked_rec) return TRUE;

  if (mask & RECORD_FIELD_LINKED_ACC_NAME)
  {
    if (a->linked_acc_name)
    {
      if (b->linked_acc_name)
      {
	if (strcmp (a->linked_acc_name, b->linked_acc_name)) return TRUE;
      }
      else
        return TRUE;
    }
    else if (b->linked_acc_name)
      return TRUE;
  }

  if (mask & RECORD_FIELD_LINK_BROKEN)
    if (a->link_broken != b->link_broken) return TRUE;
  
  if (mask & RECORD_FIELD_CATEGORY)
    if (strcmp (a->category, b->category)) return TRUE;
  
  if (mask & RECORD_FIELD_PAYEE)
    if (strcmp (a->payee, b->payee)) return TRUE;
  
  if (mask & RECORD_FIELD_MEMO)
    if (strcmp (a->memo, b->memo)) return TRUE;
  
  if (mask & RECORD_FIELD_CLEARED)
    if (a->cleared != b->cleared) return TRUE;
  
  if (mask & RECORD_FIELD_AMOUNT)
    if (a->amount != b->amount) return TRUE;
 
  if (mask & RECORD_FIELD_CLEARED_BAL)
    if (a->cleared_bal != b->cleared_bal) return TRUE;
 
  if (mask & RECORD_FIELD_OVERALL_BAL)
    if (a->overall_bal != b->overall_bal) return TRUE;

  if (mask & RECORD_FIELD_EXCHANGE_RATE)
    if (a->exchange_rate != b->exchange_rate) return TRUE;
  
  return FALSE;
}


/******************************************************************************
 * Record
 */

Record *
record_new ()
{
  Record *record;

  trace ("");

  record = g_new0 (Record, 1);
  record->exchange_rate = 1.0;
  record_ref (record);

  return record;
}

Record *
record_copy (const Record *record)
{
  Record *copy;

  trace ("");
  g_return_val_if_fail (record, NULL);

  copy = record_new ();
  memcpy (copy, record, sizeof (Record));
  copy->ref_count = 1;
  copy->parent = NULL;
  copy->link.rec = NULL;
  copy->link.acc = NULL;
  copy->link.acc_name = NULL;
  copy->overall_bal = 0;
  copy->cleared_bal = 0;

  if (copy->type)
    record_type_ref (copy->type);
  if (copy->category)
    cached_string_ref (copy->category);
  if (copy->payee)
    cached_string_ref (copy->payee);
  if (copy->memo)
    cached_string_ref (copy->memo);

  return copy;
}

void
record_ref (Record *record)
{
  trace ("%p", record);
  g_return_if_fail (record);

  record->ref_count++;
}

void
record_unref (Record *record)
{
  trace ("%p", record);

  if (record && (--record->ref_count == 0))
    record_destroy (record);
}

void
record_destroy (Record *record)
{
  trace ("%p", record);
  g_return_if_fail (record);

  record->ref_count = 0;

  if (record->parent)
    record_detach (record, TRUE);

  record_type_unref (record->type);
  cached_string_unref (record->category);
  cached_string_unref (record->payee);
  cached_string_unref (record->memo);

  if (record->link.acc_name)
    cached_string_unref (record->link.acc_name);

  g_free (record);
}

gboolean
record_attach (Record *record, Account *parent)
{
  GCompareFunc func;

  trace ("%p, %p", record, parent);
  g_return_val_if_fail (record, FALSE);
  g_return_val_if_fail (parent, FALSE);

  func = account_get_record_sort_fcn (parent);
  record->parent = parent;  /* this MUST preceed the insertion b/c
  			     * the compare function depends on being
			     * able to get the parent from the record. */
  parent->records = g_list_insert_sorted (parent->records, record, func);

  /* The parent account now references this record */
  record_ref (record);

  if (parent->parent)
    return record_attach_children (record);
  else 
    return TRUE;
}

gboolean
record_attach_children (Record *record)
{
  Bankbook *book;

  trace ("%p", record);
  g_return_val_if_fail (record, FALSE);
  g_return_val_if_fail (record->parent, FALSE);
  g_return_val_if_fail (record->parent->parent, FALSE);
  g_return_val_if_fail (record->type, FALSE);
  g_return_val_if_fail (record->category, FALSE);
  g_return_val_if_fail (record->payee, FALSE);
  g_return_val_if_fail (record->memo, FALSE);

  book = record->parent->parent;

  /* Attach cached strings to the bankbook object */
  record->category = cached_string_attach (record->category, &book->category_cache);
  record->payee = cached_string_attach (record->payee, &book->payee_cache);
  record->memo = cached_string_attach (record->memo, &book->memo_cache);

  /* Attach the type to the data object */
  record->type = record_type_attach (record->type, book);
  record->type->usage_count++;

  /* If this record has a type that is linked, then we must process
   * the link (ie. perhaps generating a new record in the linked account). */
  if (record->type->linked)
  {
    record->link.acc_name =
      cached_string_attach (record->link.acc_name, &book->link_cache);

    if (!record->link_broken)
      record_realize_link (record);
  }

  return TRUE;
}

void
record_detach (Record *record, gboolean full)
{
  trace ("%p %s", record, full ? "full" : "");
  g_return_if_fail (record);
  g_return_if_fail (record->parent);

  /* Finally, we remove this record from its parent's list of records */
  record->parent->records = g_list_remove (record->parent->records, record);
  record->parent = NULL;

  /* The parent account no longer references this record */
  record_unref (record);

  if (full)
    record_detach_children (record);
}

void
record_detach_children (Record *record)
{
  trace ("%p", record);
  g_return_if_fail (record);

  record->type->usage_count--;

  cached_string_detach (record->category);
  cached_string_detach (record->payee);
  cached_string_detach (record->memo);

  if (record->type->linked)
  {
    cached_string_detach (record->link.acc_name);

    if (!record->link_broken)
    {
      Record *linked_rec = record->link.rec;

      g_assert (linked_rec);

      if (linked_rec->parent)
	record_detach (linked_rec, TRUE);
    }
  }
}

void
record_realize_link (Record *record)
{
  Bankbook *book;
  Account *linked_acc, *linked_acc_old;
  Record *linked_rec;

  g_return_if_fail (record);
  g_return_if_fail (record->type);
  g_return_if_fail (record->type->linked);
  g_return_if_fail (record->link_broken == 0);
  g_return_if_fail (record->parent);

  book = record->parent->parent;

  /* The linked account is given by link.acc_name */
  
  linked_acc_old = record->link.acc;
  linked_rec = record->link.rec;

  /* If the named account exists, then we can go ahead and generate the 
   * corresponding record in the peer account.  Otherwise, we have to
   * mark the link as broken */
  linked_acc = get_account_by_name (book->accounts, record->link.acc_name->string);
  if (linked_acc && (linked_acc != record->parent))
  {
    //g_assert (linked_acc != record->parent);

    record->link.acc = linked_acc;

    if (linked_rec == NULL)
    {
      linked_rec = record_copy (record);
      linked_rec->link.acc = record->parent;
      linked_rec->link.rec = record;
      linked_rec->link.acc_name = cached_string_new (record->parent->name);
      record_ref (record);

      record->link.rec = linked_rec;
    }
    else
    {
      linked_rec->date = record->date;
      linked_rec->number = record->number;

      /* cleared field is skipped since linked transactions need to be
       * individually cleared in each respective account. 
       *
       * FIXME: we can't make the record fields differ at this time
       *        because the XML module is set up to ignore the first
       *        one it reads!! (it later creates the link by inserting
       *        the second one... this way the linked account is sure
       *        to exist.) unfortunately, we have to play nicely with
       *        it for now. */

      linked_rec->cleared = record->cleared;

      cached_string_unref (linked_rec->category);
      cached_string_unref (linked_rec->payee);
      cached_string_unref (linked_rec->memo);

      linked_rec->category = cached_string_copy (record->category);
      linked_rec->payee = cached_string_copy (record->payee);
      linked_rec->memo = cached_string_copy (record->memo);
    }

    /* Refresh the linked record fields since they may have changed */
    linked_rec->exchange_rate = 1.0 / record->exchange_rate;
    linked_rec->amount = -1 * rint ((gfloat) record->amount * record->exchange_rate);

    if (linked_rec->parent == NULL)
      record_attach (linked_rec, linked_acc);
  }
  else
  {
    if (linked_rec)
    {
      g_assert (linked_acc_old);

      linked_rec->link_broken = TRUE;
      
      /* Give him back to his parent */
      if (linked_rec->parent == NULL)
	record_attach (linked_rec, linked_acc_old);
    }
    record->link_broken = TRUE;
  }
}

void
record_break_link (Record *record)
{
  trace ("");
  g_return_if_fail (record);

  if (record->type->linked)
  {
    if (record->link.rec)
    {
      Record *peer = record->link.rec;

      /* We continue to cross-reference so we can unbreak the link. */

      peer->link_broken = TRUE;
      record->link_broken = TRUE;
    }
  }
}

void
record_unbreak_link (Record *record)
{
  Record *peer;

  trace ("");
  g_return_if_fail (record);
  g_return_if_fail (record->type);
  g_return_if_fail (record->type->linked);
  g_return_if_fail (record->link_broken);
  g_return_if_fail (record->link.rec);
  g_return_if_fail (record->link.acc);
  g_return_if_fail (record->parent);

  peer = record->link.rec;

  g_return_if_fail (peer->parent);
  g_return_if_fail (peer->link.rec == record);
  g_return_if_fail (peer->link.acc == record->parent);
  g_return_if_fail (record->link.acc == peer->parent);

  record->link_broken = FALSE;
  peer->link_broken = FALSE;
}

gboolean
record_set_info (Record *record, guint mask, const RecordInfo *info)
{
  /* Very minimal error checking is performed within this routine */

  trace ("");
  g_return_val_if_fail (record, FALSE);
  g_return_val_if_fail (record->parent == NULL, FALSE);
  g_return_val_if_fail (info, FALSE);
  g_return_val_if_fail (!(mask & ~RECORD_ALL_WRITABLE_FIELDS), FALSE);

  if (mask == 0)
    mask = RECORD_ALL_WRITABLE_FIELDS;

  if (mask & RECORD_FIELD_TYPE)
  {
    if (info->type != record->type)
    {
      record_type_unref (record->type);
      record->type = info->type;
      record_type_ref (record->type);
    }
  }

  if (mask & RECORD_FIELD_DATE)
  {
    g_assert (g_date_valid ((GDate *) &info->date));
    record->date = info->date;
  }

  if (mask & RECORD_FIELD_NUMBER)
    record->number = info->number;

  if (mask & RECORD_FIELD_CATEGORY)
  {
    if (record->category)
    {
      if (record->category->parent)
      {
        cached_string_unref (record->category);
        record->category = cached_string_new (info->category);
      }
      else
        cached_string_set_string (record->category, info->category);
    }
    else
      record->category = cached_string_new (info->category);
  }

  if (mask & RECORD_FIELD_PAYEE)
  {
    if (record->payee)
    {
      if (record->payee->parent)
      {
	cached_string_unref (record->payee);
	record->payee = cached_string_new (info->payee);
      }
      else
        cached_string_set_string (record->payee, info->payee);
    }
    else
      record->payee = cached_string_new (info->payee);
  }

  if (mask & RECORD_FIELD_MEMO)
  {
    if (record->memo)
    {
      if (record->memo->parent)
      {
	cached_string_unref (record->memo);
	record->memo = cached_string_new (info->memo);
      }
      else
        cached_string_set_string (record->memo, info->memo);
    }
    else
      record->memo = cached_string_new (info->memo);
  }

  if (mask & RECORD_FIELD_CLEARED)
    record->cleared = info->cleared;

  if (mask & RECORD_FIELD_AMOUNT)
    record->amount = info->amount;

  if (mask & RECORD_FIELD_EXCHANGE_RATE)
  {
    if (info->exchange_rate <= 0.0)  // not a valid exchange rate!!
    {
      trace ("invalid exchange rate specified... assuming an exchange rate of 1.0\n");
      record->exchange_rate = 1.0;
    }
    else
      record->exchange_rate = info->exchange_rate;
  }

  if (mask & RECORD_FIELD_LINKED_ACC_NAME && (info->linked_acc_name != NULL))
  {
    if (record->link.acc_name)
    {
      if (record->link.acc_name->parent)
      {
	cached_string_unref (record->link.acc_name);
	record->link.acc_name = cached_string_new (info->linked_acc_name);
      }
      else
        cached_string_set_string (record->link.acc_name, info->linked_acc_name);
    }
    else
      record->link.acc_name = cached_string_new (info->linked_acc_name);
  }

  return TRUE;
}

void
record_get_info (const Record *record, guint mask, RecordInfo *info)
{
  trace ("");
  g_return_if_fail (record);
  g_return_if_fail (info);

  if (mask == 0)
    mask = (guint) -1;

  if (mask & RECORD_FIELD_TYPE)
    info->type = record->type;
  
  if (mask & RECORD_FIELD_DATE)
    info->date = record->date;
  
  if (mask & RECORD_FIELD_NUMBER)
    info->number = record_number (record);

  if (mask & RECORD_FIELD_CATEGORY)
    info->category = record->category->string;
  
  if (mask & RECORD_FIELD_PAYEE)
    info->payee = record->payee->string;
  
  if (mask & RECORD_FIELD_MEMO)
    info->memo = record->memo->string;

  if (mask & RECORD_FIELD_LINKED_REC)
    info->linked_rec = record_linked_rec (record);

  if (mask & RECORD_FIELD_LINKED_ACC)
    info->linked_acc = record_linked_acc (record);

  if (mask & RECORD_FIELD_LINKED_ACC_NAME)
    info->linked_acc_name = record_linked_acc_name (record);

  if (mask & RECORD_FIELD_LINK_BROKEN)
    info->link_broken = record->link_broken;

  if (mask & RECORD_FIELD_CLEARED)
    info->cleared = record->cleared ? TRUE : FALSE;
  
  if (mask & RECORD_FIELD_AMOUNT)
    info->amount = record_amount (record);

  if (mask & RECORD_FIELD_OVERALL_BAL)
    info->overall_bal = record->overall_bal;

  if (mask & RECORD_FIELD_CLEARED_BAL)
    info->cleared_bal = record->cleared_bal;

  if (mask & RECORD_FIELD_EXCHANGE_RATE)
    info->exchange_rate = record->exchange_rate;
}

guint
record_number (const Record *record)
{
  trace ("");
  g_return_val_if_fail (record, 0);

  if (record->type && record->type->numbered)
    return record->number;
  else
    return 0;
}

money_t
record_amount (const Record *record)
{
  trace ("");
  g_return_val_if_fail (record, 0);

  switch (record->type->sign)
  {
  case RECORD_TYPE_SIGN_POS:
    return money_abs (record->amount);
  case RECORD_TYPE_SIGN_NEG:
    return -1 * money_abs (record->amount);
  }
  return record->amount;
}

Account *
record_linked_acc (const Record *record)
{
  trace ("");
  g_return_val_if_fail (record, NULL);

  return (record->type->linked && !record->link_broken) ? record->link.acc : NULL;
}

gchar *
record_linked_acc_name (const Record *record)
{
  trace ("");
  g_return_val_if_fail (record, NULL);

  return (record->type->linked && record->link.acc_name) ? record->link.acc_name->string : NULL;
}

Record *
record_linked_rec (const Record *record)
{
  trace ("");
  g_return_val_if_fail (record, NULL);

  return (record->type->linked && !record->link_broken) ? record->link.rec : NULL;
}

gint
record_index (const Record *record)
{
  g_return_val_if_fail (record, 0);
  g_return_val_if_fail (record->parent, 0);

  return g_list_index (record->parent->records, (gpointer) record);
}

void
record_dump (const Record *record)
{
#ifdef MONEY_IS_64BIT
  g_print ("    R - %p[rc=%u] {ty=%s,d=%d/%d/%d,n=%u,lb=%u,ln=%03d,la=%p,lr=%p,l=%03d,p=%03d,m=%03d,a=%qd,ob=%qd,cb=%qd,c=%u,x=%f}\n",
           record,
	   record->ref_count,
	   record->type ? record->type->name : NULL,
	   record->date.day,
	   record->date.month,
	   record->date.year,
	   record->number,
	   record->link_broken,
	   record->link.acc_name ? cached_string_index (record->link.acc_name) : -1,
	   record->link.acc,
	   record->link.rec,
	   record->category ? cached_string_index (record->category) : -1,
	   record->payee ? cached_string_index (record->payee) : -1,
	   record->memo ? cached_string_index (record->memo) : -1,
	   record->amount,
	   record->overall_bal,
	   record->cleared_bal,
	   record->cleared,
	   record->exchange_rate);
#else
  g_print ("    R - %p[rc=%u] {ty=%s,d=%d/%d/%d,n=%u,lb=%u,ln=%03d,la=%p,lr=%p,l=%03d,p=%03d,m=%03d,a=%ld,ob=%ld,cb=%ld,c=%u,x=%f}\n",
           record,
	   record->ref_count,
	   record->type ? record->type->name : NULL,
	   record->date.day,
	   record->date.month,
	   record->date.year,
	   record->number,
	   record->link_broken,
	   record->link.acc_name ? cached_string_index (record->link.acc_name) : -1,
	   record->link.acc,
	   record->link.rec,
	   record->category ? cached_string_index (record->category) : -1,
	   record->payee ? cached_string_index (record->payee) : -1,
	   record->memo ? cached_string_index (record->memo) : -1,
	   record->amount,
	   record->overall_bal,
	   record->cleared_bal,
	   record->cleared,
	   record->exchange_rate);
#endif
}

// vim: ts=8 sw=2
