/******************************************************************************\
 gnofin/date.c   $Revision: 1.3.2.1 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "common.h"
#include <libgnome/gnome-config.h>
#include <time.h>
#include "date.h"
#include "config-saver.h"
#include "notification-list.h"


/******************************************************************************
 * Format change notification
 */

NotificationList date_format_change_listeners = {0};


/******************************************************************************
 * Configuration
 */

typedef struct {
  gchar *date_format;
} Config;

#define CAT  "Date"
#define KEY  "/" PACKAGE "/" CAT "/"

static void
load_config (Config *config)
{
  trace ("");
  config->date_format = gnome_config_get_string (KEY "format=%x");
}

static void
save_config (const Config *config)
{
  trace ("");
  gnome_config_set_string (KEY "format", config->date_format);
}

static Config *
get_config (void)
{
  static Config config = {0};
  static gboolean init = FALSE;

  if (!init)
  {
    load_config (&config);
    config_saver_register (CAT, (ConfigSaveFunc) save_config, &config);
    init = TRUE;
  }
  return &config;
}


/******************************************************************************
 * Interface
 */

void
date_now (GDate *date)
{
  time_t t;
  struct tm *tm;

  trace ("");
  g_return_if_fail (date);

  g_date_clear (date, 1);

  t = time (NULL);
  tm = localtime (&t);

  /* Set the date:
   *   Remember that GDate expects months to start with 1
   *   and years to start with 0 (as opposed to 1900).
   */
  g_date_set_dmy (date, tm->tm_mday, tm->tm_mon + 1, tm->tm_year + 1900);
}

gchar *
date_stringize_with_format (gchar *buf, guint maxlen, const gchar *format, const GDate *date)
{
  trace ("");
  g_return_val_if_fail (date, NULL);
  g_return_val_if_fail (format, NULL);

  if (buf == NULL)
  {
    buf = g_new (gchar, 64);
    maxlen = 64;
  }
  g_date_strftime (buf, maxlen, format, (GDate *) date);
  return buf;
}

gchar *
date_stringize (gchar *buf, guint maxlen, const GDate *date)
{
  gchar *format;

  trace ("");

  format = get_config ()->date_format;
  return date_stringize_with_format (buf, maxlen, format, date);
}

gboolean
date_parse (const gchar *buf, GDate *date)
{
  trace ("");
  g_return_val_if_fail (buf, FALSE);
  g_return_val_if_fail (date, FALSE);

  g_date_clear (date, 1);
  g_date_set_parse (date, buf);
  return g_date_valid (date);
}

void
date_set_format (const gchar *format)
{
  Config *config = get_config ();

  trace ("");

  g_free (config->date_format);
  config->date_format = g_strdup (format);

  /* notify those interested in changes to the date format. */
  notification_list_notify (&date_format_change_listeners);
}

const gchar *
date_get_format (void)
{
  return get_config ()->date_format;
}
