/******************************************************************************\
 gnofin/merging.h   $Revision: 1.3 $
 Copyright (C) 2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#ifndef MERGING_H
#define MERGING_H

#include "data-types.h"

GNOFIN_API_BEGIN

/* The merging functions can be used to combine elements of two
 * distinct bankbooks.  Since an element of a bankbook (ie. Accounts
 * and Records) are always associated with a bankbook, merging
 * involves not only copying of the element's data, but also it
 * involves ensuring that everything referenced by the element is
 * also an element of the bankbook (ie. Record types, and Categories).
 *
 * These functions may have the side effect of generating record types
 * in the destination bankbook.  The force flag controls how conflicts
 * are resolved.  If true, record types that conflict by name are 
 * resolved by giving the new record type a different (but related)
 * name.  For example, if the properties of a type named ATM differ,
 * the second ATM will be called ATM1, and so on.
 *
 * These functions return false/null if the operation fails, or in the
 * case that force equals false, these functions return false/null to
 * indicate a conflict.  In the case that multiple objects are being
 * merged, the functions will attempt to keep going instead of stopping
 * at the first conflict.
 *
 * The functions that do not return a boolean, return a pointer to the
 * newly created object.
 */

Record  *merge_record_into_account    (Account *dest, const Record *src, gboolean force);
gboolean merge_account_into_account   (Account *dest, const Account *src, gboolean force);
Account *merge_account_into_bankbook  (Bankbook *dest, const Account *src, gboolean force);
gboolean merge_bankbook_into_bankbook (Bankbook *dest, const Bankbook *src, gboolean force);

RecordType *merge_record_type_into_bankbook (Bankbook *dest, const RecordType *src, gboolean force);

GNOFIN_API_END

#endif
